/* filelist.c - for drawing a scrollable filelist with size, permissions etc.
   Copyright (C) 1997 Paul Sheer

   This program is free software; you can redistribute it and/or modify
   it under the terms of the GNU General Public License as published by
   the Free Software Foundation; either version 2 of the License, or
   (at your option) any later version.

   This program is distributed in the hope that it will be useful,
   but WITHOUT ANY WARRANTY; without even the implied warranty of
   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
   GNU General Public License for more details.

   You should have received a copy of the GNU General Public License
   along with this program; if not, write to the Free Software
   Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA
   02111-1307, USA.
 */

#include <config.h>
#include <stdio.h>
#include <stdlib.h>
#include <my_string.h>
#include "stringtools.h"
#include <sys/types.h>

#if HAVE_DIRENT_H
#include <dirent.h>
#define NAMLEN(dirent) strlen((dirent)->d_name)
#else
#define dirent direct
#define NAMLEN(dirent) (dirent)->d_namlen
#if HAVE_SYS_NDIR_H
#include <sys/ndir.h>
#endif
#if HAVE_SYS_DIR_H
#include <sys/dir.h>
#endif
#if HAVE_NDIR_H
#include <ndir.h>
#endif
#endif

#include <sys/stat.h>
#if TIME_WITH_SYS_TIME
# include <sys/time.h>
# include <time.h>
#else
# if HAVE_SYS_TIME_H
#  include <sys/time.h>
# else
#  include <time.h>
# endif
#endif

#ifdef HAVE_UNISTD_H
#include <unistd.h>
#endif

#include "coolwidget.h"
#include "pool.h"
#include "mad.h"

/*
 * struct file_entry {
 *     unsigned long options;
 *     struct dirent dirent;
 *     struct stat stat;
 * };
 */

void destroy_filelist (CWidget * w)
{
    if (w->hook) {
	free (w->hook);
	w->hook = 0;
    }
}

/* This doesn't apply to solaris: */
/*
 * struct dirent {
 * 	long		d_ino;
 * 	__kernel_off_t	d_off;
 * 	unsigned short	d_reclen;
 * 	char		d_name[256];
 * };
 */

/*
 * 
 * struct new_stat {
 * 	unsigned short st_dev;
 * 	unsigned short __pad1;
 * 	unsigned long st_ino;
 * 	unsigned short st_mode;
 * 	unsigned short st_nlink;
 * 	unsigned short st_uid;
 * 	unsigned short st_gid;
 * 	unsigned short st_rdev;
 * 	unsigned short __pad2;
 * 	unsigned long  st_size;
 * 	unsigned long  st_blksize;
 * 	unsigned long  st_blocks;
 * 	unsigned long  st_atime;
 * 	unsigned long  __unused1;
 * 	unsigned long  st_mtime;
 * 	unsigned long  __unused2;
 * 	unsigned long  st_ctime;
 * 	unsigned long  __unused3;
 * 	unsigned long  __unused4;
 * 	unsigned long  __unused5;
 * };
 * 
 */

/*
 * 
 * #define S_ISLNK(m)	(((m) & S_IFMT) == S_IFLNK)
 * #define S_ISREG(m)	(((m) & S_IFMT) == S_IFREG)
 * #define S_ISDIR(m)	(((m) & S_IFMT) == S_IFDIR)
 * #define S_ISCHR(m)	(((m) & S_IFMT) == S_IFCHR)
 * #define S_ISBLK(m)	(((m) & S_IFMT) == S_IFBLK)
 * #define S_ISFIFO(m)	(((m) & S_IFMT) == S_IFIFO)
 * #define S_ISSOCK(m)	(((m) & S_IFMT) == S_IFSOCK)
 * 
 * #define S_IRWXU 00700
 * #define S_IRUSR 00400
 * #define S_IWUSR 00200
 * #define S_IXUSR 00100
 * 
 * #define S_IRWXG 00070
 * #define S_IRGRP 00040
 * #define S_IWGRP 00020
 * #define S_IXGRP 00010
 * 
 * #define S_IRWXO 00007
 * #define S_IROTH 00004
 * #define S_IWOTH 00002
 * #define S_IXOTH 00001
 */


char *dname (struct dirent *directentry);

#ifdef HAVE_STRFTIME
/* We want our own dates for NLS */
#undef HAVE_STRFTIME
#endif

#undef gettext_noop
#define gettext_noop(x) x

/* reset with timestr = 0 */
void get_file_time (char *timestr, time_t file_time)
{
#ifndef HAVE_STRFTIME
    static char monthstr[12][8] =
    {
/* Month list */
	gettext_noop("Jan"), gettext_noop("Feb"), gettext_noop("Mar"), gettext_noop("Apr"), gettext_noop("May"), gettext_noop("Jun"), gettext_noop("Jul"), gettext_noop("Aug"), gettext_noop("Sep"), gettext_noop("Oct"), gettext_noop("Nov"), gettext_noop("Dec")
    };
#endif
    static struct tm tm_current =
    {-1};
    struct tm *tm;
    static int i = 0;

    if (!timestr) {
	i = 0;
	return;
    }

    if (!i)
	for (i=0;i<12;i++)
	    strcpy (monthstr[i], _(monthstr[i]));

    if (tm_current.tm_sec == -1) {
	time_t t;
	time (&t);
	memcpy ((void *) &tm_current, (void *) localtime (&t), sizeof (struct tm));
    }
    tm = localtime (&(file_time));
#if HAVE_STRFTIME
    if (tm->tm_year == tm_current.tm_year)	/* date with year and without time */
	strftime (timestr, 31, "%b %d %H:%M", tm);
    else			/* date without year and with time */
	strftime (timestr, 31, "%Y %b %d", tm);
#else
    if (tm->tm_year == tm_current.tm_year)	/* date with year and without time */
	sprintf (timestr, "%s %.2d %.2d:%.2d", monthstr[tm->tm_mon],
		 tm->tm_mday, tm->tm_hour, tm->tm_min);
    else			/* date without year and with time */
	sprintf (timestr, "%d %s %.2d", tm->tm_year + 1900,
		 monthstr[tm->tm_mon], tm->tm_mday);
#endif
}


char **get_filelist_line (void *data, int line_number, int *num_fields, int *tagged)
{
    struct file_entry *directentry;
    static char *fields[10], size[24], mode[12], timestr[32];
    static char name[520], *n;
    mode_t m;

    *num_fields = 4;		/* name, size, date, mode only (for the mean time) */

    directentry = (struct file_entry *) data;
    if (directentry[line_number].options & FILELIST_LAST_ENTRY)
	return 0;

    n = name;
    strcpy (name, directentry[line_number].name);
    fields[0] = name;
    sprintf (size, "\t%u", (unsigned int) directentry[line_number].stat.st_size);
    fields[1] = size;

    get_file_time (timestr, directentry[line_number].stat.st_mtime);
    fields[2] = timestr;

    memset (mode, ' ', 11);
    mode[11] = 0;
    mode[0] = '-';
    m = directentry[line_number].stat.st_mode;
    switch ((int) m & S_IFMT) {
    case S_IFLNK:
	mode[0] = 'l';
	break;
    case S_IFDIR:
	mode[0] = 'd';
	break;
    case S_IFCHR:
	mode[0] = 'c';
	break;
    case S_IFBLK:
	mode[0] = 'b';
	break;
    case S_IFIFO:
	mode[0] = 'f';
	break;
    case S_IFSOCK:
	mode[0] = 's';
	break;
    }

    mode[1] = m & S_IRUSR ? 'r' : '-';
    mode[2] = m & S_IWUSR ? 'w' : '-';
    mode[3] = m & S_IXUSR ? 'x' : '-';

    mode[4] = m & S_IRGRP ? 'r' : '-';
    mode[5] = m & S_IWGRP ? 'w' : '-';
    mode[6] = m & S_IXGRP ? 'x' : '-';

    mode[7] = m & S_IROTH ? 'r' : '-';
    mode[8] = m & S_IWOTH ? 'w' : '-';
    mode[9] = m & S_IXOTH ? 'x' : '-';

    if (S_ISLNK (m)) {
	int l, i;
	char *p;
	p = directentry[line_number].name;
	l = strlen (n);
	for (i = 0; i < l; i++) {
	    *n++ = '\b';
	    *n++ = *p++;
	}
	*n++ = '\0';
    } else if (m & (S_IXUSR | S_IXGRP | S_IXOTH)) {
	int l, i;
	char *p;
	p = directentry[line_number].name;
	l = strlen (n);
	for (i = 0; i < l; i++) {
	    *n++ = '\r';
	    *n++ = *p++;
	}
	*n++ = '\0';
    }
    fields[3] = mode;
    fields[*num_fields] = 0;
    if (directentry[line_number].options & FILELIST_TAGGED_ENTRY)
	*tagged = 1;
    return fields;
}


CWidget *CDrawFilelist (const char *identifier, Window parent, int x, int y,
			int width, int height, int line, int column,
			struct file_entry *directentry,
			long options)
{
    struct file_entry e;
    CWidget *w;
    int n;

    if (!directentry) {
	memset (&e, 0, sizeof(e));
	e.options = FILELIST_LAST_ENTRY;
	directentry = &e;
	n = 0;
    } else {
	for (n = 0; !(directentry[n].options & FILELIST_LAST_ENTRY); n++);	/* count entries */
    }

    w = CDrawFieldedTextbox (identifier, parent, x, y,
			     width, height, line, column,
			     get_filelist_line,
			     options, directentry);

    w->destroy = destroy_filelist;
    w->hook = CMalloc (sizeof (struct file_entry) * (n + 1));
    memcpy (w->hook, directentry, sizeof (struct file_entry) * (n + 1));
    return w;
}

CWidget *CRedrawFilelist (const char *identifier, struct file_entry *directentry, int preserve)
{
    struct file_entry e;
    CWidget *w;
    int n;

    if (!directentry) {
	e.options = FILELIST_LAST_ENTRY;
	directentry = &e;
	n = 0;
    } else {
	for (n = 0; !(directentry[n].options & FILELIST_LAST_ENTRY); n++);	/* count entries */
    }

    w = CIdent (identifier);
    if (w->hook)
	free (w->hook);
    w->hook = CMalloc (sizeof (struct file_entry) * (n + 1));
    memcpy (w->hook, directentry, sizeof (struct file_entry) * (n + 1));

    w = CRedrawFieldedTextbox (identifier, preserve);

    return w;
}

struct file_entry *CGetFilelistLine (CWidget * w, int line)
{
    struct file_entry *e;
    static struct file_entry r;
    memset (&r, 0, sizeof (r));
    e = (struct file_entry *) w->hook;
    if (e[line].options & FILELIST_LAST_ENTRY)
	r.options = FILELIST_LAST_ENTRY;
    else
	r = e[line];
    return &r;
}


