/****************************************************************************
|                         Digital Audio Processor
|                         =======================
|
| Filename    : DPSample.h
|
| Revision    : 1.0
| Date        : 25/01/96
|
| Object      : DPSample
|
| Description : Prototypes for the DPSample class
|
| (c) Richard Kent 1996
|
| $Id$
|
****************************************************************************/

class DPSample;

#ifndef _DPSample_h
#define _DPSample_h

#include "DPDefs.h"

// Stop and release callback functions
void DPSampleStopCatch (...);
void DPSampleReleaseCatch (...);

// Replacement audiofile void handler
void DPSampleAudiofileError (long,const char *);

// Prototypes for general routines
long getInputRate ();
long getOutputRate ();
void limitQueueSize (long channels,long *queueSize);

// Dynamic memory structure
typedef struct extraListEntry
{
  void *data;
  long start;
  long end;
  extraListEntry *next;
} extraListEntry;

class DPSample
{
  public:
  
    // Constructor, destructor and copy methods
    DPSample ();
    ~DPSample ();
    DPSample (DPSample &newSample);
    DPSample &operator= (DPSample &newSample);
    
    // Functions
    long clone (DPSample &newSample);
    long cloneSimple (DPSample &newSample);
    long cloneData (DPSample &newSample);
    char *loadAIFF (char *newFilename);
    char *saveAIFF (
      char *newFilename,
      int rangeonly,
      int copysave,
      int replaceFilename);
    char *play (long playMode,FL_OBJECT *object);
    char *record (long recordMode,long trigger,FL_OBJECT *object);
    void clear ();
    long fresh (
      double newRate,
      long newWidth,
      long newChannels,
      long frames);

    // Range functions
    void rangeClear ();
    void rangeDisplay ();
    void rangeAll ();
    void setRangeSusLoop ();
    void setRangeRelLoop ();
    void setSusLoopRange ();
    void setRelLoopRange ();
    
    // Time functions
    inline long   timeToFrame (double time);
    inline double frameToTime (long frame);    

    // Zoom functions
    void zoomIn ();
    void zoomOut ();
    void zoomAll ();
    void zoomRange ();
    void zoomWidth (long width);
    
    // Scroll functions
    void scrollLstart ();
    void scrollL (long scrollFrames);
    void scrollR (long scrollFrames);
    void scrollRend ();

    // Modifiers
    inline void setChanged (long newChanged);
    inline void setFormat (long newFormat);
    inline void setCompression (long newCompression);
    inline void setRate (double newRate);
    inline void setFilename (char *newFilename);
    inline void setName (char *newName);
    inline void setCopyright (char *newCopyright);
    inline void setAuthor (char *newAuthor);
    inline void setAnnotation (char *newAnnotation);
    inline void setSusLoopMode (long newSusLoopMode);
    inline void setSusLoopStart (long newSusLoopStart);
    inline void setSusLoopEnd (long newSusLoopEnd);
    inline void setSusLoop (long newSusLoopStart,long newSusLoopEnd);
    inline void setRelLoopMode (long newRelLoopMode);
    inline void setRelLoopStart (long newRelLoopStart);
    inline void setRelLoopEnd (long newRelLoopEnd);
    inline void setRelLoop (long newRelLoopStart,long newRelLoopEnd);
    inline void setRangeValid (long newRangeValid);
    inline void setRangeStart (long newRangeStart);
    inline void setRangeEnd (long newRangeEnd);
    inline void setRange (long newRangeStart,long newRangeEnd);
    inline void setDisplayStart (long newDisplayStart);
    inline void setDisplayEnd (long newDisplayEnd);
    inline void setDisplay (long newDisplayStart,long newDisplayEnd);
    inline void setRedraw (long newRedraw);
    inline void setCurrentPos (long newCurrentPos);
   
    // Accessors
    inline long   getValid () const;
    inline long   getChanged () const;
    inline long   getFormat () const;
    inline long   getCompression () const;
    inline double getRate () const;
    inline long   getWidth () const;
    inline long   getChannels () const;
    inline long   getFrames () const;
    inline long   getSamples () const;
    inline long   getBytes () const;
    inline long   getAESused () const;
    inline unsigned char *getAES ();
    inline char   *getFilename () const;
    inline char   *getName () const;
    inline char   *getCopyright () const;
    inline char   *getAuthor () const;
    inline char   *getAnnotation () const;
    inline long   getSusLoopMode () const;
    inline long   getSusLoopStart () const;
    inline long   getSusLoopEnd () const;
    inline long   getRelLoopMode () const;
    inline long   getRelLoopStart () const;
    inline long   getRelLoopEnd () const;
    inline long   getRangeValid () const;
    inline long   getRangeStart () const;
    inline long   getRangeEnd () const;
    inline long   getDisplayStart () const;
    inline long   getDisplayEnd () const;
    inline long   getRedraw () const;
    inline long   getCurrentPos () const;
    inline long   getLzero (long frame);
    inline long   getL (long frame,long scrollFrames);
    inline long   getR (long frame,long scrollFrames);
    inline long   getRzero (long frame);
    
    // Frame functions
    inline long   getFrame24 (long frame,long channel);
    inline void   setFrame24 (long frame,long channel,long newFrame);
    inline short  getFrame16 (long frame,long channel);
    inline void   setFrame16 (long frame,long channel,short newFrame);
    inline double getFrameDb (long frame,long channel);
    inline void   setFrameDb (long frame,long channel,double newFrame);
    inline long   sign (long x);
    long          getFrame24Interp (double frame,long channel);
    short         getFrame16Interp (double frame,long channel);
    double        getFrameDbInterp (double frame,long channel);
    inline long   getFrame24Extra (long frame,long channel);
    inline short  getFrame16Extra (long frame,long channel);
    inline double getFrameDbExtra (long frame,long channel);
    inline void   setFrame24Extra (long frame,long channel,long newFrame);
    inline void   setFrame16Extra (long frame,long channel,short newFrame);
    inline void   setFrameDbExtra (long frame,long channel,double newFrame);
    
    // Conversion functions
    void to8bit   ();
    void to16bit  ();
    void toMono   (int convertMode);
    void toStereo (int convertMode);
    void toQuadro (int convertMode);
    
    // Trim function
    void trim ();

    // Cut & paste functions
    void cut (DPSample *clip,int mode);
    void del (int mode);
    void copy (DPSample *clip,int mode);
    void paste (DPSample *clip,int mode);
    void keep (DPSample *clip,int mode);
    void overwrite (DPSample *clip,int mode);
    void mix (
      DPSample *clip,
      double buffVol,
      double clipVol,
      long   mixin,
      long   mixout,
      int mode);
    void overwriteRange (DPSample *clip,int mode);
    void mixRange (
      DPSample *clip,
      double buffVol,
      double clipVol,
      long   mixin,
      long   mixout,
      int mode);
    
    // Range functions
    void reverse (int mode);
    void invert (int mode);
    void zero (int mode);
    void channelChange (int channelMode);
    void duplicate (long times,int mode);
    void panorama (long amount,long ltor,long rtol,long panChannels);
    void adjustDC (long amount,int mode);
    void addWorkspace (long newFrames,int mode);
    
    // Amplitude functions
    void percentage (
      double lamount,
      double ramount,
      double amount3,
      double amount4,
      int wrap,
      int mode);
    void normalise (int mode);
    void fadeIn (int mode);
    void fadeOut (int mode);
    void bounce (int bounceMode);
    void ramp (double startamount,double endamount,int wrap,int mode);
    void draw (double *amounts,long *frames,long n,int wrap,int mode);
    void limitValue (long *value,int wrap);
    
    // Resampling functions
    void nonTimeStretch (long newFrames,double newRate,int mode);
    int timeStretch (
      double pitchChange,
      double framesChange,
      double newRate,
      long   lowestFreq,
      long   quality,
      int    mode);
    
    // Dynamic memory functions
    extraListEntry *expand (double newTime);
    long consolidate ();
    void removeExtra ();
        
  private:
  
    // Data
    long   valid;           // true if valid sample held
    long   changed;         // true if sample changed and not saved
    long   format;          // original file format
    long   compression;     // original compresion used
    double rate;            // hertz
    long   width;           // 1=8bit=char 2=16bit=short
    long   channels;        // 1=mono 2=stereo 4=quadro
    long   frames;          // number of sample frames
    long   samples;         // frames  * channels
    long   bytes;           // samples * width
    long   AESused;         // Whether AES data bytes present
    unsigned char AES [24]; // AES data bytes;
    char   *filename;
    char   *name;
    char   *copyright;
    char   *author;
    char   *annotation;
    long   susLoopMode;     // AIFF loop mode
    long   susLoopStart;    // frames
    long   susLoopEnd;      // frames
    long   relLoopMode;     // AIFF loop mode
    long   relLoopStart;    // frames
    long   relLoopEnd;      // frames
    long   rangeValid;      // 1=on 0=off
    long   rangeStart;      // frames
    long   rangeEnd;        // frames
    long   displayStart;    // frames
    long   displayEnd;      // frames
    long   redraw;          // 0=full 1=line only 2=full + line
    long   currentPos;      // current play/record frame (-1 for none)
    void   *data;           // (width=1)=>(char*) (width=2)=>(short*)
    extraListEntry *extra;
};

// Include inline methods
#include "DPSample.inl"

// Resample headers
#include "DPResample.h"

#endif // _DPSample_h

/***************************************************************************/
