/****************************************************************************
|                         Digital Audio Processor
|                         =======================
|
| Filename    : DPSampleXforms.cc
|
| Revision    : 1.0
| Date        : 04/03/96
|
| Object      : Xforms sample
|
| Description : Xforms sample object
|
| (c) Richard Kent 1996
|
| $Id$
|
****************************************************************************/

static char DPSampleXForms_cc [] = "$Id$";

#include "forms.h"
#include "DPSample.h"
#include "DPSampleXforms.h"

#define TEMPDIRLEN 256

typedef struct
{
  DPSample  *sample;
  long      scroll;          // Whether to scroll when playing & recording
  long      axis;            // Whether to display the axis
  long      edit;            // 0=Left 1=Right 2=Both (stereo samples only)
                             // 2=All  3=Three 4=Four (quadro samples only)
  long      frames;          // Whether to display in frames or time
  long      freehand;        // Whether freehand drawing mode is on
  long      drawmode;        // 0=dots 1=lines 2=filled
  long      drawaccuracy;    // 0=quick 1=accurate
  long      undomode;        // 0=no undo 1=undo
  long      autowindows;     // close / reopen windows when doing DSP
  long      autoglobals;     // automatically adjust audio globals
  long      autoeditmode;    // Whether range select chooses mode
  long      savecopy;        // save copyright chunk in AIFF files
  long      tooltips;        // show tooltips
  long      rangealt;        // use alternative range marking method
  long      rangecross;      // use crossover when range marking
  char      tempdir [TEMPDIRLEN]; // directory for temp saves
  long      rangeMark;       // Initial range mark
  int       rangeGrab;       // 0=none 1=start 2=end
  int       oldPosx;         // Previous play or record position
  int       grabSusStart;    // Sustain start marker grabbed
  int       grabSusEnd;      // Sustain end marker grabbed
  int       grabRelStart;    // Release start marker grabbed
  int       grabRelEnd;      // Release end marker grabbed
  long      oldframe;        // Previous frame for freehand
  long      oldval;          // Previous sample value for freehand
  long      oldchan;         // Previous channel for freehand
} sampleSpec;

#define SPEC ((sampleSpec *)ob->spec)

#define TOPIXELX(frame)                                                     \
  (int)((long long) objLeftX +                                              \
  ((((long long)(frame) - displayStart) *                                   \
  (objWidth - 1)) / displayDuration))

#define TOFRAME(pixel)                                                      \
  (long)((long long) displayStart +                                         \
  ((pixel) * (long long) displayDuration / (objWidth - 1)))

#define TOFRAMEROUND(pixel)                                                 \
  (long)((long long) displayStart +                                         \
  (((pixel) * (long long) displayDuration) + ((objWidth - 1) / 2)) /        \
  (objWidth - 1))

/*---------------------------------------------------------------------------
| FUNCTION fl_create_sample
---------------------------------------------------------------------------*/
extern FL_OBJECT *fl_create_sample (
  int type,
  FL_Coord x,
  FL_Coord y, 
  FL_Coord w,
  FL_Coord h,
  const char *label)
{
  FL_OBJECT *ob;
  
  ob = fl_make_object (FL_SAMPLE,type,x,y,w,h,label,handle_sample);
  ob->boxtype = FL_FLAT_BOX;
  ob->spec = fl_calloc (1,sizeof (sampleSpec));
  
  // Set defaults
  SPEC->scroll         = 0;
  SPEC->axis           = 1;
  SPEC->edit           = 2;
  SPEC->frames         = 1;
  SPEC->freehand       = 0;
  SPEC->drawmode       = 1;
  SPEC->drawaccuracy   = 0;
  SPEC->undomode       = 1;
  SPEC->autowindows    = 1;
  SPEC->autoglobals    = 1;
  SPEC->autoeditmode   = 1;
  SPEC->savecopy       = 1;
  SPEC->tooltips       = 1;
  SPEC->rangealt       = 0;
  SPEC->rangecross     = 0;
  strcpy (SPEC->tempdir,"/tmp");
  SPEC->rangeMark      = -1;
  SPEC->rangeGrab      = 0;
  SPEC->oldPosx        = -1;
  SPEC->grabSusStart   = 0;
  SPEC->grabSusEnd     = 0;
  SPEC->grabRelStart   = 0;
  SPEC->grabRelEnd     = 0;
  SPEC->oldframe       = -1;
  SPEC->oldval         = 0;
  SPEC->oldchan        = 0;
  return ob;
}

/*---------------------------------------------------------------------------
| FUNCTION fl_add_sample
---------------------------------------------------------------------------*/
extern FL_OBJECT *fl_add_sample (
  int type,
  FL_Coord x,
  FL_Coord y, 
  FL_Coord w,
  FL_Coord h,
  const char *label)
{
  FL_OBJECT *ob = fl_create_sample (type,x,y,w,h,label);
  fl_add_object (fl_current_form,ob);
  return ob;
}

/*---------------------------------------------------------------------------
| FUNCTION fl_get_sample
---------------------------------------------------------------------------*/
extern DPSample *fl_get_sample (FL_OBJECT *ob)
{
  if (!ob || ob->objclass != FL_SAMPLE) return 0;
  return SPEC->sample;
}

/*---------------------------------------------------------------------------
| FUNCTION fl_set_sample
---------------------------------------------------------------------------*/
extern void fl_set_sample (FL_OBJECT *ob,DPSample *newSample)
{
  if (!ob || ob->objclass != FL_SAMPLE) return;
  SPEC->sample       = newSample;
  SPEC->freehand     = 0;
  SPEC->rangeMark    = -1;
  SPEC->oldPosx      = -1;
  SPEC->grabSusStart = 0;
  SPEC->grabSusEnd   = 0;
  SPEC->grabRelStart = 0;
  SPEC->grabRelEnd   = 0;
  SPEC->oldframe     = -1;
  SPEC->oldval       = 0;
  SPEC->oldchan      = 0;
  fl_set_button (mainForm->manualEdit,0);
}

/*---------------------------------------------------------------------------
| FUNCTION fl_get_sample_scroll
---------------------------------------------------------------------------*/
extern long fl_get_sample_scroll (FL_OBJECT *ob)
{
  if (!ob || ob->objclass != FL_SAMPLE) return 0;
  return SPEC->scroll;
}

/*---------------------------------------------------------------------------
| FUNCTION fl_set_sample_scroll
---------------------------------------------------------------------------*/
extern void fl_set_sample_scroll (FL_OBJECT *ob,long newScroll)
{
  if (!ob || ob->objclass != FL_SAMPLE) return;
  SPEC->scroll = newScroll;
}

/*---------------------------------------------------------------------------
| FUNCTION fl_get_sample_axis
---------------------------------------------------------------------------*/
extern long fl_get_sample_axis (FL_OBJECT *ob)
{
  if (!ob || ob->objclass != FL_SAMPLE) return 0;
  return SPEC->axis;
}

/*---------------------------------------------------------------------------
| FUNCTION fl_set_sample_axis
---------------------------------------------------------------------------*/
extern void fl_set_sample_axis (FL_OBJECT *ob,long newAxis)
{
  if (!ob || ob->objclass != FL_SAMPLE) return;
  SPEC->axis = newAxis;
}

/*---------------------------------------------------------------------------
| FUNCTION fl_get_sample_edit
---------------------------------------------------------------------------*/
extern long fl_get_sample_edit (FL_OBJECT *ob)
{
  if (!ob || ob->objclass != FL_SAMPLE) return 0;
  return SPEC->edit;
}

/*---------------------------------------------------------------------------
| FUNCTION fl_set_sample_edit
---------------------------------------------------------------------------*/
extern void fl_set_sample_edit (FL_OBJECT *ob,long newEdit)
{
  if (!ob || ob->objclass != FL_SAMPLE) return;
  SPEC->edit = newEdit;
}

/*---------------------------------------------------------------------------
| FUNCTION fl_get_sample_frames
---------------------------------------------------------------------------*/
extern long fl_get_sample_frames (FL_OBJECT *ob)
{
  if (!ob || ob->objclass != FL_SAMPLE) return 0;
  return SPEC->frames;
}

/*---------------------------------------------------------------------------
| FUNCTION fl_set_sample_frames
---------------------------------------------------------------------------*/
extern void fl_set_sample_frames (FL_OBJECT *ob,long newFrames)
{
  if (!ob || ob->objclass != FL_SAMPLE) return;
  SPEC->frames = newFrames;
}


/*---------------------------------------------------------------------------
| FUNCTION fl_get_sample_freehand
---------------------------------------------------------------------------*/
extern long fl_get_sample_freehand (FL_OBJECT *ob)
{
  if (!ob || ob->objclass != FL_SAMPLE) return 0;
  return SPEC->freehand;
}

/*---------------------------------------------------------------------------
| FUNCTION fl_set_sample_freehand
---------------------------------------------------------------------------*/
extern void fl_set_sample_freehand (FL_OBJECT *ob,long newFreehand)
{
  if (!ob || ob->objclass != FL_SAMPLE) return;
  SPEC->freehand = newFreehand;
}

/*---------------------------------------------------------------------------
| FUNCTION fl_get_sample_drawmode
---------------------------------------------------------------------------*/
extern long fl_get_sample_drawmode (FL_OBJECT *ob)
{
  if (!ob || ob->objclass != FL_SAMPLE) return 0;
  return SPEC->drawmode;
}

/*---------------------------------------------------------------------------
| FUNCTION fl_set_sample_drawmode
---------------------------------------------------------------------------*/
extern void fl_set_sample_drawmode (FL_OBJECT *ob,long newDrawmode)
{
  if (!ob || ob->objclass != FL_SAMPLE) return;
  SPEC->drawmode = newDrawmode;
}

/*---------------------------------------------------------------------------
| FUNCTION fl_get_sample_drawaccuracy
---------------------------------------------------------------------------*/
extern long fl_get_sample_drawaccuracy (FL_OBJECT *ob)
{
  if (!ob || ob->objclass != FL_SAMPLE) return 0;
  return SPEC->drawaccuracy;
}

/*---------------------------------------------------------------------------
| FUNCTION fl_set_sample_drawaccuracy
---------------------------------------------------------------------------*/
extern void fl_set_sample_drawaccuracy (FL_OBJECT *ob,long newDrawaccuracy)
{
  if (!ob || ob->objclass != FL_SAMPLE) return;
  SPEC->drawaccuracy = newDrawaccuracy;
}

/*---------------------------------------------------------------------------
| FUNCTION fl_get_sample_undomode
---------------------------------------------------------------------------*/
extern long fl_get_sample_undomode (FL_OBJECT *ob)
{
  if (!ob || ob->objclass != FL_SAMPLE) return 0;
  return SPEC->undomode;
}

/*---------------------------------------------------------------------------
| FUNCTION fl_set_sample_undomode
---------------------------------------------------------------------------*/
extern void fl_set_sample_undomode (FL_OBJECT *ob,long newUndomode)
{
  if (!ob || ob->objclass != FL_SAMPLE) return;
  SPEC->undomode = newUndomode;
}

/*---------------------------------------------------------------------------
| FUNCTION fl_get_sample_autowindows
---------------------------------------------------------------------------*/
extern long fl_get_sample_autowindows (FL_OBJECT *ob)
{
  if (!ob || ob->objclass != FL_SAMPLE) return 0;
  return SPEC->autowindows;
}

/*---------------------------------------------------------------------------
| FUNCTION fl_set_sample_autowindows
---------------------------------------------------------------------------*/
extern void fl_set_sample_autowindows (FL_OBJECT *ob,long newAutowindows)
{
  if (!ob || ob->objclass != FL_SAMPLE) return;
  SPEC->autowindows = newAutowindows;
}

/*---------------------------------------------------------------------------
| FUNCTION fl_get_sample_autoglobals
---------------------------------------------------------------------------*/
extern long fl_get_sample_autoglobals(FL_OBJECT *ob)
{
  if (!ob || ob->objclass != FL_SAMPLE) return 0;
  return SPEC->autoglobals;
}

/*---------------------------------------------------------------------------
| FUNCTION fl_set_sample_autoglobals
---------------------------------------------------------------------------*/
extern void fl_set_sample_autoglobals(FL_OBJECT *ob,long newAutoglobals)
{
  if (!ob || ob->objclass != FL_SAMPLE) return;
  SPEC->autoglobals = newAutoglobals;
}

/*---------------------------------------------------------------------------
| FUNCTION fl_get_sample_autoeditmode
---------------------------------------------------------------------------*/
extern long fl_get_sample_autoeditmode(FL_OBJECT *ob)
{
  if (!ob || ob->objclass != FL_SAMPLE) return 0;
  return SPEC->autoeditmode;
}

/*---------------------------------------------------------------------------
| FUNCTION fl_set_sample_autoeditmode
---------------------------------------------------------------------------*/
extern void fl_set_sample_autoeditmode(FL_OBJECT *ob,long newAutoeditmode)
{
  if (!ob || ob->objclass != FL_SAMPLE) return;
  SPEC->autoeditmode = newAutoeditmode;
}

/*---------------------------------------------------------------------------
| FUNCTION fl_get_sample_copysave
---------------------------------------------------------------------------*/
extern long fl_get_sample_copysave (FL_OBJECT *ob)
{
  if (!ob || ob->objclass != FL_SAMPLE) return 0;
  return SPEC->savecopy;
}

/*---------------------------------------------------------------------------
| FUNCTION fl_set_sample_copysave
---------------------------------------------------------------------------*/
extern void fl_set_sample_copysave (FL_OBJECT *ob,long newSavecopy)
{
  if (!ob || ob->objclass != FL_SAMPLE) return;
  SPEC->savecopy = newSavecopy;
}

/*---------------------------------------------------------------------------
| FUNCTION fl_get_sample_tooltips
---------------------------------------------------------------------------*/
extern long fl_get_sample_tooltips (FL_OBJECT *ob)
{
  if (!ob || ob->objclass != FL_SAMPLE) return 0;
  return SPEC->tooltips;
}

/*---------------------------------------------------------------------------
| FUNCTION fl_set_sample_tooltips
---------------------------------------------------------------------------*/
extern void fl_set_sample_tooltips (FL_OBJECT *ob,long newTooltips)
{
  if (!ob || ob->objclass != FL_SAMPLE) return;
  SPEC->tooltips = newTooltips;
}

/*---------------------------------------------------------------------------
| FUNCTION fl_get_sample_rangealt
---------------------------------------------------------------------------*/
extern long fl_get_sample_rangealt (FL_OBJECT *ob)
{
  if (!ob || ob->objclass != FL_SAMPLE) return 0;
  return SPEC->rangealt;
}

/*---------------------------------------------------------------------------
| FUNCTION fl_set_sample_rangealt
---------------------------------------------------------------------------*/
extern void fl_set_sample_rangealt (FL_OBJECT *ob,long newRangealt)
{
  if (!ob || ob->objclass != FL_SAMPLE) return;
  SPEC->rangealt = newRangealt;
}

/*---------------------------------------------------------------------------
| FUNCTION fl_get_sample_rangecross
---------------------------------------------------------------------------*/
extern long fl_get_sample_rangecross (FL_OBJECT *ob)
{
  if (!ob || ob->objclass != FL_SAMPLE) return 0;
  return SPEC->rangecross;
}

/*---------------------------------------------------------------------------
| FUNCTION fl_set_sample_rangecross
---------------------------------------------------------------------------*/
extern void fl_set_sample_rangecross (FL_OBJECT *ob,long newRangecross)
{
  if (!ob || ob->objclass != FL_SAMPLE) return;
  SPEC->rangecross = newRangecross;
}

/*---------------------------------------------------------------------------
| FUNCTION fl_get_sample_tempdir
---------------------------------------------------------------------------*/
extern const char *fl_get_sample_tempdir (FL_OBJECT *ob)
{
  if (!ob || ob->objclass != FL_SAMPLE) return 0;
  return (const char *) SPEC->tempdir;
}

/*---------------------------------------------------------------------------
| FUNCTION fl_set_sample_tempdir
---------------------------------------------------------------------------*/
extern void fl_set_sample_tempdir (FL_OBJECT *ob,const char *newTempdir)
{
  if (!ob || ob->objclass != FL_SAMPLE) return;
  strncpy (SPEC->tempdir,(char *)newTempdir,TEMPDIRLEN);
}

/*---------------------------------------------------------------------------
| FUNCTION handle_sample
---------------------------------------------------------------------------*/

int handle_sample (
  FL_OBJECT *ob,
  int event,
  FL_Coord mx,
  FL_Coord my,
  int key,
  void *xev)
{
  DPSample *sample  = SPEC->sample;
  long *rangeMark   = &SPEC->rangeMark;
  int *oldPosx      = &SPEC->oldPosx;

  switch (event)
  {
    case FL_DRAW:
    {
      long displayStart    = sample->getDisplayStart ();
      long displayEnd      = sample->getDisplayEnd ();
      long displayDuration = displayEnd - displayStart;
      long rangeStart      = sample->getRangeStart ();
      long rangeEnd        = sample->getRangeEnd ();
      int  objLeftX        = ob->x + BORDERX;
      int  objRightX       = ob->x + ob->w - 1 - BORDERX;
      int  objTopY         = ob->y + BORDERY;
      int  objBotY         = ob->y + ob->h - 1 - BORDERY;
      int  objWidth        = ob->w-(2*BORDERX);
      int  objHeight       = ob->h-(2*BORDERY);
      int  first           = TRUE;
      int  mono            = sample->getChannels () == 1;
      int  stereo          = sample->getChannels () == 2;
      int  quadro          = sample->getChannels () == 4;
      long currentPos;
      long frame;
      long sample24L;
      long sample24R;
      long sample243;
      long sample244;
      long  i;
      long px;
      long pyL;
      long pyR;
      long py3;
      long py4;
      long oldframe        = -1;
      long oldpx           = -1;
      long oldpyL          = -1;
      long oldpyR          = -1;
      long oldpy3          = -1;
      long oldpy4          = -1;
      long oldpyLmax       = -1;
      long oldpyRmax       = -1;
      long oldpy3max       = -1;
      long oldpy4max       = -1;
      GC gc                = (fl_state [fl_get_vclass()].gc)[15];
      Window win           = fl_winget ();
      Display *display     = fl_get_display ();
      int sampleTopL;
      int sampleBotL;
      int sampleTopR;
      int sampleBotR;
      int sampleTop3;
      int sampleBot3;
      int sampleTop4;
      int sampleBot4;
      int sampleHeight;
      int sampleCentreL;
      int sampleCentreR;
      int sampleCentre3;
      int sampleCentre4;
      
      if (sample->getRedraw ())
      {
        int scroll=0;
        
        // Remove old position if valid
        if (*oldPosx != -1)
        {
          XSetForeground (display,gc,fl_get_flcolor (FL_YELLOW));
          XSetFunction (display,gc,GXxor);
          XDrawLine (display,win,gc,*oldPosx,objTopY,*oldPosx,objBotY);
          XSetFunction (display,gc,GXcopy);
          *oldPosx = -1;
        }
        
        currentPos = sample->getCurrentPos ();

        // Scroll display if required
        if (fl_get_sample_scroll (ob) && currentPos != -1)
        {
          long displayMiddle = (displayStart + displayEnd) / 2;
          if (currentPos < displayStart)
          {
            sample->scrollL (displayMiddle - currentPos);
            scroll = 1;
            displayStart    = sample->getDisplayStart ();
            displayEnd      = sample->getDisplayEnd ();
            displayDuration = displayEnd - displayStart;
          }
          else if (currentPos > displayEnd)
          {
            sample->scrollR (currentPos - displayMiddle);
            scroll = 1;
            displayStart    = sample->getDisplayStart ();
            displayEnd      = sample->getDisplayEnd ();
            displayDuration = displayEnd - displayStart;
          }
        }
        
        // Draw current position if valid
        if (!scroll && sample->getRedraw () != 2)
        {
          if (currentPos >= displayStart && currentPos <= displayEnd)
          {
            *oldPosx = TOPIXELX (currentPos);
            XSetForeground (display,gc,fl_get_flcolor (FL_YELLOW));
            XSetFunction (display,gc,GXxor);
            XDrawLine (display,win,gc,*oldPosx,objTopY,*oldPosx,objBotY);
            XSetFunction (display,gc,GXcopy);
          }
          return 0;
        }
      }

      // Draw normal box (xforms)
      fl_drw_box 
        (ob->boxtype,ob->x,ob->y,ob->w,ob->h,ob->col1,ob->bw);

      // Set Xlib defaults
      XSetForeground (display,gc,fl_get_flcolor (FL_BLACK));

      // Calculate sample boundaries, draw frames, central lines and axis
      if (mono)
      {
        if (SPEC->axis)
        {
          // Mono - with axis markings
          
          sampleTopL   = objTopY;
          sampleBotL   = objBotY - AXISSPACE - MONOGAP;
          sampleHeight = sampleBotL - sampleTopL;
          fl_drw_frame (FL_ENGRAVED_FRAME,objLeftX,sampleTopL,
            objWidth,sampleHeight,FL_BLACK,ob->bw);
          fl_drw_frame (FL_ENGRAVED_FRAME,objLeftX,sampleBotL + MONOGAP,
            objWidth,AXISSPACE,FL_BLACK,ob->bw);
          sampleCentreL = (long) rint ((double) sampleBotL -
            (double)MAX23 * (sampleHeight - 1) / MAX24_1);
          XDrawLine (display,win,gc,objLeftX,sampleCentreL,
            objRightX,sampleCentreL);
          int axis = objBotY - (AXISSPACE * 2 / 3);
          XDrawLine (display,win,gc,objLeftX,axis,objRightX,axis);
          long rounding =
            (long) rint (pow (10.0,rint (log10 (displayDuration)) - 1.0));
          long x = displayDuration / 10;
          long space;
          if (displayDuration < 10) space = 1;
          else space = rounding * (long) rint ((double)x / rounding + 0.5);
          long frame = 0;
          char tempString [50];
          while (frame < displayStart) frame += space;
          while (frame < displayEnd)
          {
            px = TOPIXELX (frame);
            XDrawLine (display,win,gc,
              px,axis - AXISTICKS,px,axis + AXISTICKS);
            if (SPEC->frames)
              sprintf (tempString,"%ld",frame);
            else
              sprintf (tempString,"%.3lf",frame / sample->getRate ());
            if (px < objRightX - 50)
              fl_drw_text (FL_ALIGN_LEFT,px,axis,
                50,14,FL_BLACK,FL_NORMAL_STYLE,AXISTEXTSIZE,tempString);
            frame += space;
          }
        }
        else
        {
          // Mono - no axis markings

          sampleTopL   = objTopY;
          sampleBotL   = objBotY;
          sampleHeight = sampleBotL - sampleTopL;
          fl_drw_frame (FL_ENGRAVED_FRAME,objLeftX,sampleTopL,
            objWidth,sampleHeight,FL_BLACK,ob->bw);
          sampleCentreL = (long) rint ((double) sampleBotL -
            (double)MAX23 * (sampleHeight - 1) / MAX24_1);
          XDrawLine (display,win,gc,objLeftX,sampleCentreL,
            objRightX,sampleCentreL);
        }
      }
      else if (stereo)
      {
        if (SPEC->axis)
        {
          // Stereo - with axis markings
          
          sampleHeight = (objBotY - objTopY - AXISSPACE) / 2;
          sampleTopL   = objTopY;
          sampleBotL   = sampleTopL + sampleHeight;
          sampleBotR   = objBotY;
          sampleTopR   = sampleBotR - sampleHeight;
          fl_drw_frame (FL_ENGRAVED_FRAME,objLeftX,sampleTopL,
            objWidth,sampleHeight,FL_BLACK,ob->bw);
          fl_drw_frame (FL_ENGRAVED_FRAME,objLeftX,sampleTopR,
            objWidth,sampleHeight,FL_BLACK,ob->bw);
          sampleCentreL = (long) rint ((double) sampleBotL -
            (double)MAX23 * (sampleHeight - 1) / MAX24_1);
          sampleCentreR = (long) rint ((double) sampleBotR -
            (double)MAX23 * (sampleHeight - 1) / MAX24_1);
          XDrawLine (display,win,gc,objLeftX,sampleCentreL,
            objRightX,sampleCentreL);
          XDrawLine (display,win,gc,objLeftX,sampleCentreR,
            objRightX,sampleCentreR);
          int axis = sampleTopR - (AXISSPACE * 2 / 3);
          XDrawLine (display,win,gc,objLeftX,axis,objRightX,axis);
          long rounding =
            (long) rint (pow (10.0,rint (log10 (displayDuration)) - 1.0));
          long x = displayDuration / 10;
          long space;
          if (displayDuration < 10) space = 1;
          else space = rounding * (long) rint ((double)x / rounding + 0.5);
          long frame = 0;
          char tempString [50];
          while (frame < displayStart) frame += space;
          while (frame < displayEnd)
          {
            px = TOPIXELX (frame);
            XDrawLine (display,win,gc,
              px,axis - AXISTICKS,px,axis + AXISTICKS);
            if (SPEC->frames)
              sprintf (tempString,"%ld",frame);
            else
              sprintf (tempString,"%.3lf",frame / sample->getRate ());
            if (px < objRightX - 50)
              fl_drw_text (FL_ALIGN_LEFT,px,axis,
                50,14,FL_BLACK,FL_NORMAL_STYLE,AXISTEXTSIZE,tempString);
            frame += space;
          }
        }
        else
        {
          // Stereo - no axis markings

          sampleHeight = (objBotY - objTopY - STEREOGAP) / 2;
          sampleTopL   = objTopY;
          sampleBotL   = sampleTopL + sampleHeight;
          sampleBotR   = objBotY;
          sampleTopR   = sampleBotR - sampleHeight;
          fl_drw_frame (FL_ENGRAVED_FRAME,objLeftX,sampleTopL,
            objWidth,sampleHeight,FL_BLACK,ob->bw);
          fl_drw_frame (FL_ENGRAVED_FRAME,objLeftX,sampleTopR,
            objWidth,sampleHeight,FL_BLACK,ob->bw);
          sampleCentreL = (long) rint ((double) sampleBotL -
            (double)MAX23 * (sampleHeight - 1) / MAX24_1);
          sampleCentreR = (long) rint ((double) sampleBotR -
            (double)MAX23 * (sampleHeight - 1) / MAX24_1);
          XDrawLine (display,win,gc,objLeftX,sampleCentreL,
            objRightX,sampleCentreL);
          XDrawLine (display,win,gc,objLeftX,sampleCentreR,
            objRightX,sampleCentreR);
        }
      }
      else
      {
        if (SPEC->axis)
        {
          // Quadro - with axis markings
          
          sampleHeight = (objBotY - objTopY - AXISSPACE - 2 * QUADROGAP) / 4;
          sampleTopL   = objTopY;
          sampleBotL   = objTopY + sampleHeight;
          sampleTopR   = sampleBotL + QUADROGAP;
          sampleBotR   = sampleTopR + sampleHeight;
          sampleBot4   = objBotY;
          sampleTop4   = sampleBot4 - sampleHeight;
          sampleBot3   = sampleTop4 - QUADROGAP;
          sampleTop3   = sampleBot3 - sampleHeight;
          fl_drw_frame (FL_ENGRAVED_FRAME,objLeftX,sampleTopL,
            objWidth,sampleHeight,FL_BLACK,ob->bw);
          fl_drw_frame (FL_ENGRAVED_FRAME,objLeftX,sampleTopR,
            objWidth,sampleHeight,FL_BLACK,ob->bw);
          fl_drw_frame (FL_ENGRAVED_FRAME,objLeftX,sampleTop3,
            objWidth,sampleHeight,FL_BLACK,ob->bw);
          fl_drw_frame (FL_ENGRAVED_FRAME,objLeftX,sampleTop4,
            objWidth,sampleHeight,FL_BLACK,ob->bw);
          sampleCentreL = (long) rint ((double) sampleBotL -
            (double)MAX23 * (sampleHeight - 1) / MAX24_1);
          sampleCentreR = (long) rint ((double) sampleBotR -
            (double)MAX23 * (sampleHeight - 1) / MAX24_1);
          sampleCentre3 = (long) rint ((double) sampleBot3 -
            (double)MAX23 * (sampleHeight - 1) / MAX24_1);
          sampleCentre4 = (long) rint ((double) sampleBot4 -
            (double)MAX23 * (sampleHeight - 1) / MAX24_1);
          XDrawLine (display,win,gc,objLeftX,sampleCentreL,
            objRightX,sampleCentreL);
          XDrawLine (display,win,gc,objLeftX,sampleCentreR,
            objRightX,sampleCentreR);
          XDrawLine (display,win,gc,objLeftX,sampleCentre3,
            objRightX,sampleCentre3);
          XDrawLine (display,win,gc,objLeftX,sampleCentre4,
            objRightX,sampleCentre4);
          int axis = sampleTop3 - (AXISSPACE * 2 / 3);
          XDrawLine (display,win,gc,objLeftX,axis,objRightX,axis);
          long rounding =
            (long) rint (pow (10.0,rint (log10 (displayDuration)) - 1.0));
          long x = displayDuration / 10;
          long space;
          if (displayDuration < 10) space = 1;
          else space = rounding * (long) rint ((double)x / rounding + 0.5);
          long frame = 0;
          char tempString [50];
          while (frame < displayStart) frame += space;
          while (frame < displayEnd)
          {
            px = TOPIXELX (frame);
            XDrawLine (display,win,gc,
              px,axis - AXISTICKS,px,axis + AXISTICKS);
            if (SPEC->frames)
              sprintf (tempString,"%ld",frame);
            else
              sprintf (tempString,"%.3lf",frame / sample->getRate ());
            if (px < objRightX - 50)
              fl_drw_text (FL_ALIGN_LEFT,px,axis,
                50,14,FL_BLACK,FL_NORMAL_STYLE,AXISTEXTSIZE,tempString);
            frame += space;
          }
        }
        else
        {
          // Quadro - no axis markings

          sampleHeight = (objBotY - objTopY - 3 * QUADROGAP) / 4;
          sampleTopL   = objTopY;
          sampleBotL   = objTopY + sampleHeight;
          sampleTopR   = sampleBotL + QUADROGAP;
          sampleBotR   = sampleTopR + sampleHeight;
          sampleBot4   = objBotY;
          sampleTop4   = sampleBot4 - sampleHeight;
          sampleBot3   = sampleTop4 - QUADROGAP;
          sampleTop3   = sampleBot3 - sampleHeight;
          fl_drw_frame (FL_ENGRAVED_FRAME,objLeftX,sampleTopL,
            objWidth,sampleHeight,FL_BLACK,ob->bw);
          fl_drw_frame (FL_ENGRAVED_FRAME,objLeftX,sampleTopR,
            objWidth,sampleHeight,FL_BLACK,ob->bw);
          fl_drw_frame (FL_ENGRAVED_FRAME,objLeftX,sampleTop3,
            objWidth,sampleHeight,FL_BLACK,ob->bw);
          fl_drw_frame (FL_ENGRAVED_FRAME,objLeftX,sampleTop4,
            objWidth,sampleHeight,FL_BLACK,ob->bw);
          sampleCentreL = (long) rint ((double) sampleBotL -
            (double)MAX23 * (sampleHeight - 1) / MAX24_1);
          sampleCentreR = (long) rint ((double) sampleBotR -
            (double)MAX23 * (sampleHeight - 1) / MAX24_1);
          sampleCentre3 = (long) rint ((double) sampleBot3 -
            (double)MAX23 * (sampleHeight - 1) / MAX24_1);
          sampleCentre4 = (long) rint ((double) sampleBot4 -
            (double)MAX23 * (sampleHeight - 1) / MAX24_1);
          XDrawLine (display,win,gc,objLeftX,sampleCentreL,
            objRightX,sampleCentreL);
          XDrawLine (display,win,gc,objLeftX,sampleCentreR,
            objRightX,sampleCentreR);
          XDrawLine (display,win,gc,objLeftX,sampleCentre3,
            objRightX,sampleCentre3);
          XDrawLine (display,win,gc,objLeftX,sampleCentre4,
            objRightX,sampleCentre4);
        }
      }
      
      // Draw range box if appropriate
      if (sample->getValid () && displayDuration != 0
        && sample->getRangeValid ())
      {
        if (rangeStart <= displayEnd && rangeEnd >= displayStart)
        {
          if (rangeStart < displayStart) rangeStart = displayStart;
          if (rangeEnd > displayEnd)     rangeEnd   = displayEnd;
          int px1 = TOPIXELX (rangeStart);
          int px2 = TOPIXELX (rangeEnd);
          if (mono || (SPEC->edit == 2 || SPEC->edit == 0))
          {
            fl_rectf (px1,sampleTopL,px2-px1+1,sampleHeight,FL_BOTTOM_BCOL);
            fl_line (px1,sampleCentreL,px2,sampleCentreL,FL_BLACK);
          }
          if ((stereo || quadro) && (SPEC->edit == 2 || SPEC->edit == 1))
          {
            fl_rectf (px1,sampleTopR,px2-px1+1,sampleHeight,FL_BOTTOM_BCOL);
            fl_line (px1,sampleCentreR,px2,sampleCentreR,FL_BLACK);
          }
          if (quadro && (SPEC->edit == 2 || SPEC->edit == 3))
          {
            fl_rectf (px1,sampleTop3,px2-px1+1,sampleHeight,FL_BOTTOM_BCOL);
            fl_line (px1,sampleCentre3,px2,sampleCentre3,FL_BLACK);
          }
          if (quadro && (SPEC->edit == 2 || SPEC->edit == 4))
          {
            fl_rectf (px1,sampleTop4,px2-px1+1,sampleHeight,FL_BOTTOM_BCOL);
            fl_line (px1,sampleCentre4,px2,sampleCentre4,FL_BLACK);
          }
        }
      }
      
      // Update slider if necessary
      if (!sample->getValid () || sample->getFrames() == displayDuration)
      {
        fl_set_slider_size (mainForm->scrollBarSlider,1.0);
        fl_set_slider_value (mainForm->scrollBarSlider,0.5);
      }
      else
      {
        double frames = (double) sample->getFrames ();
        double size = (double) displayDuration / frames;
        double pos = (double) displayStart /
          (frames - (double) displayDuration);
        fl_set_slider_size (mainForm->scrollBarSlider,size);
        fl_set_slider_value (mainForm->scrollBarSlider,pos);
      }
      
      // Don't proceed if no valid sample
      if (!sample->getValid () || displayDuration == 0) return 0;
      
      // Calculate grid locking factor
      long grid = displayDuration / (long) objWidth;
      
      int rangeColour;

      // Draw sample waveform
      if (SPEC->drawaccuracy == 0)
      {
        // Quick method
        
        for (i=0; i<objWidth; i++)
        {
          // Calculate frame
          frame = TOFRAME (i);
          if (frame >= displayEnd) frame = displayEnd - 1;

          // Grid based locking
          if (grid > 1) frame = frame - (frame % grid);

          // Set colour inversion if within range bounds
          if (sample->getRangeValid () && frame >= rangeStart
            && frame <= rangeEnd-1)
            rangeColour = 1;
          else
            rangeColour = 0;

          // Retrieve frame values
          sample24L = sample->getFrame24 (frame,0);

          // Calculate x and y coordinates
          px = objLeftX + i;
          pyL = (long) rint ((double) sampleBotL -
            ((double) sample24L + MAX23) * (sampleHeight - 1) / MAX24_1);

          if (stereo)
          {
            // Stereo

            sample24R = sample->getFrame24 (frame,1);
            pyR = (long) rint ((double) sampleBotR -
            ((double) sample24R + MAX23) * (sampleHeight - 1) / MAX24_1);

            if (rangeColour && SPEC->edit != 1)
              XSetForeground (display,gc,fl_get_flcolor (FL_CYAN));
            else
              XSetForeground (display,gc,fl_get_flcolor (FL_BLACK));

            // Draw (dots, lines or filled) for left
            if (SPEC->drawmode == 0)
            {
              // Dots
              if (frame != oldframe)
                XDrawPoint (display,win,gc,px,pyL);
            }
            else if (SPEC->drawmode == 1)
            {
              // Lines
              if (first)
                XDrawPoint (display,win,gc,px,pyL);
              else
                XDrawLine (display,win,gc,oldpx+1,oldpyL,px,pyL);
            }
            else
            {
              // Filled
              XDrawLine (display,win,gc,px,sampleCentreL,px,pyL);
            }

            if (rangeColour && SPEC->edit != 0)
              XSetForeground (display,gc,fl_get_flcolor (FL_CYAN));
            else
              XSetForeground (display,gc,fl_get_flcolor (FL_BLACK));

            // Draw (dots, lines or filled) for right
            if (SPEC->drawmode == 0)
            {
              // Dots
              if (frame != oldframe)
                XDrawPoint (display,win,gc,px,pyR);
            }
            else if (SPEC->drawmode == 1)
            {
              // Lines
              if (first)
              {
                XDrawPoint (display,win,gc,px,pyR);
                first = FALSE;
              }
              else
              {
                XDrawLine (display,win,gc,oldpx+1,oldpyR,px,pyR);
              }
            }
            else
            {
              // Filled
              XDrawLine (display,win,gc,px,sampleCentreR,px,pyR);
            }

            oldframe = frame;
            oldpx    = px;
            oldpyL   = pyL;
            oldpyR   = pyR;
          }
          else if (mono)
          {
            // Mono

            if (rangeColour)
              XSetForeground (display,gc,fl_get_flcolor (FL_CYAN));
            else
              XSetForeground (display,gc,fl_get_flcolor (FL_BLACK));

            // Draw (dots, lines or filled)
            if (SPEC->drawmode == 0)
            {
              // Dots
              if (frame != oldframe)
              {
                XDrawPoint (display,win,gc,px,pyL);
              }
            }
            else if (SPEC->drawmode == 1)
            {
              // Lines
              if (first)
              {
                XDrawPoint (display,win,gc,px,pyL);
                first = FALSE;
              }
              else
              {
                XDrawLine (display,win,gc,oldpx+1,oldpyL,px,pyL);
              }
            }
            else
            {
              // Filled
              XDrawLine (display,win,gc,px,sampleCentreL,px,pyL);
            }
            oldframe = frame;
            oldpx    = px;
            oldpyL   = pyL;
          }
          else
          {
            // Quadro

            sample24R = sample->getFrame24 (frame,1);
            sample243 = sample->getFrame24 (frame,2);
            sample244 = sample->getFrame24 (frame,3);
            pyR = (long) rint ((double) sampleBotR -
            ((double) sample24R + MAX23) * (sampleHeight - 1) / MAX24_1);
            py3 = (long) rint ((double) sampleBot3 -
            ((double) sample243 + MAX23) * (sampleHeight - 1) / MAX24_1);
            py4 = (long) rint ((double) sampleBot4 -
            ((double) sample244 + MAX23) * (sampleHeight - 1) / MAX24_1);

            if (rangeColour && (SPEC->edit == 2 || SPEC->edit == 0))
              XSetForeground (display,gc,fl_get_flcolor (FL_CYAN));
            else
              XSetForeground (display,gc,fl_get_flcolor (FL_BLACK));

            // Draw (dots, lines or filled) for left
            if (SPEC->drawmode == 0)
            {
              // Dots
              if (frame != oldframe)
                XDrawPoint (display,win,gc,px,pyL);
            }
            else if (SPEC->drawmode == 1)
            {
              // Lines
              if (first)
                XDrawPoint (display,win,gc,px,pyL);
              else
                XDrawLine (display,win,gc,oldpx+1,oldpyL,px,pyL);
            }
            else
            {
              // Filled
              XDrawLine (display,win,gc,px,sampleCentreL,px,pyL);
            }

            if (rangeColour && (SPEC->edit == 2 || SPEC->edit == 1))
              XSetForeground (display,gc,fl_get_flcolor (FL_CYAN));
            else
              XSetForeground (display,gc,fl_get_flcolor (FL_BLACK));

            // Draw (dots, lines or filled) for right
            if (SPEC->drawmode == 0)
            {
              // Dots
              if (frame != oldframe)
                XDrawPoint (display,win,gc,px,pyR);
            }
            else if (SPEC->drawmode == 1)
            {
              // Lines
              if (first)
                XDrawPoint (display,win,gc,px,pyR);
              else
                XDrawLine (display,win,gc,oldpx+1,oldpyR,px,pyR);
            }
            else
            {
              // Filled
              XDrawLine (display,win,gc,px,sampleCentreR,px,pyR);
            }

            if (rangeColour && (SPEC->edit == 2 || SPEC->edit == 3))
              XSetForeground (display,gc,fl_get_flcolor (FL_CYAN));
            else
              XSetForeground (display,gc,fl_get_flcolor (FL_BLACK));

            // Draw (dots, lines or filled) for channel 3
            if (SPEC->drawmode == 0)
            {
              // Dots
              if (frame != oldframe)
                XDrawPoint (display,win,gc,px,py3);
            }
            else if (SPEC->drawmode == 1)
            {
              // Lines
              if (first)
                XDrawPoint (display,win,gc,px,py3);
              else
                XDrawLine (display,win,gc,oldpx+1,oldpy3,px,py3);
            }
            else
            {
              // Filled
              XDrawLine (display,win,gc,px,sampleCentre3,px,py3);
            }

            if (rangeColour && (SPEC->edit == 2 || SPEC->edit == 4))
              XSetForeground (display,gc,fl_get_flcolor (FL_CYAN));
            else
              XSetForeground (display,gc,fl_get_flcolor (FL_BLACK));

            // Draw (dots, lines or filled) for channel 4
            if (SPEC->drawmode == 0)
            {
              // Dots
              if (frame != oldframe)
                XDrawPoint (display,win,gc,px,py4);
            }
            else if (SPEC->drawmode == 1)
            {
              // Lines
              if (first)
              {
                XDrawPoint (display,win,gc,px,py4);
                first = FALSE;
              }
              else
              {
                XDrawLine (display,win,gc,oldpx+1,oldpy4,px,py4);
              }
            }
            else
            {
              // Filled
              XDrawLine (display,win,gc,px,sampleCentre4,px,py4);
            }

            oldframe = frame;
            oldpx    = px;
            oldpyL   = pyL;
            oldpyR   = pyR;
            oldpy3   = py3;
            oldpy4   = py4;
          }
        }
      }
      else
      {
        // Accurate method
        
        for (i=0; i<objWidth; i++)
        {
          long nextframe;
          long fLmin;
          long fLmax;
          long fRmin;
          long fRmax;
          long f3min;
          long f3max;
          long f4min;
          long f4max;
          long pyLmin;
          long pyLmax;
          long pyRmin;
          long pyRmax;
          long py3min;
          long py3max;
          long py4min;
          long py4max;
          
          // Calculate frame
          frame = TOFRAME (i);
          nextframe = TOFRAME (i+1);
          if (frame >= displayEnd) frame = displayEnd - 1;
          if (nextframe >= displayEnd) nextframe = displayEnd - 1;

          // Grid based locking
          if (grid > 1)
          {
            frame = frame - (frame % grid);
            nextframe = nextframe - (nextframe % grid);
          }

          // Set colour inversion if within range bounds
          if (sample->getRangeValid () && frame >= rangeStart
            && frame <= rangeEnd-1)
            rangeColour = 1;
          else
            rangeColour = 0;

          // Retrieve frame values
          fLmin = fLmax = sample24L = sample->getFrame24 (frame,0);
          for (long tempi=frame+1; tempi<nextframe; tempi++)
          {
            sample24L = sample->getFrame24 (tempi,0);
            if (sample24L < fLmin) fLmin = sample24L;
            else if (sample24L > fLmax) fLmax = sample24L;
          }

          // Calculate x and y coordinates
          px = objLeftX + i;
          pyLmin = (long) rint ((double) sampleBotL -
            ((double) fLmin + MAX23) * (sampleHeight - 1) / MAX24_1);
          pyLmax = (long) rint ((double) sampleBotL -
            ((double) fLmax + MAX23) * (sampleHeight - 1) / MAX24_1);

          if (stereo)
          {
            // Stereo

            fRmin = fRmax = sample24R = sample->getFrame24 (frame,1);
            for (long tempi=frame+1; tempi<nextframe; tempi++)
            {
              sample24R = sample->getFrame24 (tempi,1);
              if (sample24R < fRmin) fRmin = sample24R;
              else if (sample24R > fRmax) fRmax = sample24R;
            }

            pyRmin = (long) rint ((double) sampleBotR -
            ((double) fRmin + MAX23) * (sampleHeight - 1) / MAX24_1);
            pyRmax = (long) rint ((double) sampleBotR -
            ((double) fRmax + MAX23) * (sampleHeight - 1) / MAX24_1);

            if (rangeColour && SPEC->edit != 1)
              XSetForeground (display,gc,fl_get_flcolor (FL_CYAN));
            else
              XSetForeground (display,gc,fl_get_flcolor (FL_BLACK));

            // Draw (dots, lines or filled) for left
            if (SPEC->drawmode == 0)
            {
              // Dots
              if (frame != oldframe)
              {
                XDrawPoint (display,win,gc,px,pyLmin);
                XDrawPoint (display,win,gc,px,pyLmax);
              }
            }
            else if (SPEC->drawmode == 1)
            {
              // Lines
              XDrawLine (display,win,gc,px,pyLmin,px,pyLmax);
              if (!first)
              {
                if (grid < 1)
                {
                  XDrawLine (display,win,gc,oldpx+1,oldpyL,px,pyLmax);
                }
                else
                {
                  if (pyLmax < oldpyL)
                    XDrawLine (display,win,gc,oldpx+1,pyLmax,px,oldpyL);
                  else if (oldpyLmax < pyLmin)
                    XDrawLine (display,win,gc,oldpx+1,oldpyLmax,px,pyLmin);
                }
              }
            }
            else
            {
              // Filled
              XDrawLine (display,win,gc,px,sampleCentreL,px,pyLmin);
              XDrawLine (display,win,gc,px,sampleCentreL,px,pyLmax);
            }

            if (rangeColour && SPEC->edit != 0)
              XSetForeground (display,gc,fl_get_flcolor (FL_CYAN));
            else
              XSetForeground (display,gc,fl_get_flcolor (FL_BLACK));

            // Draw (dots, lines or filled) for right
            if (SPEC->drawmode == 0)
            {
              // Dots
              if (frame != oldframe)
              {
                XDrawPoint (display,win,gc,px,pyRmin);
                XDrawPoint (display,win,gc,px,pyRmax);
              }
            }
            else if (SPEC->drawmode == 1)
            {
              // Lines
              XDrawLine (display,win,gc,px,pyRmin,px,pyRmax);
              if (!first)
              {
                if (grid < 1)
                {
                  XDrawLine (display,win,gc,oldpx+1,oldpyR,px,pyRmax);
                }
                else
                {
                  if (pyRmax < oldpyR)
                    XDrawLine (display,win,gc,oldpx+1,pyRmax,px,oldpyR);
                  else if (oldpyRmax < pyRmin)
                    XDrawLine (display,win,gc,oldpx+1,oldpyRmax,px,pyRmin);
                }
              }
              else
              {
                first = FALSE;
              }
            }
            else
            {
              // Filled
              XDrawLine (display,win,gc,px,sampleCentreR,px,pyRmin);
              XDrawLine (display,win,gc,px,sampleCentreR,px,pyRmax);
            }

            oldframe  = frame;
            oldpx     = px;
            oldpyL    = pyLmin;
            oldpyR    = pyRmin;
            oldpyLmax = pyLmax;
            oldpyRmax = pyRmax;
          }
          else if (mono)
          {
            // Mono

            if (rangeColour)
              XSetForeground (display,gc,fl_get_flcolor (FL_CYAN));
            else
              XSetForeground (display,gc,fl_get_flcolor (FL_BLACK));

            // Draw (dots, lines or filled)
            if (SPEC->drawmode == 0)
            {
              // Dots
              if (frame != oldframe)
              {
                XDrawPoint (display,win,gc,px,pyLmin);
                XDrawPoint (display,win,gc,px,pyLmax);
              }
            }
            else if (SPEC->drawmode == 1)
            {
              // Lines
              XDrawLine (display,win,gc,px,pyLmin,px,pyLmax);
              if (!first)
              {
                if (grid < 1)
                {
                  XDrawLine (display,win,gc,oldpx+1,oldpyL,px,pyLmax);
                }
                else
                {
                  if (pyLmax < oldpyL)
                    XDrawLine (display,win,gc,oldpx+1,pyLmax,px,oldpyL);
                  else if (oldpyLmax < pyLmin)
                    XDrawLine (display,win,gc,oldpx+1,oldpyLmax,px,pyLmin);
                }
              }
              else
              {
                first = FALSE;
              }
            }
            else
            {
              // Filled
              XDrawLine (display,win,gc,px,sampleCentreL,px,pyLmin);
              XDrawLine (display,win,gc,px,sampleCentreL,px,pyLmax);
            }
            oldframe  = frame;
            oldpx     = px;
            oldpyL    = pyLmin;
            oldpyLmax = pyLmax;
          }
          else
          {
            // Quadro
            
            long tempi;

            fRmin = fRmax = sample24R = sample->getFrame24 (frame,1);
            for (tempi=frame+1; tempi<nextframe; tempi++)
            {
              sample24R = sample->getFrame24 (tempi,1);
              if (sample24R < fRmin) fRmin = sample24R;
              else if (sample24R > fRmax) fRmax = sample24R;
            }

            f3min = f3max = sample243 = sample->getFrame24 (frame,2);
            for (tempi=frame+1; tempi<nextframe; tempi++)
            {
              sample243 = sample->getFrame24 (tempi,2);
              if (sample243 < f3min) f3min = sample243;
              else if (sample243 > f3max) f3max = sample243;
            }

            f4min = f4max = sample244 = sample->getFrame24 (frame,3);
            for (tempi=frame+1; tempi<nextframe; tempi++)
            {
              sample244 = sample->getFrame24 (tempi,3);
              if (sample244 < f4min) f4min = sample244;
              else if (sample244 > f4max) f4max = sample244;
            }

            pyRmin = (long) rint ((double) sampleBotR -
            ((double) fRmin + MAX23) * (sampleHeight - 1) / MAX24_1);
            pyRmax = (long) rint ((double) sampleBotR -
            ((double) fRmax + MAX23) * (sampleHeight - 1) / MAX24_1);
            py3min = (long) rint ((double) sampleBot3 -
            ((double) f3min + MAX23) * (sampleHeight - 1) / MAX24_1);
            py3max = (long) rint ((double) sampleBot3 -
            ((double) f3max + MAX23) * (sampleHeight - 1) / MAX24_1);
            py4min = (long) rint ((double) sampleBot4 -
            ((double) f4min + MAX23) * (sampleHeight - 1) / MAX24_1);
            py4max = (long) rint ((double) sampleBot4 -
            ((double) f4max + MAX23) * (sampleHeight - 1) / MAX24_1);

            if (rangeColour && (SPEC->edit == 2 || SPEC->edit == 0))
              XSetForeground (display,gc,fl_get_flcolor (FL_CYAN));
            else
              XSetForeground (display,gc,fl_get_flcolor (FL_BLACK));

            // Draw (dots, lines or filled) for left
            if (SPEC->drawmode == 0)
            {
              // Dots
              if (frame != oldframe)
              {
                XDrawPoint (display,win,gc,px,pyLmin);
                XDrawPoint (display,win,gc,px,pyLmax);
              }
            }
            else if (SPEC->drawmode == 1)
            {
              // Lines
              XDrawLine (display,win,gc,px,pyLmin,px,pyLmax);
              if (!first)
              {
                if (grid < 1)
                {
                  XDrawLine (display,win,gc,oldpx+1,oldpyL,px,pyLmax);
                }
                else
                {
                  if (pyLmax < oldpyL)
                    XDrawLine (display,win,gc,oldpx+1,pyLmax,px,oldpyL);
                  else if (oldpyLmax < pyLmin)
                    XDrawLine (display,win,gc,oldpx+1,oldpyLmax,px,pyLmin);
                }
              }
            }
            else
            {
              // Filled
              XDrawLine (display,win,gc,px,sampleCentreL,px,pyLmin);
              XDrawLine (display,win,gc,px,sampleCentreL,px,pyLmax);
            }

            if (rangeColour && (SPEC->edit == 2 || SPEC->edit == 1))
              XSetForeground (display,gc,fl_get_flcolor (FL_CYAN));
            else
              XSetForeground (display,gc,fl_get_flcolor (FL_BLACK));

            // Draw (dots, lines or filled) for right
            if (SPEC->drawmode == 0)
            {
              // Dots
              if (frame != oldframe)
              {
                XDrawPoint (display,win,gc,px,pyRmin);
                XDrawPoint (display,win,gc,px,pyRmax);
              }
            }
            else if (SPEC->drawmode == 1)
            {
              // Lines
              XDrawLine (display,win,gc,px,pyRmin,px,pyRmax);
              if (!first)
              {
                if (grid < 1)
                {
                  XDrawLine (display,win,gc,oldpx+1,oldpyR,px,pyRmax);
                }
                else
                {
                  if (pyRmax < oldpyR)
                    XDrawLine (display,win,gc,oldpx+1,pyRmax,px,oldpyR);
                  else if (oldpyRmax < pyRmin)
                    XDrawLine (display,win,gc,oldpx+1,oldpyRmax,px,pyRmin);
                }
              }
            }
            else
            {
              // Filled
              XDrawLine (display,win,gc,px,sampleCentreR,px,pyRmin);
              XDrawLine (display,win,gc,px,sampleCentreR,px,pyRmax);
            }

            if (rangeColour && (SPEC->edit == 2 || SPEC->edit == 3))
              XSetForeground (display,gc,fl_get_flcolor (FL_CYAN));
            else
              XSetForeground (display,gc,fl_get_flcolor (FL_BLACK));

            // Draw (dots, lines or filled) for channel 3
            if (SPEC->drawmode == 0)
            {
              // Dots
              if (frame != oldframe)
              {
                XDrawPoint (display,win,gc,px,py3min);
                XDrawPoint (display,win,gc,px,py3max);
              }
            }
            else if (SPEC->drawmode == 1)
            {
              // Lines
              XDrawLine (display,win,gc,px,py3min,px,py3max);
              if (!first)
              {
                if (grid < 1)
                {
                  XDrawLine (display,win,gc,oldpx+1,oldpy3,px,py3max);
                }
                else
                {
                  if (py3max < oldpy3)
                    XDrawLine (display,win,gc,oldpx+1,py3max,px,oldpy3);
                  else if (oldpy3max < py3min)
                    XDrawLine (display,win,gc,oldpx+1,oldpy3max,px,py3min);
                }
              }
            }
            else
            {
              // Filled
              XDrawLine (display,win,gc,px,sampleCentre3,px,py3min);
              XDrawLine (display,win,gc,px,sampleCentre3,px,py3max);
            }

            if (rangeColour && (SPEC->edit == 2 || SPEC->edit == 4))
              XSetForeground (display,gc,fl_get_flcolor (FL_CYAN));
            else
              XSetForeground (display,gc,fl_get_flcolor (FL_BLACK));

            // Draw (dots, lines or filled) for channel 4
            if (SPEC->drawmode == 0)
            {
              // Dots
              if (frame != oldframe)
              {
                XDrawPoint (display,win,gc,px,py4min);
                XDrawPoint (display,win,gc,px,py4max);
              }
            }
            else if (SPEC->drawmode == 1)
            {
              // Lines
              XDrawLine (display,win,gc,px,py4min,px,py4max);
              if (!first)
              {
                if (grid < 1)
                {
                  XDrawLine (display,win,gc,oldpx+1,oldpy4,px,py4max);
                }
                else
                {
                  if (py4max < oldpy4)
                    XDrawLine (display,win,gc,oldpx+1,py4max,px,oldpy4);
                  else if (oldpy4max < py4min)
                    XDrawLine (display,win,gc,oldpx+1,oldpy4max,px,py4min);
                }
              }
              else
              {
                first = FALSE;
              }
            }
            else
            {
              // Filled
              XDrawLine (display,win,gc,px,sampleCentre4,px,py4min);
              XDrawLine (display,win,gc,px,sampleCentre4,px,py4max);
            }

            oldframe  = frame;
            oldpx     = px;
            oldpyL    = pyLmin;
            oldpyR    = pyRmin;
            oldpy3    = py3min;
            oldpy4    = py4min;
            oldpyLmax = pyLmax;
            oldpyRmax = pyRmax;
            oldpy3max = py3max;
            oldpy4max = py4max;
          }
        }
      }
      
      // Draw loop markers

      if (sample->getSusLoopMode () != AF_LOOP_MODE_NOLOOP)
      {
        long susLoopStart = sample->getSusLoopStart ();
        long susLoopEnd   = sample->getSusLoopEnd ();
        int loopStart = TOPIXELX (susLoopStart);
        int loopEnd   = TOPIXELX (susLoopEnd);
        int loopTop   = objTopY - 10;
        XSetForeground (display,gc,fl_get_flcolor (FL_RED));

        if (susLoopStart >= displayStart && susLoopStart <= displayEnd)
        {
          XDrawLine (display,win,gc,
            loopStart,loopTop,loopStart,objBotY);
          fl_rect (loopStart-9,loopTop,9,9,FL_RED);
          if (SPEC->grabSusStart)
            fl_rectf (loopStart-8,loopTop+1,8,8,FL_YELLOW);
        }

        if (susLoopEnd >= displayStart && susLoopEnd <= displayEnd)
        {
          XDrawLine (display,win,gc,
            loopEnd,loopTop,loopEnd,objBotY);
          fl_rect (loopEnd,loopTop,9,9,FL_RED);
          if (SPEC->grabSusEnd)
            fl_rectf (loopEnd+1,loopTop+1,8,8,FL_YELLOW);
        }
      }

      if (sample->getRelLoopMode () != AF_LOOP_MODE_NOLOOP)
      {
        long relLoopStart = sample->getRelLoopStart ();
        long relLoopEnd   = sample->getRelLoopEnd ();
        int loopStart = TOPIXELX (sample->getRelLoopStart ());
        int loopEnd   = TOPIXELX (sample->getRelLoopEnd ());
        int loopBot   = objBotY + 1;
        XSetForeground (display,gc,fl_get_flcolor (FL_GREEN));

        if (relLoopStart >= displayStart && relLoopStart <= displayEnd)
        {
          XDrawLine (display,win,gc,
            loopStart,objTopY,loopStart,objBotY+10);
          fl_rect (loopStart-9,loopBot,9,9,FL_GREEN);
          if (SPEC->grabRelStart)
            fl_rectf (loopStart-8,loopBot+1,8,8,FL_YELLOW);
        }

        if (relLoopEnd >= displayStart && relLoopEnd <= displayEnd)
        {
          XDrawLine (display,win,gc,
            loopEnd,objTopY,loopEnd,objBotY+10);
          fl_rect (loopEnd,loopBot,9,9,FL_GREEN);
          if (SPEC->grabRelEnd)
            fl_rectf (loopEnd+1,loopBot+1,8,8,FL_YELLOW);
        }
      }
      
      // Draw current position line
      if (sample->getRedraw () &&
        currentPos >= displayStart && currentPos <= displayEnd)
      {
        XSetForeground (display,gc,fl_get_flcolor (FL_YELLOW));
        *oldPosx = TOPIXELX (currentPos);
        XSetFunction (display,gc,GXxor);
        XDrawLine (display,win,gc,*oldPosx,objTopY,*oldPosx,objBotY);
        XSetFunction (display,gc,GXcopy);
      }
      
      return 0;
    }
      
    case FL_PUSH:
    {
      long displayStart    = sample->getDisplayStart ();
      long displayEnd      = sample->getDisplayEnd ();
      long displayDuration = displayEnd - displayStart;
      int  objLeftX        = ob->x + BORDERX;
      int  objWidth        = ob->w-(2*BORDERX);
      int  objTopY         = ob->y + BORDERY;
      int  objBotY         = ob->y + ob->h - 1 - BORDERY;      
      long frame;
      int  mono;
      int  stereo;
      int  quadro;
      long val;

      if (!sample->getValid () || displayDuration == 0) return 0;

      if (sample->getSusLoopMode () != AF_LOOP_MODE_NOLOOP
        && my < objTopY)
      {
        long loopStart = sample->getSusLoopStart ();
        long loopEnd   = sample->getSusLoopEnd ();
        int pixelStart = TOPIXELX (loopStart) + 1;
        int pixelEnd   = TOPIXELX (loopEnd) + 1;
        if (mx >= pixelStart-9 && mx <= pixelStart)
          SPEC->grabSusStart = pixelStart - mx + 1;
        else if (mx >= pixelEnd && mx <= pixelEnd+9)
          SPEC->grabSusEnd = mx - pixelEnd + 1;
        fl_redraw_object (ob);
        return 0;
      }

      if (sample->getRelLoopMode () != AF_LOOP_MODE_NOLOOP
        && my > objBotY)
      {
        long loopStart = sample->getRelLoopStart ();
        long loopEnd   = sample->getRelLoopEnd ();
        int pixelStart = TOPIXELX (loopStart) + 1;
        int pixelEnd   = TOPIXELX (loopEnd) + 1;
        if (mx >= pixelStart-9 && mx <= pixelStart)
          SPEC->grabRelStart = pixelStart - mx + 1;
        else if (mx >= pixelEnd && mx <= pixelEnd+9)
          SPEC->grabRelEnd = mx - pixelEnd + 1;
        fl_redraw_object (ob);
        return 0;
      }

      if (my >= objTopY && my <= objBotY)
      {
        if (SPEC->freehand)
        {
          if (displayDuration > objWidth)
          {
            fl_set_button (mainForm->manualEdit,0);
            SPEC->freehand = FALSE;
          }
          else
          {
            int  sampleTopL;
            int  sampleBotL;
            int  sampleTopR;
            int  sampleBotR;
            int  sampleTop3;
            int  sampleBot3;
            int  sampleTop4;
            int  sampleBot4;
            int  sampleHeight;
            
            mono   = sample->getChannels () == 1;
            stereo = sample->getChannels () == 2;
            quadro = sample->getChannels () == 4;
            frame  = TOFRAME (mx - objLeftX);
            if (mono)
            {
              if (SPEC->axis)
              {
                sampleTopL   = objTopY;
                sampleBotL   = objBotY - AXISSPACE - MONOGAP;
                sampleHeight = sampleBotL - sampleTopL;
              }
              else
              {
                sampleTopL   = objTopY;
                sampleBotL   = objBotY;
                sampleHeight = sampleBotL - sampleTopL;
              }
            }
            else if (stereo)
            {
              if (SPEC->axis)
              {
                sampleHeight = (objBotY - objTopY - AXISSPACE) / 2;
                sampleTopL   = objTopY;
                sampleBotL   = sampleTopL + sampleHeight;
                sampleBotR   = objBotY;
                sampleTopR   = sampleBotR - sampleHeight;
              }
              else
              {
                sampleHeight = (objBotY - objTopY - STEREOGAP) / 2;
                sampleTopL   = objTopY;
                sampleBotL   = sampleTopL + sampleHeight;
                sampleBotR   = objBotY;
                sampleTopR   = sampleBotR - sampleHeight;
              }
            }
            else
            {
              if (SPEC->axis)
              {
                sampleHeight =
                  (objBotY - objTopY - AXISSPACE - 2 * QUADROGAP) / 4;
                sampleTopL   = objTopY;
                sampleBotL   = objTopY + sampleHeight;
                sampleTopR   = sampleBotL + QUADROGAP;
                sampleBotR   = sampleTopR + sampleHeight;
                sampleBot4   = objBotY;
                sampleTop4   = sampleBot4 - sampleHeight;
                sampleBot3   = sampleTop4 - QUADROGAP;
                sampleTop3   = sampleBot3 - sampleHeight;
              }
              else
              {
                sampleHeight = (objBotY - objTopY - 3 * QUADROGAP) / 4;
                sampleTopL   = objTopY;
                sampleBotL   = objTopY + sampleHeight;
                sampleTopR   = sampleBotL + QUADROGAP;
                sampleBotR   = sampleTopR + sampleHeight;
                sampleBot4   = objBotY;
                sampleTop4   = sampleBot4 - sampleHeight;
                sampleBot3   = sampleTop4 - QUADROGAP;
                sampleTop3   = sampleBot3 - sampleHeight;
              }
            }
            
            if (my >= sampleTopL && my <= sampleBotL)
            {
              val = (long) rint ((sampleBotL - my) * (double) MAX24_1 /
              (sampleHeight - 1) - MAX23);
              if (val < -MAX23) val = -MAX23;
              else if (val > MAX23_1) val = MAX23_1;
              sample->setFrame24 (frame,0,val);
              sample->setChanged (TRUE);
              SPEC->oldframe = frame;
              SPEC->oldval   = val;
              SPEC->oldchan  = 0;
            }
            else if ((stereo || quadro) &&
              (my >= sampleTopR && my <= sampleBotR))
            {
              val = (long) rint ((sampleBotR - my) * (double) MAX24_1 /
              (sampleHeight - 1) - MAX23);
              if (val < -MAX23) val = -MAX23;
              else if (val > MAX23_1) val = MAX23_1;
              sample->setFrame24 (frame,1,val);
              sample->setChanged (TRUE);
              SPEC->oldframe = frame;
              SPEC->oldval   = val;
              SPEC->oldchan  = 1;
            }
            else if (quadro && (my >= sampleTop3 && my <= sampleBot3))
            {
              val = (long) rint ((sampleBot3 - my) * (double) MAX24_1 /
              (sampleHeight - 1) - MAX23);
              if (val < -MAX23) val = -MAX23;
              else if (val > MAX23_1) val = MAX23_1;
              sample->setFrame24 (frame,2,val);
              sample->setChanged (TRUE);
              SPEC->oldframe = frame;
              SPEC->oldval   = val;
              SPEC->oldchan  = 2;
            }
            else if (quadro && (my >= sampleTop4 && my <= sampleBot4))
            {
              val = (long) rint ((sampleBot4 - my) * (double) MAX24_1 /
              (sampleHeight - 1) - MAX23);
              if (val < -MAX23) val = -MAX23;
              else if (val > MAX23_1) val = MAX23_1;
              sample->setFrame24 (frame,3,val);
              sample->setChanged (TRUE);
              SPEC->oldframe = frame;
              SPEC->oldval   = val;
              SPEC->oldchan  = 3;
            }
            
            fl_redraw_object (ob);
            return 0;
          }
        }
        
        *rangeMark = TOFRAMEROUND (mx - objLeftX);
        if (*rangeMark < 0) *rangeMark = 0;
        else if (*rangeMark > displayEnd) *rangeMark = displayEnd;
        
        if (SPEC->edit != 2 && 
          SPEC->autoeditmode && sample->getChannels () != 1)
        {
          int  sampleTopL;
          int  sampleBotL;
          int  sampleTopR;
          int  sampleBotR;
          int  sampleTop3;
          int  sampleBot3;
          int  sampleTop4;
          int  sampleBot4;
          int  sampleHeight;

          stereo = sample->getChannels () == 2;
          quadro = sample->getChannels () == 4;

          if (stereo)
          {
            if (SPEC->axis)
            {
              sampleHeight = (objBotY - objTopY - AXISSPACE) / 2;
              sampleTopL   = objTopY;
              sampleBotL   = sampleTopL + sampleHeight;
              sampleBotR   = objBotY;
              sampleTopR   = sampleBotR - sampleHeight;
            }
            else
            {
              sampleHeight = (objBotY - objTopY - STEREOGAP) / 2;
              sampleTopL   = objTopY;
              sampleBotL   = sampleTopL + sampleHeight;
              sampleBotR   = objBotY;
              sampleTopR   = sampleBotR - sampleHeight;
            }
          }
          else
          {
            if (SPEC->axis)
            {
              sampleHeight =
                (objBotY - objTopY - AXISSPACE - 2 * QUADROGAP) / 4;
              sampleTopL   = objTopY;
              sampleBotL   = objTopY + sampleHeight;
              sampleTopR   = sampleBotL + QUADROGAP;
              sampleBotR   = sampleTopR + sampleHeight;
              sampleBot4   = objBotY;
              sampleTop4   = sampleBot4 - sampleHeight;
              sampleBot3   = sampleTop4 - QUADROGAP;
              sampleTop3   = sampleBot3 - sampleHeight;
            }
            else
            {
              sampleHeight = (objBotY - objTopY - 3 * QUADROGAP) / 4;
              sampleTopL   = objTopY;
              sampleBotL   = objTopY + sampleHeight;
              sampleTopR   = sampleBotL + QUADROGAP;
              sampleBotR   = sampleTopR + sampleHeight;
              sampleBot4   = objBotY;
              sampleTop4   = sampleBot4 - sampleHeight;
              sampleBot3   = sampleTop4 - QUADROGAP;
              sampleTop3   = sampleBot3 - sampleHeight;
            }
          }

          if (stereo)
          {
            if (my >= sampleTopL && my <= sampleBotL)
            {
              fl_set_choice (mainForm->editMode,1);
              SPEC->edit = 0;
            }
            else if (my >= sampleTopR && my <= sampleBotR)
            {
              fl_set_choice (mainForm->editMode,2);
              SPEC->edit = 1;
            }
          }
          else
          {
            if (my >= sampleTopL && my <= sampleBotL)
            {
              fl_set_choice (mainForm->editMode,1);
              SPEC->edit = 0;
            }
            else if (my >= sampleTopR && my <= sampleBotR)
            {
              fl_set_choice (mainForm->editMode,2);
              SPEC->edit = 1;
            }
            else if (my >= sampleTop3 && my <= sampleBot3)
            {
              fl_set_choice (mainForm->editMode,3);
              SPEC->edit = 3;
            }
            else if (my >= sampleTop4 && my <= sampleBot4)
            {
              fl_set_choice (mainForm->editMode,4);
              SPEC->edit = 4;
            }
          }
        }

        if (SPEC->rangealt)
        {
          if (key == 1)
          {
            // Dragging start - as if end was first point placed
            // Hence setting of rangeMark

            SPEC->rangeGrab = 1;
            if (!sample->getRangeValid ())
            {
              sample->setRange (*rangeMark,sample->getFrames ());
              *rangeMark = sample->getRangeEnd ();
            }
            else if (*rangeMark > sample->getRangeEnd ())
            {
              if (SPEC->rangecross)
              {
                sample->setRangeStart (*rangeMark);
                *rangeMark = sample->getRangeStart (); // Crossed over
              }
              else
              {
                sample->setRangeStart (sample->getRangeEnd ());
                *rangeMark = sample->getRangeEnd ();
              }
            }
            else
            {
              sample->setRangeStart (*rangeMark);
              *rangeMark = sample->getRangeEnd ();
            }
          }
          else if (key == 3)
          {
            // Dragging end - as if start was first point placed
            // Hence setting of rangeMark

            SPEC->rangeGrab = 2;
            if (!sample->getRangeValid ())
            {
              sample->setRange (0,*rangeMark);
              *rangeMark = sample->getRangeStart ();
            }
            else if (*rangeMark < sample->getRangeStart ())
            {
              if (SPEC->rangecross)
              {
                sample->setRangeEnd (*rangeMark);
                *rangeMark = sample->getRangeEnd (); // Crossed over
              }
              else
              {
                sample->setRangeEnd (sample->getRangeStart ());
                *rangeMark = sample->getRangeStart ();
              }
            }
            else
            {
              sample->setRangeEnd (*rangeMark);
              *rangeMark = sample->getRangeStart ();
            }
          }
        }
        else
        {
          if (key == 1)
          {
            SPEC->rangeGrab = 0;
            sample->setRange (*rangeMark,*rangeMark);
          }
          else if (key == 3)
          {
            if (!sample->getRangeValid ())
            {
              SPEC->rangeGrab = 0;
              sample->setRange (*rangeMark,*rangeMark);
            }
            else
            {
              long curRangeStart = sample->getRangeStart ();
              long curRangeEnd   = sample->getRangeEnd ();
              
              // Catch special case
              if (curRangeStart == curRangeEnd &&
                curRangeStart == *rangeMark)
              {
                SPEC->rangeGrab = 0;
                sample->setRange (*rangeMark,*rangeMark);
              }
              else if (*rangeMark <= ((curRangeStart + curRangeEnd) / 2))
              {
                SPEC->rangeGrab = 1;
                sample->setRangeStart (*rangeMark);
                
                // Dragging start - as if end was first placed
                *rangeMark = sample->getRangeEnd ();
              }
              else
              {
                SPEC->rangeGrab = 2;
                sample->setRangeEnd (*rangeMark);
                
                // Dragging end - as if start was first placed
                *rangeMark = sample->getRangeStart ();
              }
            }
          }
        }
        
        updateRangeDetails ();
        fl_redraw_object (ob);
        return 0;
      }
      return 0;
    }
      
    case FL_MOUSE:
    {
      long displayStart    = sample->getDisplayStart ();
      long displayEnd      = sample->getDisplayEnd ();
      long displayDuration = displayEnd - displayStart;
      
      int  objLeftX   = ob->x + BORDERX;
      int  objRightX  = ob->x + ob->w - 1 - BORDERX;
      int  objTopY    = ob->y + BORDERY;
      int  objBotY    = ob->y + ob->h - 1 - BORDERY;      
      int  objWidth   = ob->w-(2*BORDERX);
      long frame;
      int  mono;
      int  stereo;
      int  quadro;
      int  mxx;
      long val;
      
      if (!sample->getValid () || displayDuration == 0) return 0;
      
      if (SPEC->grabSusStart)
        mxx = mx - objLeftX + SPEC->grabSusStart - 1;
      else if (SPEC->grabSusEnd)
        mxx = mx - objLeftX - SPEC->grabSusEnd + 1;
      else if (SPEC->grabRelStart)
        mxx = mx - objLeftX + SPEC->grabRelStart - 1;
      else if (SPEC->grabRelEnd)
        mxx = mx - objLeftX - SPEC->grabRelEnd + 1;
      else if (SPEC->rangeMark != -1)
        mxx = mx - objLeftX;
      else if (SPEC->freehand)
        mxx = mx - objLeftX;
      else return 0;

      if (mxx < - BORDERX / 2)
      {
        sample->scrollL (displayDuration / 30 + 1);
        frame = sample->getDisplayStart ();
        updateDisplayDetails ();
        fl_redraw_object (mainForm->scrollBarSlider);
      }
      else if (mxx < 0)
      {
        sample->scrollL (displayDuration / 200 + 1);
        frame = sample->getDisplayStart ();
        updateDisplayDetails ();
        fl_redraw_object (mainForm->scrollBarSlider);
      }
      else if (mxx > objWidth + BORDERX / 2)
      {
        sample->scrollR (displayDuration / 30 + 1);
        frame = sample->getDisplayEnd ();
        updateDisplayDetails ();
        fl_redraw_object (mainForm->scrollBarSlider);
      }
      else if (mxx >= objWidth)
      {
        sample->scrollR (displayDuration / 200 + 1);
        frame = sample->getDisplayEnd ();
        updateDisplayDetails ();
        fl_redraw_object (mainForm->scrollBarSlider);
      }
      else
      {
        if (SPEC->grabSusStart || SPEC->grabSusEnd ||
          SPEC->grabRelStart || SPEC->grabRelEnd)
          frame = TOFRAMEROUND (mxx);
        else if (SPEC->freehand)
          frame = TOFRAME (mxx);
        else
          frame = TOFRAMEROUND (mxx);
      }
        
      char tempString [50];
      sprintf (tempString,"%ld",frame);
      fl_set_object_label (displayForm->positionText,tempString);
      sprintf (tempString,"%.3lf",frame / sample->getRate ());
      fl_set_object_label (displayForm->timeText,tempString);
      
      if (SPEC->grabSusStart)
      {
        sample->setSusLoopStart (frame);
        updateLoopDetails ();
        fl_redraw_object (ob);
        return 0;
      }

      if (SPEC->grabSusEnd)
      {
        sample->setSusLoopEnd (frame);
        updateLoopDetails ();
        fl_redraw_object (ob);
        return 0;
      }

      if (SPEC->grabRelStart)
      {
        sample->setRelLoopStart (frame);
        updateLoopDetails ();
        fl_redraw_object (ob);
        return 0;
      }

      if (SPEC->grabRelEnd)
      {
        sample->setRelLoopEnd (frame);
        updateLoopDetails ();
        fl_redraw_object (ob);
        return 0;
      }

      if (SPEC->freehand)
      {
        if (displayDuration > objWidth)
        {
          fl_set_button (mainForm->manualEdit,0);
          SPEC->freehand = FALSE;
        }
        else
        {
          int  sampleTopL;
          int  sampleBotL;
          int  sampleTopR;
          int  sampleBotR;
          int  sampleTop3;
          int  sampleBot3;
          int  sampleTop4;
          int  sampleBot4;
          int  sampleHeight;

          mono   = sample->getChannels () == 1;
          stereo = sample->getChannels () == 2;
          quadro = sample->getChannels () == 4;
          if (mono)
          {
            if (SPEC->axis)
            {
              sampleTopL   = objTopY;
              sampleBotL   = objBotY - AXISSPACE - MONOGAP;
              sampleHeight = sampleBotL - sampleTopL;
            }
            else
            {
              sampleTopL   = objTopY;
              sampleBotL   = objBotY;
              sampleHeight = sampleBotL - sampleTopL;
            }
          }
          else if (stereo)
          {
            if (SPEC->axis)
            {
              sampleHeight = (objBotY - objTopY - AXISSPACE) / 2;
              sampleTopL   = objTopY;
              sampleBotL   = sampleTopL + sampleHeight;
              sampleBotR   = objBotY;
              sampleTopR   = sampleBotR - sampleHeight;
            }
            else
            {
              sampleHeight = (objBotY - objTopY - STEREOGAP) / 2;
              sampleTopL   = objTopY;
              sampleBotL   = sampleTopL + sampleHeight;
              sampleBotR   = objBotY;
              sampleTopR   = sampleBotR - sampleHeight;
            }
          }
          else
          {
            if (SPEC->axis)
            {
              sampleHeight =
                (objBotY - objTopY - AXISSPACE - 2 * QUADROGAP) / 4;
              sampleTopL   = objTopY;
              sampleBotL   = objTopY + sampleHeight;
              sampleTopR   = sampleBotL + QUADROGAP;
              sampleBotR   = sampleTopR + sampleHeight;
              sampleBot4   = objBotY;
              sampleTop4   = sampleBot4 - sampleHeight;
              sampleBot3   = sampleTop4 - QUADROGAP;
              sampleTop3   = sampleBot3 - sampleHeight;
            }
            else
            {
              sampleHeight = (objBotY - objTopY - 3 * QUADROGAP) / 4;
              sampleTopL   = objTopY;
              sampleBotL   = objTopY + sampleHeight;
              sampleTopR   = sampleBotL + QUADROGAP;
              sampleBotR   = sampleTopR + sampleHeight;
              sampleBot4   = objBotY;
              sampleTop4   = sampleBot4 - sampleHeight;
              sampleBot3   = sampleTop4 - QUADROGAP;
              sampleTop3   = sampleBot3 - sampleHeight;
            }
          }

          if (SPEC->oldframe != -1 && SPEC->oldchan == 0)
          {
            if (my < sampleTopL)
              val = MAX23-1;
            else if (my > sampleBotL)
              val = -MAX23;
            else
              val = (long) rint ((sampleBotL - my) * (double) MAX24_1 /
              (sampleHeight - 1) - MAX23);
            if (val < -MAX23) val = -MAX23;
            else if (val > MAX23_1) val = MAX23_1;
            sample->setFrame24 (frame,0,val);
            sample->setChanged (TRUE);
            if (SPEC->oldframe != -1 && SPEC->oldframe != frame
            && SPEC->oldchan == 0)
            {
              long i;
              long j;
              long l;
              double x;
              double y;
              if (frame > SPEC->oldframe)
              {
                l = frame - SPEC->oldframe;
                for (i=SPEC->oldframe+1,j=1; i<frame; i++,j++)
                {
                  x = j / l;
                  y = 1.0 - x;
                  sample->setFrame24 (i,0,(long) (x*val + y*SPEC->oldval));
                  sample->setChanged (TRUE);
                }
              }
              else
              {
                l = SPEC->oldframe - frame;
                for (i=frame+1,j=1; i<SPEC->oldframe; i++,j++)
                {
                  x = j / l;
                  y = 1.0 - x;
                  sample->setFrame24 (i,0,(long) (x*SPEC->oldval + y*val));
                  sample->setChanged (TRUE);
                }
              }
            }
            SPEC->oldframe = frame;
            SPEC->oldval   = val;
            SPEC->oldchan  = 0;
          }
          else if ((stereo || quadro) &&
            (SPEC->oldframe != -1 && SPEC->oldchan == 1))
          {
            if (my < sampleTopR)
              val = MAX23-1;
            else if (my > sampleBotR)
              val = -MAX23;
            else
              val = (long) rint ((sampleBotR - my) * (double) MAX24_1 /
              (sampleHeight - 1) - MAX23);
            if (val < -MAX23) val = -MAX23;
            else if (val > MAX23_1) val = MAX23_1;
            sample->setFrame24 (frame,1,val);
            sample->setChanged (TRUE);
            if (SPEC->oldframe != -1 && SPEC->oldframe != frame
            && SPEC->oldchan == 1)
            {
              long i;
              long j;
              long l;
              double x;
              double y;
              if (frame > SPEC->oldframe)
              {
                l = frame - SPEC->oldframe;
                for (i=SPEC->oldframe+1,j=1; i<frame; i++,j++)
                {
                  x = j / l;
                  y = 1.0 - x;
                  sample->setFrame24 (i,1,(long) (x*val + y*SPEC->oldval));
                  sample->setChanged (TRUE);
                }
              }
              else
              {
                l = SPEC->oldframe - frame;
                for (i=frame+1,j=1; i<SPEC->oldframe; i++,j++)
                {
                  x = j / l;
                  y = 1.0 - x;
                  sample->setFrame24 (i,1,(long) (x*SPEC->oldval + y*val));
                  sample->setChanged (TRUE);
                }
              }
            }
            SPEC->oldframe = frame;
            SPEC->oldval   = val;
            SPEC->oldchan  = 1;
          }
          else if (quadro &&
            (SPEC->oldframe != -1 && SPEC->oldchan == 2))
          {
            if (my < sampleTop3)
              val = MAX23-1;
            else if (my > sampleBot3)
              val = -MAX23;
            else
              val = (long) rint ((sampleBot3 - my) * (double) MAX24_1 /
              (sampleHeight - 1) - MAX23);
            if (val < -MAX23) val = -MAX23;
            else if (val > MAX23_1) val = MAX23_1;
            sample->setFrame24 (frame,2,val);
            sample->setChanged (TRUE);
            if (SPEC->oldframe != -1 && SPEC->oldframe != frame
            && SPEC->oldchan == 2)
            {
              long i;
              long j;
              long l;
              double x;
              double y;
              if (frame > SPEC->oldframe)
              {
                l = frame - SPEC->oldframe;
                for (i=SPEC->oldframe+1,j=1; i<frame; i++,j++)
                {
                  x = j / l;
                  y = 1.0 - x;
                  sample->setFrame24 (i,2,(long) (x*val + y*SPEC->oldval));
                  sample->setChanged (TRUE);
                }
              }
              else
              {
                l = SPEC->oldframe - frame;
                for (i=frame+1,j=1; i<SPEC->oldframe; i++,j++)
                {
                  x = j / l;
                  y = 1.0 - x;
                  sample->setFrame24 (i,2,(long) (x*SPEC->oldval + y*val));
                  sample->setChanged (TRUE);
                }
              }
            }
            SPEC->oldframe = frame;
            SPEC->oldval   = val;
            SPEC->oldchan  = 2;
          }
          else if (quadro &&
            (SPEC->oldframe != -1 && SPEC->oldchan == 3))
          {
            if (my < sampleTop4)
              val = MAX23-1;
            else if (my > sampleBot4)
              val = -MAX23;
            else
              val = (long) rint ((sampleBot4 - my) * (double) MAX24_1 /
              (sampleHeight - 1) - MAX23);
            if (val < -MAX23) val = -MAX23;
            else if (val > MAX23_1) val = MAX23_1;
            sample->setFrame24 (frame,3,val);
            sample->setChanged (TRUE);
            if (SPEC->oldframe != -1 && SPEC->oldframe != frame
            && SPEC->oldchan == 3)
            {
              long i;
              long j;
              long l;
              double x;
              double y;
              if (frame > SPEC->oldframe)
              {
                l = frame - SPEC->oldframe;
                for (i=SPEC->oldframe+1,j=1; i<frame; i++,j++)
                {
                  x = j / l;
                  y = 1.0 - x;
                  sample->setFrame24 (i,3,(long) (x*val + y*SPEC->oldval));
                  sample->setChanged (TRUE);
                }
              }
              else
              {
                l = SPEC->oldframe - frame;
                for (i=frame+1,j=1; i<SPEC->oldframe; i++,j++)
                {
                  x = j / l;
                  y = 1.0 - x;
                  sample->setFrame24 (i,3,(long) (x*SPEC->oldval + y*val));
                  sample->setChanged (TRUE);
                }
              }
            }
            SPEC->oldframe = frame;
            SPEC->oldval   = val;
            SPEC->oldchan  = 3;
          }
          else
          {
            SPEC->oldframe = -1;
            SPEC->oldval   = 0;
            SPEC->oldchan  = 0;
          }

          fl_redraw_object (ob);
          return 0;
        }
      }
          
      if (SPEC->rangeMark != -1)
      {
        if (SPEC->rangeGrab == 1)
        {
          if (frame >= *rangeMark)
          {
            if (SPEC->rangecross)
            {
              if (frame >= *rangeMark)
                sample->setRange (*rangeMark,frame);
              else
                sample->setRange (frame,*rangeMark);
            }
            else
              sample->setRangeStart (sample->getRangeEnd ());
          }
          else
          {
            sample->setRange (frame,*rangeMark);
          }
        }
        else if (SPEC->rangeGrab == 2)
        {
          if (frame <= *rangeMark)
          {
            if (SPEC->rangecross)
            {
              if (frame >= *rangeMark)
                sample->setRange (*rangeMark,frame);
              else
                sample->setRange (frame,*rangeMark);
            }
            else
              sample->setRangeEnd (sample->getRangeStart ());
          }
          else
          {
            sample->setRange (*rangeMark,frame);
          }
        }
        else
        {
          if (frame >= *rangeMark)
            sample->setRange (*rangeMark,frame);
          else
            sample->setRange (frame,*rangeMark);
        }

        updateRangeDetails ();
        fl_redraw_object (ob);
        return 0;
      }
      return 0;
    }
    
    case FL_RELEASE:
    {
      SPEC->rangeMark    = -1;
      SPEC->rangeGrab    = 0;
      SPEC->grabSusStart = 0;
      SPEC->grabSusEnd   = 0;
      SPEC->grabRelStart = 0;
      SPEC->grabRelEnd   = 0;
      SPEC->oldframe     = -1;
      SPEC->oldval       = 0;
      SPEC->oldchan      = 0;
      fl_redraw_object (ob);
      return 0;
    }
    
    case FL_ENTER :
    case FL_MOTION :
    {
      long displayStart    = sample->getDisplayStart ();
      long displayEnd      = sample->getDisplayEnd ();
      long displayDuration = displayEnd - displayStart;

      int objLeftX   = ob->x + BORDERX;
      int objRightX  = ob->x + ob->w - 1 - BORDERX;
      int objTopY    = ob->y + BORDERY;
      int objBotY    = ob->y + ob->h - 1 - BORDERY;      
      int objWidth   = ob->w-(2*BORDERX);
      long frame;
      
      if (!sample->getValid () || displayDuration == 0) return 0;
      
      if (mx < objLeftX || mx > objRightX
        || my < objTopY || my > objBotY)
      {
        fl_set_object_label (displayForm->positionText,"");
        fl_set_object_label (displayForm->timeText,"");
       return 0;
      }

      frame = TOFRAME (mx - objLeftX);
      char tempString [50];
      sprintf (tempString,"%ld",frame);
      fl_set_object_label (displayForm->positionText,tempString);
      sprintf (tempString,"%.3lf",frame / sample->getRate ());
      fl_set_object_label (displayForm->timeText,tempString);
      return 0;
    }
    
    case FL_LEAVE :
    {
      fl_set_object_label (displayForm->positionText,"");
      fl_set_object_label (displayForm->timeText,"");
      return 0;
    }
    
    case FL_FREEMEM:
    {
      fl_free ((sampleSpec *)ob->spec);
      return 0;
    }
  }
  return 0;
}

/***************************************************************************/
