/* GtkEditor - a source editor widget for GTK
 * Copyright (C) 1998 Thomas Mailund.
 *
 * The editor widget was written by Thomas Mailund, so bugs should be
 * reported to <mailund@daimi.au.dk>, not the gtk ppl.
 *
 * This library is free software; you can redistribute it and/or
 * modify it under the terms of the GNU Library General Public
 * License as published by the Free Software Foundation; either
 * version 2 of the License, or (at your option) any later version.
 *
 * This library is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 * Library General Public License for more details.
 *
 * You should have received a copy of the GNU Library General Public
 * License along with this library; if not, write to the
 * Free Software Foundation, Inc., 59 Temple Place - Suite 330,
 * Boston, MA 02111-1307, USA.
 */

#include <vdkb/syntaxtable.h>
#include <vdkb/internal.h>
#include <vdkb/gtkeditor-regex.h>

#define INIT_BUFFER_SIZE 512	/* this should do for most cases...I think */

/* --<creation, destruction and manipulation>------------------------------- */
/* new_stable -- inserts the entries in the 'entries' list into a
 * freshly allocated syntax table.  Do *not* touch the data in the
 * entries after calling this function, unless you really want to!
 */
GtkEditorSyntaxTable*
_gtk_editor_new_stable (GList *entries)
{
  GtkEditorSyntaxTable *stable = g_new (GtkEditorSyntaxTable, 1);
  int i;

  int size, used;
  char *all_start;
  int len;

  if (!entries)
    return NULL;		/* what else? */

  /* first we get number of entries, and malloc the array */
  stable->size = g_list_length (entries);
  stable->entries = g_new (GtkEditorSTEntry*, stable->size);

  /* now we build all_start and initialise the array */
  all_start = g_new (char, INIT_BUFFER_SIZE);
  size = INIT_BUFFER_SIZE;
  used = 0;

  for (i = 0; entries; entries = entries->next, i++) {
    stable->entries[i] = entries->data;
    stable->entries[i]->refcount++;

    len = strlen (stable->entries[i]->start);

    /* do we have room for more? */
    if ( ((len+strlen ("\\|")) > (size - used)) ) {
      size *= 2;
      g_realloc (all_start, size);
    }

    strcpy (all_start+used, stable->entries[i]->start);
    used += len;

    strcpy (all_start+used, "\\|");
    used += strlen ("\\|");

  }

  /* remove last "\\|" */
  all_start[used-strlen ("\\|")] = '\0';

  /* now compile the regex and we're done */
  if (!_gtk_editor_compile_regex (all_start, &stable->regex)) {
    /* critical error...we tested this a long time ago! */
    g_error ("all start fucked up!!!");
  }

  return stable;
}

/* gtk_editor_stentry_new -- creates new stentry for insertion into a
 * stable.  Designed for for easy use with install_stable. */
GList*
gtk_editor_stentry_new (const gchar *name, const gchar *start,
			const gchar *end, gboolean nest,
			GdkFont *font, const GdkColor *fore,
			const GdkColor  *back, GList *next)
{
  GList *tmp = g_list_alloc ();
  GtkEditorSTEntry *new = g_new (GtkEditorSTEntry, 1);

  /* we first try to compile the patterns...if it doesn't work
   * there's no need to do anything else. */
  if (!_gtk_editor_compile_regex (start, &new->start_regex)) {
    /* FIXME: better error msg */
    g_warning ("couldn't compile %s", start);
    g_free (new);
    return next;
  }
  if (!_gtk_editor_compile_regex (end, &new->end_regex)) {
    /* FIXME: better error msg */
    g_warning ("couldn't compile %s", end);
    g_free (new);
    return next;
  }

  new->name = g_strdup (name);
  new->start = g_strdup (start);
  new->end = g_strdup (end);
  new->esc = NULL;
  new->nest = nest;
  new->font = _gtk_editor_fontdup (font);
  new->fore = _gtk_editor_coldup (fore);
  new->back = _gtk_editor_coldup (back);
  new->refcount = 1;		/* the list is refering to this one. */

  tmp->data = (gpointer)new;
  tmp->next = next;

  return tmp;
}

/* gtk_editor_stentry_new -- creates new stentry for insertion into a
 * stable.  Designed for for easy use with install_stable. */
GList*
gtk_editor_stentry_new_with_esc (const gchar *name, const gchar *start,
				 const gchar *end, gboolean nest,
				 GdkFont *font, const GdkColor *fore,
				 const GdkColor  *back,
				 const gchar *esc,
				 GList *next)
{
  GList *tmp = g_list_alloc ();
  GtkEditorSTEntry *new = g_new (GtkEditorSTEntry, 1);

  /* we first try to compile the patterns...if it doesn't work
   * there's no need to do anything else. */
  if (!_gtk_editor_compile_regex (start, &new->start_regex)) {
    /* FIXME: better error msg */
    g_warning ("couldn't compile %s", start);
    g_free (new);
    return next;
  }

  if (!_gtk_editor_compile_regex (end, &new->end_regex)) {
    /* FIXME: better error msg */
    g_warning ("couldn't compile %s", end);
    g_free (new);
    return next;
  }

  new->name = g_strdup (name);
  new->start = g_strdup (start);
  new->end = g_strdup (end);
  new->esc = g_strdup (esc);
  new->nest = nest;
  new->font = _gtk_editor_fontdup (font);
  new->fore = _gtk_editor_coldup (fore);
  new->back = _gtk_editor_coldup (back);
  new->refcount = 1;		/* the list is refering to this one. */

  tmp->data = (gpointer)new;
  tmp->next = next;

  return tmp;
}


static void
destroy_stentry (GtkEditorSTEntry *ste)
{
  if ( --(ste->refcount) <= 0) {
    g_free (ste->name);
    g_free (ste->start);
    g_free (ste->end);
    g_free (ste->esc);
    _gtk_editor_regex_free (&ste->start_regex);
    _gtk_editor_regex_free (&ste->end_regex);
    if (ste->font) gdk_font_unref (ste->font);
    g_free (ste->fore);
    g_free (ste->back);
    g_free (ste);
  }
}

/* free_stentries -- frees list of 'entries' */
void
gtk_editor_free_stentries (GList *entries)
{
  for (; entries; entries = entries->next) {
    destroy_stentry ((GtkEditorSTEntry*)entries->data);
  }

  g_list_free (entries);
}

/* destroy_stable -- frees memory */
void
_gtk_editor_destroy_stable (GtkEditorSyntaxTable *stable)
{
  int i;

  if ( !stable )
    return;			/* pretty much destroyed */

  for (i = 0; i < stable->size; i++) {
    destroy_stentry (stable->entries[i]);
  }

  _gtk_editor_regex_free (&stable->regex);

  g_free (stable);		/* free table */
}

/* --<highlighting>--------------------------------------------------------- */
/* finish_syntax_block -- highligts from 'from' and forward to the end
 * of the syntax block as described in 'ste', or the end of editor's
 * text, which ever comes first.  'from' is the beginnig of the syntax block
 * 'cont' is the point we should search from. */
static gint
finish_syntax_block (GtkEditor *editor, guint from, guint cont, guint nlvl,
		     GtkEditorSTEntry *ste)
{

#if SUPPORT_NESTED_SYNTAX_HIGHLIGHT
  Match m1, m2;
  gint succ;
  int tlen = gtk_sctext_get_length (GTK_SCTEXT (editor));
  int esclen, i, j;

  while ( (nlvl > 0) && (cont < tlen) ) {
    /* find next ending */
    if (_gtk_editor_regex_search(GTK_SCTEXT(editor), cont,
				 &ste->end_regex, TRUE, &m1) < 0) {
      /* unbounded syntax block */
      m1.to = tlen;

    } else if (ste->esc) {
      esclen = strlen (ste->esc);
      for (i = 0; i < esclen; i++) {
	if (GTK_SCTEXT_INDEX (GTK_SCTEXT (editor), m1.from - 1) == ste->esc[i]) {

	  /* it is escaped...find out if the escape char is escaped... */
	  j = 1;
	  for (i = 0; i < esclen; i++) {
	    if (GTK_SCTEXT_INDEX (GTK_SCTEXT (editor), m1.from - j - 1)
		== ste->esc[i]) {
	      j++; i = -1;
	      continue;
	    }
	  }

	  /* an uneven j value indicates escaped */
	  if ((j%2)==1) {
	    return finish_syntax_block (editor, from, m1.to, nlvl, ste);
	  } else {
	    break;		/* leave for-loop */
	  }

	}
      }
    }

    /* find next start */
    if ( ste->nest ) {
      succ = _gtk_editor_regex_search(GTK_SCTEXT(editor), cont,
				      &ste->start_regex, TRUE, &m2);
    }
    /* check nesting */
    if ( ste->nest &&                /* nesting allowed */
	 (m2.from >= 0) &&           /* a new start exists */
	 (m2.from <= m1.from)) {     /* start <= end */
      /* Highlight up to "here" */
      gtk_sctext_set_property (GTK_SCTEXT (editor), from, m2.from,
			       ste->font, ste->fore, ste->back,
			       GINT_TO_POINTER (TRUE)); /* we're inside a block */

      /* Finish off */
      cont = finish_syntax_block(editor, m2.from, m2.to, nlvl+1, ste);

      return cont;
    } else { /* no nesting */

      gtk_sctext_set_property (GTK_SCTEXT (editor), from, m1.to,
			       ste->font, ste->fore, ste->back,
			       GINT_TO_POINTER (TRUE));

      cont = m1.to;
      nlvl--; /* we took the level one down */
    }
  }

  return cont;

#else /* SUPPORT_NESTED_SYNTAX_HIGHLIGHT */

  Match m;
  int tlen = gtk_sctext_get_length (GTK_SCTEXT (editor));
  int i, j, esclen;

  if (ste->nest) {
    g_warning ("not implemented yet!");
    return tlen;

  } else {
    /* no nesting */

    if ( _gtk_editor_regex_search (GTK_SCTEXT (editor), cont,
				   &ste->end_regex, TRUE, &m)
	 < 0) {
      /* no match...set rest of text */
      m.to = tlen;
    } else if (ste->esc) {
      esclen = strlen (ste->esc);
      for (i = 0; i < esclen; i++) {
	if (GTK_SCTEXT_INDEX (GTK_SCTEXT (editor), m.from - 1) == ste->esc[i]) {

	  /* it is escaped...find out if the escape char is escaped... */
	  j = 1;
	  for (i = 0; i < esclen; i++) {
	    if (GTK_SCTEXT_INDEX (GTK_SCTEXT (editor), m.from - j - 1)
		== ste->esc[i]) {
	      j++; i = -1;
	      continue;
	    }
	  }

	  /* an uneven j value indicates escaped */
	  if ((j%2)==1) {
	    return finish_syntax_block (editor, from, m.to, nlvl, ste);
	  } else {
	    break;		/* leave for-loop */
	  }

	}
      }
    }

    gtk_sctext_set_property (GTK_SCTEXT (editor), from, m.to,
			     ste->font, ste->fore, ste->back,
			     GINT_TO_POINTER (TRUE));
  }

  return m.to;
#endif /* SUPPORT_NESTED_SYNTAX_HIGHLIGHT */

}

void
_gtk_editor_sthilite_interval (GtkEditor *editor, guint from, guint to)
{
  int i, j, k;
  int cont;
  int cycle;
  gint tlen;
  Match m;

  if (from == to)
    return;			/* nothing to do really */

  /* if this editor has no syntax table we simply jump to pattern hiliting */
  if (!editor->stable) {
    /* first we clear everyting just in case */
    gtk_sctext_set_property (GTK_SCTEXT (editor), from, to,
			   GTK_WIDGET (editor)->style->font,
			   &GTK_WIDGET (editor)->style->text[GTK_STATE_NORMAL],
			   &GTK_WIDGET (editor)->style->base[GTK_STATE_NORMAL],
			   NULL);
    _gtk_editor_philite_interval (editor, from, to);
    return;
  }

  /* Start condition, backwards to basic state */
  /* FIXME: there must be a faster way to do this */
  i = cont = from;
  while ( gtk_sctext_get_property_data (GTK_SCTEXT (editor), cont)
	  && (cont != 0)) {
    cont-=1;
  }

  tlen = gtk_sctext_get_length (GTK_SCTEXT (editor));

  cycle = 1;
  /* now we're in basic state */
  while ( cycle ) {
    /* search forward to the first match */
    i = _gtk_editor_regex_search(GTK_SCTEXT(editor), cont,
				 &(editor->stable->regex), TRUE, &m);

    /* we stop if one of these are met:
       1. cont >= to && (!metadata(cont)) -- he have come to a basic state
       2. cont > buffer length
       3. no continuation could be found
    */
    if ( (cont >= to &&
	  !(gtk_sctext_get_property_data (GTK_SCTEXT (editor), cont))) ||
	 (cont > tlen) ) {
      cycle = 0;
      break; /* skip cycling */
    }


    if ( i >= 0 ) {
      /* there was a match.  Find out which pattern.  We can't use
       * the registers since we don't know how many each pattern uses. */
      for (j = 0; j < editor->stable->size; j++) {
	if ( (k = _gtk_editor_regex_match (GTK_SCTEXT (editor), i,
					   &editor->stable->entries[j]->start_regex))
	     > 0) {
	  /* found...k is the length of the match */

	  if (cont < i) {
	    /* now we set the basic state up to 'i' then we set the
	     * interval [i,...]  to the entry properties */
	    gtk_sctext_set_property (GTK_SCTEXT (editor), cont, i,
				   GTK_WIDGET (editor)->style->font,
				   &GTK_WIDGET (editor)->style->text[GTK_STATE_NORMAL],
				   &GTK_WIDGET (editor)->style->base[GTK_STATE_NORMAL],
				   NULL);

	    /* hilite patterns in basic block */
	    _gtk_editor_philite_interval (editor, cont, i);
	  }

	  /* 'i' is the start of the new block, but we must search for
	   * the end from 'cont' which is i + k. */
	  cont = i+k;
	  cont = finish_syntax_block (editor, i, cont, 1,
				      editor->stable->entries[j]);
	  break;		/* break the for loop. */
	}
      }
    } else {
      if (cont <= tlen) {
	/* we set the rest of the buffer as basic */
	gtk_sctext_set_property (GTK_SCTEXT (editor), cont, tlen,
			       GTK_WIDGET (editor)->style->font,
			       &GTK_WIDGET (editor)->style->text[GTK_STATE_NORMAL],
			       &GTK_WIDGET (editor)->style->base[GTK_STATE_NORMAL],
			       NULL);
	/* hilite patterns in basic block */
	_gtk_editor_philite_interval (editor, cont, tlen);
	cycle = 0;
	break;
      }
    }
  }
}


