/* 
   Written by Pieter J. Schoenmakers <tiggr@ics.ele.tue.nl>

   Copyright (C) 1996-1998 Pieter J. Schoenmakers.

   This file is part of TOM.  TOM is distributed under the terms of the
   TOM License, a copy of which can be found in the TOM distribution; see
   the file LICENSE.

   $Id: trt.h,v 1.65 1999/08/29 19:35:16 tiggr Exp $  */

#include <config/target.h>
#ifndef CURRENT_UNIT
#define CURRENT_UNIT tom
#endif
#include "tom/trt.h"
#include "tom/util.h"
#include <string.h>

/* Concatenation macros.  */
#define C2(A, B)  D9 (A, B, , , , , , , )
#define C3(A, B, C)  D9 (A, B, C, , , , , , )
#define C4(A, B, C, D)  D9 (A, B, C, D, , , , , )
#define C5(A, B, C, D, E)  D9 (A, B, C, D, E, , , , )
#define C6(A, B, C, D, E, F)  D9 (A, B, C, D, E, F, , , )
#define C7(A, B, C, D, E, F, G)  D9 (A, B, C, D, E, F, G, , )
#define C8(A, B, C, D, E, F, G, H)  D9 (A, B, C, D, E, F, G, H, )
#define C9(A, B, C, D, E, F, G, H, I)  D9 (A, B, C, D, E, F, G, H, I)
#define D9(A, B, C, D, E, F, G, H, I)  A ## B ## C ## D ## E ## F ## G ## H ## I

void init_streams (void);
void tini_streams (void);
void alloc_init (void);

/* Obscure stuff for GNU CC's apply.  */
union apply_args
{
  void **stack;
  char reg[0];
};

typedef BUILTIN_RETURN_TYPE builtin_return_type;

#define NUM_SELECTORS  (trt_selectors->st.num)
#define NUM_BUCKETS  ((NUM_SELECTORS + TRT_BUCKET_SIZE - 1) / TRT_BUCKET_SIZE)

GENERIC_RETURN_TYPE trt_forward (tom_object self, selector cmd, ...);

/* Declarations for the runtime structures.  */

/* Set as the first statement in main.  */
extern char trt_started_main;

/* Set by trt_resolve_module if it is invoked before TRT_STARTED_MAIN is
   set.  */
extern char trt_module_constructors;

/* The default dispatch table.  */
extern struct trt_method_dispatch_table *dtable_default;

/* All known selectors.  */
extern struct trtd_selectors *trt_selectors;

/* The known metas.  The metas include only the class objects (the meta
   class objects are available through these.  */
extern struct trt_metas *trt_metas;

extern tom_object trt_all_classes;

/* The module information of the program.  The reason for the main program's
   module info to have another name than the usual (dynamic loading) name is
   because rld(3) on the NeXT is braindead.  */
extern struct trtd_module trt_main_module_info;

/* The main class.  */
extern struct trt_class **trt_main_class;

/* The main selector.  */
extern selector trt_main_selector;

/* The number of extensions id's in use.  */
extern int trt_num_eids;

/* The mark used when traversing the inheritance graph.  This is
   increamented for each following traversal.  */
extern int trt_search_mark;

/* The recorded dynamic selectors.  */
extern struct trt_dynamic_selectors *trt_dynamic_selectors;

/* Grow the stalloc governed memory at *PST to be able to accomodate NUM
   entries of ELT_SIZE bytes each.  The size of the struct with zero
   elements is the ZERO_SIZE.  Iff GRANULARITY, it indicates the increment
   in elements allowed for the capacity.  If it is 0, the capacity starts
   (from 0) at 4 and is doubled in size at other times.  */
void *stalloc (struct stalloc **pst, int zero_size, int elt_size,
	       int num, int granularity);

/* Return whether the SUPER is a supermeta of the SUB meta.  The first
   function searches from the subclass towards the superclass; the second
   one (with reversed arguments!) searches towards the subclass.  */
tom_byte trt_supermeta_star (struct trt_class *sub, struct trt_class *super,
			     int mark);
tom_byte trt_submeta_star (struct trt_class *super, struct trt_class *sub,
			   int mark);

struct trtd_method *trt_extension_get_imp (struct trtd_extension *x,
					   selector sel);

/* Functions exported by array.  */

/* Return a new bytestring which can hold LEN bytes.  Return in THIS the
   pointer to the array extension of the string.  */
struct _es_i_tom_Array;
tom_object new_byte_string (struct _es_i_tom_Array **this, tom_int len);

/* Adjust the capacity of the mutable array defined by A and M which holds
   ELT_SIZE elements, so it can accomodate an extra NUM elements.  */
struct _es_i_tom_MutableArray;
void trt_array_resize (struct _es_i_tom_Array *a,
		       struct _es_i_tom_MutableArray *m,
		       int elt_size, int num);

/* Functions exported by dload.  */

void trt_resolve_module (struct trtd_module *mi);

void trt_execute_all_load_imps (void *args);

void trt_execute_load_imps (struct trtd_module *mi, void *args);

/* List functionality.  */

struct cons
{
  void *car, *cdr;
};

static __inline__ struct cons *
cons (void *a, void *d)
{
  struct cons *c = xmalloc (sizeof (*c));
  c->car = a;
  c->cdr = d;
  return c;
}

/* Hashtable functionality.  */

struct bucket
{
  void *cdr, *key, *value;
};

struct hashtable
{
  int (*hash) (void *);
  int (*equal) (void *, void *);

  /* The 2log of the number of buckets, and the number of contained
     elements.  */
  int size_shift, num_elts;

  struct bucket **buckets;
};

/* Pointer and string hashtable functions.  */
int hash_hash_pointer (void *elt);
int hash_hash_string (void *elt);
int hash_equal_pointer (void *a, void *b);
int hash_equal_string (void *a, void *b);
int hash_hash_selector (selector sel);

struct bucket *hash_add (struct hashtable *ht, void *key, void *value);
void *hash_find (struct hashtable *ht, void *key);

/********** invocations **********/

/* Full information to builtin_apply a method invocation.  */
struct trt_invocation
{
  /* Everything for an invocation of this invocation, just feed it to
     __builtin_apply.  */
  union apply_args *args;

  /* The stack memory in ARGS, but now usable as an argument to free.  */
  void *stack;

  /* The size of the stack arguments in ARGS.  */
  int args_size;

  /* The number of arguments actually supplied.  */
  int next_arg;

  /* Pointers to each argument in ARGS, being the receiver, selector, and
     the in arguments.  The out argument pointers are not recorded since
     they reside, fully described, in the RESULT.  These pointers are
     valid, even if the invocation is incomplete.  */
  void **arg_pointers;

  /* The stored type of each of the arguments (as a TRT_TE_*).  This
     usually is equal to the type provided, but, for instance, can be a
     double for float arguments.  */
  char *arg_types;

  /* The result of performing the invocation.  This should not be changed
     since the multiple return values in the ARGS refer to the fields in
     this result.  */
  struct trt_invocation_result *result;
};

#define INV_RECEIVER(X)  (*(void **) (X)->arg_pointers[0])
#define INV_SELECTOR(X)  ((selector) (*(void **) (X)->arg_pointers[1]))
#define INV_OFFSET_TO_ARGS  2

/* Full information on the result of an invocation.  */
struct trt_invocation_result
{
  /* The description.  */
  struct trtd_selector_args *desc;

  union
    {
      double d;
      float f;
      void *a;
      tom_int i;
      tom_long l;
    } cret;

  /* The offsets from return value at index I (thus 0 is unused, and at
     index 1, the offset is 0) to the byte offset in MRV.  */
  int *offsets;

  /* The size of the MRV, in bytes.  */
  int size;

  /* The extra returned values, in sequence, as described by CMD->OUT.  */
  void *mrv;
};

struct trt_thread_invocation
{
  struct trt_invocation inv;

  trt_thread_t th_id;
};

/* Free the structure and what dangles underneath.  */
void trt_invocation_free (struct trt_invocation *inv);
void trt_invocation_result_free (struct trt_invocation_result *inv);

/* Copy the structure and what dangles underneath.  */
struct trt_invocation_result *trt_invocation_result_copy
					(struct trt_invocation_result *in);

/* Return an invocation structure for invoking the SEL on the OBJECT,
   getting the arguments through the AP, checking with the argument types
   from CMD (if not NULL), skipping the first SKIP_IN incoming arguments.  */
struct trt_invocation *
invocation_build_args (struct trt_invocation *inv, tom_object object,
		       selector sel, selector cmd, int skip_in, va_list *ap);

/* Fire the invocation INV and return an InvocationResult, unless
   NO_RESULT in which case NULL is returned (and no result object is
   created).  Iff CHECK_COMPLETE is false, do not check for completeness
   of the invocation.  SELF and CMD are used when raising on an incomplete
   invocation.  */
tom_object invocation_fire (tom_object self, selector cmd,
			    struct trt_invocation *inv, int check_complete,
			    int no_result);

/* Invoke the IMP for the OBJECT with the SEL, retrieving arguments
   through PAP as described by CMD ignoring the first SKIP_IN arguments in
   CMD.  Return the result in RETURN_RESULT.  */
void perform_args (builtin_return_type *return_result, int_imp imp,
		   tom_object object, selector sel, selector cmd,
		   int skip_in, va_list *pap);

/********** threads **********/

/* Machine independent information maintained on each thread.  */
typedef struct
{
  /* The Thread object for this thread.  */
  tom_object object;

  /* Local data pointer for this thread.  */
  void *data;

  /* bind/catch/unwind stack for this thread.  */
  struct trt_condition_info **cond_stack;

  /* The number of entries in COND_STACK, and its capacity.  */
  int cond_num, cond_cap;

#if STACK_REF_STRUCT
#error STACK_REF_STRUCT does not fit multi threading
#else
  /* The bottom of this thread's stack.  */
  void *bottom;

  /* And a top of stack, measured at some moment in time.  */
  void *top;
#endif
} trt_thread_info;

/* Initialize the thread stuff up to a level that object allocation can
   work.  */
void trt_thread_pre_init (void *bottom_of_stack);

/* Finish initializing the thread stuff, now that object allocation works.  */
void trt_thread_post_init (void);

/* Initialize the machine dependent part of the threads package.  Return !0
   on success.  */
int trt_thread_mach_init (void);

/* The main function for _every_ thread we create.  */
void trt_thread_main (struct trt_thread_invocation *inv);

/* Run a new thread which will have TH_ID as its id.  The thread will call
   trt_thread_main() with the ARGS.  Return !0 upon success.  */
int trt_thread_create (trt_thread_t th_id, struct trt_thread_invocation *args);

/* Exit this thread.  */
__volatile__ void trt_thread_exit (void);

/* Return the thread identifier of this thread.  */
trt_thread_t trt_thread_self (void);

/* Set the pointer to the local data of this thread to BASE.  */
void trt_thread_set_local_data (void *base);

/* Notify the thread package that the number of locals has changed.  */
void trt_thread_update_locals (void);

/* The number of threads.  */
extern int trt_thread_num;

/* The capacity of the threads arrays.  */
extern int trt_thread_cap;

/* The threads, indexed on their id.  */
extern trt_thread_info *trt_threads;

/* The size of each thread's local data.  */
extern int trt_thread_local_size;

/* The capacity of each thread's local data block.  */
extern int trt_thread_local_cap;

/* The thread data as generated by the resolver.  */
extern char *trt_start_thread_data;

/* The data for the first thread.  Only used to communicate to
   trt_thread_init and trt_thread_mach_init routines.  */
extern char *trt_init_thread_data;

/******************** debugging ********************/

#if DEBUG_MESSAGE_MONITORING

/* Iff !0, message monitoring is currently on.  */
extern char trt_msg_monitor;

/* Invoked for each trt_lookup when TRT_MSG_MONITOR is 1.  */
void trt_monitor_message (tom_object receiver, selector cmd);

/* Invoked when trt_monitor_message finds a match.  */
void trt_monitor_match (tom_object receiver, selector cmd);

#endif

/******************** misc ********************/

struct trt_units
{
  struct stalloc st;

  /* The actual units.  */
  struct trtd_unit *units[0];
};

/* The collection of known of units.  */
extern struct trt_units *trt_units;

struct trt_modules
{
  struct stalloc st;

  /* The actual units.  */
  struct trtd_module *modules[0];
};

/* The collection of known of modules.  */
extern struct trt_modules *trt_modules;

/* Run the garbage collector for at most SECONDS time.  */
void tgc_collect_garbage (double seconds);

#define ABORT()  \
  do									\
    {									\
      fprintf (stderr, __FILE__ ":%d: abort\n", __LINE__);		\
      abort ();								\
    } while (0)

/* Return 1 if the ADDRESS denotes a valid object, -1 if the object was
   deallocated, and 0 if it does not denote the address of an object.  */
int address_status (void *address);

/* Return O if it is an object's address.  Otherwise, moan to STDERR and
   return NULL.  */
tom_object moan_address_status (tom_object o);
