/*
   Written by Pieter J. Schoenmakers <tiggr@gerbil.org>

   Copyright (C) 1996-1999 Pieter J. Schoenmakers.

   This file is part of TOM.  TOM is distributed under the terms of the
   TOM License, a copy of which can be found in the TOM distribution; see
   the file LICENSE.

   $Id: main.c,v 1.113 1999/09/11 12:17:43 tiggr Exp $  */

#include "trt.h"
#include <tom/tom-r.h>
#include <stdio.h>

#if HAVE_UNISTD_H
#include <unistd.h>
#endif

#if HAVE_LIBC_H
#include <libc.h>
#endif

/* The module information of the program.  The reason for the main program's
   module info to have another name than the usual (dynamic loading) name is
   because rld(3) on the NeXT is braindead.

   This is a common declaration which can be mapped onto a definition when
   in use, or becomes all 0's when not used.  */
struct trtd_module trt_main_module_info;

/* The main class.  */
struct trt_class **trt_main_class;

/* The main selector.  */
selector trt_main_selector;

/* The registration function of the unit containing the main method.  All
   0's when not used.  */
void (*trt_main_registration) (void);

/* Set as the first statement in main.  */
char trt_started_main;

/* Set by trt_resolve_module if it is invoked before TRT_STARTED_MAIN is
   set.  */
char trt_module_constructors;

int trt_bucket_size = TRT_BUCKET_SIZE;

/* Common declarations for the runtime structures.  These can be defined
   by static resolving.  */
struct trt_method_dispatch_table *dtable_default;
int trt_num_eids, trt_search_mark;
struct trtd_selectors *trt_selectors;
struct trt_metas *trt_metas;
struct trt_dynamic_selectors *trt_dynamic_selectors;
char *trt_start_thread_data;
char *trt_init_thread_data;

/* The collection of classes.  */
tom_object trt_all_classes;

/* The environment, as passed to main.  */
char **trt_environ;

/* Iff !0, the [Runtime environment] was modified.  */
int trt_environment_modified;

#if DEBUG_MESSAGE_MONITORING

/* Type in which to register message monitors.  */
typedef struct
{
  /* The object to watch, or NIL if the receiver does not matter.  */
  tom_object receiver;

  /* The selector to watch, or NIL if the selector does not matter.  */
  selector cmd;
} trt_monitor;

/* And the adminstration.  */
struct trt_monitors
{
  struct stalloc st;

  trt_monitor monitors[0];
};

/* Iff !0, message monitoring is currently on.  */
char trt_msg_monitor;

/* For use from the debugger.  */
struct trt_monitors *trt_monitors;

#define DISABLE_MONITORING()  \
  do { char foo = trt_msg_monitor; \
       int previous_msg_monitor = (trt_msg_monitor = 0, foo)
#define REENABLE_MONITORING()  \
  trt_msg_monitor = previous_msg_monitor; } while (0)

#else
#define DISABLE_MONITORING() do while (0)
#define REENABLE_MONITORING() do while (0)
#endif

/* If O is an object, print the object O to STDIO's ERR; otherwise issue
   an error on STDERR.  */
void
d (tom_object o)
{
  o = moan_address_status (o);
  if (o)
    {
      DISABLE_MONITORING ();
      TRT_SEND (, c_tom_stdio_err, SEL (r_print_r), o);
      TRT_SEND (, c_tom_stdio_err, SEL (r_nl));
      REENABLE_MONITORING ();
    }
}

/* If O is an object, dump O.  */
void
u (tom_object o, char allow_simple, int level)
{
  o = moan_address_status (o);
  if (o)
    {
      DISABLE_MONITORING ();
      TRT_SEND (, o, SEL (v_dump__oo__level_i), 1, allow_simple, level);
      REENABLE_MONITORING ();
    }
}

/* If O is an object, dump O.  */
void
v (tom_object o, char allow_self, char allow_simple, int level)
{
  o = moan_address_status (o);
  if (o)
    {
      DISABLE_MONITORING ();
      TRT_SEND (, o, SEL (v_dump__oo__level_i),
		allow_self, allow_simple, level);
      REENABLE_MONITORING ();
    }
}

/* Different name for D(), in case we're in a function employing a
   variable named D.  */
void
print_object (tom_object o)
{
  d (o);
}

/* Different name for U(), in case we're in a function employing a
   variable named U.  */
void
dump_object (tom_object o, char allow_simple, int level)
{
  u (o, allow_simple, level);
}

/* Different name for V(), in case we're in a function employing a
   variable named V.  */
void
dump_object2 (tom_object o, char allow_self, char allow_simple, int level)
{
  v (o, allow_self, allow_simple, level);
}

/* Return 1 iff the string OBJECT equals the STRING.  This can be used,
   for example, in breakpoint conditions.  */
int
tom_streq (tom_object object, char *string)
{
  DISABLE_MONITORING ();
  tom_object str = byte_string_with_c_string (string);
  return TRT_SEND (_BI_, object, SEL (o_equal_r), str);
  REENABLE_MONITORING ();
}

#if DEBUG_MESSAGE_MONITORING

static void
trt_monitor_add_receiver_selector (tom_object o, selector c)
{
  int new_num = 1;

  if (trt_monitors)
    new_num = trt_monitors->st.num + 1;
  trt_monitors = stalloc ((void *) &trt_monitors, sizeof (*trt_monitors),
			  sizeof (trt_monitors->monitors[0]), new_num, 1);
  trt_monitors->monitors[new_num - 1].receiver = o;
  trt_monitors->monitors[new_num - 1].cmd = c;
  trt_msg_monitor = 1;
}

/* For use in the debugger: monitor all messages to the object O.  */
void
trt_monitor_add_receiver (tom_object o)
{
  trt_monitor_add_receiver_selector (o, 0);
}

/* For use in the debugger: monitor all messages CMD.  */
void
trt_monitor_add_selector (selector cmd)
{
  trt_monitor_add_receiver_selector (0, cmd);
}

void
trt_monitor_message (tom_object receiver, selector cmd)
{
  int i;

  for (i = 0; i < trt_monitors->st.num; i++)
    {
      trt_monitor *m = &trt_monitors->monitors[i];
      if ((!m->receiver || m->receiver == receiver)
	  && (!m->cmd || trt_selectors_equal (m->cmd, cmd)))
	{
	  trt_monitor_match (receiver, cmd);
	  break;
	}
    }
}

void
trt_monitor_match (tom_object receiver, selector cmd)
{
  fprintf (stderr, "MATCH %s %p %s\n", receiver->isa->info.name.s,
	   receiver, cmd->name.s);
}

#endif

static void
idc (struct trt_class *m)
{
  fprintf (stderr, "%c %p %s", TGC_CLASS_P (m->asi) ? 'c' : 'm',
	   m, m->info.name.s);
}

static void
dc (struct trt_class *m)
{
  idc (m);
  fprintf (stderr, "\n");
}

static void
dcl (struct trt_metas *ml)
{
  if (ml)
    {
      int i;

      for (i = 0; i < ml->st.num; i++)
	{
	  if (i)
	    fprintf (stderr, ", ");
	  idc (ml->metas[i]);
	}
      fprintf (stderr, "\n");
    }
}

static void
ide (struct trtd_extension *x)
{
  idc (x->meta);
  fprintf (stderr, " %s", x->name.s ?: "MAIN");
}

static void
de (struct trtd_extension *x)
{
  ide (x);
  fprintf (stderr, "\n");
}

static void
del (struct trt_meta_extensions *el)
{
  if (el)
    {
      int i;

      for (i = 0; i < el->st.num; i++)
	{
	  if (i)
	    fprintf (stderr, ", ");
	  ide (el->extensions[i]);
	}
      fprintf (stderr, "\n");
    }
}

void *
i_tom_All_p_address (tom_object self, selector cmd)
{
  return self;
}

tom_byte
i_tom_All_o_gc_dead_p (tom_object self, selector cmd)
{
  return (!(TGC_CLASS_P (self->asi) || TGC_META_CLASS_P (self->asi))
	  && TGC_COLOUR (self->asi) == TGC_WHITE);
}

void
i_tom_All_v_gc_mark (tom_object self, selector cmd)
{
  if (TGC_COLOUR (self->asi) == TGC_WHITE)
    trt_make_gray (self);
}

tom_int
i_tom_All_i_hash (tom_object self, selector cmd)
{
  return HASH_POINTER (self);
}

tom_int
i_tom_All_i_hashq (tom_object self, selector cmd)
{
  return HASH_POINTER (self);
}

selector
trt_selector_named (char *s, int len)
{
  int i;

  /* XXX This is slow.  */
  for (i = 0; i < trt_selectors->st.num; i++)
    if (len == trt_selectors->selectors[i]->name.len
	&& !memcmp (s, trt_selectors->selectors[i]->name.s, len))
      return trt_selectors->selectors[i];

  return 0;
}

int
trt_all_arguments_size (selector sel)
{
  int size = 2 * sizeof (void *);
  int i;

  if (sel->out->num > 1)
    size += (sel->out->num - 1) * sizeof (void *);

  for (i = 0; i < sel->in->num; i++)
    switch (sel->in->args[i])
      {
      case TRT_TE_BOOLEAN:
      case TRT_TE_BYTE:
      case TRT_TE_CHAR:
      case TRT_TE_INT:
	size += sizeof (tom_int);
	break;

      case TRT_TE_LONG:
	size = (size + 7) & ~7;
	size += sizeof (tom_long);
	break;

      case TRT_TE_FLOAT:
	size += sizeof (tom_float);
	break;

      case TRT_TE_DOUBLE:
	size = (size + 7) & ~7;
	size += sizeof (tom_double);
	break;

      case TRT_TE_POINTER:
      case TRT_TE_SELECTOR:
      case TRT_TE_REFERENCE:
	size += sizeof (void *);
	break;

      default:
	ABORT ();
      }

  return (size + 7) & ~7;
}

tom_object
c_tom_Runtime_r_environment (tom_object set, selector cmd)
{
  if (c_tom_Runtime_environment)
    return c_tom_Runtime_environment;

  LOCK_RUNTIME ();
  if (!c_tom_Runtime_environment)
    {
      char **e = trt_environ, *s, *t;

      c_tom_Runtime_environment
	= TRT_SEND ((reference_imp), _mr_c_tom_MutableDictionary, SEL (r_new));
      while ((s = *e++))
	{
	  t = strchr (s, '=');
	  if (t)
	    TRT_SEND ((void_imp), c_tom_Runtime_environment,
		      SEL (v_set_r_at_r), byte_string_with_c_string (t + 1),
		      byte_string_with_string (s, t - s));
	}
    }
  UNLOCK_RUNTIME ();

  return c_tom_Runtime_environment;
}

void
c_tom_Runtime_v_runtimeStatistics_r (tom_object self, selector cmd,
				     tom_object s)
{
  int i, j;

#define INFO_INT(VAR) \
  TRT_SEND (, s, SEL (r_print_r), byte_string_with_c_string (#VAR ": "));  \
  TRT_SEND (, s, SEL (r_print_i), (tom_int) (VAR));			   \
  TRT_SEND (, s, SEL (r_nl));

  extern int num_buckets;
  extern int trt_num_buckets;

  INFO_INT (trt_units->st.num);
  INFO_INT (trt_metas->st.num);
  INFO_INT (trt_selectors->st.num);
  INFO_INT (TRT_BUCKET_SIZE);
  INFO_INT (num_buckets);
  INFO_INT (trt_num_buckets);
  INFO_INT (trt_dynamic_selectors->st.num);
  INFO_INT (trt_thread_local_size);
  INFO_INT (trt_num_eids);
  INFO_INT (trt_search_mark);

#define PRINT_INT(MSG,VAR) \
  TRT_SEND (, s, SEL (r_print_i), (tom_int) (VAR));			  \
  TRT_SEND (, s, SEL (r_print_r), byte_string_with_c_string ("\t" MSG));  \
  TRT_SEND (, s, SEL (r_nl));

  i = j = trt_metas->st.num * 2 * trt_num_eids;
  PRINT_INT ("eid array overhead limit (num_metas * 2 * num_eids)", i);

  i = trt_metas->st.num * 2 * num_buckets;
  j += i;
  PRINT_INT ("sparse array bucket index limit (num_metas * 2 * num_buckets)",
	     i);

  i = trt_num_buckets * TRT_BUCKET_SIZE * sizeof (void *);
  j += i;
  PRINT_INT ("bucket space (trt_num_buckets * TRT_BUCKET_SIZE * sizeof void*)",
	     i);

  PRINT_INT ("total space overhead", j);
}

void
c_tom_Runtime_v_setenv__rr_ (tom_object self, selector cmd,
			     tom_object var, tom_object val)
{
  LOCK_RUNTIME ();
  trt_environment_modified = 1;
  TRT_SEND ((void_imp), c_tom_Runtime_environment,
	    SEL (v_set_r_at_r), var, val);
  UNLOCK_RUNTIME ();
}

void
c_tom_Runtime_v_fastExit_i (tom_object self, selector cmd, tom_int rc)
{
  exit (rc);
}

tom_object
c_tom_Runtime_r_nameOfSelector_s (tom_object self, selector cmd,
				  selector sel)
{
  return byte_string_with_string (sel->name.s, sel->name.len);
}

selector
c_tom_Runtime_s_nullSelector (tom_object self, selector cmd)
{
  return 0;
}

selector
c_tom_Runtime_s_selectorNamed_r (tom_object self, selector cmd,
				 tom_object name)
{
  selector sel;
  tom_int len;
  void *s;

  s = TRT_SEND ((pointer_imp), name, SEL (_pi__byteStringContents), &len);
  sel = trt_selector_named (s, len);

  if (!sel)
    trt_raise (0, self, cmd, c_tom_Conditions_program_condition,
	       "unknown selector `%s' (length=%d)", s, (int) len);

  return sel;
}

tom_byte
c_tom_Runtime_o_selector_s_equals_s (tom_object self, selector cmd,
				     selector s1, selector s2)
{
  return trt_selectors_equal (s1, s2);
}

tom_int
i_tom_Selector_i_hash (tom_object self, selector cmd)
{
  struct _es_i_tom_Selector *this =
    trt_ext_address (self, _ei_i_tom_Selector);

  return this->sel->sel_id;
}

tom_object
c_tom_Runtime_r_classes (tom_object self, selector cmd)
{
  if (!trt_all_classes)
    {
      int i;

      trt_all_classes
	= TRT_SEND ((reference_imp), _mr_c_tom_MutableObjectArray,
		    SEL (r_withCapacity_i), (tom_int) trt_metas->st.num);

      for (i = 0; i < trt_metas->st.num; i++)
	if (trt_metas->metas[i] != &_md_c__builtin__Top
	    && trt_metas->metas[i] != &_md_c__builtin__Any)
	  TRT_SEND ((reference_imp), trt_all_classes, SEL (v_add_r),
		    trt_metas->metas[i]);
    }

  return trt_all_classes;
}

tom_object
c_tom_Runtime_r_hostname (tom_object self, selector cmd)
{
  if (!c_tom_Runtime_hostname)
    {
      char *s = alloca (1024);

      /* This can only fail if S is an invalid address, which it isn't.  */
      gethostname (s, 1024);

      c_tom_Runtime_hostname = byte_string_with_c_string (s);
    }
  return c_tom_Runtime_hostname;
}

void
c_tom_C_v_free_p (tom_object self, selector cmd, void *p)
{
  xfree (p);
}

void *
c_tom_C_p_malloc_i (tom_object self, selector cmd, tom_int l)
{
  return xmalloc (l);
}

tom_int
c_tom_C_i_memcmp__ppi_ (tom_object self, selector cmd,
			void *a, void *b, tom_int l)
{
  return memcmp (a, b, l);
}

void *
c_tom_C_p_memcpy__ppi_ (tom_object self, selector cmd,
			void *a, void *b, tom_int l)
{
  return memcpy (a, b, l);
}

void *
c_tom_C_p_memmove__ppi_ (tom_object self, selector cmd,
			 void *a, void *b, tom_int l)
{
  return memmove (a, b, l);
}

void
c_tom_C_v_bzero__pi_ (tom_object self, selector cmd, void *a, tom_int l)
{
  BZERO (a, l);
}

void *
c_tom_C_p_realloc__pi_ (tom_object self, selector cmd, void *a, tom_int l)
{
  return xrealloc (a, l);
}

/* Return the address in the object SELF of the variable named NAME in the
   extension X.  Return in TYPE the type of the variable.  */
static void *
trt_address_of_ext_var (tom_object self, struct trtd_extension *x,
			tom_object name, enum trt_type_encoding *type)
{
  tom_int nl;
  void *ns;
  int i;

  ns = TRT_SEND (_PI_, name, SEL (_pi__byteStringContents), &nl);

  for (i = 0; i < x->num_vars; i++)
    if (nl == x->vars[i].c.name.len
	&& !memcmp (ns, x->vars[i].c.name.s, nl))
      {
	*type = x->vars[i].c.type;
	return ((char *) trt_ext_address (self, *x->eid_in_a_global)
		+ x->vars[i].offset);
      }

  for (i = 0; i < x->num_statics; i++)
    if (nl == x->statics[i].c.name.len
	&& !memcmp (ns, x->statics[i].c.name.s, nl))
      {
	*type = x->statics[i].c.type;
	return x->statics[i].address;
      }

  return NULL;
}

void *
trt_address_of_ivar (tom_object self, tom_object name,
		     enum trt_type_encoding type)
{
  struct trt_meta_extensions *exts = self->isa->info.extensions;
  struct trtd_extension *ext;
  void *ns;
  tom_int nl;
  int i, j;

  ns = TRT_SEND ((pointer_imp), name, SEL (_pi__byteStringContents), &nl);

  for (i = 0; i < exts->st.num; i++)
    {
      for (j = 0, ext = exts->extensions[i]; j < ext->num_vars; j++)
	if (nl == ext->vars[j].c.name.len
	    && !memcmp (ns, ext->vars[j].c.name.s, nl)
	    && type == ext->vars[j].c.type)
	  return ((char *) trt_ext_address (self, *ext->eid_in_a_global)
		  + ext->vars[j].offset);

      for (j = 0; j < ext->num_statics; j++)
	if (nl == ext->statics[j].c.name.len
	    && !memcmp (ns, ext->statics[j].c.name.s, nl)
	    && type == ext->statics[j].c.type)
	  return ext->statics[j].address;
    }

  fatal ("object 0x%x, of kind %s, has no variable named `%s' typed %s",
	 self, self->isa->info.name.s, c_string_with_length (ns, nl),
	 trt_type_name (type));
  return 0;
}

GENERIC_RETURN_TYPE
i_tom_All_x_valueOfVariableNamed_r (tom_object self, selector cmd,
				    tom_object name)
{
  builtin_return_type rt;
  void *address;

  BZERO (&rt, sizeof (rt));

  if (cmd->out->num != 1)
    fatal ("return type of `%s' bad type for variable", cmd->name.s);

  address = trt_address_of_ivar (self, name, cmd->out->args[0]);

  switch (cmd->out->args[0])
    {
    case TRT_TE_CHAR: rt.i.i = *(tom_char *) address; break;
    case TRT_TE_INT: rt.i.i = *(tom_int *) address; break;
    case TRT_TE_LONG: rt.l.l = *(tom_long *) address; break;
    case TRT_TE_FLOAT: RETURN_SET_FLOAT (&rt, *(tom_float *) address); break;
    case TRT_TE_DOUBLE: RETURN_SET_DOUBLE (&rt, *(tom_double *) address); break;

    case TRT_TE_BOOLEAN:
    case TRT_TE_BYTE:
      rt.i.i = *(tom_byte *) address;
      break;

    case TRT_TE_SELECTOR:
    case TRT_TE_POINTER:
    case TRT_TE_REFERENCE: rt.p.p = *(void **) address; break;

    default:
      fatal ("valueOfVariableNamed: unhandled type %s",
	     trt_type_name (cmd->out->args[0]));
    }

  APPLY_ARGS_RETURN (&rt);
}

void
i_tom_Extension_v_setValue_x_ofVariableNamed_r_in_r (tom_object self,
						     selector cmd, ...)
{
  struct _es_i_tom_Extension *this
    = trt_ext_address (self, _ei_i_tom_Extension);
  struct trtd_extension *x = this->rti;
  void *address;
  va_list ap;
  tom_object name;
  tom_object o;

  /* This is not very `clean', but it is better than having to come up
     with a strange name for this method just to have the value as a last
     argument.  */
  /* The assignments to zero are only here to make the compiler shut up.  */
  tom_byte b = 0;
  tom_char c = 0;
  tom_int i = 0;
  tom_long l = 0;
  tom_float f = 0;
  tom_double d = 0;
  void *p;
  enum trt_type_encoding te, vte;
  
  if (cmd->in->num != 3)
    fatal ("bad argument 1 of `%s' for set variable", cmd->name.s);

  va_start (ap, cmd);;

  te = cmd->in->args[0];
  switch (te)
    {
    case TRT_TE_CHAR: c = VA_ARG_CHAR (ap); break;
    case TRT_TE_INT: i = va_arg (ap, tom_int); break;
    case TRT_TE_LONG: l = va_arg (ap, tom_long); break;
    case TRT_TE_FLOAT: f = VA_ARG_FLOAT (ap); break;
    case TRT_TE_DOUBLE: d = va_arg (ap, tom_double); break;

    case TRT_TE_BOOLEAN:
    case TRT_TE_BYTE:
      b = VA_ARG_BYTE (ap);
      break;

    case TRT_TE_SELECTOR:
    case TRT_TE_POINTER:
    case TRT_TE_REFERENCE:
      p = va_arg (ap, void *);
      break;

    default:
      fatal ("setValue: ofVariableNamed: unhandled type %s",
	     trt_type_name (cmd->out->args[0]));
    }

  name = va_arg (ap, tom_object);
  o = va_arg (ap, tom_object);
  va_end (ap);

  address = trt_address_of_ext_var (o, x, name, &vte);
  if (vte != cmd->in->args[0])
    fatal ("setValue: ofVariableNamed: unhandled type %s",
	   trt_type_name (cmd->out->args[0]));

  switch (te)
    {
    case TRT_TE_CHAR: *(tom_char *) address = c; break;
    case TRT_TE_INT: *(tom_int *) address = i; break;
    case TRT_TE_LONG: *(tom_long *) address = l; break;
    case TRT_TE_FLOAT: *(tom_float *) address = f; break;
    case TRT_TE_DOUBLE: *(tom_double *) address = d; break;

    case TRT_TE_BOOLEAN:
    case TRT_TE_BYTE:
      *(tom_byte *) address = b;
      break;

    case TRT_TE_SELECTOR:
    case TRT_TE_POINTER:
    case TRT_TE_REFERENCE:
      *(void **) address = p;
      break;

    default:
      fatal ("setValueOfVariableNamed to: : unhandled type %s",
	     trt_type_name (cmd->out->args[1]));
    }
}
    
void
i_tom_All_v_setValue_x_ofVariableNamed_r
  (tom_object self, selector cmd, ...)
{
  void *address;
  va_list ap;
  tom_object name;

  /* This is not very `clean', but it is better than having to come up
     with a strange name for this method just to have the value as a last
     argument.  */
  /* The assignments to zero are only here to make the compiler shut up.  */
  tom_byte b = 0;
  tom_char c = 0;
  tom_int i = 0;
  tom_long l = 0;
  tom_float f = 0;
  tom_double d = 0;
  void *p;
  enum trt_type_encoding te;
  
  if (cmd->in->num != 2)
    fatal ("bad argument 1 of `%s' for set variable", cmd->name.s);

  va_start (ap, cmd);;

  te = cmd->in->args[0];
  switch (te)
    {
    case TRT_TE_CHAR: c = VA_ARG_CHAR (ap); break;
    case TRT_TE_INT: i = va_arg (ap, tom_int); break;
    case TRT_TE_LONG: l = va_arg (ap, tom_long); break;
    case TRT_TE_FLOAT: f = VA_ARG_FLOAT (ap); break;
    case TRT_TE_DOUBLE: d = va_arg (ap, tom_double); break;

    case TRT_TE_BOOLEAN:
    case TRT_TE_BYTE:
      b = VA_ARG_BYTE (ap);
      break;

    case TRT_TE_SELECTOR:
    case TRT_TE_POINTER:
    case TRT_TE_REFERENCE:
      p = va_arg (ap, void *);
      break;

    default:
      fatal ("setValue: ofVariableNamed: unhandled type %s",
	     trt_type_name (cmd->out->args[0]));
    }

  name = va_arg (ap, tom_object);
  va_end (ap);

  address = trt_address_of_ivar (self, name, cmd->in->args[0]);

  switch (te)
    {
    case TRT_TE_CHAR: *(tom_char *) address = c; break;
    case TRT_TE_INT: *(tom_int *) address = i; break;
    case TRT_TE_LONG: *(tom_long *) address = l; break;
    case TRT_TE_FLOAT: *(tom_float *) address = f; break;
    case TRT_TE_DOUBLE: *(tom_double *) address = d; break;

    case TRT_TE_BOOLEAN:
    case TRT_TE_BYTE:
      *(tom_byte *) address = b;
      break;

    case TRT_TE_SELECTOR:
    case TRT_TE_POINTER:
    case TRT_TE_REFERENCE:
      *(void **) address = p;
      break;

    default:
      fatal ("setValueOfVariableNamed to: : unhandled type %s",
	     trt_type_name (cmd->out->args[1]));
    }
}

tom_byte
i_tom_All_o_respondsTo_s (tom_object self, selector cmd, selector sel)
{
  if (!sel)
    return 0;

  return (trt_lookup (self, sel) != (int_imp) trt_forward);
}

tom_byte
trt_submeta_star (struct trt_class *super, struct trt_class *sub, int k)
{
  if (super == sub)
    return 1;
  if (super->info.mark == k)
    return 0;
  super->info.mark = k;

  if (super->info.subs)
    {
      int i, n = super->info.subs->st.num;

      for (i = 0; i < n; i++)
	if (trt_submeta_star (super->info.subs->metas[i], sub, k))
	  return 1;
    }

  return 0;
}

tom_byte
trt_supermeta_star (struct trt_class *sub, struct trt_class *super, int k)
{
  if (sub == super)
    return 1;
  if (sub->info.mark == k)
    return 0;
  sub->info.mark = k;

  if (sub->info.supers)
    {
      int i, n = sub->info.supers->st.num;

      for (i = 0; i < n; i++)
	if (trt_supermeta_star (sub->info.supers->metas[i], super, k))
	  return 1;
    }

  return 0;
}

tom_byte
i_tom_All_o_isKindOf_r (tom_object self, selector cmd, tom_object class)
{
  return trt_supermeta_star ((void *) self->isa, (void *) class,
			     ++trt_search_mark);
}

/* Search for the extension with the indicated name within this meta.  If
   !NAME, searching is to be done for the main extension.  */
static struct trtd_extension *
trt_meta_extension_named (struct trt_class *class, char *name, int len)
{
  struct trt_meta_extensions *x = class->info.extensions;

  /* _builtin_.Top does not have any extensions.  */
  if (x)
    {
      int i, n = x->st.num;

      if (!name)
	{
	  for (i = 0; i < n; i++)
	    if (!x->extensions[i]->name.len)
	      return x->extensions[i];
	}
      else
	for (i = 0; i < n; i++)
	  if (x->extensions[i]->name.len == len
	      && !memcmp (x->extensions[i]->name.s, name, len))
	    return x->extensions[i];
    }

  return 0;
}

struct trtd_extension *
trt_metas_extension_named (struct trt_class *class, char *name, int len, int k)
{
  struct trtd_extension *x;

  if (class->info.mark == k)
    return 0;
  class->info.mark = k;

  x = trt_meta_extension_named (class, name, len);
  if (x)
    return x;

  if (class->info.supers)
    {
      struct trt_metas *supers = class->info.supers;
      int i, n = class->info.supers->st.num;

      for (i = 0; i < n; i++)
	{
	  x = trt_metas_extension_named (supers->metas[i], name, len, k);
	  if (x)
	    return x;
	}
    }

  return 0;
}

tom_object
i_tom_All_r_extensionNamed_r_inherited__o (tom_object self, selector cmd,
					   tom_object name, tom_byte supers)
{
  struct trtd_extension *x;
  tom_int l = 0;
  void *s = 0;

  if (name)
    s = TRT_SEND ((pointer_imp), name, SEL (_pi__byteStringContents), &l);

  if (!supers)
    x = trt_meta_extension_named (self->isa, s, l);
  else
    x = trt_metas_extension_named (self->isa, s, l, ++trt_search_mark);

  if (!x)
    return 0;

  if (!x->extension_object)
    x->extension_object = TRT_SEND ((reference_imp), _mr_c_tom_Extension,
				    SEL (r_new_p), x);
  return x->extension_object;
}

void
trt_metas_collect_state_extensions (struct trt_class *class, int k,
				    tom_object array)
{
  int i, n;

  if (class->info.mark == k)
    return;
  class->info.mark = k;

  if (class->info.supers)
    {
      struct trt_metas *supers = class->info.supers;

      for (i = 0, n = class->info.supers->st.num; i < n; i++)
	trt_metas_collect_state_extensions (supers->metas[i], k, array);
    }

  if (class->info.extensions)
    {
      struct trt_meta_extensions *x = class->info.extensions;

      for (i = 0, n = x->st.num; i < n; i++)
	{
	  struct trtd_extension *e = x->extensions[i];

	  if (e->state_size || e->num_statics)
	    {
	      if (!e->extension_object)
		e->extension_object = TRT_SEND (_PI_, _mr_c_tom_Extension,
						SEL (r_new_p), e);
	      TRT_SEND (, array, SEL (v_add_r), e->extension_object);
	    }
	}
    }
}

tom_object
i_tom_All_r_stateExtensions (tom_object self, selector cmd)
{
  TRT_PANIC_MODE_P ();

  if (!self->isa->info.state_extensions)
    {
      self->isa->info.state_extensions
	= TRT_SEND (_PI_, _mr_c_tom_MutableObjectArray, SEL (r_new));

      trt_metas_collect_state_extensions (self->isa, ++trt_search_mark,
					  self->isa->info.state_extensions);
    }

  TRT_PANIC_MODE_V ();

  return self->isa->info.state_extensions;
}

struct trtd_method *
trt_extension_get_imp (struct trtd_extension *x, selector sel)
{
  int i, n;

  if (x->methods)
    for (i = 0, n = x->methods->num_methods; i < n; i++)
      if (trt_selectors_equal (sel, x->methods->methods[i].sel))
	return &x->methods->methods[i];

  return 0;
}

tom_byte
i_tom_Extension_o_implements_s (tom_object self, selector cmd, selector sel)
{
  struct _es_i_tom_Extension *this
    = trt_ext_address (self, _ei_i_tom_Extension);

  return !!trt_extension_get_imp (this->rti, sel);
}

tom_object
i_tom_Extension_r_meta (tom_object self, selector cmd)
{
  struct _es_i_tom_Extension *this
    = trt_ext_address (self, _ei_i_tom_Extension);
  struct trtd_extension *x = this->rti;

  return (void *) x->meta;
}

tom_object
i_tom_Extension_r_name (tom_object self, selector cmd)
{
  struct _es_i_tom_Extension *this
    = trt_ext_address (self, _ei_i_tom_Extension);
  struct trtd_extension *x = this->rti;

  return x->name.len ? byte_string_with_string (x->name.s, x->name.len) : 0;
}

tom_object
i_tom_Extension_r_variables (tom_object self, selector cmd)
{
  struct _es_i_tom_Extension *this
    = trt_ext_address (self, _ei_i_tom_Extension);
  tom_object array;

  TRT_PANIC_MODE_P ();

  array = (void *) this->var_names;
  if (!array)
    {
      struct trtd_extension *x = this->rti;
      int i;

      array = TRT_SEND (_PI_, CREF (tom_MutableObjectArray), SEL (r_new));

      for (i = 0; i < x->num_vars; i++)
	{
	  tom_object o = byte_string_with_string (x->vars[i].c.name.s,
						  x->vars[i].c.name.len);
	  TRT_SEND (, array, SEL (v_add_r), o);
	}

      for (i = 0; i < x->num_statics; i++)
	{
	  tom_object o = byte_string_with_string (x->statics[i].c.name.s,
						  x->statics[i].c.name.len);
	  TRT_SEND (, array, SEL (v_add_r), o);
	}

      this->var_names = (void *) array;
    }

  TRT_PANIC_MODE_V ();

  return array;
}

tom_int
i_tom_Extension_i_typeOfVariableNamed_r_in_r (tom_object self, selector cmd,
					      tom_object name, tom_object o)
{
  struct _es_i_tom_Extension *this
    = trt_ext_address (self, _ei_i_tom_Extension);
  struct trtd_extension *x = this->rti;
  enum trt_type_encoding type;

  if (!trt_address_of_ext_var (o, x, name, &type))
    trt_raise (1, self, cmd, c_tom_Conditions_program_condition,
	       "unknown variable");

  return type;
}

tom_object
i_tom_Extension_r_valueOfVariableNamed_r_in_r (tom_object self, selector cmd,
					       tom_object name, tom_object o)
{
  struct _es_i_tom_Extension *this
    = trt_ext_address (self, _ei_i_tom_Extension);
  struct trtd_extension *x = this->rti;
  enum trt_type_encoding type;
  tom_object result = NULL;
  void *address;

  address = trt_address_of_ext_var (o, x, name, &type);
  if (!address)
    trt_raise (1, self, cmd, c_tom_Conditions_program_condition,
	       "unknown variable");
  else switch (type)
    {
    case TRT_TE_BOOLEAN:
    case TRT_TE_BYTE:
      result = TRT_SEND ((void * (*) (void *, void *, tom_byte)),
			 CREF (tom_ByteNumber), SEL (r_with_b),
			 *(tom_byte *) address);
      break;

    case TRT_TE_CHAR:
      result = TRT_SEND ((void * (*) (void *, void *, tom_char)),
			 CREF (tom_CharNumber), SEL (r_with_c),
			 *(tom_char *) address);
      break;
    case TRT_TE_INT:
      result = TRT_SEND ((void * (*) (void *, void *, tom_int)),
			 CREF (tom_IntNumber), SEL (r_with_i),
			 *(tom_int *) address);
      break;
    case TRT_TE_LONG:
      result = TRT_SEND ((void * (*) (void *, void *, tom_long)),
			 CREF (tom_LongNumber), SEL (r_with_l),
			 *(tom_long *) address);
      break;
    case TRT_TE_FLOAT:
      result = TRT_SEND ((void * (*) (void *, void *, tom_float)),
			 CREF (tom_FloatNumber), SEL (r_with_f),
			 *(tom_float *) address);
      break;
    case TRT_TE_DOUBLE:
      result = TRT_SEND ((void * (*) (void *, void *, tom_double)),
			 CREF (tom_DoubleNumber), SEL (r_with_d),
			 *(tom_double *) address);
      break;

    case TRT_TE_REFERENCE:
      result = *(tom_object *) address;
      break;

    case TRT_TE_POINTER:
      result = TRT_SEND (_PI_, CREF (tom_Pointer), SEL (r_with_p),
			 *(void **) address);
      break;

    case TRT_TE_SELECTOR:
      result = TRT_SEND (_PI_, CREF (tom_Selector), SEL (r_with_s),
			 *(selector *) address);
      break;

    default:
      fatal ("valueOfVariableNamed in: unhandled type %s",
	     trt_type_name (type));
    }

  return result;
}

char *
trt_type_name (enum trt_type_encoding type)
{
  switch (type)
    {
    case TRT_TE_VOID: return "void";
    case TRT_TE_BOOLEAN: return "boolean";
    case TRT_TE_BYTE: return "byte";
    case TRT_TE_CHAR: return "char";
    case TRT_TE_INT: return "int";
    case TRT_TE_LONG: return "long";
    case TRT_TE_FLOAT: return "float";
    case TRT_TE_DOUBLE: return "double";
    case TRT_TE_POINTER: return "pointer";
    case TRT_TE_SELECTOR: return "selector";
    case TRT_TE_REFERENCE: return "reference";
    case TRT_TE_DYNAMIC: return "dynamic";
    default:
      break;
    }

  {
    /* XXX This is a leak.  */
    char *s = xmalloc (30);

    sprintf (s, "<unknown type %d>", type);
    return s;
  }
}

tom_object
trt_raise (int signal_p, tom_object object, selector cmd,
	   void *class, char *fmt, ...)
{
  tom_object c, m;

  if (!fmt)
    m = 0;
  else
    {
      char buf[2000];
      va_list ap;
      int n;

      va_start (ap, fmt);
      n = vsprintf (buf, fmt, ap);
      va_end (ap);
      if (n >= sizeof (buf))
	ABORT ();
      m = byte_string_with_string (buf, n);
    }


  if (cmd)
    c = TRT_SEND (_PI_, _mr_c_tom_SelectorCondition,
		  SEL (r_for_r_class_r_message_r_selector_s),
		  object, class, m, cmd);
  else
    c = TRT_SEND (_PI_, _mr_c_tom_Condition, SEL (r_for_r_class_r_message_r),
		  object, class, m);

  if (!signal_p)
    TRT_SEND (, c, SEL (v_raise));

  return TRT_SEND (_PI_, c, SEL (r_signal));
}

void
trt_unrecognized (tom_object self, selector cmd)
{
  trt_raise (0, self, cmd, c_tom_Conditions_unrecognized_selector,
	     "unrecognised selector");
}

void
vquit (char *fmt, va_list ap)
{
  vfprintf (stderr, fmt, ap);
  fprintf (stderr, "\n");
  fflush (stderr);

  /* This _must_ abort.  Raising a condition isn't good enough.  */
  if (c_tom_Runtime_core_on_fatal)
    abort ();
  exit (42);
}

void
vfatal (char *fmt, va_list ap)
{
  fprintf (stderr, "trt: fatal error: ");
  vquit (fmt, ap);
}

void
fatal (char *fmt, ...)
{
  va_list ap;

  va_start (ap, fmt);
  vfatal (fmt, ap);
  va_end (ap);
}

void
unimplemented (char *fmt, ...)
{
  va_list ap;

  va_start (ap, fmt);
  fprintf (stderr, "trt: unimplemented feature: ");
  vquit (fmt, ap);
  va_end (ap);
}

void
trt_init_main (int argc, char **argv)
{
  i_tom_MutableObjectArray *args;
  int i, r, no_main_module;

  /* Act as if this is a constructor.  */
  if (trt_main_registration)
    trt_main_registration ();

  /* This will only work if trt_main_module_info is actually defined.
     If it is, it will contain units.  */
  no_main_module = !trt_main_module_info.units;
    
  trt_started_main = 1;
  setvbuf (stderr, NULL, _IOLBF, BUFSIZ);

  if (!trt_metas || trt_module_constructors)
    {
      if (!no_main_module)
	trt_resolve_module (&trt_main_module_info);
    }
  else
    trt_init_thread_data = trt_start_thread_data;

  trt_thread_pre_init (&argc);
  alloc_init ();
  trt_thread_post_init ();

  init_streams ();

  /* Install a dummy encoding for the ByteString's default encoding.  */
  c_tom_ByteString_default_encoding = TRT_SEND (_PI_, _mr_c_tom_USASCIIEncoding,
						SEL (r_shared));

  c_tom_Runtime_long_program_name
    = (void *) byte_string_with_c_string (argv[0]);

  /* XXX Removing everything up to, including, the last `/' can be done
     slightly faster than this...  */
  {
    tom_object a = TRT_SEND ((reference_imp), c_tom_Runtime_long_program_name,
  SEL (r_componentsSeparatedBy_c_limit__i_excludeEmpty__o_substringSelector__s),
			     (tom_char) '/', -1, 0, SEL (r_substring__ii_));
    /* GGG This expression can not be passed instead of the L in the
       TRT_SEND below, since cpp does not expand it there...  */
    int l = TRT_SEND (, a, SEL (i_length));

    c_tom_Runtime_program_name
      = TRT_SEND ((reference_imp), a, SEL (r_at_i), l - 1);
  }

  args = TRT_SEND (_PI_, _mr_c_tom_MutableObjectArray, SEL (r_alloc));
  for (i = 1; i < argc; i++)
    TRT_SEND (, args, SEL (v_add_r), byte_string_with_c_string (argv[i]));

  c_tom_Runtime_all_arguments = TRT_SEND ((reference_imp),
					  _mr_c_tom_ObjectArray,
					  SEL (r_withEnumerable_r), args);

  TRT_SEND (, _mr_c_tom_Runtime, SEL (v_preload_r), args);

  if (trt_module_constructors)
    trt_execute_all_load_imps (args);
  else if (!no_main_module)
    trt_execute_load_imps (&trt_main_module_info, args);

  c_tom_Runtime_arguments = TRT_SEND ((reference_imp), _mr_c_tom_ObjectArray,
				      SEL (r_withEnumerable_r), args);
}
