%{/* -*- C -*- */
/* OTM lexer.
   Written by Pieter J. Schoenmakers <tiggr@ics.ele.tue.nl>

   Copyright (C) 1996 Pieter J. Schoenmakers.

   This file is part of TOM.  TOM is distributed under the terms of the
   TOM License, a copy of which can be found in the TOM distribution; see
   the file LICENSE.

   $Id: lex.l,v 1.76 1998/05/31 18:17:40 tiggr Exp $  */

/* Get RE_DUP_MAX before it simply redefines what tl/regex.h tells the value
   should be.  */
#import <limits.h>
#import "global.h"
#import <ctype.h>
#import <float.h>
#import <limits.h>
#import "parse.h"
#import "OTMAsm.h"
#import "OTMBasic.h"
#import "OTMBuiltinMethod.h"
#import "OTMInstance.h"
#import "OTMNumberCST.h"
#import "OTMVariable.h"
#import "OTMForeignExpr.h"
#import <tom/trt.h>

#define yywrap()  1

/* Identity counter for generating string names.  */
static unsigned int next_string_id;

int otm_debug;

%}

%%

"bind"			{ return BIND; }
"break"			{ return BREAK; }
"class"			{ return CLASS; }
"catch"			{ return CATCH; }
"const"			{ return CONST; }
"continue"		{ return CONTINUE; }
"deferred"		{ return DEFERRED; }
"do"			{ return DO; }
"dynamic"		{ return DYNAMIC; }
"end"			{ return END; }
"else"			{ return ELSE; }
"extension"		{ return EXTENSION; }
"extern"		{ return EXTERN; }
"for"			{ return FOR; }
"id"			{ return RECEIVER; }
"implementation"	{ return IMPLEMENTATION; }
"instance"		{ return INSTANCE; }
"if"			{ return IF; }
"interface"		{ return INTERFACE; }
"local"			{ return LOCAL; }
"mutable"		{ return MUTABLE; }
"nil"			{ return NIL; }
"old"			{ return OLD; }
"posing"		{ return POSING; }
"post"			{ return POST; }
"public"		{ return PUBLIC; }
"pre"			{ return PRE; }
"private"		{ return PRIVATE; }
"protected"		{ return PROTECTED; }
"redeclare"		{ return REDECLARE; }
"redefine"		{ return REDEFINE; }
"return"		{ return RETURN; }
"super"			{ return SUPER; }
"static"		{ return STATIC; }
"typedef"		{ return TYPEDEF; }
"unwind"		{ return UNWIND; }
"void"			{ return VOID; }
"while"			{ return WHILE; }

"++"			{ return PLUSPLUS; }
"--"			{ return MINMIN; }
"*="			{ lvalp->i = BO_MUL; return ASSIGN; }
"/="			{ lvalp->i = BO_DIV; return ASSIGN; }
"%="			{ lvalp->i = BO_MOD; return ASSIGN; }
"+="			{ lvalp->i = BO_ADD; return ASSIGN; }
"-="			{ lvalp->i = BO_SUB; return ASSIGN; }
"<<="			{ lvalp->i = BO_SHL; return ASSIGN; }
">>="			{ lvalp->i = BO_SHR; return ASSIGN; }
">>>="			{ lvalp->i = BO_LSR; return ASSIGN; }
"&="			{ lvalp->i = BO_AND; return ASSIGN; }
"|="			{ lvalp->i = BO_OR; return ASSIGN; }
"^="			{ lvalp->i = BO_EOR; return ASSIGN; }
"&&="			{ lvalp->i = BO_SC_AND; return ASSIGN; }
"||="			{ lvalp->i = BO_SC_OR; return ASSIGN; }
"=="			{ return EQ; }
"!="			{ return NE; }
">="			{ return GE; }
"<="			{ return LE; }
"->"			{ return IMPLIES; }
"&&"			{ return AND; }
"||"			{ return OR; }
"<<"			{ return SHL; }
">>"			{ return SHR; }
">>>"			{ return LSR; }

"boolean"		{
			  lvalp->v = basic_type[BT_BOOLEAN];
			  return BASIC_TYPE;
			}
"byte"			{
			  lvalp->v = basic_type[BT_BYTE];
			  return BASIC_TYPE;
			}
"char"			{
			  lvalp->v = basic_type[BT_CHAR];
			  return BASIC_TYPE;
			}
"double"		{
			  lvalp->v = basic_type[BT_DOUBLE];
			  return BASIC_TYPE;
			}
"float"			{
			  lvalp->v = basic_type[BT_FLOAT];
			  return BASIC_TYPE;
			}
"int"			{
			  lvalp->v = basic_type[BT_INT];
			  return BASIC_TYPE;
			}
"long"			{
			  lvalp->v = basic_type[BT_LONG];
			  return BASIC_TYPE;
			}
"pointer"		{
			  lvalp->v = basic_type[BT_POINTER];
			  return BASIC_TYPE;
			}
"selector"		{
			  lvalp->v = basic_type[BT_SELECTOR];
			  return BASIC_TYPE;
			}

"//".*\n		{
			  static last_slash_slash_line;

			  if (last_slash_slash_line != current_line - 1
			      && !flag_inhibit_comment)
			    warning (@"// comment");
			  last_slash_slash_line = current_line;
			  current_line++;
			}
[ \t\f]*\n		{ current_line++; }
[ \t\f]*		{ /* Retry.  */ }

[A-Za-z][-A-Za-z_0-9]*[A-Za-z_0-9]|[A-Za-z][A-Za-z_0-9]*	{
			  TLString *s = [CO_TLString stringWithCString: yytext
					 length: yyleng];
			  TLVector *v = [CO_LTTUnit instancesNamed: s];

			  if (v)
			    {
			      lvalp->v = v;
			      return TYPE;
			    }

			  /* It is an identifier.  */
			  lvalp->v = unique_identifier (s);
			  return IDENTIFIER;
			}

(0[xX])?[0-9a-fA-F]+[lL]? {
			  char *buf = yytext, *next;
			  C_INT v, base = 10, d;
			  int long_p = 0;
			  char *n;

			  if (yyleng > 1)
			    if (yytext[1] == 'x' || yytext[1] == 'X')
			      {
				buf += 2;
				base = 16;
			      }
			    else if (yytext[0] == '0')
			      base = 8;
			  if (yyleng > 0 && (yytext[yyleng - 1] == 'l'
					     || yytext[yyleng - 1] == 'L'))
			    {
			      long_p = 1;
			      yyleng--;
			    }

			  next = yytext + yyleng;

			  /* Avoid warning about unused yyunput.  */
			  { void *v = yyunput; v = *&v; }

			  if (!long_p)
			    {
			      /* See if this'll fit an int.  */
			      for (v = 0, n = buf; n < next; n++)
				{
				  d = *n - '0';
				  if (d > 9)
				    {
				      d -= 'A' - '9' - 1;
				      if (d > 15)
					d -= 'a' - 'A';
				    }
				  if (d >= base)
				    {
				      error (@"bad digit %c for base %d",
					     *n, base);
				      n = next;
				      break;
				    }
				  if ((INT_MAX - d) / base >= v)
				    v = v * base + d;
				  else
				    break;
				}

			      if (n == next)
				{
				  if (v >= TL_SMALL_INT_MIN
				      && v <= TL_SMALL_INT_MAX)
				    lvalp->v = tll_small_int[v];
				  else
				    lvalp->v = [CO_TLNumber numberWithInt: v];
				  lvalp->v = [CO_OTMNumberCST
					      numberWithValue: lvalp->v 
					      type: basic_type[BT_INT]];
				  return NUMBER;
				}
			    }

			    {
			      /* Hopefully, this'll fit a long long.  */
			      C_LONG lv, nlv;

			      for (lv = 0, n = buf; n < next; n++)
				{
				  d = *n - '0';
				  if (d > 9)
				    {
				      d -= 'A' - '9' - 1;
				      if (d > 15)
					d -= 'a' - 'A';
				    }
				  if (d >= base)
				    {
				      error (@"bad digit %c for base %d",
					     *n, base);
				      break;
				    }

				  nlv = lv * base + d;
				  if ((nlv - d) / base != lv
				      && (base != 16
					  || ((unsigned long long) (nlv - d)
					      / base != lv)))
				    {
				      *next = 0;
				      error (@"number too large: %s", buf);
				    }
				  lv = nlv;
				}
			      lvalp->v = [CO_TLNumber numberWithLongLong: lv];
			      lvalp->v = [CO_OTMNumberCST
					   numberWithValue: lvalp->v 
					   type: basic_type[BT_LONG]];
			      return NUMBER;
			    }
			}

[0-9]+(\.[0-9]+)?([eE][-+]?[0-9]*)? {
  lvalp->v = [CO_OTMNumberCST numberWithValue: [CO_TLString stringWithCString: yytext
						      length: yyleng]
			   type: basic_type[BT_FLOAT]];
  return NUMBER;
}

[0-9]+(\.[0-9]+)?([dD]("-"|"+")?[0-9]*)? {
  char *s;

  for (s = yytext + yyleng - 1; s > yytext; s--)
    if (*s == 'd' || *s == 'D')
      {
	if (s == yytext + yyleng - 1)
	  yyleng--;
	else
	  *s = 'e';
	break;
      }

  lvalp->v = [CO_OTMNumberCST numberWithValue: [CO_TLString stringWithCString: yytext
						      length: yyleng]
			   type: basic_type[BT_DOUBLE]];
  return NUMBER;
}


"/*"			{
			  int start_line = current_line;
			  int b = 0, c = input ();

			  while (c != EOF && !(c == '/' && b == '*'))
			    {
			      if (c == '\n')
				current_line++;
			      else if (b == '/' && c == '*')
				warning (@"`/*' in comment");
			      b = c;
			      c = input ();
			    }
			  if (c == EOF)
			    {
			      int l = current_line;

			      error (@"unterminated comment");
			      current_line = start_line;
			      cerror (@"comment started here");
			      current_line = l;
			    }
			}

"<"[a-zA-Z0-9_]*">" {
			  int c_lit = yyleng == 3 && yytext[1] == 'c';
			  int c = input (), state = -1;
			  int start_line = current_line;
			  char *text = alloca (1 + yyleng);
			  static int c_cap, c_len;
			  static char *c_text;

			  /* Copy yytext, as that pointer can be mangled by
                             input().  */
			  memcpy (text, yytext, yyleng);
			  c_len = 0;
			  while (c != EOF)
			    {
			      if (c == '\n')
				current_line++;

			      if (c == '<')
				state = 0;
			      else if (state == 0)
				state = c == '/' ? state + 1 : -1;
			      else if (state > 0)
				if (text[state] == c)
				  {
				    if (c == '>')
				      break;
				    state++;
				  }
				else
				  state = -1;

			      if (c_lit)
				{
				  if (c_len == c_cap)
				    {
				      c_cap = c_cap ? 2 * c_cap : 1024;
				      c_text = xrealloc (c_text, c_cap);
				    }
				  c_text[c_len++] = c;
				}

			      c = input ();
			    }

			  if (c == EOF)
			    {
			      int l = current_line;

			      error (@"unterminated %s documentation", text);
			      current_line = start_line;
			      cerror (@"%s started here", text);
			      current_line = l;
			    }

			  if (c_lit)
			    {
			      int l = current_line;

			      current_line = start_line;
			      lvalp->v = [[CO_OTMAsm alloc]
					 initWithType: basic_type[BT_VOID]
					 string: [TLString stringWithCString:
						 c_text length: c_len - 3]];
			      current_line = l;
			      return C_LITERAL;
			    }
			}

'(\\.|[^'\\])*'	{
			  int nnl;
			  TLString *str
			    = ltt_lex_string_cst (yytext, yyleng, '\'', &nnl);
			  current_line += nnl;
			  if ([str length] != 1)
			    error (@"bad ASCII byte constant");
			  lvalp->v = [CO_OTMNumberCST numberWithValue:
				      [CO_TLString stringWithCString: yytext
						length: yyleng]
				      type: basic_type[BT_BYTE]];
			  return NUMBER;
}

\"(\\.|\\\\|\\[0-7][0-7]*|[^\\\"])*\"	{
			  int nnl;
			  TLString *str
			    = ltt_lex_string_cst (yytext, yyleng, '"', &nnl);
			  id <TLString> cst_name;
			  LTTStringCST *cst;

			  current_line += nnl;
			  cst_name = formac (nil, @"%@_%@_%@_%d",
					     top_unit_name, input_basename,
					     (current_extension
					      ? [current_extension outputName]
					      : quote (current_filename)),
					     next_string_id++);

			  cst = [CO_LTTStringCST stringWithString: str
				 name: cst_name];

			  {
			    LTTStringCST *u = [strings objectForKey: str];

			    if (u)
			      cst = u;
			    else
			      [strings setObject: cst forKey: str];
			  }

			  lvalp->v = [cst semantics];

			  if (![tom_string_instance fullyLoaded])
			    load_interface ((id) [tom_string_instance
						  structure]);

			  return STRING_CST;
			}

.			{ return yytext[0]; }

%%
/* Parse the file called NAME to be found in the UNIT along the
   searchpath.  */
void
parse_file (id <TLString> name, LTTUnit *unit, FILE *f)
{
  /* Hack needed when switching back from top-level buffer.  It is not
     allowed to switch to the NULL buffer.  Sigh.  */
  static YY_BUFFER_STATE top;
  YY_BUFFER_STATE pyy_buf = YY_CURRENT_BUFFER;
  GCDECL2;

  id previous_either = current_either;
  id previous_class = current_class;
  id previous_instance = current_instance;
  id previous_extension = current_extension;
  id previous_method = current_method;
  id previous_compound = current_compound;
  id previous_unit = current_unit;
  int previous_tmp = next_tmp;
  int previous_otm_debug = otm_debug;
  void *lex_buffer;

  if (flag_verbose > 2)
    formac (V_stderr_, @"entering %@ from %d %@\n",
	    name, current_line, current_filename);

  PARSE_START_CONTEXT (name);

  current_class = nil;
  current_instance = nil;
  current_either = nil;
  current_unit = unit;

  GCPRO2 (current_filename, current_unit);

  lex_buffer = yy_create_buffer (f, YY_BUF_SIZE);
  yy_switch_to_buffer (lex_buffer);

  if (of)
    {
      if (!flag_readable)
	formac (of, @"# 1 %#\n", current_filename);

      formac (of, @"/* Generated by otmc (%@)\n   version %s.  */\n",
	      prog, long_version);
      if (flag_1)
	formac (of, @"#define CURRENT_UNIT %@\n", [current_unit outputName]);
      formac (of, @"\n#include <tom/trt.h>\n");
      if (flag_check_extension_address)
	formac (of, @"\n#include <tom/util.h>\n");
      formac (of, @"\nstatic char _otmc_created[0] __attribute__ ((unused));");
      formac (of, @"\n");
      OUTPUT_FILE_PREAMBLE (of);
      formac (of, @"\n");
    }

  if (otm_debug > 0)
    otm_debug--;
  otm_parse ();
  otm_debug = previous_otm_debug;

  PARSE_STOP_CONTEXT;

  if (!top)
    top = yy_create_buffer (NULL, 1);
  yy_switch_to_buffer (pyy_buf ? pyy_buf : top);
  yy_delete_buffer (lex_buffer);

  current_class = previous_class;
  current_instance = previous_instance;
  current_either = previous_either;
  current_extension = previous_extension;
  current_method = previous_method;
  current_compound = previous_compound;
  current_unit = previous_unit;
  next_tmp = previous_tmp;

  /* No need to GC if we're about to exit.  */
  if (!flag_inhibit_gc && !of)
    [TLObject gc];

  PARSE_END_CONTEXT;

  GCUNPRO;
}
