/*
   Written by Pieter J. Schoenmakers <tiggr@ics.ele.tue.nl>

   Copyright (C) 1996 Pieter J. Schoenmakers.

   This file is part of TOM.  TOM is distributed under the terms of the
   TOM License, a copy of which can be found in the TOM distribution; see
   the file LICENSE.

   $Id: OTMMethod.h,v 1.47 1998/03/16 14:09:07 tiggr Exp $  */

#import "OTMTop.h"

@class OTMCompound, OTMExtension, OTMInvocation, OTMType, OTMVariable;
@class OTMArgument, OTMTuple, OTMTypeTuple;

@interface OTMMethod: OTMTop
{
  /* The extension we're in.  */
  OTMExtension *extension;

  /* The type being returned.  */
  OTMType *return_type;

  /* The (explicit and implicit) arguments to this method.  Each element
     is a single argument, i.e. either a variable or a tuple.  */
  TLVector *arguments;

  /* The default values of the arguments, as matched by the index.  */
  TLVector *arg_values;

  /* The number of non-optional arguments.  */
  int num_fixed_arguments;

  /* The output arguments, i.e. all those fields in the return tuple
     except the first (which is returned straight away).  */
  TLVector *out_args;

  /* The elements of this method's name.  For a method without any
     explicit arguments, this is nil.  This contains uniqued strings, for
     fast comparison.  */
  TLVector *name_elements;

  /* The internal name, used for hashing and such.  */
  TLString *internal_name;

  /* Our selector.  Assigned as needed.  */
  LTTSelector *the_selector;

  /* The output name; as seen by the output language.  */
  TLString *output_name;

  /* Iff !0, some of the argument types or the return type is dynamic.  */
  unsigned int dynamic_typing: 1;

  /* Iff !0, we do not have implicit arguments.  */
  unsigned int flat: 1;
}

+(OTMMethod *) methodWithExtension: (OTMExtension *) extension
			      name: (TLString *) n
			returnType: (id) rt
			     flatp: (BOOL) flatp;

/* Each element in the list NT is a (name-part . (argument . value)),
   where VALUE is the expression for the default argument value (and nil
   if the argument does not have a default value).  */
+(OTMMethod *) methodWithExtension: (OTMExtension *) extension
			 nameTypes: (TLCons *) nt
			returnType: (id) rt
			     flatp: (BOOL) flatp;

+(LTTSelector *) selectorForMethod: (OTMMethod *) m
			invocation: (OTMInvocation *) rt
			      used: (BOOL) used_p;

/* Return YES iff the NAME_PARTS match for an invocation of the receiving
   method.  */
-(BOOL) namePartsMatch: (TLVector *) name_parts;

/* Compare the return types and the types of all arguments of the
   receiving method with those of the method M.  Return YES upon a match.  */
-(BOOL) typesMatch: (OTMMethod *) m;

/* Compare the types of the explicit arguments of the receiving method
   with the types of the arguments in A.  */
-(BOOL) argumentsTypeMatch: (TLVector *) a;

-(BOOL) allowedRedeclaration: (OTMMethod *) m
		  inSubclass: (BOOL) subclass_p;

-(OTMVariable *) argumentNamed: (id <TLString>) name;

/* Return the argument with the number N.  */
-(OTMArgument *) argumentNumbered: (int) n;

/* Return a vector containing the ARGUMENTS, interspersed with default
   values, as can be deduced from the provided NAME_PARTS.  */
-(TLVector *) argumentsForNameParts: (TLVector *) name_parts
			  arguments: (TLVector *) arguments;

/* Return, in the context of the EITHER, the type of the argument for the
   NAME_PART, which, in the invocation is at index I (and thus, in the
   method name should reside at an index >= I).  */
-(OTMType *) typeForArgumentAtIndex: (int) i
			  inContext: (OTMMeta *) either
		       withNamePart: (TLString *) np;

-(BOOL) fitsInvocationNumArguments: (int) n;

/* Return YES iff the types and number of arguments to the receiving
   method are identical (`eq') to those of the method M in the context of
   the EITHER.  */
-(BOOL) identical: m inContext: (OTMMeta *) either;

-(void) dumpInfo: (id <TLOutputStream>) s;

-(id <TLString>) firstNamePart;

-(int) implicitArguments;

-(OTMVariable *) searchVariableNamed: (id <TLString>) name;

-(TLVector *) arguments;

-(OTMExpr *) argumentDefaultAt: (int) index;

-(int) numExplicitArguments;

-(OTMExtension *) extension;
-(id <TLString>) methodName;
-(TLVector *) nameParts;
-(id <TLString>) outputName;
-(LTTSelector *) selector;
-(OTMType *) returnType;

-(BOOL) dynamicTyped;

-(BOOL) builtinp;

-(void) warnDifferingArguments: (OTMMethod *) orig;

-(void) resolveIdentifiers: (OTMMeta *) meta;

/********** compilation **********/

/* Assign to the RefVar out-arguments the rights values from the tuple
   TUP.  */
-(void) assignOutArgsFrom: (OTMTuple *) tup;

/* Make all types of arguments and the return value compile their
   declaration.  */
-(void) compileDeclarationTypes;

/* Output the declaration of the method, sans semicolon.  */
-(void) compileDeclaration;

-(id) resultOfInvocation: (OTMInvocation *) inv;

/* Output the invocation of this method as described by the INV, where the
   result is assigned to the tuple TUP.  */
-(id) resultOfInvocation: (OTMInvocation *) inv
		 toTuple: (OTMTuple *) tup;

/* Primarilily here for OTMCustomMethod to be notified of arguments with a
   reference type (which need to be protected).  */
-(void) setHaveStackReferences: (OTMVariable *) v;

#ifdef OTMMETHOD_DECLARE_PRIVATE_METHODS

-(void) addOutArgsFrom: (OTMTypeTuple *) tup skip: (BOOL) skip;

-(void) assignOutArgsFrom: (OTMTuple *) tup
		       to: (id <TLEnumerator>) to
		     skip: (BOOL) skip;

-initWithExtension: (OTMExtension *) ext
	returnType: (id) rt
	     flatp: (BOOL) flatp;

-initWithExtension: (OTMExtension *) extension
	      name: (TLString *) n
	returnType: (id) rt
	     flatp: (BOOL) flatp;

-initWithExtension: (OTMExtension *) extension
	 nameTypes: (TLCons *) nt
	returnType: (id) rt
	     flatp: (BOOL) flatp;

#endif

@end
