<copyright> Collection behavioural classes.
    Written by <a href="mailto:tiggr@gerbil.org">Pieter J. Schoenmakers</a>

    Copyright &copy; 1995-1999 Pieter J. Schoenmakers.

    This file is part of TOM.  TOM is distributed under the terms of the
    TOM License, a copy of which can be found in the TOM distribution; see
    the file LICENSE.

    <id>$Id: collections.t,v 1.42 1999/09/13 12:13:09 miro Exp $</id>
    </copyright>

/******************** Collection ********************/

// Collection inherits from state to avoid conflicts with several methods,
// which otherwise need to be directed to Collection instead of State for
// every (concrete) Collection subclass.
// Mon Sep 2 20:04:10 1996, tiggr@tricky.es.ele.tue.nl
implementation class
Collection: State, Conditions, Copying, Enumerable

end;

implementation instance
Collection

<doc> Return {YES}.  </doc>
boolean
  dump_simple_p
{
  = YES;
}

<doc> Evaluate the {block} for each object element in this {Collection}.
    Subclasses can provide a faster implementation.  </doc>
void
  do Block block
{
  Enumerator e = [self enumerator];
  boolean b;
  All elt;

  while ({(b, elt) = [e next]; b;})
    [block eval elt];
}

<doc> Two collections consider themselves equal if they are the same
    object or when their elements are equal.  </doc>
boolean
  equal id other
{
  if (self == other)
    return TRUE;

  Enumerator self_e = [self enumerator], other_e = [other enumerator];

  for (;;)
    {
      boolean b, c;
      All o, p;

      (b, o) = [self_e next];
      (c, p) = [other_e next];

      if (!b && !c)
	return TRUE;
      if (!(b && c) || ![o equal p])
	return FALSE;
    }
}

<doc> Make the receiving collection immutable.  This is irreversible.  It
    is a no-op for immutable collections.  </doc>
void
  freeze
{
  void;
}

<doc> Return {self} if this is a non-mutable {Collection}.  Otherwise,
    return a non-mutable collection with the same contents.  </doc>
Collection
  frozen
{
  = self;
}

<doc> Return the element contained in this collection, which is {equal} to
    the {object}.  The default implementation by {Collection} visits the
    elements using an enumerator.  </doc>
Any
  member All object
{
  Enumerator e = [self enumerator];
  boolean b;
  Any o;

  while ({(b, o) = [e next]; b;})
    if ([object equal o])
      return o;
}

<doc> Like {member}, but the element is identified on reference equality.
    </doc>
Any
  memq All object
{
  Enumerator e = [self enumerator];
  boolean b;
  Any o;

  while ({(b, o) = [e next]; b;})
    if (o == object)
      return o;
}

<doc> Return the number of elements in this {Collection}.  </doc>
deferred int
  length;

<doc> Fire the {invocation} at the elements contained in the receiving
    collection.  </doc>
void
  makeElementsPerform Invocation invocation
{
  Enumerator e = [self enumerator];
  boolean b;
  All o;

  while ({(b, o) = [e next]; b;})
    [invocation fireAt o];
}

<doc> Send the argumentless {message} to the elements contained in the
    receiving collection.  </doc>
void
  makeElementsPerform selector message
{
  Enumerator e = [self enumerator];
  boolean b;
  All o;

  while ({(b, o) = [e next]; b;})
    [o perform message];
}

<doc> Send the {message} with the object {argument} to the elements
    contained in the receiving collection.  </doc>
void
  makeElementsPerform selector message
		 with All argument
{
  Enumerator e = [self enumerator];
  boolean b;
  All o;

  while ({(b, o) = [e next]; b;})
    [o perform message with argument];
}

<doc> A {Collection} is not mutable.  </doc>
boolean
  mutable
{
  = FALSE;
}

<doc> Fire the invocation repeatedly, each time with the next object from
    the collection completing the invocation.  </doc>
void
  passElementsTo Invocation inv
{
  Enumerator e = [self enumerator];
  boolean b;
  All o;

  while ({(b, o) = [e next]; b;})
    [inv fireWith o];
}

OutputStream
  write OutputStream s
{
  Enumerator e = [self enumerator];
  boolean b;
  All o;

  s = [s print ('(', [[self kind] name])];

  while ({(b, o) = [e next]; b;})
    s = [s print (' ', o)];

  = [s print ')'];
}

end;

implementation class
MutableCollection: Collection

end;

implementation instance
MutableCollection

deferred void
  add All object;

void
  addElementsFrom Enumerable other
{
  [self addElementsFromEnumerator [other enumerator]];
}

void
  addElementsFromEnumerator Enumerator e
{
  Any object;

  while ({boolean b; (b, object) = [e next]; b;})
    [self add object];
}

<doc> Remove all elements from the receiving collection.  </doc>
deferred void
  empty;

// CCC Actually, redeclaring something as deferred does not yet have the
// desired effect...  Wed Aug 7 14:28:22 1996, tiggr@cobra.es.ele.tue.nl
<doc> Force {freeze} to be undefined since each particular subclass must
    itself implement it.  </doc>
deferred void
  freeze;

<doc> Initialize by feeding {self} the elements from the {enumerator}.
    </doc>
id (self)
  initWithEnumerator Enumerator e
{
  [self addElementsFromEnumerator e]; 
}

<doc> A {MutableCollection} is mutable.  </doc>
boolean
  mutable
{
  = TRUE;
}

end;

/******************** Keyed ********************/

<doc> A {Keyed} {Collection} stores elements on a key.  </doc>

implementation class
Keyed: Collection

end;

implementation instance
Keyed

deferred Any
  at All key
pre
  key != nil;

<doc> Return an enumerator on the keys of this mapped collection.  </doc>
deferred Enumerator
  keyEnumerator;

<doc> Member for a Keyed collection can be implemented efficiently.
    </doc>
Any
  member All object
{
  return self[object];
}

<doc> This is a less-efficient abstract implementation.  </doc>
Any
  memq All object
{
  return self[object] == object ? Any (object) : nil;
}

end;

implementation class
MutableKeyed: Keyed, MutableCollection

end;

implementation instance
MutableKeyed

<doc> Add the {object}.  </doc>
deferred void
  add All object;

<doc> Remove the {object}.  </doc>
deferred void
  remove All object;

<doc> Remove all the objects contained in the collection {c}.  </doc>
void
  removeElementsFrom Collection c
{
  [self removeElementsFromEnumerator [c enumerator]];
}

void
  removeElementsFromEnumerator Enumerator e
{
  boolean b; All o;
  while ({(b, o) = [e next]; b;})
    [self remove o];
}

<doc> Remove all the objects not contained in the collection {c}, i.e.,
    change the receiving collection into the result of intersecting {self}
    and {c}.

    Keyed collections that can not handle losing elements while being
    enumerated must reimplement this.  There is no method
    {keepElementsFromEnumerator} because set intersection is not
    meaningful with an arbitrary enumerator.  </doc>
void
  keepElementsFrom Collection c
{
  Enumerator e = [self enumerator];
  boolean b; All o;

  while ({(b, o) = [e next]; b;})
    if (![c member o])
      [self remove o];
}

<doc> All the keys in a convenient format.  </doc>
Indexed
  allKeys
{
  = [[ObjectArray alloc] initWithEnumerator [self keyEnumerator]];
}

end;

/******************** Mapped ********************/

<doc> A {Mapped} {Collection} is a {Keyed} collection which stores (key,
    value) associations.  </doc>
implementation class
Mapped: Keyed

end;

implementation instance
Mapped

<doc> Evaluate the {block} for each key.  Subclasses can provide a faster
    implementation.  </doc>
void
  doKeys Block block
{
  Enumerator e = [self keyEnumerator];
  boolean b;
  All key;

  while ({(b, key) = [e next]; b;})
    [block eval key];
}

deferred MapEnumerator
  valueEnumerator;

<doc> Check not only the values, but also the keys.  </doc>
boolean
  equal id other
{
  if (self == other)
    return TRUE;

  Enumerator self_e = [self keyEnumerator], other_e = [other keyEnumerator];

  for (;;)
    {
      boolean b, c;
      All o, p;

      (b, o) = [self_e next];
      (c, p) = [other_e next];

      if (!b && !c)
	return TRUE;
      if (!(b && c) || ![o equal p])
	return FALSE;

      All vo = self[o];
      if (![vo equal other[p]])
	return FALSE;
    }
}

<doc> Return an enumerator on the keys of this mapped collection.  </doc>
deferred Enumerator
  keyEnumerator;

end;

implementation class
MutableMapped: Mapped, MutableKeyed
{
  <doc> Directives to guide {addPairsFrom}.  </doc>
  const MAPPED_KEEP = 0;
  const MAPPED_ERROR = 1;
  const MAPPED_CLOBBER = 2;
}

end;

implementation instance
MutableMapped

<doc> Use the {value}  as the value and the key.  </doc>
void
  add All value
{
  self[value] = value;
}

deferred void
  set All value
   at All key
pre
  key != nil && value != nil;

void
   addPairsFrom Mapped m
  onContention: int action = MAPPED_KEEP
{
  MapEnumerator e = [m valueEnumerator];
  boolean b;
  All k, v;

  while ({(b, k, v) = [e next]; b;})
    if ([self member k] != nil)
      {
	if (action != MAPPED_KEEP)
	  if (action == MAPPED_CLOBBER)
	    self[k] = v;
	  else
	    [[Condition for self class type-condition
			message "rewriting protected key"] raise];
      }
    else
      self[k] = v;
}

<doc> Return all values in a convenient format.  </doc>
Indexed
  allValues
{
  = [[ObjectArray alloc] initWithEnumerator [self valueEnumerator]];
}

end;

/******************** Ordered ********************/

<doc> An {Ordered} {Collection} maintains its elements in a specific
    order, though the time complexity or retrieving the nth object not
    necessarily independent of n.  </doc>
implementation class
Ordered: Collection

end;

implementation instance
Ordered

<doc> Adjust the range ({start}, {len}) to fit the length of the receiving
    {Indexed} collection.  </doc>
(int, int)
  adjustRange (int, int) (start, len)
{
  int length = [self length];

  if (start >= length)
    {
      start = length;
      len = 0;
    }
  else
    {
      if (len == -1)
	len = length - start;

      if (start < 0)
	{
	  len += start;
	  start = 0;

	  if (len < 0)
	    len = 0;
	}

      if (start + len > length)
	len = length - start;
    }

  = (start, len);
}

<doc> Return the element at {index}.  If the receiving collection
    stores unboxed values, such as integers, the value returned is the
    element boxed. Returns nil on index overflow (precondition should
    whine about it, though - nil may only be returned if [self length] is
    bugged).  </doc>
Any
  at int index
pre
  index >= 0 && index < [self length]
{
  Enumerator e = [self enumerator];
  boolean b;
  Any obj;

  while (index-- != 0) {
    (b, obj) = [e next];
    if (!b)
      return nil;
  }

  = obj;
}  

<doc> Return the byte value of the element at {index}.  If the receiving
    collection stores objects, the {byteValue} of the element retrieved is
    actually returned.  </doc>
byte
  at int index
{
  // Ancient excuse-as-reason, probably:
  // Sat Sep 11 14:09:02 1999, tiggr@gerbil.org
  // The object at INDEX is explicitly referred to as a Number here, since
  // the compiler must read the definition of Number to have seen the
  // `byteValue' method.
  Number n = self[index];

  = [n byteValue];
}

<doc> The following all follow the <basic type> {at} <index> stanza.  </doc>
char
  at int index
{
  = [self[index] charValue];
}

int
  at int index
{
  = [self[index] intValue];
}

long
  at int index
{
  = [self[index] longValue];
}

float
  at int index
{
  = [self[index] floatValue];
}

double
  at int index
{
  = [self[index] doubleValue];
}

<doc> Return the index of the first {element}, or -1 if it could not be
    found.  </doc>
int
  indexOf All element
{
  Enumerator e = [self enumerator];
  boolean b;
  All elt;
  int i = 0;

  while ({(b, elt) = [e next]; b;}) {
    if ([elt equal element])
      return i;
    i++;
  }

  = -1;
}

<doc> Return the index of the first identical {element}, or -1 if it could
    not be found.  </doc>
int
  indexOfIdentical All element
{
  Enumerator e = [self enumerator];
  boolean b;
  All elt;
  int i = 0;

  while ({(b, elt) = [e next]; b;}) {
    if (elt == element)
      return i;
    i++;
  }

  = -1;
}

end;

implementation class
MutableOrdered: Ordered, MutableCollection

end;

implementation instance
MutableOrdered

<doc> Store the {object} at {index} in the receiving collection.  </doc>
deferred void
  set All object
   at int index;

<doc> Swap the elements at the indices {i} and {j}.  </doc>
deferred void
  swap (int, int) (i, j);

<doc> Reverse the elements in the range starting at {start}, with length
    {len}.  </doc>
void
  reverse (int, int) (start, len)
{
  (start, len) = [self adjustRange (start, len)];
  int last = start + len - 1;

  while (start < last)
    {
      [self swap (start, last)];
      start++;
      last--;
    }
}

<doc> Reverse the entire collection. </doc>
void reverse
{
    [self reverse (0, [self length])];
}

<doc> Remove the element at {index}, decreasing by 1 the indices of
    the elements further in the collection.  </doc>
deferred void
  removeElementAt int index;

<doc> Remove the first occurence of {element}.  </doc>
void
  removeElement All element
{
  int i = [self indexOf element];

  if (i >= 0)
    [self removeElementAt i];
}

<doc> Remove the first occurence of the identical {element}.  </doc>
void
  removeIdenticalElement All element
{
  int i = [self indexOfIdentical element];

  if (i >= 0)
    [self removeElementAt i];
}

end;

/******************** Indexed ********************/

<doc> An {Indexed} {Collection} maintains an association between integer
    indices and the objects it contains, with the promise that retrieving
    an object through the index is O(1) in time complexity.  </doc>

implementation class
Indexed: Ordered

end;

implementation instance
Indexed

<doc> Evaluate the {block} for each object element in this {Indexed}.
    </doc>
void
  do Block block
{
  int i, n = [self length];
  for (i = 0; i < n; i++)
    {
      All elt = self[i];
      [block eval elt];
    }
}

<doc> Extract the elements from the receiving collection, as indicated by
    the return type.  The number of elements in the collection must match
    the number of expected elements.  </doc>
extern dynamic
  elements;

<doc> Like {elements}, but extract only the {num} elements starting at
    index {start}.  </doc>
extern dynamic
  elements (int, int) (start, num);

<doc> Return an {Enumerator} on this {Indexed}.  </doc>
Enumerator
  enumerator
{
  = [IndexedEnumerator with self];
}

<doc> A faster implementation than the one inherited from {Collection}.
    </doc>
boolean
  equal id other
{
  if (self == other)
    return TRUE;

  int i, len = [other length], length = [self length];

  if (len != length)
    return FALSE;

  for (i = 0; i < length; i++)
    {
      All o = self[i];
      if (![o equal other[i]])
	return FALSE;
    }

  return TRUE;
}

<doc> Return the index of the first {element}, or -1 if it could not be
    found.  </doc>
int
  indexOf All element
{
  int i, n = [self length];

  for (i = 0; i < n; i++)
    if ([element equal self[i]])
      return i;

  = -1;
}

<doc> Return the index of the first identical {element}, or -1 if it could
    not be found.  </doc>
int
  indexOfIdentical All element
{
  int i, n = [self length];

  for (i = 0; i < n; i++)
    if (self[i] == element)
      return i;

  = -1;
}

<doc><h4>Performing</h4></doc>

<doc> A faster implementation than the one inherited from {Collection},
    without using an {Enumerator}.  </doc>
void
  makeElementsPerform Invocation inv
{
  int i, l = [self length];

  while (i < l)
    [inv fireAt self[i++]];
}

<doc> Likewise.  </doc>
void
  makeElementsPerform selector message
{
  int i, l = [self length];

  while (i < l)
    [self[i++] perform message];
}

<doc> Likewise.  </doc>
void
  makeElementsPerform selector message
		 with All argument
{
  int i, l = [self length];

  while (i < l)
    [self[i++] perform message with argument];
}

<doc> Likewise.  </doc>
void
  passElementsTo Invocation inv
{
  int i, l = [self length];

  while (i < l)
    {
      All object = self[i++];
      [inv fireWith object];
    }
}

end;

implementation class
MutableIndexed: Indexed, MutableOrdered

end;

implementation instance
MutableIndexed

end;

/******************** IndexedEnumerator ********************/

<doc> The {IndexedEnumerator} enumerates any {Indexed} collection,
    returning the elements boxed.  </doc>
implementation class
IndexedEnumerator: State, Enumerator

instance (id)
  with Indexed indexed
{
  = [[self alloc] init indexed length [indexed length]];
}

end;

implementation instance
IndexedEnumerator
{
  <doc> The index of the next element to be returned.  </doc>
  int next;

  <doc> The index one beyond the last element to be returned.  </doc>
  int num;

  <doc> The actual indexed collection.  </doc>
  Indexed indexed;
}

<doc> Designated initializer.  </doc>
id (self)
    init Indexed a
  start: int start = 0
  length int length
{
  (indexed, next, num) = (a, start, length);
}

(boolean, Any)
  next
{
  if (next != num)
    = (TRUE, indexed[next++]);
}

<doc> All these {next} methods are not really necessary, as they are
    provided by the {Enumerator} behaviour.  However, binding them here
    directly greatly enhances speed and reduces memory requirements for
    non-object indexeds, which now also do not need their own enumerator to
    obtain speed.  </doc>
(boolean, byte)
  next
{
  if (next != num)
    = (TRUE, indexed[next++]);
}

(boolean, char)
  next
{
  if (next != num)
    = (TRUE, indexed[next++]);
}

(boolean, int)
  next
{
  if (next != num)
    = (TRUE, indexed[next++]);
}

(boolean, long)
  next
{
  if (next != num)
    = (TRUE, indexed[next++]);
}

(boolean, float)
  next
{
  if (next != num)
    = (TRUE, indexed[next++]);
}

(boolean, double)
  next
{
  if (next != num)
    = (TRUE, indexed[next++]);
}

end;
