<copyright> Miscellaneous behavioural classes.
    Written by <a href="mailto:tiggr@ics.ele.tue.nl">Pieter J. Schoenmakers</a>

    Copyright &copy; 1995-1997 Pieter J. Schoenmakers.

    This file is part of TOM.  TOM is distributed under the terms of the
    TOM License, a copy of which can be found in the TOM distribution; see
    the file LICENSE.

    <id>$Id: behaviours.t,v 1.34 1999/09/11 12:27:57 tiggr Exp $</id>
    </copyright>

/******************** Comparable ********************/

implementation class
Comparable: instance (All)
end;

implementation instance
Comparable: instance (All)

<doc> Return 0 if the {other} is considered equal by the receiving object.
    1 if the receiver considers himself larger, and -1 when smaller.
    </doc>
deferred int
  compare id other
pre
  other != nil;

end;

/******************** Container ********************/

<doc> A container is an object which gets to mark its elements after
    normal marking has been done.  This is very usable for unique string
    tables, DO proxies, etc; actually: all cases where an object having
    become garbage implying it should be removed from its container, and
    the container itself is not allowed to reference the object in a
    normal way (since then it would never become garbage).

    In short, a the combo of garbage collector and container implements
    weak referencing.</doc>
implementation class
Container
//{
//  <doc> The collection of all containers.  Note that this collection
//      itself is a container too.  </doc>
//  static MutableEqSet all_containers;
//}
// 
// What the hell is <cod>?!  It _is_ documentation, but not supposed to go
// into the distributed header files.  Do come up with a better name.
// Thu May 30 15:03:33 1996, tiggr@tom.es.ele.tue.nl
<cod> This is what the load method could look like, but actually this
    functionality is implemented in the runtime, as we want containers
    available before the very first load method, irrespective of which it
    is.

    void
      load Array arguments
    {
      all_containers = [MutableEqSet new];

      [all_containers setIsContainer YES]
    }
</cod>

end;

implementation instance Container

<doc> The container mark method.  </doc>
deferred void
  gc_container_mark_elements;

<doc> Return {TRUE} iff the receiving object is a container.  </doc>
extern boolean
  isContainer;

<doc> Set this object to be a container, or not, depending on
    {container_p}.  </doc>
extern void
  setIsContainer boolean container_p;

<doc> State that this container wants to be notified when it is
    conservatively pinpointed.  </doc>
extern void
  setStackNotify boolean notify_p;

<doc> Does this container receive stack notifications? </doc>
extern boolean
  wantsStackNotify;

<doc> Be notified of a reference from the stack, as request by
    {setStackNotify}.  Default implementation does nothing.  </doc>
void
  gc_stack_notify
{
}

end;

/******************** Copying ********************/

<doc> The {Copying} class defines an interface to copying objects.

    {Copying} inherits from {State} since class objects should not be
    copyable.  Inheriting from {State} ensures that the {Copying} instance
    methods can not be inherited by class objects.  </doc>
implementation class
Copying: State

end;

implementation instance
Copying

<doc> Return a shallow copy of the receiving object.  </doc>
extern id
  copy;

<doc> Return a deep copy of the receiving object.  </doc>
id
  deepCopy
{
  = [[self copy] deepen -1];
}

<doc> Intended to be called on a recently acquired copy of an object,
    {deepen mutable:} deepens the copy.  Iff the optional {mutable_p} is
    {TRUE}, the deepened copies will also be mutable.  The default
    implementation does nothing.

    The {level} should be less than 0 for an infinite deepen.  {length ==
    0} is a nop; iff {length > 0}, every element of the copy is copied and
    deepened with {level - 1}.

    The value returned is {self}.  </doc>
id
    deepen int level
  mutably: boolean mutable_p = NO
{
  = self;
}

<doc> Initialize the receiver just after it has been created as the result
    of a {copy}.  The default implementation does nothing but return
    {self}.  </doc>
id
  initCopy
{
  = self;
}

<doc> Initialize the receiver just after it has been created as the result
    of a {mutableCopy} of the {other} object.  The default implementation
    does nothing but returning {self}.  </doc>
id
  initAsCopyOf All other
{
  = self;
}

<doc> Return a mutable (shallow) copy of the receiving object.  For
    objects which do not discern between mutable and immutable variants,
    the default implementation returns {[self copy]}.

    Mutable copying asks the receiving object for its {mutableCopyClass}.
    If this class is {isa}, {self} is sent a {copy}.  Otherwise, an
    instance of the class is allocated and sent an {initAsCopyOf}.  </doc>
Any
  mutableCopy
{
  class (id) copy_class = class (Copying) ([self mutableCopyClass]);

  = Any ((copy_class == isa) ? [self copy]
	 : [[copy_class alloc] initAsCopyOf self]);
}

<doc> Return the class of the object resulting from a mutable copy of this
    object.  The default implementation simply returns {isa}.  </doc>
class (State)
  mutableCopyClass
{
  = isa;
}

end;

/******************** Enumerable ********************/

implementation class
Enumerable: State

<doc> Invoke {self}'s {withEnumerator} with an {enumerator} from the
    {other}.  </doc>
instance (id)
  withEnumerable Enumerable other
{
  = [self withEnumerator [other enumerator]];
}

<doc> Return a newly allocated instance of the receiving class, filled
    with the elements from the {Enumerator} {e}.  </doc>
instance (id)
  withEnumerator Enumerator e
{
  = [[self alloc] initWithEnumerator e];
}

end;

implementation instance
Enumerable

<doc> Initialize with the elements from the {Enumerator} {e}.  </doc>
deferred protected id
  initWithEnumerator Enumerator e;

<doc> Return an {Enumerator} on the receiving object.  </doc>
deferred Enumerator
  enumerator;

end;

/******************** Enumerator ********************/

implementation class Enumerator end;

implementation instance
Enumerator

<doc> Return a tuple containing the next object, preceded by a boolean
    value indicating whether the end of the enumerable has been reached;
    if the boolean is {TRUE}, the end has not yet been reached.  </doc>
deferred (boolean, Any)
  next;

<doc> Default implementations for direct value retrieving enumerators.  </doc>
(boolean, byte)
  next
{
  boolean b;
  Number o;

  (b, o) = [self next];

  = b ? (b, [o byteValue]) : (b, byte (0));
}

(boolean, char)
  next
{
  boolean b;
  Number o;

  (b, o) = [self next];

  = b ? (b, [o charValue]) : (b, char (0));
}

(boolean, int)
  next
{
  boolean b;
  Number o;

  (b, o) = [self next];

  = b ? (b, [o intValue]) : (b, 0);
}

(boolean, long)
  next
{
  boolean b;
  Number o;

  (b, o) = [self next];

  = b ? (b, [o longValue]) : (b, long (0));
}

(boolean, float)
  next
{
  boolean b;
  Number o;

  (b, o) = [self next];

  = b ? (b, [o floatValue]) : (b, 0.0);
}

(boolean, double)
  next
{
  boolean b;
  Number o;

  (b, o) = [self next];

  = b ? (b, [o doubleValue]) : (b, 0.0d);
}

end;

/*************** MapEnumerator *****************/

implementation class MapEnumerator: Enumerator end;

implementation instance
MapEnumerator

<doc> MapEnumerator allows iteration over both keys and values.  </doc>
deferred (boolean, Any, Any)
  next;

end;
