<copyright> Buffered streams classes.
    Written by <a href="mailto:tiggr@ics.ele.tue.nl">Pieter J. Schoenmakers</a>

    Copyright &copy; 1996-1998 Pieter J. Schoenmakers.

    This file is part of TOM.  TOM is distributed under the terms of the
    TOM License, a copy of which can be found in the TOM distribution; see
    the file LICENSE.

    <id>$Id: StreamBuffer.t,v 1.20 1998/05/31 18:16:54 tiggr Exp $</id>
    </copyright>

// Switching between reading and writing does not work.
// Fri May 24 14:42:16 1996, tiggr@cobra.es.ele.tue.nl
// Maybe that is a feature---try to define precise semantics if not so.
// Sun Sep 15 22:46:57 1996, tiggr@tricky.es.ele.tue.nl

implementation class
BufferedStream: StreamStream, InputOutputStream, Conditions
{
  <doc> The default value of the default size for the buffers of our
      instances.  </doc>
  const DEFAULT_BUFFER_SIZE = 8192;

  <doc> The default size for the buffers of our instances.  If anyone sets
      this to a negative number, s/he should be ni'd.  </doc>
  static int default_buffer_size;
}

void
  load MutableArray arguments
{
  default_buffer_size = DEFAULT_BUFFER_SIZE;
}

end;

implementation instance
BufferedStream
{
  <doc> The buffer we use.  </doc>
  MutableByteArray buffer;

  <doc> The number of elements in the {buffer}. </doc>
  int num;

  <doc> The index of the first character not yet handled (i.e. read or
      written).  </doc>
  int next;
}

<doc> Initialize the newly allocated instance to buffer the stream {s}
      with a buffer sized the {default_buffer_size}.  </doc>
id
  init Stream s
{
  = [self init s bufferSize default_buffer_size];
}

<doc> Designated initializer.  Initialize the newly allocated instance to
    buffer the stream {s} with a buffer sized {cap}.  </doc>
id
	init Stream s
  bufferSize int cap
{
  buffer = [MutableByteArray withCapacity cap];

  = [super init s];
}

<doc> Return the value of the next byte to be returned by {read}, or -1
    upon an error or end-of-file.  This does not actually read the byte.
    </doc>
int
  peek
{
  int in_buffer = num - next;

  if (in_buffer == 0)
    in_buffer = [self readBuffer];

  = in_buffer != 0 ? ({byte b = buffer[next];}) : -1;
}

<doc> Stuff the byte {b} back (sort-of) into the stream.  It will be the
    next byte to be read.  </doc>
void
  unget byte b
{
  if (next > 0)
    buffer[--next] = b;
  else
    {
      [buffer resize (0, 1)];
      buffer[0] = b;
    }
}

/******************** OutputStream ********************/
<doc><h4>{OutputStream} methods</h4></doc>

<doc> Flush any bytes buffered to the stream this instance is buffering.
      </doc>
id
  flushOutput
{
  [self writeBuffer];
  [stream flushOutput];

  = self;
}

<doc> Write the byte {b}, raising a {stream-error} on error.  </doc>
void
  write byte b
{
  int i = [self write b];

  if (!i)
    [[Condition for self class stream-error message "write failed"] raise];
}

<doc> Write the byte {b}, returning 1 upon success.  </doc>
int
  write byte b
{
  int cap = [buffer capacity];

  if (next == cap)
    [self writeBuffer];

  if (next == cap)
    // Return 0?
    // Mon Feb 10 17:55:21 1997, tiggr@akebono.ics.ele.tue.nl
    return 0;

  buffer[num++] = b;
  = 1;
}

<doc> Write to this stream the {length} bytes residing in memory at
    {address}.  </doc>
int
  writeBytes int length
	from pointer address
{
  int in_buffer = num - next;

  if (length > 0)
    {
      int cap = [buffer capacity];

      if (cap < in_buffer + length)
	in_buffer -= [self writeBuffer];

      if (length > cap)
	if (!in_buffer)
	  return [stream writeBytes length from address];
	else
	  [self unimplemented cmd message: "large writes while in_buffer"];
      else
	num += [buffer writeBytes length from address at num];
    }

  = length;
}

/******************** InputStream ********************/
<doc> <h4>{InputStream} methods</h4> </doc>

<doc> Return the next byte, raising a {stream-eos} upon an error or
    end-of-file.  </doc>
byte
  read
{
  int b = [self read];

  if (b == -1)
    [[Condition for self class stream-eos message "end of stream"] raise];

  = byte (b);
}

<doc> Return the value of the next byte read, or -1 upon an error or
    end-of-file.  </doc>
int
  read
{
  int in_buffer = num - next;

  if (in_buffer == 0)
    in_buffer = [self readBuffer];

  = in_buffer != 0 ? ({byte b = buffer[next++];}) : -1;
}

<doc> Read at most {length} bytes into the {destination}, writing them
    from {start}.  Return the number of bytes actually read.  </doc>
int (num_read)
  readRange (int, int) (start, length)
       into MutableByteArray destination
{
  int in_buffer = num - next;

  while (num_read < length)
    {
      if (!in_buffer)
	{
	  in_buffer = [self readBuffer];
	  if (!in_buffer)
	    break;
	}

      int this_read = (length - num_read < in_buffer
		       ? length - num_read : in_buffer);
      int n = [buffer writeRange (next, this_read) into destination at start];
      next += n;
      start += n;
      num_read += n;
      in_buffer -= n;

      if (n < this_read)
	{
	  /* Something bad happened.  */
	  break;
	}
    }
}

<doc><h4>Internal methods</h4></doc>

<doc> Fill the buffer by reading more bytes from the {stream}.  Return the
    number of bytes read.  </doc>
protected int
  readBuffer
{
  if (next == num)
    next = num = 0;

  int n = [stream readRange (num, [buffer capacity] - num) into buffer];
  num += n;

  = n;
}

<doc> Write any bytes needing to be output to the {stream}.  Return the
    number of bytes written.  </doc>
protected int
  writeBuffer
{
  int n;

  if (next != num)
    {
      n = [stream writeRange (next, num - next) from buffer];
      next += n;
      if (next == num)
	next = num = 0;
    }

  = n;
}

end;
