<copyright> Heap class.
    Written by <a href="mailto:tiggr@ics.ele.tue.nl">Pieter J. Schoenmakers</a>

    Copyright &copy; 1997 Pieter J. Schoenmakers.

    This file is part of TOM.  TOM is distributed under the terms of the
    TOM License, a copy of which can be found in the TOM distribution; see
    the file LICENSE.

    <id>$Id: Heap.t,v 1.8 1998/07/21 14:56:04 tiggr Exp $</id>
    </copyright>

implementation class
Heap: State, MutableCollection

end;

implementation instance
Heap
{
  <doc> The array used to store the elements.  </doc>
  MutableArray elements;

  <doc> The number of elements.  This is equal to {[elements length]}.
      </doc>
  public int length;

  <doc> The selector used to have two elements compare themselves.  If
      this isn't set, {i_compare_r} will be used.  </doc>
  public mutable selector compare_selector;

  <doc> If this is {TRUE}, this is a heap of which the root node is the
      largest.  Otherwise, the root is the smallest node.  </doc>
  boolean max_heap;
}

<doc> Invoke {[self init TRUE]}.  </doc>
id
  init
{
  = [self init TRUE];
}

<doc> Designated initializer.  </doc>
id (self)
  init boolean root_is_max
{
  max_heap = root_is_max;
  elements = [MutableObjectArray new];
}

<doc> Return {NO}.  </doc>
boolean
  dump_simple_p
{
}

<doc> Extract the root of the heap, which must be a heap storing the
    minimum value at its root.  </doc>
Any
  extract_min
pre
  !max_heap
{
  = [self extract_root];
}

<doc> Return the minimum value of the heap, which must be a heap storing
    the minimum value at its root.  Return {nil} if the heap is empty.
    </doc>
Any (object)
  min
pre
  !max_heap
{
  if (length > 0)
    object = elements[0];
}

<doc> Extract the root of the heap, which must be a heap which stores the
    maximum value at its root.  </doc>
Any
  extract_max
pre
  max_heap
{
  = [self extract_root];
}

<doc> Return the maximum value of the heap, which must be a heap storing
    the maximum value at its root.  Return {nil} if the heap is empty.
    </doc>
Any (object)
  max
pre
  max_heap
{
  if (length > 0)
    object = elements[0];
}

<doc> Extract and return the root {object} of the heap.  </doc>
Any (object)
  extract_root
pre
  length > 0
post
  length >= 0 && [object heap_index] == 0
{
  object = elements[0];
  [object set_heap_index 0];

  if (--length > 0)
    {
      HeapElement elt = elements[length];

      elements[0] = elt;
      [elt set_heap_index 1];

      if (length > 1)
	[self heapify 1];
    }

  [elements truncate length];
}

<doc> Return the root {object} of the heap, without extracting it.  </doc>
Any (object)
  root
pre
  length > 0
{
  object = elements[0];
}

<doc> Remove the {elt} from the receiving heap.  The {elt} must be an
    element of the heap.  </doc>
void
  remove HeapElement elt
pre
  [elt heap_index] > 0 && elements[[elt heap_index] - 1] == elt
post
  [elt heap_index] == 0
{
  int index = [elt heap_index];

  [elt set_heap_index 0];
  if (index <= --length)
    {
      HeapElement elt = elements[length];

      elements[index - 1] = elt;
      [elt set_heap_index index];

      [self heapify index];
    }

  [elements truncate length];
}

/******************** MutableCollection ********************/

<doc><h4>MutableCollection</h4></doc>

<doc> Add the {object} to this Heap.  It may not yet be part of any
    (other) heap, as stated by the precondition.  </doc>
void
  add HeapElement object
pre
  [object heap_index] == 0
post
  length == 1 + old (length)
{
  int i = ++length;

  while (i > 1)
    {
      HeapElement elt = elements[i / 2 - 1];
      int comp = [self compare (object, elt)];

      if (max_heap ? comp <= 0 : comp >= 0)
	break;

      elements[i - 1] = elt;
      [elt set_heap_index i];
      i /= 2;
    }

  elements[i - 1] = object;
  [object set_heap_index i];
}

void
  addElementsFromEnumerator Enumerator e
{
  if (length != 0)
    [super addElementsFromEnumerator e];
  else
    {
      [elements addElementsFromEnumerator e];
      [self build_heap];
    }
}

void
  empty
{
  length = 0;
  [elements makeElementsPerform
	    [Invocation of selector (void set_heap_index int) with 0]];
  [elements truncate 0];
}

/******************** Enumerable ********************/

<doc><h4>Enumerable</h4></doc>

<doc> Return an enumerator on the elements of this heap.  Note that the
    order of the elements returned is undefined.  </doc>
Enumerator
  enumerator
{
  = [elements enumerator];
}

/******************** internal methods ********************/

<doc><h4>Internal methods</h4></doc>

<doc> Build a heap from the elements already in {elements}.  </doc>
protected void
  build_heap
{
  int i;

  length = [elements length];
  for (i = length / 2; i > 0; i--)
    [self heapify i];
}

<doc> Let {one} compare itself with the {other}, either using the
    {compare_selector}, or, if not set, the {int compare id} method.
    </doc>
int
  compare (Comparable, Comparable) (one, other)
{
  = (!compare_selector ? [one compare other]
     : [one perform compare_selector with other]);
}

<doc> Heapify from the node at index {i} (which is off by 1 compared to
    the index of the element in the {elements} array).  </doc>
protected void
  heapify int index
{
  int left = 2 * index;

  if (left <= length)
    {
      HeapElement elt = elements[left - 1], index_elt = elements[index - 1];
      int comp = [self compare (elt, index_elt)];
      int largest = left;

      if ((max_heap && comp <= 0) || (!max_heap && comp >= 0))
	{
	  largest = index;
	  elt = index_elt;
	}

      int right = left + 1;
      if (right <= length)
	{
	  HeapElement right_elt = elements[right - 1];
	  int comp = [self compare (elt, right_elt)];

	  if ((max_heap && comp <= 0) || (!max_heap && comp >= 0))
	    {
	      largest = right;
	      elt = right_elt;
	    }
	}

      if (largest != index)
	{
	  [elements swap (index - 1, largest - 1)];
	  [elt set_heap_index index];
	  [index_elt set_heap_index largest];
	  [self heapify largest];
	}
    }
}

end;
