/* Implementation of TLTrie class.
   This file is part of TL, Tiggr's Library.
   Written by Pieter J. Schoenmakers<tiggr@es.ele.tue.nl>
   Copyright (C) 1995, 1996 Pieter J. Schoenmakers
   TL is distributed WITHOUT ANY WARRANTY.
   See the file LICENSE in the TL distribution for details.

   $Id: TLTrie.m,v 1.1 1998/01/08 16:12:05 tiggr Exp $  */

#define TLTRIE_DECLARE_PRIVATE_METHODS
#import "tl/support.h"
#import "tl/TLTrie.h"
#import "tl/TLString.h"
#import <ctype.h>

#define NEXT_SINGLE_SUFFIX_P	(start == end)
#define SINGLE_SUFFIX		((TLString *) values)
#define NEXT_AT_INDEX(I)	(((id *) values)[(I)])
#define NEXT_FOR_CHAR(C)	(((id *) values)[(C) - start])

#define PREFIX_P(O)		((O) & TLTO_PREFIX)
#define REVERSED_P(O)		((O) & TLTO_REVERSED)
#define FOLD_CASE_P(O)		((O) & TLTO_FOLD_CASE)

@implementation TLTrie

/* Return a new empty trie.  */
+(TLTrie *) trie
{
  return [[self gcAlloc] init];
} /* +trie */

-(void) gcReference
{
  MARK (object);
  if (NEXT_SINGLE_SUFFIX_P)
    MARK (SINGLE_SUFFIX);
  else
    {
      int i;

      for (i = end - start - 1; i >= 0; i--)
	MARK (NEXT_AT_INDEX (i));
    }
} /* -gcReference */

-(id) objectForChars: (const char *) s limit: (const char *) e
	     options: (enum tltrie_options) options
{
  if (NEXT_SINGLE_SUFFIX_P && SINGLE_SUFFIX)
    if (s == e)
      {
	/* Can't match the OBJECT if we're here for a non-empty suffix.  */
	return nil;
      }
    else
      {
	int sl = [SINGLE_SUFFIX length];
	const char *ss;

	if (PREFIX_P (options) ? e - s < sl : e - s != sl)
	  return nil;

	ss = [SINGLE_SUFFIX cString];
	if (!FOLD_CASE_P (options))
	  return (memcmp (REVERSED_P (options) ? s + (e - s - sl) : s, ss, sl)
		  ? nil : object);
	else
	  {
	    int i, j;

	    for (i = 0, j = e - s - sl; i < sl; i++, j++)
	      if (s[j] != ss[i] && (!isalpha (s[j])
				    || tolower (s[j]) != tolower (ss[i])))
		return nil;
	    return object;
	  }
      }
  else if (s == e)
    return object;
  else
    {
      unsigned char c = REVERSED_P (options) ? *--e : *s++;

      if (FOLD_CASE_P (options) && isupper (c))
	c = tolower (c);

      if (c < start || c >= end)
	return PREFIX_P (options) ? object : nil;
      else
	{
	  id o = [NEXT_FOR_CHAR (c) objectForChars: s limit: e
				options: options];
	  return o ? o : PREFIX_P (options) ? object : nil;
	}
    }
} /* -objectForChars:limit:options: */

-(id) objectForKey: (TLString *) k options: (enum tltrie_options) options
{
  const char *s;
  int l;

  if (!k)
    return nil;

  s = [k cString];
  l = [k length];
  return [self objectForChars: s limit: s + l options: options];
} /* -objectForKey:options: */

-(id) objectForKey: (TLString *) k
{
  return [self objectForKey: k options: 0];
} /* -objectForKey: */

-(id) objectForReversedKey: (TLString *) k
{
  return [self objectForKey: k options: TLTO_REVERSED];
} /* -objectForReversedKey: */

-(id) objectForPrefix: (TLString *) k
{
  return [self objectForKey: k options: TLTO_PREFIX];
} /* -objectForPrefix: */

-(id) objectForReversedPrefix: (TLString *) k
{
  return [self objectForKey: k options: TLTO_PREFIX | TLTO_REVERSED];
} /* -objectForReversedPrefix: */

-(void) pushSuffixWithOptions: (enum tltrie_options) options
{
  TLString *suffix = SINGLE_SUFFIX;
  const char *ss = [suffix cString];
  int sl = [suffix length];
  const char *se = ss + sl;
  unsigned char c = REVERSED_P (options) ? *--se : *ss++;

  if (!NEXT_SINGLE_SUFFIX_P || !SINGLE_SUFFIX)
    abort ();

  if (FOLD_CASE_P (options) && isupper (c))
    c = tolower (c);

  start = c;
  end = start + 1;

  values = xmalloc (sizeof (*values));
  NEXT_AT_INDEX (0) = [TLTrie trie];
  [NEXT_AT_INDEX (0) setObject: object forChars: ss limit: se options: options];
  object = nil;
} /* pushSuffixWithOptions: */

-(id) setObject: (id) o forChars: (const char *) s limit: (const char *) e
	options: (enum tltrie_options) options
{
  /* Push the suffix, in case we have one, one level further.  */
  if (NEXT_SINGLE_SUFFIX_P && SINGLE_SUFFIX)
    [self pushSuffixWithOptions: options];

  /* We could have a match.  */
  if (s == e)
    {
      id previous_object = object;
      ASGN_IVAR (object, o);
      return previous_object;
    }

  /* If we're empty, we can store the suffix, but only if we haven't got
     an object here already.  */
  if (NEXT_SINGLE_SUFFIX_P && !object)
    {
      if (SINGLE_SUFFIX)
	abort ();

      SINGLE_SUFFIX = [CO_TLString stringWithCString: s length: e - s];
      ASGN_IVAR (object, o);
      return nil;
    }

  /* We can't handle this as a suffix.  Face it like a trie.  */
  {
    unsigned char c = REVERSED_P (options) ? *--e : *s++;

    if (FOLD_CASE_P (options) && isupper (c))
      c = tolower (c);

    if (start == end)
      {
	start = c;
	end = start + 1;
	values = xmalloc (sizeof (*values));
	*values = 0;
      }
    else if (c >= end)
      {
	values = xrealloc (values, (c + 1 - start) * sizeof (*values));
	bzero (values + end - start, (c + 1 - end) * sizeof (*values));
	end = c + 1;
      }
    else if (c < start)
      {
	values = xrealloc (values, (end - c) * sizeof (*values));
	memmove (values + start - c, values, (end - start) * sizeof (*values));
	bzero (values, (start - c) * sizeof (*values));
	start = c;
      }

    if (!NEXT_FOR_CHAR (c))
      NEXT_FOR_CHAR (c) = [TLTrie trie];

    return [NEXT_FOR_CHAR (c) setObject: o forChars: s limit: e
			  options: options];
  }
} /* -setObject:forChars:limit:options: */

-(id) setObject: (id) o forKey: (TLString *) k
	options: (enum tltrie_options) options
{
  const char *s;
  int l;

  if (!k)
    return nil;

  s = [k cString];
  l = [k length];
  return [self setObject: o forChars: s limit: s + l options: options];
} /* -setObject:forKey:options: */

-setObject: (id) o forKey: (TLString *) k
{
  return [self setObject: o forKey: k options: 0];
} /* -setObject:forKey: */

-setObject: (id) o forReversedKey: (TLString *) k
{
  return [self setObject: o forKey: k options: TLTO_REVERSED];
} /* -setObject:forReversedKey: */

@end
