/* Implementation of TLSet class.
   This file is part of TL, Tiggr's Library.
   Written by Tiggr <tiggr@es.ele.tue.nl>
   Copyright (C) 1995, 1996 Pieter J. Schoenmakers
   TL is distributed WITHOUT ANY WARRANTY.
   See the file LICENSE in the TL distribution for details.

   $Id: TLSet.m,v 1.1 1998/01/08 16:11:55 tiggr Exp $  */

#define TLSET_DECLARE_PRIVATE_METHODS
#import "tl/support.h"
#import "tl/TLSet.h"
#import "tl/TLSymbol.h"
#import <stdlib.h>

struct element
{
  struct element *cdr;
  id elt;
};

@interface TLSetEnumerator: TLObject
{
  TLSet *set;
  int bucket, depth;
  struct element *e;
}

+(id <TLEnumerator>) enumeratorWithSet: (TLSet *) s;

@end

@implementation TLSet

+(TLSet *) set
{
  return ([[self gcAlloc] init]);
} /* +set */

+(TLSet *) setWithEnumerator: (id <TLEnumerator>) e
{
  TLSet *set = [self set];
  id o;

  while ((o = [e nextObject]) || [e notEndP])
    [set addElement: o];
  
  return (set);
} /* +setWithEnumerator: */

+(TLSet *) setWithElement: (id) o
{
  TLSet *set = [self set];

  [set addElement: o];
  return (set);
} /* +setWithElement: */

+(TLSet *) setWithSequence: seq
{
  return ([self setWithEnumerator: [seq enumerator]]);
} /* +setWithSequence: */

-addElement: e
{
  struct element **pe;
  id po;

  if (2 * nbucket <= nobject)
    [self reconfigure];

  for (pe = &buckets[[e hash] % nbucket];
       *pe && ![e equal: (*pe)->elt]; pe = &(*pe)->cdr);
  if (!*pe)
    {
      *pe = xmalloc (sizeof (**pe));
      (*pe)->cdr = NULL;
      ASGN_IVAR ((*pe)->elt, e);
      nobject++;
      po = nil;
    }
  else
    {
      po = (*pe)->elt;
      ASGN_IVAR ((*pe)->elt, e);
    }
  return (po);
} /* -addElement: */

-(struct element *) _elementAtIndex: (int *) bucket : (int *) depth
{
  struct element *e;
  int i;

  if (*bucket < nbucket)
    {
      for (i = 0, e = buckets[*bucket]; e && i < *depth; e = e->cdr, i++);
      if (!e)
	{
	  *depth = 0;
	  while (++*bucket < nbucket)
	    if (buckets[*bucket])
	      {
		e = buckets[*bucket];
		break;
	      }
	}
      if (e)
	{
	  if (e->cdr)
	    (*depth)++;
	  else
	    {
	      (*bucket)++;
	      *depth = 0;
	    }
	  return (e);
	}
    }
  return (NULL);
} /* _elementAtIndex:: */

-(id <TLEnumerator>) enumerator
{
  static id tlset_enumerator_class;
  if (!tlset_enumerator_class)
    tlset_enumerator_class = [TLSetEnumerator self];

  return ((id) [tlset_enumerator_class enumeratorWithSet: self]);
}

-equal: (id) o
{
  id <TLEnumerator> e;
  int i;

  if (![o isKindOf: isa] || [(TLSet *) o length] != nobject)
    return (nil);

  for (i = 0, e = [o enumerator], o = [e nextObject];
       o || [e notEndP]; i++, o = [e nextObject])
    if (![self member: o])
      return (nil);

  return (Qt);
} /* -equal: */

-(id) member: (id) e
{
  unsigned int hv = [e hash];
  struct element *el;

  if (nbucket)
    for (el = buckets[hv % nbucket]; el; el = el->cdr)
      if ([e equal: el->elt])
	return (el->elt);
  return (nil);
} /* -member: */

-(id) memq: (id) e
{
  unsigned int hv = [e hash];
  struct element *el;

  if (nbucket)
    for (el = buckets[hv % nbucket]; el; el = el->cdr)
      if (el->elt == e)
	return (e);
  return (nil);
} /* -memq: */

-(int) length
{
  return (nobject);
} /* -length */

-objectLength
{
  return ([CO_TLNumber numberWithInt: nobject]);
} /* -objectLength */

-(void) print: (id <TLMutableStream>) stream quoted: (BOOL) qp
{
  id <TLString> format = qp ? @" %#" : @" %@";

  [stream writeBytes: 6 fromBuffer: "(TLSet"];

  if (nbucket)
    {
      struct element *e;
      int i;

      for (i = 0; i < nbucket; i++)
	for (e = buckets[i]; e; e = e->cdr)
	  formac (stream, format, e->elt);
    }

  [stream writeByte: ')'];
} /* -print:quoted: */

-(void) reconfigure
{
  unsigned int i, nn = 2 * nbucket + 1;
  struct element **nb = xcalloc (nn, sizeof (*nb));
  struct element *e, *ne;

  for (i = 0; i < nbucket; i++)
    for (e = buckets[i]; e; e = ne)
      {
	unsigned int hvb = [e->elt hash] % nn;
	ne = e->cdr;
	e->cdr = nb[hvb];
	nb[hvb] = e;
      }

  xfree (buckets);
  buckets = nb;
  nbucket = nn;
} /* -reconfigure */

-removeElement: e;
{
  struct element **pe;
  id o;

  if (!nbucket)
    return (nil);

  for (pe = &buckets[[e hash] % nbucket];
       *pe && ![e equal: (*pe)->elt]; pe = &(*pe)->cdr);
  if (!*pe)
    o = nil;
  else
    {
      struct element *e = *pe;
      nobject--;
      *pe = e->cdr;
      o = e->elt;
      xfree (e);
    }
  return (o);
} /* -removeElement: */

-removeElementIdenticalTo: e;
{
  struct element **pe;
  id o;

  if (!nbucket)
    return (nil);

  for (pe = &buckets[[e hash] % nbucket];
       *pe && e != (*pe)->elt; pe = &(*pe)->cdr);
  if (!*pe)
    o = nil;
  else
    {
      struct element *e = *pe;
      nobject--;
      *pe = e->cdr;
      o = e->elt;
      xfree (e);
    }
  return (o);
} /* -removeElementIdenticalTo: */

-(TLSet *) setByIntersectionWithSequence: (id) seq
{
  id <TLEnumerator> e = [seq enumerator];

  return ([self setByIntersectionWithEnumerator: e]);
} /* -setByIntersectionWithSequence: */

-(TLSet *) setByIntersectionWithEnumerator: (id <TLEnumerator>) e
{
  TLSet *s = [TLSet set];
  id o;

  while ((o = [e nextObject]) || [e notEndP])
    if ([self member: o])
      [s addElement: o];
  return (s);
} /* -setByIntersectionWithEnumerator: */

-(TLSet *) setByUnionWithSequence: (id) seq
{
  id <TLEnumerator> e = [seq enumerator];

  return ([self setByUnionWithEnumerator: e]);
} /* -setByUnionWithSequence: */

-(TLSet *) setByUnionWithEnumerator: (id <TLEnumerator>) e
{
  return ([[TLSet setWithSequence: self] unionWithEnumerator: e]);
} /* -setByUnionWithEnumerator: */

-(TLSet *) unionWithSequence: (id) seq
{
  id <TLEnumerator> e = [seq enumerator];

  return ([self unionWithEnumerator: e]);
} /* -unionWithSequence: */

-(TLSet *) unionWithEnumerator: (id <TLEnumerator>) e
{
  id o;

  while ((o = [e nextObject]) || [e notEndP])
    [self addElement: o];
  return (self);
} /* -unionWithEnumerator: */

/******************** garbage collection ********************/

-(void) dealloc
{
  xfree (buckets);
} /* -dealloc */

-(void) gcReference
{
  struct element *e;
  unsigned int i;

  for (i = 0; i < nbucket; i++)
    for (e = buckets[i]; e; e = e->cdr)
      MARK (e->elt);
} /* -gcReference */

@end

@implementation TLSetEnumerator

/******************** private methods ********************/

-initWithSet: (TLSet *) d
{
  ASGN_IVAR (set, d);
  return (self);
} /* -initWithSet: */

/******************** public methods ********************/

+(TLSetEnumerator *) enumeratorWithSet: (TLSet *) d
{
  return ([[self gcAlloc] initWithSet: d]);
} /* +enumeratorWithSet: */

/******************** TLEnumerator ********************/

-notEndP
{
  return (e ? Qt : nil);
} /* -notEndP */

-nextObject
{
  e = [set _elementAtIndex: &bucket : &depth];
  return (e ? e->elt : nil);
} /* -nextObject */

/******************** garbage collection ********************/

-(void) gcReference
{
  MARK (set);
} /* -gcReference */

@end
