/* Implementation of TLFILEStream class.
   This file is part of TL, Tiggr's Library.
   Written by Tiggr <tiggr@es.ele.tue.nl>
   Copyright (C) 1995, 1996 Pieter J. Schoenmakers
   TL is distributed WITHOUT ANY WARRANTY.
   See the file LICENSE in the TL distribution for details.

   $Id: TLFILEStream.m,v 1.1 1998/01/08 16:11:35 tiggr Exp $  */

#import "tl/support.h"
#import "tl/TLFILEStream.h"
#import "tl/TLString.h"
#import "tl/TLPatchedRoots.h"
#import <string.h>

id <TLInputStream> V_stdin_;
id <TLOutputStream> V_stdout_;
id <TLInputOutputStream> V_stderr_;

@implementation TLFILEStream

+init
{
  if (!V_stdin_)
    {
      V_stdin_ = [self mutableStreamWithFILE: stdin];
      V_stdout_ = [self mutableStreamWithFILE: stdout];
      V_stderr_ = [self mutableStreamWithFILE: stderr];
    }
  return (self);
} /* +init */

+(id <TLStream>) streamWithFILE: (FILE *) an_f
{
  return ([[self gcAlloc] initWithFILE: an_f]);
} /* -streamWithFILE: */

+(id <TLMutableStream>) mutableStreamWithFILE: (FILE *) an_f
{
  return ([[self gcAlloc] initWithFILE: an_f]);
} /* -mutableStreamWithFILE: */

+(id <TLStream>) streamWithFileNamed: (id <TLString>) name
{
  FILE *af = fopen ([name cString], "r");
  if (!af)
    {
      [self warning: "open %#: %s", name, ERRMSG];
      return (nil);
    }
  return ([self streamWithFILE: af]);
} /* +streamWithFileNamed: */

+(id <TLStream>) streamWithFileNamed: (id <TLString>) name
 alongPath: (TLCons *) in_path
{
  TLCons *dir, *path = in_path;
  TLString *filename;
  FILE *af;

  if (!path || [[name string] fileExistsP])
    return ([self streamWithFileNamed: name]);

  while (path)
    {
      DECONS (path, dir, path);
      filename = [CO_TLString stringWithFormat: @"%@/%@", dir, name];
      af = fopen ([filename cString], "r");
      if (af)
	return ([self streamWithFILE: af]);
    }
  [self warning: "file %# not found along %#", name, in_path];
  return (nil);
} /* +streamWithFileNamed:alongPath: */

+(id <TLMutableStream>) mutableStreamWithFileNamed: (id <TLString>) name
{
  const char *s = [name cString];
  FILE *af = fopen (s, "w+");

  if (!af)
    {
      [self warning: "open %#: %s", name, ERRMSG];
      return (nil);
    }
  return ([self mutableStreamWithFILE: af]);
} /* +mutableStreamWithFileNamed: */

-initWithFILE: (FILE *) an_f
{
  f = an_f;
  return (self);
} /* -initWithFILE: */

-(void) print: (id <TLMutableStream>) s quoted: (BOOL) qp
{
  formac (s, @"#<%s %p on %p fd=%d eof=%d error=%d>",
	  class_get_class_name (isa), self, f,
	  f ? fileno (f) : 0, f ? feof (f) : 0, f ? ferror(f) : 0);
} /* -print:quoted: */

/******************** TLStream ********************/

-close
{
  FILE *file = f;
  f = NULL;
  return (fclose (file) ? nil : self);
} /* -fclose */

-(int) fileDescriptor
{
  return (f ? fileno (f) : -1);
} /* -fileDescriptor */

-streamp
{
  return (self);
} /* -streamp */

/******************** TLInputStream ********************/

-flushInput
{
  /* We don't do flush input, because of the horrible semantics of
     read/write FILEs.  */
  return (self);
} /* -flushInput */

-(int) readByte
{
  return (getc (f));
} /* -readByte */

-(int) readBytes: (int) n intoBuffer: (char *) b
{
  return (fread (b, 1, n, f));
} /* -readBytes:intoBuffer: */

/******************** TLOutputStream ********************/

-flushOutput
{
  return (fflush (f) ? nil : self);
} /* -flushOutput */

-(int) writeByte: (char) c
{
  return (putc (c, f));
} /* -writeByte: */

-(int) writeBytes: (int) n fromBuffer: (const char *) b
{
  return (fwrite (b, 1, n, f));
} /* -writeBytes:fromBuffer: */

/******************** TLMutableStream ********************/

-(long) _tell
{
  return (ftell (f));
} /* -_tell */

-(long) _seek: (long) offset from: (int) pos
{
  int r;

  switch (pos)
    {
    case TLSEEK_ABSOLUTE: r = SEEK_SET; break;
    case TLSEEK_FROM_END: r = SEEK_END; break;
    case TLSEEK_RELATIVE: r = SEEK_CUR; break;
    default: [self error: "bad TLSEEK %d", pos]; r = 0; break;
    }

  return (fseek (f, offset, r));
} /* -_seek:from: */

@end
