/* Implementation of TLCons class.
   This file is part of TL, Tiggr's Library.
   Written by Tiggr <tiggr@es.ele.tue.nl>
   Copyright (C) 1995, 1996 Pieter J. Schoenmakers
   TL is distributed WITHOUT ANY WARRANTY.
   See the file LICENSE in the TL distribution for details.

   $Id: TLCons.m,v 1.1 1998/01/08 16:11:32 tiggr Exp $  */

#import "tl/support.h"
#import "tl/TLCons.h"
#import "tl/TLSymbol.h"
#import "tl/subr.h"
#import "tl/TLGC.h"
#import "tl/predicates.h"

id tlcons_class;

@interface TLConsEnumerator: TLObject <TLEnumerator>
{
  TLCons *cell;
}

/******************** creation ********************/

-initWithCons: (TLCons *) cons;

@end

@implementation TLCons

+initialize
{
  tlcons_class = self;
  return (self);
} /* +initialize */

+(TLCons *) cons: o_car : o_cdr
{
  return ([[self gcAlloc] initWithCar: o_car cdr: o_cdr]);
} /* +cons:: */

+(TLCons *) listWithEnumerator: (id <TLEnumerator>) e
{
  TLCons *list = nil, *last = nil;
  id o;

  while ((o = [e nextObject]) || [e notEndP])
    {
      TLCons *c = CONS (o, nil);

      if (last)
	{
	  [last setCdr: c];
	  last = c;
	}
      else
	last = list = c;
    }

  return list;
} /* +listWithEnumerator: */

+(TLCons *) listWithSequence: (id) s
{
  return [self listWithEnumerator: [s enumerator]];
} /* +listWithSequence: */

-car
{
  return (car);
} /* -car */

-(void) car: (id *) a cdr: (id *) d
{
  *a = car;
  *d = cdr;
} /* -car:cdr: */

-cdr
{
  return (cdr);
} /* -cdr */

-(int) compare: o
{
  int r;

  if (!CONSP (o))
    return (1);

  r = [car compare: [o car]];
  return (r ? r : [cdr compare: [o cdr]]);
} /* -compare: */

-consp
{
  return (Qt);
} /* -consp */

-delq: o
{
  if (cdr)
    cdr = [cdr delq: o];
  if (car == o)
    return (cdr);
  return (self);
} /* -delq: */

-delq: o count: (unsigned int) n
{
  if (!n)
    return (self);
  if (car != o)
    {
      cdr = [cdr delq: o count: n];
      return (self);
    }
  cdr = [cdr delq: o count: n - 1];
  return (cdr);
} /* -delq:count: */

-(id <TLEnumerator>) enumerator
{
  static id tlcons_enumerator_class;
  if (!tlcons_enumerator_class)
    tlcons_enumerator_class = [TLConsEnumerator self];

  return ([[tlcons_enumerator_class gcAlloc] initWithCons: self]);
}

-eval
{
  if (tlgc_total_threshold
      && (TLGC_TOTAL_ALLOC_SINCE_COMPLETE >= tlgc_total_threshold))
    if (tlgc_alloc_limit && tlgc_num_alloc >= tlgc_alloc_limit)
      tl_garbage_collect (0);
    else
      tl_garbage_collect (tlgc_total_time_limit);
  else if (tlgc_partial_threshold
	   && (tlgc_alloc_since_partial >= tlgc_partial_threshold))
    if (tlgc_alloc_limit && tlgc_num_alloc >= tlgc_alloc_limit)
      tl_garbage_collect (0);
    else
      tl_garbage_collect (tlgc_partial_time_limit);

  return (EVAL_WITH_ARGS (car, cdr));
} /* -eval */

-evalWithArguments: (TLCons *) args
{
  return (EVAL_WITH_ARGS (EVAL_WITH_ARGS (car, cdr), args));
} /* -evalWithArguments: */

-(void) gcReference
{
  /* Do not invoke super's for the sake of speed.  */
  MARK (car);
  MARK (cdr);
} /* -gcReference */

-initWithCar: o_car cdr: o_cdr
{
  /* Do not invoke super's init for the sake of speed.  */
  ASGN_IVAR (car, o_car);
  ASGN_IVAR (cdr, o_cdr);
  return (self);
} /* -initWithCar:cdr: */

-(int) _lengthPlus: (int) prefix_length
{
  return (cdr ? [cdr _lengthPlus: 1 + prefix_length] : 1 + prefix_length);
} /* -_lengthPlus: */

-(int) length
{
  return (cdr ? [cdr _lengthPlus: 1] : 1);
} /* -length */

-objectLength
{
  return ([CO_TLNumber numberWithInt: [self length]]);
} /* -objectLength */

-(int) _lengthWithoutMembersOf: (TLCons *) list
{
  return ((list && [list memq: car] ? 0 : 1) + (cdr ? [cdr length] : 0));
} /* -lengthWithoutMembersOf: */

-mapcar: (TLSymbol *) sym
{
  id retval;
  GCDECL1;

  GCPRO1 (sym);
  retval = CONS (EVAL_WITH_ARGS (sym, CONS (CONS (Qquote,
						  CONS (car, nil)), nil)),
		 (cdr ? [cdr mapcar: sym] : nil));
  GCUNPRO;
  return (retval);
} /* -mapcar: */

-memq: elt
{
  if (car == elt)
    return (self);
  return (cdr ? [cdr memq: elt] : nil);
} /* -memq: */

-nconc: o
{
  if (cdr)
    [cdr nconc: o];
  else
    ASGN_IVAR (cdr, o);
  return (self);
} /* -nconc: */

-(void) print: (id <TLMutableStream>) stream quoted: (BOOL) qp
{
  [stream writeByte: '('];
  print (car, stream, qp);
  if (cdr)
    print_list_element (cdr, stream, qp);
  else
    [stream writeByte: ')'];
} /* -print:quoted: */

-(void) printListElement: (id <TLMutableStream>) stream quoted: (BOOL) qp
{
  [stream writeByte: ' '];
  print (car, stream, qp);
  if (cdr)
    print_list_element (cdr, stream, qp);
  else
    [stream writeByte: ')'];
} /* -printListElement:quoted: */

-rplaca: o
{
  ASGN_IVAR (car, o);
  return (self);
} /* -rplaca: */

-rplacd: o
{
  ASGN_IVAR (cdr, o);
  return (self);
} /* -rplacd: */

-(void) setCar: o
{
  ASGN_IVAR (car, o);
} /* -setCar: */

-(void) setCdr: o
{
  ASGN_IVAR (cdr, o);
} /* -setCdr: */

@end

@implementation TLConsEnumerator

-initWithCons: (TLCons *) cons
{
  ASGN_IVAR (cell, cons);
  return (self);
} /* -initWithCons: */

/******************** TLEnumerator ********************/

-nextObject
{
  id r;

  if (cell)
    DECONS (cell, r, cell);
  else
    r = nil;
  return (r);
} /* -nextObject */

-notEndP
{
  return (cell ? Qt : nil);
} /* -notEndP */

/******************** garbage collection ********************/

-(void) gcReference
{
  MARK (cell);
} /* -gcReference */

@end
