/* Implementation of TLBufferedStream class.
   This file is part of TL, Tiggr's Library.
   Written by Tiggr <tiggr@es.ele.tue.nl>
   Copyright (C) 1995, 1996 Pieter J. Schoenmakers
   TL is distributed WITHOUT ANY WARRANTY.
   See the file LICENSE in the TL distribution for details.

   $Id: TLBufferedStream.m,v 1.1 1998/01/08 16:11:31 tiggr Exp $  */

#define TLBUFFEREDSTREAM_DECLARE_PRIVATE_METHODS

#import "tl/support.h"
#import "tl/TLBufferedStream.h"
#import "tl/TLString.h"
#import <string.h>

@implementation TLBufferedStream

-(int) copyBytes: (int) n fromBuffer: (const char *) b
{
  return [stream writeBytes: n fromBuffer: b];
} /* -copyBytes:fromBuffer: */

-(int) copyBytes: (int) n intoBuffer: (char *) b
{
  return [stream readBytes: n intoBuffer: b];
} /* -copyBytes:intoBuffer: */

-(void) emptyBuffer
{
  next = num = 0;
} /* -emptyBuffer */

-initWithStream: (id) str
{
  if (![super initWithStream: str])
    return (nil);

  cap = 512;
  buf = xmalloc (cap);
  return (self);
} /* -initWithStream: */

-(int) readBytes: (int) n toOffset: (int) off
{
  return [stream readBytes: n intoBuffer: buf + off];
} /* -readBytes:toOffset: */

-(TLString *) readLine
{
  return ([self readLineIntoInstanceOf: [CO_TLString class]]);
} /* -readLine */

-(id <TLString>) readLineIntoInstanceOf: (id <TLStringCreationUsage>) class;
{
  int i, n;
  id r;

  if (eof)
    return (0);

  if (buf_write_p)
    [self flushOutput];

  for (i = next; i < num; i++)
    if (buf[i] == '\n')
      break;

  if (i < num)
    i++;
  else while (i == num && !eof)
    {
      cap *= 2;
      buf = xrealloc (buf, cap);
      n = [self readBytes: cap - num toOffset: num];
      if (!n && num - next)
	break;
      else if (n <= 0)
	eof = 1;

      for (i = num; i < num + n; i++)
	if (buf[i] == '\n')
	  {
	    i++;
	    break;
	  }
      num += n;
    }

  r = [class stringWithCString: buf + next length: i - next];
  next = i;
  return (r);
} /* -readLineIntoInstanceOf: */

-(TLMutableString *) readMutableLine
{
  return ([self readLineIntoInstanceOf: [CO_TLMutableString class]]);
} /* -readMutableLine */

/******************** TLStream ********************/

-close
{
  [self flushOutput];
  eof = 1;
  return [super close];
} /* -close */

/******************** TLInputStream ********************/

-(int) readByte
{
  unsigned char c;
  int i;

  i = [self readBytes: 1 intoBuffer: &c];
  return (i == 1 ? c : TL_EOF);
} /* -readByte */

-(int) readBytes: (int) n intoBuffer: (char *) b
{
  int i, inbuf;

  if (eof)
    return 0;

  if (buf_write_p)
    [self flushOutput];

  inbuf = num - next;
  if (n <= inbuf)
    {
      memcpy (b, buf + next, n);
      next += n;
      return n;
    }

  /* XXX For this kind of non-atomic reads, what to do upon errors?  For
     now, return -1 and set EOF.  */
  memcpy (b, buf + next, inbuf);
  [self emptyBuffer];
  if (n - inbuf > cap)
    /* Read straight into the caller's buffer.  */
    i = [self copyBytes: n - inbuf intoBuffer: b + inbuf];
  else
    {
      i = [self readBytes: cap toOffset: 0];
      if (i > 0)
	{
	  num = i;
	  if (n - inbuf <= num)
	    i = next = n - inbuf;
	  else
	    next = i;
	  memcpy (b + inbuf, buf, next);
	}
    }

  if (i <= 0)
    {
      [self emptyBuffer];
      eof = 1;
      if (i < 0)
	return (-1);
    }

  return (i + inbuf);
} /* -readBytes:intoBuffer: */

/******************** TLBufferedInputStream ********************/

-(int) unreadByte: (char) c
{
  return ([self unreadBytes: 1 fromBuffer: &c] == 1 ? c : TL_EOF);
} /* -readByte */

-(int) unreadBytes: (int) n fromBuffer: (const char *) b
{
  if (buf_write_p)
    [self flushOutput];

  if (n > next)
    {
      int new_num = num - next + n;

      if (new_num > cap)
	{
	  cap = new_num;
	  buf = xrealloc (buf, cap);
	}
      memmove (buf + n, buf + next, num - next);
      num += n - next;
      next = n;
    }
  memcpy (buf + next - n, b, n);
  next -= n;
  return n;
} /* -unreadBytes:fromBuffer: */

/******************** TLOutputStream ********************/

-flushOutput
{
  if (!buf_write_p)
    return self;

  {
    BOOL ok = YES;
    int n = num;

    [self emptyBuffer];
    buf_write_p = 0;
    if ([stream writeBytes: n fromBuffer: buf] != n)
      {
	eof = 1;
	ok = NO;
      }
    return ([stream flushOutput] && ok ? self : nil);
  }
} /* -flushOutput */

-(int) writeByte: (char) c
{
  int i = [self writeBytes: 1 fromBuffer: &c];
  return (i == 1 ? c : TL_EOF);
} /* -writeByte: */

-(int) writeBytes: (int) n fromBuffer: (const char *) b
{
  if (eof)
    return (0);

  if (!buf_write_p && !num)
    buf_write_p = 1;

  if (buf_write_p)
    {
      /* If it's not too much, copy it into our buffer.  */
      if (num + n <= cap)
	{
	  memcpy (buf + num, b, n);
	  num += n;
	  return n;
	}

      if (![self flushOutput])
	{
	  eof = 1;
	  return -1;
	}
    }

  return [self copyBytes: n fromBuffer: b];
} /* -writeBytes:fromBuffer: */

-(int) _writeBytes: (int) n toStream: (id <TLOutputStream>) s
{
  /* To do.  */
  abort ();
} /* _writeBytes:toStream: */

/******************** garbage collection ********************/

-(void) dealloc
{
  xfree (buf);
} /* -dealloc */

@end
