/* Implementation of TLBufferedMutableStream class.
   This file is part of TL, Tiggr's Library.
   Written by Tiggr <tiggr@es.ele.tue.nl>
   Copyright (C) 1995, 1996 Pieter J. Schoenmakers
   TL is distributed WITHOUT ANY WARRANTY.
   See the file LICENSE in the TL distribution for details.

   $Id: TLBufferedMutableStream.m,v 1.1 1998/01/08 16:11:30 tiggr Exp $  */

#define TLBUFFEREDMUTABLESTREAM_DECLARE_PRIVATE_METHODS

#import "tl/support.h"
#import "tl/TLBufferedMutableStream.h"
#import "tl/TLString.h"
#import <string.h>

@implementation TLBufferedMutableStream

-(int) copyBytes: (int) n fromBuffer: (const char *) b
{
  int i;

  if ([stream _seek: off from: TLSEEK_ABSOLUTE] != off)
    return -1;

  i = [super copyBytes: n fromBuffer: b];

  if (i > 0)
    off += i;
  return i;
} /* -copyBytes:fromBuffer: */

-(int) copyBytes: (int) n intoBuffer: (char *) b
{
  int i;

  if ([stream _seek: off from: TLSEEK_ABSOLUTE] != off)
    return -1;

  i = [super copyBytes: n intoBuffer: b];

  if (i > 0)
    off += i;
  return i;
} /* -copyBytes:intoBuffer: */

-(void) emptyBuffer
{
  off += num;
  [super emptyBuffer];
} /* -emptyBuffer */

-(int) readBytes: (int) n toOffset: (int) buf_off
{
  if ([stream _seek: off + buf_off from: TLSEEK_ABSOLUTE] != off + buf_off)
    return -1;

  return [stream readBytes: n intoBuffer: buf + buf_off];
} /* -readBytes:toOffset: */

-(long) _seek: (long) new_offset from: (int) pos
{
  switch (pos)
    {
    case TLSEEK_ABSOLUTE:
      break;
    case TLSEEK_FROM_END:
      new_offset = [stream _seek: new_offset from: pos];
      break;
    case TLSEEK_RELATIVE:
      new_offset = off + new_offset;
      if (new_offset >= 0)
	break;
      /* Fall through.  */
    default:
      [self error: "bad TLSEEK %d (to %d)", pos, new_offset];
      return -1;
    }

  if (new_offset >= off && new_offset < off + num)
    next = new_offset - off;
  else
    {
      [self emptyBuffer];
      off = new_offset;
    }
  return off + next;
} /* -_seek:from: */

-(long) _tell
{
  return off + next;
} /* _tell */

-(int) unreadBytes: (int) n fromBuffer: (const char *) b
{
  int r, prev_num;

  if (buf_write_p)
    [self flushOutput];

  prev_num = num;
  r = [super unreadBytes: n fromBuffer: b];
  off -= num - prev_num;
  return r;
} /* -unreadBytes:fromBuffer: */

-(int) _writeRange: (id <TLRange>) r toStream: (id <TLOutputStream>) s
{
  int start = [r _start];
  int n;

  /* Iff the buffered data does not overlap with what is to be written,
     flush it.  */
  if (start < off || start >= off + num)
    [self emptyBuffer];

  /* Now the buffer is empty, we might as well use a straight write.
     XXX If the STREAM understands that...  */
  if (!num && [stream respondsTo: @selector (_writeRange:toStream:)])
    {
      n = [stream _writeRange: r toStream: s];
      if (n > 0)
	off += n;
    }
  else
    {
      int to_be_read, length = [r length];

      /* Fill the buffer, resizing iff needed.  */
      to_be_read = start + length - (off + num);
      if (num + to_be_read > cap)
	{
	  cap = num + to_be_read;
	  buf = xrealloc (buf, cap);
	}
      n = [self readBytes: to_be_read toOffset: num];
      if (n > 0)
	num += n;

      next = start - off;
      length = num - next;
      n = [s writeBytes: length fromBuffer: buf + next];
      if (n > 0)
	next += n;
    }
  return n;
} /* _writeRange:toStream: */

@end
