#|------------------------------------------------------------*-Scheme-*--|
 | File:	    packages/threads/manager/processio.scm
 |
 |          Copyright (C)1998 Donovan Kolbly <d.kolbly@rscheme.org>
 |          as part of the RScheme project, licensed for free use.
 |	    See <http://www.rscheme.org/> for the latest info.
 |
 | File version:     1.5
 | File mod date:    1999.01.08 13:32:52
 | System build:     v0.7.3.1-b39, 1999-12-25
 | Owned by module:  rs.sys.threads.manager
 |
 | Purpose:          Thread-safe Process I/O (pipes & stuff)
 `------------------------------------------------------------------------|#

;;;-----------------------------------------------------------------------
;;;			 Process Environment
;;;-----------------------------------------------------------------------


(define *process-environment* #f)
(define *process-environment-as-vector* #f)

(set! *process-environment* #f) ;; reset when process starts
(set! *process-environment-as-vector* #f)

(define (process-environment)
  (if (not *process-environment*)
      (let ((tbl (make-string-table)))
	(for-each (lambda ((e <pair>))
		    (table-insert! tbl (car e) (cdr e)))
		  (the-environ))
	(set! *process-environment* tbl)))
  *process-environment*)

(define (process-environment-as-vector)
  (if (not *process-environment-as-vector*)
      (set! *process-environment-as-vector*
	    (let (((tbl <string-table>) (process-environment)))
	      (vector-map
	       (lambda (k)
		 (string-append k "=" (table-lookup tbl k)))
	       (sort (list->vector (key-sequence tbl)) string<?)))))
  *process-environment-as-vector*)

;;;-----------------------------------------------------------------------
;;;		       Ways to run a subprocess
;;;-----------------------------------------------------------------------

(define-syntax (as-string object)
  (if (string? object)
      object
      (to-string object)))

(define (run cmd . args)
  (run* cmd args '#(0 1 2)))

;; returns a port which accesses the output of the command

(define (run->port cmd . args)
  (bind ((r w (pipe))
	 (p (run* cmd args (vector 0 w 2))))
    (fd-close w)
    (values (open-mbox-input-port r) p)))

;; returns a port which feeds the input of the command

(define (port->run cmd . args)
  (bind ((r w (pipe))
	 (p (run* cmd args (vector r 1 2))))
    (fd-close r)
    (values (open-queued-output w) p)))

;;

(define (run* cmd args fds)
  (let* ((cmd-str (as-string cmd))
	 (cmd-file (if (string-search cmd-str #\/)
		       (if (file-access? cmd-str (access-mask execute))
			   cmd-str
			   (error "~a: not accessible in `execute' mode" cmd))
		       (find-in-path cmd-str)))
	 (p (make <process>
		  name: cmd)))
    (fork-and-exec
     p
     cmd-file
     (list->vector (cons cmd-file
			 (map (lambda (a)
				(as-string a))
			      args)))
     (process-environment-as-vector)
     fds)
    p))

;;;-----------------------------------------------------------------------
;;;	     Thread-aware implementations of process I/O
;;;-----------------------------------------------------------------------

;;;			 --- Input Ports ---

(define-class <process-input-port> (<input-port>)
  (name type: <string>)
  underlying-input-port
  process)

(define (open-input-process/thread (str <string>))
  (bind ((port proc (run->port "sh" "-c" str)))
    (set-name! proc str)
    (make <process-input-port>
	  name: str
	  underlying-input-port: port
	  process: proc)))

(define-delegation ((self <process-input-port>) (underlying-input-port self))
  (input-port-read-char self)
  (input-port-peek-char self)
  (input-port-scan-token self)
  (input-port-read self)
  (input-port-read-line self)
  (input-port-read-rest self)     ;; / constituents of internal
  (input-port-read-len self len)) ;; \ read-string protocol


(define-method close-input-port ((self <process-input-port>))
  ;; this might cause it to die on a SIGPIPE...
  (let ((fd (fd (underlying-input-port self))))
    ;; this might cause it to die on a SIGPIPE...
    (close-input-port (underlying-input-port self))
    (fd-close fd)
    (check-exit-status (process self))))

;;;			 --- Output Ports ---

(define-class <process-output-port> (<output-port>)
  (name type: <string>)
  underlying-output-port
  process)

(define-delegation ((self <process-output-port>) (underlying-output-port self))
  (flush-output-port self)
  (output-port-write-char self char)
  (write-string self string))

(define-method close-output-port ((self <process-output-port>))
  (let ((fd (fd (underlying-output-port self))))
    (close-output-port (underlying-output-port self))
    (fd-close fd)
    (check-exit-status (process self))))

(define (open-output-process/thread (str <string>))
  (bind ((port proc (port->run "sh" "-c" str)))
    (set-name! proc str)
    (make <process-output-port>
	  name: str
	  underlying-output-port: port
	  process: proc)))

;;;
;;;  install the hooks
;;;

(%early-once-only
 (set-process-io-proc! 'open-input-process open-input-process/thread)
 (set-process-io-proc! 'open-output-process open-output-process/thread))
