
(define (xterm-readline prompt inp out (state <readline-state>))
  (let ((reset-thunk (init-for-readline 0)))
    (if (not (ever-read-from? state))
	(begin
	  (on-exit reset-thunk)
	  (set-ever-read-from?! state #t)))
    (let ((s (xterm-readline* prompt inp out state)))
      (reset-thunk)
      s)))

(define (xterm-readline* prompt inp out (state <readline-state>))
  ;
  (let ((left (make-dequeue))
	(right (make-dequeue))
	(prev (map (lambda (h)
		     (cons h (string-length h)))
		   (history state)))
	(succ '())
	(undo '()))
    ;;
    (define (move-left)
      (dequeue-push-front! right (dequeue-pop-back! left))
      (write-string out "\10"))
    ;
    (define (move-right)
      (dequeue-push-back! left (dequeue-pop-front! right))
      (write-string out "\033[C"))
    ;
    (define (insert-right ch)
      (if (not (dequeue-empty? right))
	  (write-string out "\033[@"))
      (write-char ch out)
      (dequeue-push-back! left ch)
      (values))
    ;
    (define (clear-to-eol)
      (write-string out "\033[K"))
    ;
    (define (beep)
      (write-string out "\7"))
    ;
    (define (can-undo)
      (set! undo (cons (current-state) undo)))
    ;
    (define (delete-right)
      (dequeue-pop-front! right)
      (write-string out "\033[P"))
    ;
    (define (current-state)
      (cons (list->string
	     (vector->list
	      (vector-append (dequeue-state left)
			     (dequeue-state right))))
	    (dequeue-count left)))
    ;
    (define (load-state s)
      ;; first, erase what we have
      (write-string out "\033\.8") ;; restore cursor
      (clear-to-eol)
      ;(write-string out "\033[1m") ;; bold
      (write-string out prompt)
      ;(write-string out "\033[m") ;; end bold
      (write-string out (car s))
      (set! left (substring->dequeue (car s) 0 (cdr s)))
      (set! right (substring->dequeue (car s) (cdr s) (string-length (car s))))
      (let ((goback (- (string-length (car s)) (cdr s))))
	(if (not (zero? goback))
	    (write-string out (format #f "\033[~dD" goback))))) ;; move left
    ;
    (write-string out "\033\.7") ;; save cursor
    (load-state '("" . 0))
    ;;
    (let loop ()
      (flush-output-port out)
      (let ((ch (read-char inp)))
	;(format #t "(~s)" ch)
	(case ch
	  ((#\return #\newline)
	   (newline)
	   (car (current-state)))
	  ;; ------------------------------ QUERY -------------------------
	  ((#\C-o)
	   (write-string out "\033[L\015")
	   (write-string out "+--------------------------------+\n")
	   (write-string out "| A message from the sponsor...  |\n")
	   (format out "| current => ~s\n" (current-state))
	   (format out "| prev => ~s\n" (if (pair? prev)
					    (car prev)
					    'none))
	   (format out "| succ => ~s\n" (if (pair? succ)
					    (car succ)
					    'none))
	   (write-string out "+--------------------------------+\n")
	   ;(write-string out (make-string (dequeue-count right) #\bs))
	   (write-string out "\033\.7") ;; re-save cursor on this line
	   (load-state (current-state))
	   (loop))
	  ;; ------------------------------ CURSOR -------------------------
	  ((#\C-a)
	   (while (not (dequeue-empty? left))
	     (move-left))
	   (loop))
	  ((#\C-e)
	   (while (not (dequeue-empty? right))
	     (move-right))
	   (loop))
	  ((#\C-b)
	   (if (not (dequeue-empty? left))
	       (move-left))
	   (loop))
	  ((#\C-f)
	   (if (not (dequeue-empty? right))
	       (move-right))
	   (loop))
	  ;; ------------------------------ COMPLETION -----------------------
	  ((#\tab)
	   (let* ((pre (collect-prefix left state))
		  (c (collect-completions state pre)))
	     (cond
	      ((null? c)
	       ;; no completions (c will be null if pre is empty)
	       (write-string out "\7"))
	      ((null? (cdr c))
	       ;; unique completion
	       (can-undo)
	       (for-each insert-right 
			 (list-tail (string->list (car c))
				    (string-length pre)))
	       (insert-right #\space))
	      (else
	       (let ((com (common-prefix c)))
		 (if (<= (string-length com) (string-length pre))
		     (write-string out "\7") ;; no more to complete
		     (begin
		       (can-undo)
		       (for-each insert-right 
				 (string->list (substring com
							  (string-length pre)))))))))
	     (loop)))
	  ;; ------------------------------ EDITING -------------------------
	  ((#\C-k)
	   (can-undo)
	   (set! right (make-dequeue))
	   (clear-to-eol)
	   (loop))
	  ((#\C-u)
	   (can-undo)
	   (load-state '("" . 0))
	   (loop))
	  ((#\del #\C-h)
	   (if (not (dequeue-empty? left))
	       (begin
		 (can-undo)
		 (move-left)
		 (delete-right)))
	   (loop))
	  ((#\C-d)
	   (if (and (dequeue-empty? right)
		    (dequeue-empty? left))
	       (with-input-from-string "" read-char)
	       (begin
		 (if (not (dequeue-empty? right))
		     (begin
		       (can-undo)
		       (delete-right)))
		 (loop))))
	  ;; ------------------------------ HISTORY --------------------
	  ((#\C-_)
	   (if (pair? undo)
	       (begin
		 (load-state (car undo))
		 (set! undo (cdr undo))))
	   (loop))
	  ((#\C-p)
	   (if (null? prev)
	       (beep)
	       (begin
		 (set! succ (cons (current-state) succ))
		 (load-state (car prev))
		 (set! prev (cdr prev))))
	   (loop))
	   ;;
	  ((#\C-n)
	   (if (null? succ)
	       (beep)
	       (begin
		 (set! prev (cons (current-state) prev))
		 (load-state (car succ))
		 (set! succ (cdr succ))))
	   (loop))
	  ;; ------------------------------ CONTROL -----------------
	  ((#\C-z)
	   (suspend-process)
	   (loop))
	  ((#\C-c)
	   (interrupt-process)
	   (loop))
	  ((#\C-\)
	   (destroy-process)
	   (loop))
	  ;; ------------------------------ CHARACTERS -----------------
	  (else
	   (if (char>=? ch #\space)
	       (insert-right ch))
	   (loop)))))))

;;;

(define (suspend-process)
  (with-module unixm
    (kill (getpid) (vmemq 'SIGTSTP (os-signal-name-vector)))))

(define (interrupt-process)
  (with-module unixm
    (kill (getpid) (vmemq 'SIGINT (os-signal-name-vector)))))

(define (destroy-process)
  (with-module unixm
    (kill (getpid) (vmemq 'SIGQUIT (os-signal-name-vector)))))

;;;


(define (substring->dequeue (str <string>) (from <fixnum>) (to <fixnum>))
  (if (eq? from to)
      (make-dequeue)
      (let (((v <vector>) (make-vector (max 4 (+ (- to from) 1)))))
	(let loop (((i <fixnum>) 0)
		   ((x <fixnum>) from))
	  (if (eq? x to)
	      (make <dequeue>
		    state: v
		    front: 0
		    back: i)
	      (begin
		(vector-set! v i (string-ref str x))
		(loop (add1 i) (add1 x))))))))

