
(define (encode-runs key-vec index num-runs run-widths)
  (let ((encoded (bvec-alloc <byte-vector> (* num-runs 4))))
    (let loop ((i (* num-runs 4))
	       (w run-widths)
	       (j (vector-length key-vec)))
      (if (null? w)
	  encoded
	  (let ((k (vector-ref key-vec (vector-ref index (- j (car w))))))
	    (bvec-write-unsigned-16 encoded (- i 4) k)
	    (bvec-write-unsigned-16 encoded (- i 2) (car w))
	    (loop (- i 4) (cdr w) (- j (car w))))))))
      

(define (permute-val-vec val-vec index num-runs run-widths)
  (let ((permuted (make-vector num-runs)))
    (let loop ((i num-runs)
	       (w run-widths)
	       (j (vector-length val-vec)))
      (if (null? w)
	  permuted
	  (let ((v (vector-ref val-vec (vector-ref index (- j (car w))))))
	    (vector-set! permuted (- i 1) v)
	    (loop (- i 1) (cdr w) (- j (car w))))))))


(define (pack-keys/rle key-vec val-vec)
  (let* ((n (vector-length key-vec))
	 (index (list->vector (range n))))
    ;
    (vector-sort! index
		  (lambda ((a <fixnum> :trust-me)
			   (b <fixnum> :trust-me))
		    (fixnum<? (vector-ref key-vec a)
			      (vector-ref key-vec b))))
    ;
    (let loop ((i 0)
	       (run-lengths '())
	       (num-runs 0))
      (if (eq? i n)
	  (values (encode-runs key-vec index num-runs run-lengths)
		  (if val-vec
		      (permute-val-vec val-vec index num-runs run-lengths)
		      #f))
	  (let ((start-val (vector-ref key-vec (vector-ref index i))))
	    (let run-loop ((run-len 1))
	      (if (and (< (+ run-len i) n)
		       (eq? (vector-ref key-vec 
					(vector-ref index (+ i run-len)))
			    (+ start-val run-len))
		       (or (not val-vec)
			   (eq? (vector-ref val-vec
					    (vector-ref index (+ i run-len)))
				(vector-ref val-vec (vector-ref index i)))))
		  (run-loop (+ run-len 1))
		  (loop (+ i run-len)
			(cons run-len run-lengths)
			(+ num-runs 1)))))))))

;;;

(define (rle-find-index (tbl <byte-vector>) (key-code <fixnum>))
  (let search (((start <fixnum>) 0)
	       ((end <fixnum>) (fixnum-quotient (bvec-length tbl) 4)))
    (let* (((i <fixnum>) (div2 (fixnum+ start end)))
	   ((v <fixnum>) (bvec-read-unsigned-16
			  tbl
			  (fixnum* i 4))))
      (if (fixnum>? v key-code)
	  (if (eq? i end)
	      #f
	      (search start i))
	  (if (fixnum<? key-code
			(fixnum+ v (bvec-read-unsigned-16
			      tbl
			      (fixnum+ (fixnum* i 4) 2))))
	      i
	      (if (eq? i start)
		  #f
		  (search i end)))))))

;;;

(define-class <packed-char-table> (<table>)
  (cache type: <vector>)
  (full-table type: <byte-vector>)
  (tbl-size type: <fixnum>))

(define-syntax (make-cache)
  (make-vector 16))

(define (key-code-sequence (self <packed-char-table>))
  (expand-rle->vector (full-table self)
		      (tbl-size self)
		      (lambda (k i)
			k)))

(define (expand-rle->vector (ft <byte-vector>)
			    (num <fixnum>)
			    (proc <function>))
  (let (((r <vector>) (make-vector num)))
    (let loop ((i 0)
	       (k 0)
	       (vti 0))
      (if (< i (bvec-length ft))
	  (let ((code (bvec-read-unsigned-16 ft i))
		(n (bvec-read-unsigned-16 ft (+ i 2))))
	    (let run-loop ((j n))
	      (if (eq? j 0)
		  (loop (+ i 4) (+ k n) (+ vti 1))
		  (let ((j (sub1 j)))
		    (vector-set! r (+ j k) (proc (+ j code) vti))
		    (run-loop j)))))
	  r))))

;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;
;;;
;;;    Common methods
;;;
;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;

(define-method table-size ((self <packed-char-table>))
  (tbl-size self))

(define-method key-sequence ((self <packed-char-table>))
  (vector-map integer->char (key-code-sequence self)))

;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;
;;;
;;;    Character Sets
;;;
;;;       Domain: characters
;;;       Range:  { #t, #f }
;;;
;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;

(define-class <char-set> (<packed-char-table>))

(define (make-char-set)
  (make <char-set>
	full-table: (bvec-alloc <byte-vector> 0)
	tbl-size: 0
	cache: (make-cache)))

(define (set-char-set-from-elemv! (self <char-set>) (set <vector>))
  (set-cache! self (make-cache))
  (set-full-table! self (pack-keys/rle set #f))
  (values))

(define-method table-insert! ((self <char-set>) (key <char>) (val <boolean>))
  (let ((was (table-remove! self key)))
    (if val
	(begin
	  (set-char-set-from-elemv! self
				    (vector-append
				     (key-code-sequence self)
				     (vector (char->integer key))))
	  (set-tbl-size! self (add1 (tbl-size self)))))
    was))

(define-method table-remove! ((self <char-set>) (key <char>))
  ; it might be worth checking to see if the keycode is present, first
  (let* ((v (key-code-sequence self))
	 (i (vmemq (char->integer key) v)))
    (if i
	(begin
	  (set-char-set-from-elemv! self
				    (vector-append (subvector v 0 i)
						   (subvector v (+ i 1))))
	  (set-tbl-size! self (sub1 (tbl-size self)))
	  #t)
	#f)))

(define-method table-lookup ((self <char-set>) (key <char>))
  (let ((i (rle-find-index (full-table self) (char->integer key))))
    (if i
	#t
	#f)))

(define-method value-sequence ((self <char-set>))
  (make-vector (tbl-size self) #t))

(define char-set-union
  (nlambda
   (() (make-char-set))
   ((a #rest r)
    (let loop ((v (key-code-sequence a))
	       (r r))
      (if (null? r)
          (make <char-set>                                                 
                full-table: (pack-keys/rle v #f)
                tbl-size: (vector-length v)
                cache: (make-cache))
          (loop (union-merge-sorted v (key-code-sequence (car r)))
		(cdr r)))))))

(define (union-merge-sorted (s1 <vector>) (s2 <vector>))
  (let ((r (make-dequeue))
	(n1 (vector-length s1))
	(n2 (vector-length s2)))
    (let loop (((k1 <fixnum>) 0)
	       ((k2 <fixnum>) 0))
      (cond
       ((eq? k1 n1)
	(vector-append (dequeue-state r) (subvector s2 k2)))
       ((eq? k2 n2)
	(vector-append (dequeue-state r) (subvector s1 k1)))
       ((eq? (vector-ref s1 k1) (vector-ref s2 k2))
	(dequeue-push-back! r (vector-ref s1 k1))
	(loop (add1 k1) (add1 k2)))
       ((< (vector-ref s1 k1) (vector-ref s2 k2))
	(dequeue-push-back! r (vector-ref s1 k1))
	(loop (add1 k1) k2))
       (else
	(dequeue-push-back! r (vector-ref s2 k2))
	(loop k1 (add1 k2)))))))

;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;
;;;
;;;    Character Tables
;;;
;;;       Domain: characters
;;;       Range:  objects
;;;
;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;

(define-class <char-table> (<packed-char-table>)
  (values-table type: <vector>))

(define (make-char-table)
  (make <char-table>
	full-table: (bvec-alloc <byte-vector> 0)
	values-table: '#()
	tbl-size: 0
	cache: (make-cache)))

(define-method members->char-table ((a-list <pair>))
  (let ((t (make-char-table))
	(v (list->vector a-list)))
    (set-char-tbl-from-elemv! t 
			      (vector-map (lambda (tuple)
					    (char->integer (car tuple)))
					  v)
			      (vector-map cdr v))
    t))

(define (set-char-tbl-from-elemv! (self <char-table>)
				  (keys <vector>)
				  (vals <vector>))
  (set-cache! self (make-cache))
  (bind ((k v (pack-keys/rle keys vals)))
    (set-full-table! self k)
    (set-values-table! self v)
    (values)))

(define-method table-insert! ((self <char-table>) (key <char>) val)
  (let ((was (table-remove! self key)))
    (set-char-tbl-from-elemv! self
			      (vector-append
			       (key-code-sequence self)
			       (vector (char->integer key)))
			      (vector-append
			       (value-sequence self)
			       (vector val)))
    (set-tbl-size! self (add1 (tbl-size self)))
    was))

(define-method table-remove! ((self <char-table>) (key <char>))
  ; it might be worth checking to see if the keycode is present, first
  (let* ((v (key-code-sequence self))
	 (vt (value-sequence self))
	 (i (vmemq (char->integer key) v)))
    (if i
	(begin
	  (set-char-tbl-from-elemv! self
				    (vector-append (subvector v 0 i)
						   (subvector v (+ i 1)))
				    (vector-append (subvector vt 0 i)
						   (subvector vt (+ i 1))))
	  (set-tbl-size! self (sub1 (tbl-size self)))
	  (vector-ref vt i))
	#f)))

(define-method table-lookup ((self <char-table>) (key <char>))
  (let ((i (rle-find-index (full-table self) (char->integer key))))
    (if i
	(vector-ref (values-table self) i)
	#f)))

(define-method value-sequence ((self <char-table>))
  (expand-rle->vector (full-table self)
		      (tbl-size self)
		      (lambda (k i)
			(vector-ref (values-table self) i))))

;;;

(define-method members->char-set ((self <string>))
  (members->char-set (string->list self)))

(define-method members->char-set ((self <vector>))
  (make <char-set>
	full-table: (pack-keys/rle (vector-map char->integer self) #f)
	tbl-size: (vector-length self) ;; what about dups?
	cache: (make-cache)))

(define-method members->char-set ((self <list>))
  (members->char-set (list->vector self)))
