;;;    <text-graphic>

(define-class <text-graphic> (<leaf-object>)
  (text-runs type: <vector> init-value: '#())
  (text-width-cache init-value: #f))

(define-class <text-run> (<object>)
  (text-run-string type: <string>)
  (text-run-font type: <text-font>))

;;;

(define-method status-line-when-sel ((self <text-graphic>))
  (format #f "Text ~d" (id self)))

;;;    drawing

(define (text-width (self <text-graphic>))
  (or (text-width-cache self)
      (let ((w 0))
	(vector-for-each
	 (lambda (run)
	   (set! w (+ w (string-width (text-run-font run)
				      (text-run-string run)))))
	 (text-runs self))
	(set-text-width-cache! self w)
	w)))

(define (enumerate-text-runs (self <text-graphic>) dev)
  (let* ((w (text-width self))
	 (pt (case (get-property self 'alignment 'left)
	       ((left) 0)
	       ((right) (- w))
	       ((center) (/ w -2)))))
    (moveto dev (make-point pt 0))
    ;;
    (vector-for-each
     (lambda (run)
       (setfont dev (text-run-font run))
       (show dev (text-run-string run)))
     (text-runs self))
    (values w pt)))

(define (text-alignment self)
  (get-property self 'alignment 'left))

(define-method pick-list* ((self <text-graphic>) pt ctm)
  (let* ((w (text-width self))
	 (p (case (text-alignment self)
	      ((left) (list $zero-point (make-point w 0)))
	      ((center) (list (make-point (/ w -2) 0) (make-point (/ w 2) 0)))
	      ((right) (list (make-point (- w) 0) $zero-point)))))
    (pick-on-path self pt ctm p)))

(define-method paint-artwork* ((self <text-graphic>) dev)
  (bind ((w x (enumerate-text-runs self dev)))
    (moveto dev (make-point x 0))
    (lineto dev (make-point (+ x w) 0))
    (stroke dev)))

(define-method paint-object* ((self <text-graphic>) dev)
  (enumerate-text-runs self dev))

(define-method accum-handles ((self <text-graphic>) accum)
  (accum self (make-point 0 0) 0))

(define-method start-active-drag ((self <text-graphic>) 
				  (in-view <open-view>)
				  (initial-pt <point>))
  (let ((initial-posn (origin self))
	(ctm (invert-transform (compute-view-ctm-for self in-view))))
    (lambda ((new-pt <point>) flags)
      (set-origin! self
		   (point+ initial-posn
			   (transform (point- new-pt initial-pt) ctm)))
      (mark-as-dirty (in-document in-view))
      ; expensive, but effective...
      (clear-all-areas (in-document in-view)))))


(define-method start-active-drag-handle ((self <text-graphic>)
					 (in-view <open-view>)
					 handle-id
					 (initial-pt <point>))
  (let ((initial-posn (origin self))
	(ctm (invert-transform (compute-view-ctm-for self in-view))))
    (lambda ((new-pt <point>) flags)
      (let ((p (transform (point- new-pt initial-pt) ctm)))
	;(dm 142 "device ~s -> user ~s" new-pt p)
	(set-origin! self (point+ initial-posn p))
	(mark-as-dirty (in-document in-view))
	; expensive, but effective...
	(clear-all-areas (in-document in-view))))))


;;;

(define-interactive (place-text-mode view)
  (interactive (owner))
  (set-major-mode! view (get-major-mode 'place-text)))

(graphic-set-key #\a place-text-mode)

(define (place-text-button-press (in-view <open-view>)
				 (at <point>)
				 modifier-state)
  (bg
   (let* ((str (read-from-minibuffer "String: "))
	  (par (page-contents (view-page (underlying-object in-view))))
	  (at (window->user-point in-view at))
	  (txt (make <text-graphic>
		     text-runs: (vector 
				 (make <text-run>
				       text-run-font: (active-font
						       (current-client))
				       text-run-string: str))
		     in-document: (in-document par)
		     parent-object: par
		     origin: at
		     graphic-bounding-box: (make-rect 0 0 0 0))))
     (clear-all-areas (in-document in-view))
     (do-select in-view txt 0)
     (update-handles in-view))))

(add-major-mode!
 (make <major-mode>
       name: 'place-text
       button-press-proc: place-text-button-press))

;;;

(define-method externalize ((self <text-font>))
  `(font ,(font-name self) ,(font-style self) ,(font-size self)))

(define (paste-font-from-extern extern group offset)
  (apply get-text-font (cdr extern)))

;;;

(define-method externalize ((self <text-graphic>))
  (let ((r (vector-ref (text-runs self) 0)))
    `(text origin-x: ,(x (origin self))
	   origin-y: ,(y (origin self))
	   string: ,(text-run-string r)
	   font: ,(externalize (text-run-font r)))))

(define (paste-text-from-extern extern group offset)
  (apply (lambda (#key (origin-x default: 0)
		       (origin-y default: 0)
		       (alignment default: #f)
		       string font)
	   (let* ((fnt (paste-font-from-extern font group offset))
		  (g (make <text-graphic>
			   in-document: (in-document group)
			   parent-object: group
			   graphic-bounding-box: $zero-rect
			   origin: (point+ (make-point origin-x origin-y)
					   offset)
			   text-runs: (vector
				       (make <text-run>
					     text-run-font: fnt
					     text-run-string: string)))))
	     (if alignment 
		 (if (memq alignment '(left center right))
		     (set-property! g 'alignment alignment)
		     (wm "text: ignored invalid text alignment of '~s'" 
			 alignment)))
	     (recompute-graphic-bounding-box! g)
	     g))
	 (cdr extern)))

(define-method recompute-graphic-bounding-box! ((self <text-graphic>))
  (let* ((w (text-width self))
	 (h (apply max (map (lambda (r)
			      (font-size (text-run-font r)))
			    (vector->list (text-runs self)))))
	 (ox (case (text-alignment self)
	       ((left) 0)
	       ((right) (- w))
	       ((center) (/ w -2))))
	 (r (make-rect ox 0 w h)))
    (set-graphic-bounding-box! self r)
    r))

