
(define (num-strips img k)
  (vector-length (get-tag-array img k 'strip-offsets)))

(define (strip-data-getter img k)
  (let (((si <tiff-subimage>) (vector-ref (tiff-subimages img) k)))
    (if (get-property si '*strips* #f)
	; if the property *strips* is defined, then the image content
	; has already been loaded into the heap (as by `load-into-heap')
	(let (((strips <vector>) (get-property si '*strips*)))
	  (lambda (strip)
	    (vector-ref strips strip)))
	; otherwise, we're expected to read it off the disk on the fly
	(let (((strip-bytes <vector>) (get-tag-array img k
						     'strip-byte-counts))
	      ((strip-offsets <vector>) (get-tag-array img k 
						       'strip-offsets)))
	  (lambda (strip)
	    (read-bytes (tiff-fd img)
			(vector-ref strip-bytes strip)
			(vector-ref strip-offsets strip)))))))

(define (for-each-row (img <tiff-image>) k proc)
  (let* ((get-strip (strip-data-getter img k))
	 (num-strips (num-strips img k))
	 (image-length (get-tag-scalar img k 'image-length))
	 (rows-per-strip (get-tag-scalar img k 'rows-per-strip))
	 (bits-per-pixel (apply + (vector->list
				   (get-tag-array img k 'bits-per-sample))))
	 (bytes-per-row (quotient (* bits-per-pixel
				     (get-tag-scalar img k 'image-width))
				  8))
	 (row-buffer (make-string bytes-per-row))
	 (uncompressor (case (get-tag-scalar img k 'compression)
			 ((0) (null-uncompressor row-buffer))
			 ((#x8005) (packbits-uncompressor row-buffer))
			 (else
			  (error "unsupported compression scheme: #x~x"
				 (get-tag-scalar img k 'compression))))))
    (let loop ((strip 0)
	       (row 0))
      (if (< strip num-strips)
	  (let ((strip-data (get-strip strip))
		(row-limit (min image-length (+ row rows-per-strip))))
	    (let row-loop ((row row)
			   (data-offset 0)
			   (n 0))
	      (if (< row row-limit)
		  (let ((x2 (uncompressor strip-data data-offset)))
		    (proc row row-buffer)
		    (row-loop (+ row 1) x2 (+ n 1)))
		  (loop (+ strip 1) row))))
	  (values)))))

(define (null-uncompressor (buf <string>))
  (lambda ((data <string>) offset)
    (bvec-copy buf 0 data offset (string-length buf))
    (+ offset (string-length buf))))

(define (packbits-uncompressor (buf <string>))
  (lambda ((data <string>) offset)
    (let loop ((s offset)
	       (d 0))
      ;; Loop until we get the number of unpacked bytes we are expecting
      (if (< d (string-length buf))
	  ;; Read the next source byte into `n'
	  (let ((n (bvec-read-signed-8 data s)))
	    (cond
	     ;; If n is between 0 and 127, incl..
	     ((>= n 0)
	      ;; Copy the next n+1 bytes literally
	      (let ((m (+ n 1)))
		(bvec-copy buf d data (+ s 1) m)
		(loop (+ s 1 m) (+ d m))))
	     ;; else if n is between -127 and -1, incl.,
	     ((>= n -127)
	      ;; Copy the next byte -n+1 times
	      (let ((nxt (bvec-ref data (+ s 1)))
		    (m (- 1 n)))
		(let copy-loop ((i 0)
				(d d))
		  (if (< i m)
		      (begin
			(bvec-set! buf d nxt)
			(copy-loop (+ i 1) (+ d 1)))
		      (loop (+ s 2) d)))))
	     ;; else, noop
	     (else (loop (+ s 1) d))))
	  s))))
