(define-class <transform> (<geometric>) :abstract)

(define-class <affine-transform> (<transform>)
  (matrix type: <vector>))

(define (make-affine-transform)
  (make <affine-transform>
	matrix: '#(1 0 0 1 0 0)))

(define-constant $identity-transform (make-affine-transform))

(define (transform-matrix t)
  (cond
   ((vector? t)
    (vector->values t))
   ((instance? t <affine-transform>)
    (vector->values (matrix t)))
   ((list? t)
    (list->values t))
   (else
    (error "expected a <vector>, <list>, or <transform>: ~s" t))))

(define (concatenate-transform t1 t2)
  (bind ((a1 b1 c1 d1 tx1 ty1 (transform-matrix t2))
	 (a2 b2 c2 d2 tx2 ty2 (transform-matrix t1)))
    (make <affine-transform>
	  matrix: (vector
		   (+ (* a1 a2) (* b1 c2))
		   (+ (* a1 b2) (* b1 d2))
		   (+ (* a2 c1) (* c2 d1))
		   (+ (* b2 c1) (* d1 d2))
		   (+ (* a2 tx1) tx2 (* c2 ty1))
		   (+ (* b2 tx1) ty2 (* d2 ty1))))))
	
(define-method transform1 ((self <affine-transform>) x y)
  (let (((v <vector>) (matrix self)))
    (values (+ (* (vector-ref v 0) x)
	       (* (vector-ref v 2) y)
	       (vector-ref v 4))
	    (+ (* (vector-ref v 1) x)
	       (* (vector-ref v 3) y)
	       (vector-ref v 5)))))

(define-method dtransform1 ((self <affine-transform>) dx dy)
  (let (((v <vector>) (matrix self)))
    (values (+ (* (vector-ref v 0) dx)
	       (* (vector-ref v 2) dy))
	    (+ (* (vector-ref v 1) dx)
	       (* (vector-ref v 3) dy)))))

(define-method transform ((self <point>) xf)
  (bind ((x1 y1 (transform1 xf (x self) (y self))))
    (make-point x1 y1)))

(define-method transform ((self <size>) xf)
  (bind ((dx1 dy1 (dtransform1 xf (dx self) (dy self))))
    (make-size dx1 dy1)))

(define-method transform ((self <rect>) ctm)
  (let (((p1 <point>) (transform (lower-left self) ctm))
	((p2 <point>) (transform (lower-right self) ctm))
	((p3 <point>) (transform (upper-right self) ctm))
	((p4 <point>) (transform (upper-left self) ctm)))
    (bbox-rect (min (x p1) (x p2) (x p3) (x p4))
	       (min (y p1) (y p2) (y p3) (y p4))
	       (max (x p1) (x p2) (x p3) (x p4))
	       (max (y p1) (y p2) (y p3) (y p4)))))

;;;

(define-method translate ((self <affine-transform>) (delta <point>))
  (concatenate-transform 
   self
   (make <affine-transform>
	 matrix: (vector 1 0 0 1 (x delta) (y delta)))))

(define-method scale ((self <affine-transform>) scale)
  (let ((sx (if (real? scale) scale (x scale)))
	(sy (if (real? scale) scale (y scale))))
    (concatenate-transform
     self
     (make <affine-transform>
	   matrix: (vector sx 0 0 sy 0 0)))))

(define-method rotate ((self <size>) (angle <real>))
  (transform self (rotate $identity-transform angle)))

;;(define-constant $Pi (atan 0 -1))

(define (compute-rotation-kernel angle)
  (cond
   ((= angle 0) (values 1 0))
   ((= angle 90) (values 0 1))
   ((= angle 180) (values -1 0))
   ((= angle 270) (values 0 -1))
   (else
    (let ((angle (* $Pi (/ angle 180))))
      (values (cos angle) (sin angle))))))

(define-method rotate ((self <affine-transform>) (theta <real>))
  ;; some exact values
  (bind ((c s (compute-rotation-kernel theta)))
    (concatenate-transform 
     self
     (make <affine-transform>
       matrix: (vector c s (- s) c 0 0)))))

(define-method invert-transform ((self <affine-transform>))
  (bind ((a b c d tx ty (list->values (vector->list (matrix self))))
	 (z (/ (- (* a d) (* b c)))))
    (make <affine-transform>
	  matrix: (vector (* z d)     (* z (- b))
			  (* z (- c)) (* z a)
			  (* z (- (* c ty) (* d tx)))  
			  (* z (- (* b tx) (* a ty)))))))
