------------------------------------------------------------------------------
--                                                                          --
--                         GNAT COMPILER COMPONENTS                         --
--                                                                          --
--            G N A T . D I R E C T O R Y _ O P E R A T I O N S             --
--                                                                          --
--                                 S p e c                                  --
--                                                                          --
--                            $Revision: 1.4 $                              --
--                                                                          --
--            Copyright (C) 1998-1999 Ada Core Technologies, Inc.           --
--                                                                          --
-- GNAT is free software;  you can  redistribute it  and/or modify it under --
-- terms of the  GNU General Public License as published  by the Free Soft- --
-- ware  Foundation;  either version 2,  or (at your option) any later ver- --
-- sion.  GNAT is distributed in the hope that it will be useful, but WITH- --
-- OUT ANY WARRANTY;  without even the  implied warranty of MERCHANTABILITY --
-- or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU General Public License --
-- for  more details.  You should have  received  a copy of the GNU General --
-- Public License  distributed with GNAT;  see file COPYING.  If not, write --
-- to  the Free Software Foundation,  59 Temple Place - Suite 330,  Boston, --
-- MA 02111-1307, USA.                                                      --
--                                                                          --
-- As a special exception,  if other files  instantiate  generics from this --
-- unit, or you link  this unit with other files  to produce an executable, --
-- this  unit  does not  by itself cause  the resulting  executable  to  be --
-- covered  by the  GNU  General  Public  License.  This exception does not --
-- however invalidate  any other reasons why  the executable file  might be --
-- covered by the  GNU Public License.                                      --
--                                                                          --
-- GNAT is maintained by Ada Core Technologies Inc (http://www.gnat.com).   --
--                                                                          --
------------------------------------------------------------------------------

--  Directory operations

--  This package provides routines for manipulating directories. A directory
--  can be treated as a file, using open and close routines, and a scanning
--  routine is provided for iterating through the entries in a directory.

package GNAT.Directory_Operations is

   type Dir_Type is limited private;
   --  A value used to reference a directory. Conceptually this value includes
   --  the identity of the directory, and a sequential position within it.

   Directory_Error : exception;
   --  Exception raised if the directory cannot be opened.

   procedure Change_Dir (Dir_Name : String);
   --  Changes the working directory of the current execution environment
   --  to the directory named by Dir_Name.

   procedure Close (Dir : in out Dir_Type);
   --  Closes the directory stream refered to by Dir. On return, Dir is set
   --  to Null_Dir. Close_Dir raises Directory_Error if Dir = Null_Dir.

   function Get_Current_Dir return String;
   --  Returns the current working directory for the running process. The
   --  returned value is the prefix to be prepended to the simple file name
   --  to obtain a full file name (in the case of Unix and DOS based systems,
   --  this means that it will contain a closing file separator).

   procedure Get_Current_Dir (Dir : out String; Last : out Natural);
   --  Returns the current working directory for the running process. The name
   --  is returned in Dir; Last is the index in Dir such that Dir (Last) is the
   --  last character written. If Dir is too small for the dir name, the name
   --  will be truncated before beeing copied to Dir.
   --  The returned value is the prefix to be prepended to the  simple file
   --  name to obtain a full file name (in the case of Unix and DOS based
   --  systems, this means that it will contain a closing file separator).

   function Is_Open (Dir : Dir_Type) return Boolean;
   --  Returns True if Dir is open, or False otherwise.

   procedure Open (Dir : out Dir_Type; Dir_Name : in String);
   --  Opens the directory named by Dir_Name and returns a Dir_Type value
   --  that refers to this directory, and is positioned at the first entry.
   --  Directory_Error is raised if Dir_Name cannot be accessed.

   procedure Read
     (Dir  : in out Dir_Type;
      Str  : out String;
      Last : out Natural);
   --  Reads the next entry from the directory and sets Str to the name
   --  of that entry. Last is the index in Str such that Str (Last) is the
   --  last character written. Last is 0 when there is no more file in the
   --  directory. If Str is too small for the file name, the file name will
   --  be truncated before beeing copied to Str. Directory_Error is raised
   --  if Dir is not open. It is unspecified if entries for the current
   --  and parent directories are returned (e.g. '.' and '..' in Unix).
   --  The directory is returned in target-OS form

   function Read_Is_Thread_Safe return Boolean;
   --  Indicates if procedure Read is thread safe. On systems where the
   --  target system supports this functionality, Read is thread safe,
   --  and this function returns True (e.g. this will be the case on any
   --  Unix or Unix-like system providing a correct implementation of the
   --  function readdir_r). If the system cannot provide a thread safe
   --  implementation of Read, then this function returns False.

private

   type Dir_Type_Value;
   type Dir_Type is access Dir_Type_Value;

end GNAT.Directory_Operations;
