(**************************************************************************)
(*                                coq2html                                *)
(**************************************************************************)

open Lexer;;
open Filter;;

let one_space = " ";;
let real_space = "&nbsp;";;
let line_header = "";;
let line_trailer = "<br>";;
let tab = ref 8;;
let filter = ref false;;
let std_input = ref false;;
let std_output = ref false;;
let current_file = ref "";;
let defs = ref true;;

let path = ref (["."] : string list);;
let assocl = ref ([] : (string * string) list);;
let add_assoc (id,f) =
  if not (List.mem_assoc id !assocl) then
    assocl := (id,f) :: !assocl
;;
let assoc id =
  List.assoc id !assocl
;;

let read_labels filename =
  let chan_in = open_in filename in
  try while true do 
        let line = input_line chan_in in
        if String.length line > 9 then
        if (String.sub line 0 9) = "<A NAME=\"" (* "\"" *) then
          let name' = String.sub line 9 ((String.length line) - 9) in
          let name  = String.sub name' 0 ((String.length name') - 2) in
          add_assoc (name,filename)
      done
  with End_of_file -> close_in chan_in
;;

let find_file id =
  let rec find_rec = function
    d::reste -> let filename = Filename.concat d (id^".html") in
                if Sys.file_exists filename then begin
                  read_labels filename ;
                  filename
                end else
                  find_rec reste
  | []       -> raise Not_found
  in find_rec !path
;;

let chan_out = ref stdout;;
let print_s s = output_string !chan_out s;;
let print_c c = output_char !chan_out c;;

let rec initial_spaces =
    let col = ref 0 in
    let rec ispaces = parser
      [< 'Char ' '  ; s >] ->
        print_s real_space ;
        col := succ !col ;
        ispaces s
    | [< 'Char '\t' ; s >] ->
        print_s real_space ; 
        col := succ !col ;
        while (!col mod !tab) != 0 do
          print_s real_space ; 
          col := succ !col
        done ; 
        ispaces s
    | [< >]               -> () in
    function s -> col := 0 ; ispaces s
;;

let require = ref false;;
let definition = ref false;;

let href (f,s) =
  "<A HREF=\"" ^ f ^ "\">" ^ s ^ "</A>"
;;

let hreflabel (f,id) =
  "<A HREF=\"" ^ f ^ "#" ^ id ^ "\">" ^ id ^ "</A>"
;;

let label id =
  "\n<A NAME=\"" ^ id ^ "\">\n"
;;

let is_definition s =
     s="Theorem"
  or s="Lemma"
  or (  (  (* s="Syntax" or s="Grammar" or *) s="Mutual" or s="Inductive" 
        or s="CoInductive" or s="Recursive" or s="Definition"
        or s="Fixpoint" or s="Record" or s="CoFixpoint" or s="Syntactic") 
      & !defs)
;;

let print_token_html tok =
    let com = !clevel > 0 in
    match tok with
      Special "->"  -> print_s "->"
    | Special "<-"  -> print_s "<-"
    | Special "=>"  -> print_s "=>"
    | Special "<->" -> print_s "<->"
    | Special "/\\" -> print_s "/\\"
    | Special "\\/" -> print_s "\\/"
    | Special s     -> print_s s

    | Kwd s      -> if com then 
                      print_s s 
                    else begin
                      print_s ("<B>" ^ s ^ "</B>") ;
                      require := (s="Require" or s="Export") ;
                      definition := is_definition s
                    end

    | Ident s    -> if com then 
                      print_s s
                    else if !require then begin
                      require := (s="Export") ;
                      try  let f = find_file s in
                           print_s( href (f,s))
                      with Not_found -> print_s s
                    end else if !definition then begin
                      print_s (label s) ;
                      if not !std_output then add_assoc (s,!current_file) ;
                      definition := false ;
                      print_s s
                    end else begin
                      try  let f = assoc s in
                           print_s (hreflabel (f,s))
                      with Not_found -> print_s s
                    end

    | String s   -> print_s ("\"" ^ (String.escaped s) ^ "\"")

    | Char ' '   -> print_s (if com then real_space else " ")
    | Char '<'   -> print_s "&lt;" 
    | Char '>'   -> print_s "&gt;" 
    | Char '&'   -> print_s "&amp;" 
    | Char '"'   -> print_s "&quot;" (* "\"" *)
    | Char c     -> print_c c

    | CB         -> if !clevel = 1 then print_s "<tt>" ;
                    print_s "(*" (**)

    | CE         -> print_s "*)" ;
                    if !clevel = 0 then print_s "</tt>"
;;

module P : PRINT = struct
    let initial = initial_spaces ;;
    let print_token = print_token_html ;;
end;;
 
module Filter = FFilter(P);;

let print_html_line line =
  let st = lexer (Stream.of_string line) in
  print_s line_header ;
  Filter.filter st ;
  print_s line_trailer
;;

let html_header titre ="\
<HTML>
<head>
<title>" ^ titre ^ "</title>
</head>

<body>";;

let html_trailer = "
</body>
</HTML>
";;

let process_channel chan_in =
  try 
    while true do
      let line = input_line chan_in in
      print_html_line line ;
      print_s "\n"
    done
  with End_of_file -> ()
;;

let string2_of_int n =
  if n<10 then
    "0" ^ (string_of_int n)
  else
    string_of_int n
;;

let format_time mtime =
  let tm = Unix.localtime mtime in
  let date = (string2_of_int tm.Unix.tm_mday) 
        ^ "/" ^ (string2_of_int (succ tm.Unix.tm_mon))
        ^ "/" ^ (string2_of_int tm.Unix.tm_year) in
  let time = (string2_of_int tm.Unix.tm_hour)
        ^ ":" ^ (string2_of_int tm.Unix.tm_min) in
  time,date
;;

let treat_one_file filename =
  let vfilename = filename ^ ".v"
  and htmlfilename = filename ^ ".html" in
  let mtime = (Unix.stat vfilename).Unix.st_mtime in
  let time,date = format_time mtime in

  chan_out := if !std_output then stdout else open_out htmlfilename ;
  if not !std_output then current_file := htmlfilename ;
  print_s (html_header vfilename) ;
  print_s ("<H3>" ^ vfilename ^ "</H3>\n<HR>\n") ;

  let chan_in = open_in vfilename in
  process_channel chan_in ;
  close_in chan_in ;

  print_s "<HR>\n" ;
  print_s ("<I>" ^ date ^ ", " ^ time ^ "</I>\n") ;
  print_s html_trailer ;

  flush !chan_out ;
  if not !std_output then close_out !chan_out
;;

let treat_all_files files =
  List.iter treat_one_file files  
;;

let treat_stdin () =
  chan_out := stdout ;
  let time,date = format_time (Unix.time()) in
  print_s (html_header "Anonymous Coq module") ;
  process_channel stdin ;
  print_s "<HR>\n" ;
  print_s ("<I>" ^ date ^ ", " ^ time ^ "</I>\n") ;
  print_s html_trailer ;
  flush stdout
;;

let usage () =
  prerr_string 
  "Usage: coq2html [-] [-stdout] [-f] [-filter] [-nodefs] [-I dir] [-t n] files\n";
  flush stderr;
  exit 1
;;

let coq2html () =
  let lg_command = Array.length Sys.argv in
  if lg_command < 2 then 
    usage ()
  else
    let rec parse = function
        ("-" | "-filter" | "-f") :: ll
            -> std_input := true ; std_output := true ; parse ll
      | "-stdout" :: ll   -> std_output := true ; parse ll
      | "-nodefs" :: ll   -> defs := false ; parse ll
      | "-t" :: (n :: ll) -> tab := (int_of_string n) ; parse ll
      | "-t" :: []        -> usage ()
      | "-I" :: (d :: ll) -> path := (d :: !path) ; parse ll
      | "-I" :: []        -> usage ()
      | f :: ll           -> if Filename.check_suffix f ".v" then
                               (Filename.chop_suffix f ".v")::(parse ll)
                             else
                               parse ll
      | []                -> []
    in
    let files = parse (List.tl (Array.to_list Sys.argv)) in
    
    if !std_input then
      treat_stdin ()
    else
      treat_all_files files ;
;;

Printexc.catch coq2html ();;




