(****************************************************************************)
(*                 The Calculus of Inductive Constructions                  *)
(*                                                                          *)
(*                                Projet Coq                                *)
(*                                                                          *)
(*                     INRIA        LRI-CNRS        ENS-CNRS                *)
(*              Rocquencourt         Orsay          Lyon                    *)
(*                                                                          *)
(*                                 Coq V6.3                                 *)
(*                               July 1st 1999                              *)
(*                                                                          *)
(****************************************************************************)
(*                               install.ml4                                *)
(****************************************************************************)


(* This little tool installs a list of files in a given directory,
   and writes a log message in $COQTOP/COQFILES 
   Equivalent Usages :
   
1) install Truc.vo Truc.v Bidule.v /usr/local/lib/coq/theories

  (copy Truc.v and Truv.vo and Bidule.vin /usr/local/lib/coq/theories and
  writes :

    /usr/local/lib/coq/theories/Truc.v
    /usr/local/lib/coq/theories/Truc.vo
    /usr/local/lib/coq/theories/Bidule.v

  in the $COQTOP/COQFILES log file)

2) install Truc.v /usr/local/lib/coq/theories/Machin.v

  (copy Truc.v in /usr/local/lib/coq/theories/Machin.v and
  writes :

    /usr/local/lib/coq/theories/Machin.v

  in the $COQTOP/COQFILES log file)

*)

open Unix
open Sys
open Filename

let cp file dest = command ("cp " ^ file ^ " " ^ dest)

let error msg =
  prerr_endline ("coqinstall: "^msg);
  exit 2;;

let rec process_option_d file =
  if (file_exists file) then
    if (stat file).st_kind <> S_DIR then failwith "not a dir"
    else ()
  else
   (process_option_d (dirname file);
    mkdir file 0o755)

let is_a_directory file =
  (file_exists file) & ((stat file).st_kind  = S_DIR)

let usage () =
  print_string 
    "[ usage: coqinstall [-d] [-prefix path] filename1 ... filenameN dirname \
    \n         coqinstall [-d] [-prefix path] filename1 filename2             ]\n";
  flush Pervasives.stdout;
  exit 1
;;

let prefix = ref ""
let create_dir = ref false

(* Possibilit d'installer dans un rpertoire temporaire au lieu de /
   tout en mettant le chemin dfinitif dans COQFILES. A cause d'un bug
   de rpm, ce dernier passe le prfixe temporaire via une var globale *)

let rec parse = function
    "-prefix" :: path :: rem -> prefix := path ; parse rem
  | "-d" :: rem              -> create_dir := true ; parse rem
  | [ ]                      -> usage ()
  | [ _ ]                    -> usage ()
  | rem                      -> Array.of_list rem

let get_args () =
   (try prefix := Sys.getenv "COQINSTALLPREFIX" with Not_found -> ());
   parse (List.tl (Array.to_list argv))
;;

(* main *)
let main _ =
   let argv = get_args () in
   let nargs = Array.length argv in
   let dest = Filename.concat !prefix argv.(nargs-1) in
   let coqfiles =
     open_out_gen [Open_creat; Open_append; Open_wronly] 0o644
       (Filename.concat !prefix (Filename.concat Coq_config.coqlib "COQFILES"))
   in
   (if !create_dir then
     try process_option_d dest
     with Failure "not a dir" ->
       error "-d option needs last argument is a directory");
   if is_a_directory dest
   then (for i=0 to nargs-2 do
         if (cp argv.(i) dest)<>0
	    then (close_out coqfiles;
		  error ("error during copy of "^argv.(i)^" to "^dest))
	     else output_string coqfiles 
	      ((Filename.concat argv.(nargs-1) argv.(i))^"\n")	    
         done;
         close_out coqfiles;
   	 exit 0)
   else if nargs = 2 
   then 
     if  (cp argv.(0) dest) <> 0
     then (close_out coqfiles;
           error ("error during copy of "^argv.(0)^" to "^dest))
     else (output_string coqfiles (argv.(1)^"\n"); 
	   close_out coqfiles;
	   exit 0)
   else ((* n>2, dest not a dir *) usage (); close_out coqfiles; exit 1)
;;

Printexc.catch main ();;




