(****************************************************************************)
(*                 The Calculus of Inductive Constructions                  *)
(*                                                                          *)
(*                                Projet Coq                                *)
(*                                                                          *)
(*                     INRIA        LRI-CNRS        ENS-CNRS                *)
(*              Rocquencourt         Orsay          Lyon                    *)
(*                                                                          *)
(*                                 Coq V6.3                                 *)
(*                               July 1st 1999                              *)
(*                                                                          *)
(****************************************************************************)
(*                                mlterm.ml                                 *)
(****************************************************************************)

open Std
open More_util
open Names

(* ML type expressions. *)

type typeid = TYPEparam of identifier
            | TYPEname  of identifier

type mLtype = TYvar  of typeid
      	    | TYapp  of mLtype list
	    | TYarr  of mLtype * mLtype
	    | TYglob of identifier

(* ML inductive types. *)

type mLind = identifier list * identifier * (identifier * mLtype list) list

(* ML terms. *)

type mLast = MLrel  of int
      	   | MLapp  of mLast * mLast list
	   | MLlam  of identifier * mLast
	   | MLglob of identifier
	   | MLcons of int * identifier * mLast list
	   | MLcase of mLast * (identifier * identifier list * mLast) array
	   | MLfix  of int * bool * (identifier list) * (mLast list)
	   | MLexn  of identifier

(* ML declarations. *)

type mLdecl = DECLtype   of mLind list
            | DECLabbrev of identifier * (identifier list) * mLtype
            | DECLglob   of identifier * mLast


(* Renaming problems *)

(* functions to get correct caml names out of identifiers
 * TODO : identifiers like _foo and foo_ are not accepted by caml !
 *)

let up_string_of_id id = String.capitalize (string_of_id id)

let lo_string_of_id id = String.uncapitalize (string_of_id id)

let name_cpt = ref 0

let reset_caml_names () = name_cpt := 0

let caml_name_of_int n = 
 let rec gen n = 
    let su = string_of_char (Char.chr (97+(n mod 26))) in
    match n / 26 with
      0 -> su
    | d -> (gen d)^su
 in 
   make_ident (gen n) (-1)

let caml_new_name () =
  incr name_cpt;
  caml_name_of_int !name_cpt

let caml_reserved = [
  "and";       "as";         "assert";    "asr";       "begin";     "class";
  "closed";    "constraint"; "do";        "done";      "downto";    "else";
  "end";       "exception";  "external";  "false";     "for";       "fun";
  "function";  "functor";    "if";        "in";        "include";   "inherit";
  "land";      "lazy";       "let";       "lor";       "lsl";       "lsr";
  "lxor";      "match";      "method";    "mod";       "module";    "mutable";
  "new";       "of";         "open";      "or";        "parser";    "private";
  "rec";       "sig";        "struct";    "then";      "to";        "true";
  "try";       "type";       "val";       "virtual";   "when";      "while";
  "with";      
  (* mots cles haskell *)
  "Left"; "Right"
  ]

let is_caml_keyword =
  let t = (Hashtabl.create 17 : (string, unit) Hashtabl.t) in
  List.iter (fun s -> Hashtabl.add t s ()) caml_reserved ;
  function s -> try Hashtabl.find t s ; true
      	       	with Not_found -> false

let gen_caml_name_of stofid avoid = function
    Anonymous -> next_ident_away (caml_new_name()) avoid
  | Name id   -> let s = stofid id in
      	       	 if (String.get s 0) = '_' then 
      	       	   next_ident_away (caml_new_name()) avoid
      	       	 else if is_caml_keyword s then
		   next_ident_away (id_of_string (s^"_ren")) avoid
		 else
      	       	   next_ident_away (id_of_string s) avoid

let caml_name_of = gen_caml_name_of string_of_id
let lo_caml_name_of = gen_caml_name_of lo_string_of_id
let up_caml_name_of = gen_caml_name_of up_string_of_id

let rec new_vars avoid = function
    0 -> []
  | n -> let id = make_ident "x" (-1) in
      	 let f = next_ident_away id avoid in
	 f::(new_vars (f::avoid) (n-1))

(* IMPORTANT:

   Le terme avec ses indices de dB est sans ambigut, mais les noms des
   lieurs peuvent introduire des ambiguts plus tard au parsing.
   On choisit donc de conserver cet invariant (suffisant) :

       Le long d'une squence descendante de lieurs, les noms sont distincts.

   Cette proprit est vrifie pour les termes qui sortent de la
   traduction FW -> ML.
   Pour que cette proprit reste vrifie, il faut qu'elle soit conserve
   lors de :
   - l'expansion de constantes ;
   - la normalisation qui suit cette expansion.

   Pour la substitution, on choisit de renommer les lieurs du terme substitu
   (et non du terme dans lequel on substitue). Cf rename_bindings et
   ml_subst1 ci-dessous, ainsi que Optimise.ml_subst_glob et Optimise.norm.

*)

(* rename_list : identifier list -> identifier list -> identifier list
 * [rename_list V l] renames the list l with names out of V. *)

let union l1 l2 =
  let rec union_rec = function
    [] -> l2
  | a::l -> if List.mem a l2 then union_rec l else a :: union_rec l
  in union_rec l1

let rec rename_list avoid = function
    []    -> []
  | a::l1 -> 
      if List.mem a avoid then
      	let nv = caml_name_of avoid (Name a) in
	  nv :: (rename_list (nv::avoid) l1)
      else
       	a :: (rename_list (a::avoid) l1)

(* rename_bindings : identifier list -> MLast -> MLast
   [rename_bindings avoid t] renames the bindings of t out of avoid. *)

let rec rename_bindings avoid = function
    MLlam(id,t) ->
      let id' = caml_name_of avoid (Name id) in
      MLlam(id', rename_bindings (id'::avoid) t)
  | MLcase(t,pl) ->
      MLcase(rename_bindings avoid t,
	     Array.map (fun (id,ids,p) -> 
			  let ids' = rename_list avoid ids in
			  (id,ids',rename_bindings (ids'@avoid) p)) pl )
  | MLfix(n0,b,idl,pl) ->
      let idl' = rename_list avoid idl in
      MLfix(n0,b,idl',List.map (rename_bindings (idl'@avoid)) pl)
      
  | MLapp(t,args) ->
      MLapp(rename_bindings avoid t, List.map (rename_bindings avoid) args)
  | MLcons(j,id,args) ->
      MLcons(j,id, List.map (rename_bindings avoid) args)

  | x -> x

(* lifting on terms. ml_lift : int -> MLast -> MLast
 * [ml_lift k M] lifts the binding depth of M across k bindings. *)

let ml_liftn k n c = 
 let rec liftrec n = function
    MLrel i as c      -> if i<n then c else MLrel(i+k)
  | MLapp(t,args)     -> MLapp(liftrec n t, List.map (liftrec n) args)
  | MLlam(id,t)       -> MLlam(id,liftrec (n+1) t)
  | MLcons(j,id,args) -> MLcons(j,id,List.map (liftrec n) args)
  | MLcase(t,pl)      -> MLcase(liftrec n t,
      	       	    Array.map (fun (id,idl,p) -> let k = List.length idl in
			              (id,idl,liftrec (n+k) p)) pl)
  | MLfix (n0,b,idl,pl) -> MLfix(n0,b,idl,let k = List.length idl in
                               List.map (liftrec (n+k)) pl)
  | x                 -> x
 in 
   if k=0 then c else liftrec n c


let ml_lift k c = ml_liftn k 1 c

let ml_pop c = ml_lift (-1) c

let ml_liftn_branch k (id,ids,v) =
  (id,ids,ml_liftn k (List.length ids+1) v)

(* ml_subst1 : identifier list -> MLast -> MLast -> MLast
 * [ml_subst1 V M t] substitutes M for (Rel 1) in t,
 * renaming the bindings of M out of V. *)

let rec ml_subst1 v =
  let rec subst n av m = function
    MLrel i ->
      if i=n then
	rename_bindings av m
      else 
	if i<n then MLrel i else MLrel (i-1)
  | MLapp(t,argl) ->
      MLapp(subst n av m t, List.map (subst n av m) argl)
  | MLlam(id,t) ->
      MLlam(id, subst (n+1) (id::av) (ml_lift 1 m) t)
  | MLcons(i,id,argl) ->
      MLcons(i,id,List.map (subst n av m) argl)
  | MLcase(t,pv) ->
      MLcase(subst n av m t,
	     Array.map (fun (id,ids,t) ->
			  let k = List.length ids in
      	       		  (id,ids,subst (n+k) (ids@av) (ml_lift k m) t))
	       pv)
  | MLfix(i,b,ids,cl) -> 
      MLfix(i,b,ids, 
	    let k = List.length ids in
	    List.map (subst (n+k) (ids@av) (ml_lift k m)) cl)

  | x -> x

  in 
    subst 1 v

(* occurs : int -> MLast -> bool
 * [occurs k M] returns true if (Rel k) occurs in M. *)

let rec occurs k = function
    MLrel i          -> i=k
  | MLapp(t,argl)    -> (occurs k t) or (occurs_list k argl)
  | MLlam(_,t)       -> occurs (k+1) t
  | MLcons(_,_,argl) -> occurs_list k argl
  | MLcase(t,pv)     -> (occurs k t) or
      	       	       	(List.exists (fun (k',t') -> occurs (k+k') t')
      	       	            (map_vect_list (fun (_,l,t') -> 
      	       	             let k' = List.length l in (k',t')) pv))
  | MLfix(_,_,l,cl)  -> let k' = List.length l in occurs_list (k+k') cl
  | _                -> false
and occurs_list k l =
  List.exists (fun t -> occurs k t) l

(* collect_lambda MLlam(id1,...MLlam(idn,t)...) = [id1;...;idn],t *)

let collect_lambda = 
 let rec collect acc = function
     MLlam(id,t) -> collect (id::acc) t
   | x           -> acc,x
 in collect []

(* $Id: mlterm.ml,v 1.14 1999/06/29 07:48:05 loiseleu Exp $ *)
