(****************************************************************************)
(*                 The Calculus of InductiveConstructions                  *)
(*                                                                          *)
(*                                Projet Coq                                *)
(*                                                                          *)
(*                     INRIA                        ENS-CNRS                *)
(*              Rocquencourt                        Lyon                    *)
(*                                                                          *)
(*                                 Coq V6.1                                 *)
(*                               Oct 1st 1996                               *)
(*                                                                          *)
(****************************************************************************)
(*                                 gofer.ml                                 *)
(****************************************************************************)

(**************************************************************************)
(*        production of gofer syntax out of an ML environment             *)
(**************************************************************************)

open Pp;;
open Std;;
open More_util;;
open Names;;
open Vernacinterp;;
open Ml_import;;
open Mlterm;;
open Genpp;;


(**************************************************************************)
(*          translation of type expressions and inductive types           *)
(**************************************************************************)

let print_typeid = function
    TYPEname id  -> [< 'sTR(string_of_id id) >]
  | TYPEparam id -> [< 'sTR(string_of_id id) >]
;;

(* gofer_of_type : bool -> MLtype -> std_ppcmds
 * [gofer_of_type b t] formats the type expression t.
 * b=true if we need parentheses around the result. *)

let rec gofer_of_type paren_p = function
     TYarr(t1,t2) ->
       	[< open_par paren_p ;
	   gofer_of_type true t1 ; 'sTR" -> " ;
           gofer_of_type false t2 ;
	   close_par paren_p
        >]

  | TYapp cl ->
      	let n = List.length cl in
	if n=1 then 
	  gofer_of_type false (List.hd cl)
        else
	  [< open_par paren_p ;
             gofer_of_type false (List.hd cl) ; 'sTR" " ;
	     prlist_with_sep (fun () -> [< 'sTR" " >])
      	       	       	     (fun c -> gofer_of_type true c) (List.tl cl) ;
	     close_par paren_p
	  >]

  | TYvar tid ->
      	[< print_typeid tid >]

  | TYglob id ->
      	[< 'sTR(string_of_id id) >]
;;


(* gofer_of_one_inductive : MLind -> std_ppcmds
 * [gofer_of_one_inductive mi] formats the declaration of mutual
 * inductive mi. *)

let gofer_of_one_inductive (pl,name,cl) =
  let fmt_constructor (id,l) =
    [< 'sTR(string_of_id id) ;
       match l with
         [] -> [< >] 
       | _  -> [< 'sTR" " ;
      	       	  prlist_with_sep (fun () -> [< 'sTR" " >]) 
      	       	       	          (fun s -> gofer_of_type true s) l
	       >]
    >] in

  [< 'sTR(string_of_id name) ; 'sTR" " ;
     prlist_with_sep (fun () -> [< 'sTR" " >])
      	       	     (fun id -> [< 'sTR(string_of_id id) >]) pl ;
     if pl=[] then [< >] else [< 'sTR" " >] ;
     
     v 0 [< 'sTR"= " ;
            prlist_with_sep (fun () -> [< 'fNL ; 'sTR"| ">])
                            (fun c -> fmt_constructor c)
		            cl
         >]
  >]
;;

let gofer_of_inductive il =
  [< 'sTR"data " ;
     prlist_with_sep 
      	  (fun () -> [< 'fNL ; 'sTR"data " >])
       	  (fun i -> gofer_of_one_inductive i)
	  il ;
     'fNL
  >]
;;


(**************************************************************************)
(*                  production of gofer syntax for terms                   *)
(**************************************************************************)

let abst = function
    [] -> [< >]
  | l  -> [< 'sTR"\\" ;
             prlist_with_sep (fun  ()-> [< 'sTR" " >])
      	       	       	     (fun id -> [< 'sTR(string_of_id id) >]) l ;
             'sTR" -> " >]
;;

let pr_binding = function
    [] -> [< >]
  | l  -> [< 'sTR" " ; prlist_with_sep (fun () -> [< 'sTR" " >])
      	       	       	(fun id -> [< 'sTR(string_of_id id) >]) l >]
;;

(* pp_mlast : identifier list -> bool -> std_ppcmds list
 *	      -> MLast -> std_ppcmds
 * [pp_mlast idl b args t] formats the Ml term (t a1...am)
 * in the de Bruijn environment idl, where args=[a1...am].
 * b=true if we need parentheses around the result. *)

let rec pp_mlast idl paren_p args = 

  let apply st = match args with
     [] -> st
   | _  -> hOV 2 [< open_par paren_p ; st ; 'sTR" " ;
                    prlist_with_sep (fun () -> [< 'sPC >]) (fun s -> s) args ;
                    close_par paren_p
                 >] in

  function
    MLrel n ->
      	 apply [< 'sTR(string_of_id (nth idl n)) >]

  | MLapp (h, args') ->
      	 let stl = List.map (fun t -> pp_mlast idl true [] t) args' in
         pp_mlast idl paren_p (stl@args) h

  | MLlam _ as t ->
      	 let fl,t' = collect_lambda t in
	 let st = [< abst (List.rev fl) ; pp_mlast (fl@idl) false [] t' >] in
	 if args=[] then
           [< open_par paren_p ; st ; close_par paren_p >]
         else
           apply [< 'sTR"(" ; st ; 'sTR")" >]

  | MLglob id ->
      	 apply [< 'sTR(string_of_id id) >]
	
  | MLcons (_,id,args') ->
      	 if args'=[] then
	   [< 'sTR(string_of_id id) >]
	 else
	   [< open_par paren_p ; 'sTR(string_of_id id) ; 'sTR" " ;
	      prlist_with_sep (fun () -> [< 'sTR" " >])
	                      (fun t -> pp_mlast idl true [] t) args' ;
	      close_par paren_p
           >]

  | MLcase (t, pv) ->
      	 apply
      	 [< if args<>[] then [< 'sTR"(" >]  else open_par paren_p ;
      	    v 0 [< 'sTR"case " ; pp_mlast idl false [] t ; 'sTR" of" ;
		   'fNL ; 'sTR"  " ;
		   pp_mlpat idl pv >] ;
	    if args<>[] then [< 'sTR")" >] else close_par paren_p >]

  | MLfix (x_0,x_1,x_2,x_3) ->
      	 pp_mlfix idl paren_p (x_0,x_1,x_2,x_3) args

  | MLexn id ->
      	 [< open_par paren_p ; 'sTR"error \"" ; print_id id ; 'sTR"\"" ;
	    close_par paren_p >]

and pp_mlfix idl paren_p (j,in_p,fid,bl) args =
  let idl' = fid@idl in
  [< open_par paren_p ; v 0 [< 'sTR"let { " ;
       prlist_with_sep
      	  (fun () -> [< 'sTR";" ; 'fNL >])
	  (fun (fi,ti) -> pp_mlfunction idl' fi ti)
	  (List.combine fid bl) ;
       'sTR" }" ; 'fNL ;
       if in_p then 
      	 hOV 2 [< 'sTR"in " ; 'sTR(string_of_id (nth fid (j+1))) ;
                  if args<>[] then
                    [< 'sTR" "; prlist_with_sep (fun () -> [<'sTR" ">])
                                 (fun s -> s) args >]
                  else [< >]
      	       >]
       else 
         [< >] >] ;
     close_par paren_p >]

and pp_mlfunction idl f t =
  let bl,t' = collect_lambda t in

 (***TO REMOVE
  let is_function pv =
    let ktl = map_vect_list (fun (_,l,t0) -> (list_length l,t0)) pv in
    not(exists (fun (k,t0) -> occurs (k+1) t0) ktl) in

  match t' with 
    MLcase(MLrel 1,pv) ->
       if is_function pv then
	 [< 'S(gofer_of_id f) ; pr_binding (rev (tl bl)) ;
       	    'S" = \\" ; 'FNL ;
	    V 0 [< 'S"  " ; pp_mlpat (bl@idl) pv >] >]
       else
         [< 'S(gofer_of_id f) ; pr_binding (rev bl) ; 'S" = case " ;
	    'S(gofer_of_id (hd bl)) ; 'S" of" ; 'FNL ;
	    V 0 [< 'S"  " ; pp_mlpat (bl@idl) pv >] >]
	   
  | _ -> ***)

  [< 'sTR(string_of_id f) ; pr_binding (List.rev bl) ;
     'sTR" =" ; 'fNL ; 'sTR"  " ;
     hOV 2 (pp_mlast (bl@idl) false [] t') >]

and pp_mlpat idl pv =
  let pp_one_pat (name,ids,t) =
      let paren_p = match t with
                      MLlam _  -> true
                    | MLcase _ -> true
                    | _        -> false in

      hOV 2 [< 'sTR(string_of_id name) ;
      	       if ids=[] then [< >]
      	       else [< 'sTR" " ; prlist_with_sep (fun () -> [<'sTR" ">])
      	       	       (fun id -> [< 'sTR(string_of_id id) >]) (List.rev ids) >] ;
	       'sTR" ->" ; 'sPC ; pp_mlast (ids@idl) paren_p [] t
            >]

  in [< prvect_with_sep (fun () -> [< 'fNL ; 'sTR"  " >])
                        (fun p -> pp_one_pat p)
	                pv >]
;;


(* gofer_of_decl : MLdecl -> std_ppcmds
 * [gofer_of_decl d] formats the declaration d. *)

let gofer_of_decl = function
    DECLtype il -> gofer_of_inductive il

  | DECLabbrev (id, idl, t) ->
          [< 'sTR"type " ; 'sTR(string_of_id id) ; 'sTR" " ;
	     prlist_with_sep (fun () -> [< 'sTR" " >])
	                     (fun id -> [< 'sTR(string_of_id id) >]) idl;
      	     if idl<>[] then [< 'sTR" " >] else [< >] ;'sTR"= " ;
	     gofer_of_type false t ; 'fNL >]

  | DECLglob (id0 , MLfix(n,_,idl,fl)) ->
      let id' = Std.nth idl (succ n) in
      [< prlist_with_sep (fun () -> [< 'fNL >])
	   (fun (id,f) -> pp_mlfunction (List.rev idl) id f)
	   (List.combine idl fl) ;
	 'fNL ; 'fNL ;
         'sTR(string_of_id id0) ; 'sTR" = " ; 'sTR(string_of_id id') ;
         'fNL >]

  | DECLglob (id, t) ->
      	  [< pp_mlfunction [] id t ; 'fNL >]       
;;


(**************************************************************************)
(*             translation of an environment into gofer syntax.            *)
(**************************************************************************)

let gofer_of_env env =
  prlist (fun d -> [< gofer_of_decl d ; 'fNL >]) env
;;


(* Optimisation step added to Gofer extraction. Eduardo 9/6/96 *)

module Gofer_renaming =
  struct
    let rename_type_parameter = lo_caml_name_of;;
    let rename_type           = up_caml_name_of;;
    let rename_term           = lo_caml_name_of;;
    let rename_global_type    = up_caml_name_of;;
    let rename_global_constructor = up_caml_name_of;;
    let rename_global_term    = lo_caml_name_of;;
  end;;

module Gofer_pp : MLPP = 
  struct
    let opt = Optimise.gofer_optimise;;
    let suffixe = ".gs";;
    let cofix = true;;
    let pp_of_env = gofer_of_env;;
    module Renaming = Gofer_renaming;;
  end;;

module Gofer_pp_file = Pp_to_file(Gofer_pp);;


(**************************************************************************)
(*            Write Gofer File filename [ ident1 ... identn ].            *)
(**************************************************************************)

(*** TODO: remove overwriting ***)
overwriting_vinterp_add("WRITEGOFERFILE",
  function [VARG_STRING file ; VARG_VARGLIST l] ->
      	      let idl = List.map (fun (VARG_IDENTIFIER id) -> id) l in
	      fun () -> Gofer_pp_file.write_extraction_file file idl
    | _ -> anomaly "WRITEGOFERFILE called with bad arguments.")
;;

(* $Id: gofer.ml,v 1.10 1997/03/07 12:22:24 jcfillia Exp $ *)
