(****************************************************************************)
(*                 The Calculus of Inductive Constructions                  *)
(*                                                                          *)
(*                                Projet Coq                                *)
(*                                                                          *)
(*                     INRIA        LRI-CNRS        ENS-CNRS                *)
(*              Rocquencourt         Orsay          Lyon                    *)
(*                                                                          *)
(*                                 Coq V6.3                                 *)
(*                               July 1st 1999                              *)
(*                                                                          *)
(****************************************************************************)
(*                                termenv.ml                                *)
(****************************************************************************)

open Std;;
open Pp;;
open Names;;
open Vectops;;
open Generic;;
open Evd;;
open Term;;
open Constrtypes;;
open Environ;;


(* Variables *)

type mind_specif = 
{sp:section_path;
 mib:mutual_inductive_body;
 tyi:int;
 args:constr array;
 mip:mutual_inductive_packet};;

let initial_sign = Vartab.initial_sign;;
let initial_fsign = Vartab.initial_fsign;;
let initial_assumptions = Vartab.initial_assumptions;;
let initial_sigma_sign () = (mt_evd(), initial_sign());;
let initial_sigma_assumptions () = (mt_evd(), initial_assumptions());;
let auto_save_variables = Vartab.auto_save_variables;;

let inf_app f = function Inf (c) -> Inf (f c) | Logic -> Logic;;

let is_id_inst id c =
  match c with
      VAR id' -> id=id'
    | _ -> false
;;

let inst_constr (ids,c) args =
    if for_all2eq is_id_inst ids args then c
    else if List.length ids <> List.length args then
        invalid_arg "Hum - a constant with a wrong number of arguments"
    else replace_vars (List.combine ids (List.map make_substituend args)) c
;;

let inst_type (ids,{body=t;typ=s}) args =
  mkCast (inst_constr (ids,t) args) (mkSort s)

let inst_fterm (ids,ft) fargs =
    if for_all2eq is_id_inst ids fargs then ft
    else if List.length ids <> List.length fargs then
        invalid_arg "Hum - an fconstant with a wrong number of arguments"
    else replace_vars (List.combine ids (List.map make_substituend fargs)) ft
;;

(* Constants and Existentials *)

(* There are 3 kinds of existential variables
 - The ones of the first kind are represented by the constructor Meta and 
     are generated by refinement tactics in clenv.ml
     "Meta" variables looks like "$23" in concrete syntax
 - The two others are constants with a name starting with "?"
   - In "evar" are stored a second kind of variables generated by refinement
     tactics (in clenv.ml) (what is the difference with Meta ??)
     They also appears as "$23" in concrete syntax
   - In "isevar" are stored the implicit arguments to be infered by trad.ml
     This variables corresponds "?" in concrete syntax

  Currently, it seems there is no interference between these 3 kinds
  of existential var, but when Meta and evar will be referrable by user
  by some common "?n" notation, their number will have to be distinct

  La diffrence entre isevar et Meta dans clenv.ml m'chappe...
*)

let existential_id_prefix = "?"

let new_isevar_path =
  let existential_ctr = ref 0 in
  fun k ->
    let n = (incr existential_ctr; !existential_ctr) in
    make_path ["isevar"] (make_ident existential_id_prefix n) k

let new_evar_path =
  let existential_ctr = ref 0 in
  fun k ->
    let n = (incr existential_ctr; !existential_ctr) in
    make_path ["evar"] (make_ident existential_id_prefix n) k

let newMETA = 
  let meta_ctr = ref 0 in
  fun () -> (incr meta_ctr; !meta_ctr)

let mkNewMeta () = mkMeta (newMETA ())

let evar_of_id k id = make_path ["evar"] id k

(*******************************************)
(* Selector functions on defined constants *)
(*******************************************)

(* A constant is an existential if its name has the existential id prefix *)
(* is_existential_id : identifier -> bool *)
let is_existential_id id =
  (atompart_of_id id = existential_id_prefix);;

(* is_existential : 'a oper -> bool *)
let is_existential_oper = function
    Const sp -> is_existential_id (basename sp)
  |  _       -> false
;;

(* is_existential constr -> bool *)
let is_existential = function
    DOPN (oper,_) -> is_existential_oper oper
  |  _            -> false
;;

let occur_existential = 
 let rec occrec = function
    DOPN(Const sp,_) as k -> is_existential k
  | DOPN(_,cl) -> exists_vect occrec cl
  | DOPL(_,cl) -> List.exists occrec cl
  | DOP2(_,c1,c2) -> occrec c1 or occrec c2
  | DOP1(_,c) -> occrec c
  | DOP0 _ -> false
  | DLAM(_,c) -> occrec c
  | DLAMV(_,cl) -> exists_vect occrec cl
  | Rel _ -> false
  | VAR _ -> false
 in occrec
;;  


(* defined_existential : 'a evar_map -> constr -> bool *)
let defined_existential sigma = function
  DOPN(Const sp,_) ->
    (try (match (Evd.map sigma sp).Evd.body with
         EVAR_DEFINED _ -> true
       | _ -> false)
     with Not_found -> false)
  | _ -> invalid_arg "termenv__defined_existential"
;;

(* defined_constant : 'a evar_map -> constr -> bool 
   does not use sigma !! *)
let defined_constant = function
    (DOPN(Const sp,_) as k) ->
    (try let (_,cb) = const_of_path sp
         in (match cb.cONSTBODY with
             None -> false
           | Some _ -> true)
     with Not_found -> false)
  | _ -> invalid_arg "termenv__defined_constant"
;;

(* opaque_constant : 'a evar_map -> constr -> bool *)
let opaque_constant sigma = function
    (DOPN(Const sp,_) as k) ->
    (try let (_,cb) = const_of_path sp
         in (match cb.cONSTBODY with
             None -> true
           | Some _ -> cb.cONSTOPAQUE)
     with Not_found -> false)
  | _ -> invalid_arg "termenv__opaque_constant"
;;
(* A const is defined if it is a defined existential or a constant
with a non empty body *)
(* defined_const : 'a evar_map -> constr -> bool *)
let defined_const sigma = function
    (DOPN(Const sp,_) as k) ->
     (defined_constant  k)
     or (is_existential k & (defined_existential sigma k))
  | _ -> invalid_arg "termenv__defined_const"
;;

(* A const is opaque if it is a non-defined existential or
a non-existential opaque constant *)
let opaque_const (sigma : 'a evar_map) = function
    (DOPN(Const sp,_) as k) ->
        if is_existential k
         then not (defined_existential sigma k)
         else (opaque_constant sigma k)
  | _ -> invalid_arg "termenv__opaque_const"
;;

(* A const is evaluable if it is defined and not opaque *)
let evaluable_const sigma k =
    try  defined_const sigma k & not (opaque_const sigma k)
    with Not_found -> false
;;

(* A translucent const is a defined existential *)
let translucent_const sigma k =
  is_existential k & defined_existential sigma k
;;

(* A cookable constant is a defined constant. Raises an error if sp is not
a constant *)
let cookable_constant sp =
  try let cb = snd (const_of_path sp) in cb.cONSTBODY <> None
  with Invalid_argument _ ->
    errorlabstrm "Termenv.cookable_constant"
      [< 'sTR"Can not cook a non constant" >]
;;

(* existential_value gives the value of a defined existential *)
let existential_value sigma k =
  let (sp,args) = destConst k in
  if defined_const sigma k then
    let evd = Evd.map sigma sp in
    match evd.Evd.body with
      EVAR_DEFINED c ->
 	inst_constr (ids_of_sign evd.hyps,c) (Array.to_list args)
    | _ -> anomalylabstrm "termenv__existential_value"
          [< 'sTR"The existential variable code just registered a" ;
            'sPC ; 'sTR"grave internal error." >]
  else failwith "undefined existential"
;;

(* constant_value gives the value of a defined constant *)
let constant_value sigma k =
  let (sp,args) = destConst k in
  let (_,cb) = const_of_path sp in
  if not cb.cONSTOPAQUE & defined_const sigma k then
    match cb.cONSTBODY with
      Some{contents=COOKED body} -> 
        inst_constr (ids_of_sign cb.cONSTHYPS,body) (Array.to_list args)
    | Some{contents=RECIPE _} ->
        anomalylabstrm "termenv__constant_value"
          [< 'sTR"a transparent constant which was not cooked">]
    | None -> anomalylabstrm "termenv__constant_value"
          [< 'sTR"a defined constant as no body.">]
  else failwith "opaque"
;;

(* existential_type gives the type of an existential *)
let existential_type sigma k = 
  let (sp,args)=destConst k in
  let evd = Evd.map sigma sp in
  inst_constr (ids_of_sign evd.hyps,evd.concl) (Array.to_list args)
;;

(* constant_type gives the type of a constant *)
let constant_type sigma k =
  let (sp,args) = destConst k in
  let (_,cb) = const_of_path sp in
  inst_type (ids_of_sign cb.cONSTHYPS,cb.cONSTTYPE) (Array.to_list args)
;;


(* gives the implicits of a constant *)
let constant_implicits k =
  let (sp,args) = destConst k in
  try let (_,cb) = const_of_path sp in list_of_implicits cb.cONSTIMPARGS
  with Not_found (* existential variables *) -> []
;;

(* gives the value of a const *)
let const_value sigma k = 
  let (sp,_)=destConst k in
    if is_existential k then
        existential_value sigma k
    else constant_value sigma k
;;

(* gives the type of a const *)
let const_type sigma k = 
  let (sp,_)=destConst k in
    if is_existential k then
      existential_type sigma k
    else constant_type sigma k
;;

let has_ise = 
 let rec hasrec = function
    DOP0(XTRA("ISEVAR",[])) -> true
  | DOPN(XTRA("REC",[]),_) -> true
  | DOPN(XTRA("MLCASE",_),_) -> true
  | DOPN(XTRA("MULTCASE",_),_) -> true
  | DOP1(_,c) -> hasrec c
  | DOP2(_,c1,c2) -> hasrec c1 or hasrec c2
  | DOPN(_,cl) -> exists_vect hasrec cl
  | DLAM(_,c) -> hasrec c
  | DLAMV(_,cl) -> exists_vect hasrec cl
  | _ -> false
 in hasrec
;;


(* Mutual Inductives *)

let mind_path = 
 function
    (DOPN(MutInd (sp,tyi),_)) -> if tyi = 0 then sp else
    let (_,mib) = mind_of_path sp in
    let mip = mind_nth_type_packet mib tyi
    in let (pa,_,k) = repr_path sp 
    in make_path pa (mip.mINDTYPENAME) k 
  | (DOPN(MutConstruct ((sp,tyi),ind),_)) -> 
    let (_,mib) = mind_of_path sp in
    let mip = mind_nth_type_packet mib tyi
    in let (pa,_,k) = repr_path sp 
    in make_path pa (mip.mINDCONSNAMES.(ind-1)) k 

  | _ -> invalid_arg "mind_path called with invalid argument"
;;

let mind_specif_of_mind = function 
      (DOPN(MutInd (sp,tyi),args)) -> let (_,mib) = mind_of_path sp in
   {sp=sp;mib=mib;tyi=tyi;args=args;mip=mind_nth_type_packet mib tyi}
    | _ ->  invalid_arg "mind_specif_of_mind";;

let mis_singl mispec = mispec.mib.mINDSINGL;;

let is_singl sp = let (_,mib) = mind_of_path sp in mib.mINDSINGL <> None;;

let mis_lc_arity' mispec = 
    let idhyps = ids_of_sign mispec.mib.mINDHYPS 
    and largs = Array.to_list mispec.args in
    (inst_constr (idhyps,mispec.mip.mINDLC) largs,
     {body = inst_constr (idhyps,mispec.mip.mINDARITY.body) largs;
      typ = mispec.mip.mINDARITY.typ});;

let mis_lc_arity mispec =
  let c, {body=b; typ=t} = mis_lc_arity' mispec in
    (c, DOP2 (Cast, b, DOP0 (Sort t)));;

let mis_lc mispec = 
    inst_constr (ids_of_sign mispec.mib.mINDHYPS,
                    mispec.mip.mINDLC) (Array.to_list mispec.args);;

let mis_lc_without_abstractions mispec = 
     let rec strip_DLAM c =
            match c with
              (DLAM  (n,c1)) -> strip_DLAM c1 
            | (DLAMV (n,v))  -> v
	    | _ -> assert false
     in strip_DLAM (mis_lc mispec)
;;

let mis_arity' mispec =
    let idhyps = ids_of_sign mispec.mib.mINDHYPS 
    and largs = Array.to_list mispec.args in 
    {body = inst_constr (idhyps,mispec.mip.mINDARITY.body) largs;
     typ = mispec.mip.mINDARITY.typ};;

let mis_arity mispec =
  let {body=b; typ=t} = mis_arity' mispec in
    DOP2 (Cast, b, DOP0 (Sort t));;

let mind_arity = (comp mis_arity mind_specif_of_mind);;

(* gives implicits of a inductive *)
let mind_implicits i = 
  list_of_implicits (mind_specif_of_mind i).mip.mINDIMPLICITS;;

(* gives the implicits of a constructor *)
let mconstr_implicits mc = 
  let (x, y, i, cl) = destMutConstruct mc in
  let i_0 = DOPN(MutInd (x,y),cl) in
    list_of_implicits (mind_specif_of_mind i_0).mip.mINDCONSIMPLICITS.(i-1)
;;

let mis_nparams mispec = mispec.mib.mINDNPARAMS;;
let mis_ntypes mispec = mispec.mib.mINDNTYPES;;
let mis_consnames mispec = mispec.mip.mINDCONSNAMES;;
let mis_nconstr mispec = Array.length (mispec.mip.mINDCONSNAMES);;
let mis_typename mispec = mispec.mip.mINDTYPENAME;;

let mis_kd mispec = mispec.mip.mINDKD;;
let mis_kn mispec = mispec.mip.mINDKN;;

let mis_is_finite mispec = mispec.mip.mINDFINITE;;

let mis_lamexp mispec = 
    match mispec.mip.mINDLAMEXP with
        None -> None
      | Some c -> Some(inst_constr (ids_of_sign mispec.mib.mINDHYPS,c)
                                   (Array.to_list mispec.args));;

let mind_lamexp = (comp mis_lamexp mind_specif_of_mind);;

let mis_recargs mispec =  
    Array.map (fun mip -> mip.mINDLISTREC) mispec.mib.mINDPACKETS;;

let mis_recarg mispec =  mispec.mip.mINDLISTREC;;


let mind_recargs = (comp mis_recargs mind_specif_of_mind);;
let mind_nparams = (comp mis_nparams mind_specif_of_mind);;
let mind_is_finite  = (comp mis_is_finite mind_specif_of_mind);;

let mindsp_nparams sp = (snd (mind_of_path sp)).mINDNPARAMS;;

let mis_is_recursive sp = 
 (is_recursive (interval 0 ((mis_ntypes sp)-1)) (mis_recarg sp));;

let mind_is_recursive = (comp mis_is_recursive mind_specif_of_mind);;

(* functions working for both notions of inductive definitions *)

(* TODO: Implement opaque abstractions *)
(* TODO: Understand what opaque abstractions _mean_! *)
let evaluable_abst = function
    (DOPN(Abst _,_)) -> true
  | _ -> invalid_arg "termenv__evaluable_abst";;

let translucent_abst = function
    (DOPN(Abst _,_)) -> false
  | _ -> invalid_arg "termenv__translucent_abst";;


let abst_value a = Abstraction.contract_abstraction a;;
let global_abst sp args = Abstraction.make_abstraction sp args;;

let search_abst pk id args =
let sp = Nametab.sp_of_id pk id
in if Abstraction.is_abstraction sp then
    global_abst sp args
   else failwith "search_abst"
;;



let const_abst_opt_value sigma c =
  match c with
  | DOPN(Const sp,_) ->
     if evaluable_const sigma c then Some (const_value sigma c)
     else None
  | DOPN(Abst sp,_) ->
     if evaluable_abst c then Some (abst_value c)
     else None
  | _ -> invalid_arg "Termenv.const_abst_opt_value: not a constant or abstraction"
;;   

let rec new_name_from_type default = function 
  |  DOPN(Const _,_) as x     -> basename (path_of_const x)
  |  DOP2(Cast,_,x)           -> new_name_from_type default x
  |  DOPN(MutInd _,_) as x    -> mis_typename (mind_specif_of_mind x)
  |  DOPN(MutConstruct ((sp,tyi),i),_) ->
        let (_,mib) = mind_of_path sp in
        mib.Constrtypes.mINDPACKETS.(tyi).Constrtypes.mINDCONSNAMES.(i-1)
  |  VAR id                   -> id
  |  _                        -> default;;

(* $Id: termenv.ml,v 1.18 1999/08/06 20:49:17 herbelin Exp $ *)
