(****************************************************************************)
(*                 The Calculus of Inductive Constructions                  *)
(*                                                                          *)
(*                                Projet Coq                                *)
(*                                                                          *)
(*                     INRIA        LRI-CNRS        ENS-CNRS                *)
(*              Rocquencourt         Orsay          Lyon                    *)
(*                                                                          *)
(*                                 Coq V6.3                                 *)
(*                               July 1st 1999                              *)
(*                                                                          *)
(****************************************************************************)
(*                                 stock.ml                                 *)
(****************************************************************************)

open Std;;
open More_util;;
open Pp;;
open Initial;;
open Names;;
open Library;;


(* The pattern table for tactics *)

(* The idea is that we want to write tactic files which are only
   "activated" when certain modules are loaded and imported.  Already,
   the question of how to store the tactics is hard, and we will not
   address that here.  However, the question arises of how to store
   the patterns that we will want to use for term-destructuring, and
   the solution proposed is that we will store patterns with a
   "module-marker", telling us which modules have to be open in order
   to use the pattern.  So we will write:

   let mark = make_module_marker ["<module-name>";<module-name>;...];;

   let p1 = put_pat mark "<parseable pattern goes here>";;

   And now, we can use:

             (get p1)

   to get the term which corresponds to this pattern, already parsed
   and with the global names adjusted.

   In other words, we will have the term which we would have had if we
   had done an:

        constr_of_com mt_ctxt (initial_sign()) "<text here>"

   except that it will be computed at module-opening time, rather than
   at tactic-application time.  The ONLY difference will be that
   no implicit syntax resolution will happen.

   So the entries we provide are:

   type module_mark;;

   value make_module_marker : string list -> module_mark;;

   type marked_term;;

   value put_pat : module_mark -> string -> marked_term;;

   value get_pat : marked_term -> constr;;

 *)

type module_mark = MMK of int;;
let path_mark_bij = (Mhb.create 17 : (section_path list,module_mark) Mhb.t);;

let mmk_ctr = ref 0;;
let new_mmk () = (incr mmk_ctr; !mmk_ctr);;

let make_module_marker stock sl =
  let spl =
    (try List.map path_of_string sl
     with UserError("Names.path_of_string",_) -> 
       error "make_module_marker: malformed section_path in marker") in
  let sorted_spl = Sort.list (fun sp1 sp2 -> sp_gt(sp1,sp2)) spl
  in (try Mhb.map path_mark_bij sorted_spl
      with Not_found ->
        let mmk = MMK(new_mmk())
        in (Mhb.add path_mark_bij (sorted_spl,mmk);
            mmk))
;;

let mark_satisfied mmk =
let spl = Mhb.pam path_mark_bij mmk
in subset spl (search_imports())
;;

(* src_tab: for each module_mark, stores the tickets of objects which
   need to be compiled when that mark becomes active.

   obj_tab: for each ticket, stores the (possibly nonexistent)
   compiled object

   ticket_tab: for each ticket, stores its module_mark and the string
   (source)

   string_tab: for each string * module_mark, stores the ticket.

*)

type 'a stock_args =
    {name : string;
     proc : string -> 'a};;

type 'a stock =
    {src_tab : (module_mark,int) Mhmfs.t;
     obj_tab : (int,'a) Mhm.t;
     ticket_string_bij : (int,string * module_mark) Mhb.t;
     args : 'a stock_args}
;;

type 'a stocked = int;;

let stock_ctr = ref 0;;
let new_stock () = (incr stock_ctr; !stock_ctr);;

let make_stock args =
  let stock =
    {src_tab = Mhmfs.create 17;
     obj_tab = Mhm.create 17;
     ticket_string_bij = Mhb.create 17;
     args = args}
  in stock
;;

let stock stock mmk s =
  try Mhb.pam stock.ticket_string_bij (s,mmk)
  with Not_found ->
  let idx = new_stock()
  in (Mhmfs.add stock.src_tab (mmk,idx);
      Mhb.add stock.ticket_string_bij (idx,(s,mmk));
      idx)
;;

let pr_mm mm =
  let spl = Mhb.pam path_mark_bij mm in
    prlist_with_sep pr_spc (fun p -> [< 'sTR(string_of_path p) >]) spl
;;


(* TODO: traiter a part les erreurs provenant de stock.args.proc
   ( = parsing quand [so]pattern appelle retrieve)
    -> eviter d'avoir l'erreur stocked datum *)
let retrieve stock idx =
  try Mhm.map stock.obj_tab idx
  with Not_found ->
    let (s,mmk) = Mhb.map stock.ticket_string_bij idx in 
      if mark_satisfied mmk then
        (try
           let c = stock.args.proc s in 
	     (Mhm.add stock.obj_tab (idx,c);
              c)
         with e ->
           (mSGNL [< 'sTR"Processing of the stocked datum " ; 'sTR s ;
                     'sTR" failed: " ; Errors.explain_user_exn e; 'fNL;
                     Library.fmt_module_state() >];
            Errors.reraise_user_exn e))
      else begin
          mSGERRNL [< 'sTR"The stocked object " ; 'sTR s ; 'sTR" was not compi\
lable" ; 'fNL ;
             'sTR"Its module mark was: " ; pr_mm mmk ; 'fNL ;
             Library.fmt_module_state() >] ;
          errorlabstrm "Stock.retrieve"
        [< 'sTR"The stocked object "; 'sTR s; 'sTR" was not compilable"; 
	   'fNL; 'sTR"Its module mark was: " ; pr_mm mmk ; 'fNL ;
           Library.fmt_module_state() >]
	    end
;;
        


(* $Id: stock.ml,v 1.16 1999/06/29 07:47:27 loiseleu Exp $ *)
