(****************************************************************************)
(*                 The Calculus of Inductive Constructions                  *)
(*                                                                          *)
(*                                Projet Coq                                *)
(*                                                                          *)
(*                     INRIA        LRI-CNRS        ENS-CNRS                *)
(*              Rocquencourt         Orsay          Lyon                    *)
(*                                                                          *)
(*                                 Coq V6.3                                 *)
(*                               July 1st 1999                              *)
(*                                                                          *)
(****************************************************************************)
(*                                states.ml                                 *)
(****************************************************************************)

open Std;;
open Names;;
open System;;
open Pp;;
open Library;;
open Summary;;

type state = Lib.frozen_t * Summary.frozen_t;;

type externable_states = (string * (string * state)) list;;

let sTATES = ref ([]:externable_states);;


(* D2| State saving/restoring *)

let raw_save_state name desc = 
  if List.mem_assoc name !sTATES then error(name ^ " already exists");
  force_freeze_caches();
  let state = (desc,(Lib.freeze(),Summary.get_frozen_summaries())) in 
  sTATES := (name,state) :: !sTATES;;

let reset_state (env,sum) = 
  Lib.unfreeze env;
  Summary.set_frozen_summaries sum;
  backtrack_caches ();;

let raw_restore_state name = 
  try reset_state (snd (List.assoc name (!sTATES)))
  with Not_found -> error(name ^ ": unknown state");;

let raw_restore_last_saved_state () =
  match !sTATES with
    ((_,(_,s))::_) -> reset_state s
  | _ -> errorlabstrm "states__raw_restore_last_saved_state"
     	       	       	[< 'sTR"Can not restore the last saved state,"; 'sPC;
			   'sTR"since no state has been saved." >]
;;

let forget_state verbose name =
  if List.mem_assoc name !sTATES then
        (sTATES := except_assoc name !sTATES;
         if verbose then warning(name ^ " forgotten"))
  else warning ("There is no state named " ^ name)
;;

let list_saved_states () =
    List.map (fun (n,(desc,_)) -> (n,desc)) (!sTATES);;

(* Total resetting *)

(*
let mAGIC = 19755;;
let aTOM_MAGIC = 19600;;
*)
let mAGIC = 19763;;

let (extern_state,intern_state) =
  let (raw_extern_state,raw_intern_state) = System.extern_intern(mAGIC,".coq") 
  in
  let extern_state s = raw_extern_state s !sTATES

  and intern_state s =
    if !sTATES <> [] then
        error "cannot intern a state when we already have some"
    else sTATES := raw_intern_state s

  in (extern_state,intern_state)
;;


(* $Id: states.ml,v 1.9 1999/06/29 07:47:27 loiseleu Exp $ *)
