(****************************************************************************)
(*                 The Calculus of Inductive Constructions                  *)
(*                                                                          *)
(*                                Projet Coq                                *)
(*                                                                          *)
(*                     INRIA        LRI-CNRS        ENS-CNRS                *)
(*              Rocquencourt         Orsay          Lyon                    *)
(*                                                                          *)
(*                                 Coq V6.3                                 *)
(*                               July 1st 1999                              *)
(*                                                                          *)
(****************************************************************************)
(*                                library.ml                                *)
(****************************************************************************)

open Std;;
open Names;;
open Pp;;
open More_util;;
open System;;
open Summary;;
open Libobject;;


let lookup_module s =
  try
    Lib.find_entry_P
      (function (sp,Lib.ClosedDir(s',{Lib.module_p=true},_,_)) -> (s = s')
       	| _ -> false)
  with Failure _ ->
    errorlabstrm "lookup_module" [< 'sTR s ; 'sTR " does not exist." >]
;;

let search_modules () =
  let ctxt = Lib.contents_after None in
  map_succeed (function (sp,Lib.ClosedDir(s,{Lib.module_p=true},_,_)) -> (s,sp)
                 | _ -> failwith "caught") ctxt
;;

(* Modules are successively read, loaded and possibly opened.  A
 * module is read if it a closed section exists. Reading is not
 * synchronized because it consists of adding it at the beginning of
 * the context segment.
 * Loading and opening (= caching) a module is synchronized.
 *)
let module_is_read sp =
  try
    match Lib.map sp with
      Lib.ClosedDir(_,{Lib.module_p=true},_,_) -> true
    | _ -> false
  with _ -> false
;;

(* Same with a module name instead of a section path. *)
let module_is_known s =
    try let _ = lookup_module s in true
    with UserError _ -> false
;;


(* Dealing with Implementation/Specification. *)

(* If we required an implementation, then require implementation of
 * dependencies, otherwise require what we can find (impl. or spec.).
 * Is it really what we want ? Searchisos uses another strategy.
 *)
let dependency_spec = function
    false -> Some false
  | _ -> None
;;

(* Given a filename returns true if only the .vi file was found, or
 * false if the .vo file exists.
 *)
let spec_on_path spec filename =
  match spec with
    Some only_spec -> only_spec
  | None ->
      if is_in_path (search_paths()) (make_suffix filename ".vo") then false
      else
 	if is_in_path (search_paths()) (make_suffix filename ".vi") then true
  	else
	  errorlabstrm "Library.intern_module"
	    (hOV 0 [< 'sTR"Can't find module" ; 'sPC ; 'sTR filename ; 'sPC ;
                     'sTR"on loadpath" >])
;;

let rec specification_of_module = function
  (Lib.ClosedDir(s,odc,cdc,ctxt)) ->
    if not odc.Lib.module_p then
      error "Only modules can be converted to specifications"
    else
      let ctxt' =
 	map_succeed
	  (function (sp,Lib.LEAF obj) -> 
      	        (sp,Lib.LEAF (extract_object_specification obj))
	    | (_,Lib.OpenDir _) -> 
      	       	anomaly "specification_of_module: found an OpenDir"
	    | (sp,(Lib.ClosedDir(_,odc',_,_) as cdir)) ->
               	if odc'.Lib.module_p then (sp,specification_of_module cdir)
               	else failwith "caught"
	    | (_,Lib.Import _ as x) -> x) 
	  ctxt
      in Lib.ClosedDir(s,odc,cdc,ctxt')
  | _ -> invalid_arg "specification_of_module"
;;


(* Numeros magiques :
   1001 : extern_module de la V5.10
   1002 : extern_module_specification de la V5.10
   1003 : extern_module de la V6.1
   1004 : extern_module de la V6.1 sans reinterning
   1014 : extern_module_specification de la V6.1 sans reinterning
   1005 : extern_module de la V6.2
   1015 : extern_module_specification de la V6.2
   1006 : extern_module pour la V6.2.3
   1016 : extern_module_specification pour la V6.2.3
   1007 : extern_module pour la V6.3.1
   1017 : extern_module_specification pour la V6.3.1
 *)

let vo_magic_number = 1007;;
let vi_magic_number = 1017;;

let (raw_extern_module, raw_intern_module) =
      System.extern_intern(vo_magic_number,".vo")
and (raw_extern_module_specification, raw_intern_module_specification) =
      System.extern_intern(vi_magic_number,".vi")
;;

let disk_intern_module spec_only =
  if spec_only then raw_intern_module_specification else raw_intern_module
;;

let disk_extern_module spec_only file (sp,md) =
  if spec_only
  then raw_extern_module_specification file (sp,(specification_of_module md))
  else raw_extern_module file (sp,md)
;;


let needed_modules = function
    Lib.ClosedDir(_,odc,_,ctxt) ->
      let imports = map_succeed 
		      (function (_,Lib.Import(_,isp,_)) -> isp
                         | _ -> failwith "caught") 
		      ctxt in
      	uniquize((List.rev odc.Lib.imports)@imports)
  | _ -> invalid_arg "needed_modules"
;;

(* Externing a module *)

let extern_module spec s filename =
  let f = match filename with Some f -> f | None -> s in
  let sp = lookup_module s in
  match Lib.map sp with
    Lib.ClosedDir _ as d -> disk_extern_module spec f (sp,d)
  | _ -> errorlabstrm "Library.extern_module"
        [< 'sTR"Can only extern a module-object" >]
;;

(* Interning a module *)

let rec intern_module spec s ofile =
  if not (module_is_known s) then begin
    let filename = match ofile with Some f -> f | None -> s in
    let spec_only = spec_on_path spec filename in
    let (sp,d) = disk_intern_module spec_only filename in
    pP [< '(vB 0) ; '(vB 2) ; 
	 h 0 [< 'sTR"[Reinterning " ;
	       'sTR(if spec_only then "specification " else "");
	       'sTR s ; 'sTR"..." >] >] ;
    match d with
      Lib.ClosedDir(_,odc,_,_) ->
      	pP [< 'cLOSE ; 'sTR"done]" ; 'cLOSE ; 'fNL >];
      	let impbefore = needed_modules d in
      	List.iter
	  (fun sp ->
	    let name = string_of_id(basename sp) in
	    intern_module (dependency_spec spec_only) name None)
          impbefore;
      	Lib.add_module (sp,d)
    | _ -> failwith "Library.intern_module"
  end
;;


(* Synchronized operations: load and open *)

(* Summary of loaded modules.
 * Read modules are also loaded unless a Reset occured, which is not very safe
 * since the typechecker may type constant which constraint haven't been
 * merged. The toplevel is safe since we can only access constants with short
 * names, and opened modules have all been loaded.
 *)
let loaded_modules = ref ([] : section_path list);;
let module_is_loaded sp = List.mem sp !loaded_modules;;
declare_summary "loaded"
  {freeze_function = (fun () -> !loaded_modules);
   unfreeze_function = (fun l -> loaded_modules := l);
   init_function = (fun () -> loaded_modules := [])}
;;

(* Imports Summary
 * Opened modules have first been loaded.
 *)
let opened_modules = ref ([] : section_path list);;
let module_is_opened sp = List.mem sp !opened_modules;;
let search_imports () = !opened_modules;;
declare_summary "opened"
  {freeze_function = (fun () -> !opened_modules);
   unfreeze_function = (fun l -> opened_modules := l);
   init_function = (fun () -> opened_modules := [])}
;;


(* Initializing the environment *)
let reset_library () = Lib.init(); init_caches();;

let load_closeddir sp (s,odc,cdc) = ();;
let cache_closeddir sp (s,odc,cdc) = ();;

(* Some objects must be cached even though the module is not opened.
   ML modules. Assume:
     1- A.v uses A.ml which defines an object TOTO
     2- B.v imports A and defines TOTO objects
     3- C.v requires B, but not A (A is loaded but not opened)
     Then if A.cmo is not linked, the caching of TOTO objects in B will fail.
   Universe constraints: for consistency reasons, the universes of a loaded
     module which is not opened must be added anyway.

  The following function scans a loaded module to cache such objects.
*)
let rec load_import sp =
  if not (module_is_loaded sp)
  then (load_decl (sp, Lib.map sp);
	loaded_modules := sp :: !loaded_modules)

and load_decl = function
    (sp,Lib.LEAF o) -> load_object o
  | (sp,Lib.ClosedDir (_,odc,_,ctxt)) ->
      if odc.Lib.module_p then
	(do_listRL load_import odc.Lib.imports;
	 List.iter load_decl ctxt)
  | (sp,Lib.Import(_,isp,_)) -> load_import isp
  | _ -> ()
;;

let rec cache_import sp =
  match Lib.map sp with
    Lib.ClosedDir(_,odc,cdc,ctxt) ->
      if not (module_is_opened sp) then
      if List.for_all module_is_read odc.Lib.imports then
             (List.iter (fun (sp',_ as d) ->
                         if List.mem sp' cdc.Lib.exports then recache_decl d)
		ctxt;
             opened_modules := sp:: !opened_modules)
      else error ("Some needed imports haven't been done for module "^
      	       	   (string_of_path sp))
  | _ -> errorlabstrm "library__cache_import"
      	       	      [< 'sTR"Cannot import a non-directory" >]

(* Cache a declaration appearing in a module. Does not cache Imports that
 * are not re-exported.
 *)
and recache_decl = function
    (_ ,Lib.Import(_,isp,unexport)) -> if not unexport then cache_import isp
  | (sp,Lib.LEAF x) -> cache_object (sp,x)
  | (_ ,Lib.OpenDir(_,_)) -> anomaly "library__recache_decl: OpenDir"
  | (sp,Lib.ClosedDir(s,odc,cdc,_)) -> cache_closeddir sp (s,odc,cdc)
;;

let cache_decl = function
    (sp,Lib.OpenDir(_,_)) -> freeze_to_key sp
  | (sp,Lib.LEAF x) -> cache_object (sp,x)
  | (_ ,Lib.Import(_,isp,_)) -> cache_import isp
  | (sp,Lib.ClosedDir(s,odc,cdc,_)) -> cache_closeddir sp (s,odc,cdc);;

let load_cache_decl d = (load_decl d; cache_decl d);;

(* Most recent objects are in the head of l *)
let load_cache_ctxt l = do_listRL load_cache_decl l;;


(* require_module 
    -  spec:bool option	        Some (true if Specification, false if Implem.)
      	       	       	       	None if missing
    -  name:string 		the module name
    -  filename:string option	the filename 
    -  locally:bool	        true if not exported, false otherwise *)

let rec require_module spec name filename locally =
  intern_module spec name filename;
  if module_is_opened (lookup_module name) then
    warning (name^" already imported");
  import_export_module name locally

(* The warning is issued in require_module because Searchisos uses these
 * functions and it must be silent.
 *)
and import_export_module s import_p =
  let isp = lookup_module s in
  if not (module_is_opened isp) then
    let anon_id = Lib.next_anonymous_id() in
    load_cache_decl (Lib.add((anon_id,OBJ),Lib.Import(s,isp,import_p)))
;;

(* Read a module. The constants are not made visible.
 * Also load it to avoid universe inconsistencies, but no Import object
 * is added to the context.
 *)
let read_module name ofile =
  intern_module None name ofile;
  load_import (lookup_module name)
;;



(* Other synchronization operations *)

let rebuild_caches () =
  init_caches();
  Lib.app load_cache_decl;;

(* There are so many functions called init_caches... *)
let init_caches = Summary.init_caches;;

(* Protocol for unfreezing states and verifying that the tables are
 * in sync with the library.

 * When we are "out_of_sync", we need to unfreeze the latest valid
 * summary (summary__find_latest), and then re-cache all the entries
 * in the library which follow.  Then we can try to resynchronize, and
 * if this reports that there was something to resync, we save that
 * frozen state under the key of the last lib-entry.

 * If we aren't out of sync, then "sum_sp" contains the latest valid
   frozen summary.  "sp" contains the latest library-entry-name.  If
   the last thing unfrozen wasn't sum_sp, then we treat this as being
   out of sync.

   If the last thing unfrozen _was_ sum_sp, then we still need to
   check if the caches are in sync or not, and if they aren't we need
   to resynchronize them.

 *)

let last_unfrozen = ref (None : section_path option);;

let backtrack_caches () = 
  match Lib.last_entry() with
     None    -> ()
   | Some sp ->
       try let sum_sp = Summary.find_latest () in
           Summary.unfreeze_latest sum_sp;
           last_unfrozen := Some sum_sp;
           load_cache_ctxt (Lib.contents_after (Some sum_sp));

           if [] = Summary.list_frozen_summaries() then begin
              Summary.freeze_to_key sp;
              last_unfrozen := Some sp;
              mSGNL [< 'sTR"[Froze to " ; 'sTR(string_of_path sp) ; 'sTR"]" >] 
           end

       with Failure _ ->
          init_caches();
          rebuild_caches();
          if [] = Summary.list_frozen_summaries() then begin
             Summary.freeze_to_key sp;
             last_unfrozen := Some sp;
             mSGNL [< 'sTR"[Froze to " ; 'sTR(string_of_path sp) ; 'sTR"]" >] 
          end
;;

let force_freeze_caches () =
  try if (not((Lib.last_entry () = Some(Summary.find_latest())))) then
        failwith "caught"
  with Failure _ ->
        (match Lib.last_entry() with
            None -> ()
          | Some sp ->
             (Summary.freeze_to_key sp;
              last_unfrozen := Some sp;
              mSGNL [< 'sTR"[Froze to " ; 'sTR(string_of_path sp) ; 'sTR"]" >]))
;;

let open_section name =
  cache_decl (Lib.open_dir name {Lib.module_p = false;
                                  Lib.imports = []})
;;

(* This function does not rebuild caches; hence, it assumes that if
   something goes wrong, the offending function understands how to
   rollback it's modifs.  Since this is restricted to the functions
   which manipulate the tables, it is a safe assumption.
 *)
let rollback f arg =
  let fs = Lib.freeze()
  in try f arg
     with e -> (Lib.unfreeze fs; raise e)
;;

let add_anonymous_object obj =
  let anon_id = Lib.next_anonymous_id() in
    rollback (fun () -> load_cache_decl (Lib.add_leaf((anon_id,OBJ),obj)))
      ()
;;

let add_named_object idp obj =
  rollback (fun () -> load_cache_decl(Lib.add_leaf(idp,obj)))
    ()
;;

let cwd () = Lib.cwd();;

(* can only open a module when the only top-level entries around are
   other (closed) modules, and import declarations
 *)
let open_module name =
  let ctxt = Lib.contents_after None in
  if List.for_all (function (_,Lib.ClosedDir _) -> true
                | (_,Lib.Import _) -> true
                | _ -> false) ctxt then
    cache_decl (Lib.open_dir name {Lib.module_p = true;
                                    Lib.imports = !loaded_modules})
  else errorlabstrm "open_module"
       [< 'sTR"You cannot open a module" ; 'sPC ;
          'sTR"when there are things other than" ; 'sPC ;
          'sTR"Modules and Imports in the context." >]
;;

(* can only close a module when:
 * (1) all the imported names are currently loaded modules
 *)
let close_module sp contents =
  (if List.for_all
       (function (_,Lib.Import(_,isp,_)) ->
                    (try (match Lib.map isp with
                            Lib.ClosedDir(_,odc,_,_) -> odc.Lib.module_p
                          | _ -> false)
                     with Not_found -> false)
                 | _ -> true)
       contents
  then 
    let exports =
      map_succeed
         (function (sp,Lib.LEAF _) -> sp
                 | (sp,Lib.Import(_,_,false)) -> sp
                 | _ -> failwith "caught")
         contents in
    Lib.close_dir sp {Lib.exports = exports; Lib.export_fn = None}
  else failwith "Cannot close a module right now");
  backtrack_caches ()
;;


let with_heavy_rollback f x =
    let sum = freeze_summary()
    and flib = Lib.freeze() in
    try f x
    with reraise -> (unfreeze_summary sum;
                     Lib.unfreeze flib;
                     raise reraise)
;;

let fmt_module_state () =
  let opened = search_imports ()
  and loaded = search_modules () in
       [< 'sTR"Imported (open) Modules: " ;
          prlist_with_sep pr_spc (fun sp -> [< 'sTR(string_of_path sp) >])
      	       opened ;
      	  'fNL ;
          'sTR"Loaded Modules: " ;
          prlist_with_sep pr_spc
            (fun (s,sp) -> [< 'sTR s ; 'sTR" = " ; 'sTR(string_of_path sp) >])
      	    loaded ;
      	  'fNL >]
;;

(* D1 | Resetting *)

let raw_reset_section str =
  (try let sp = Lib.find_dir (Some str) in
       Lib.reset_to sp
   with Failure _ -> error ("Section "^str^" not declared"));
  backtrack_caches()
;;

let reset_keeping str =
  Lib.reset_keeping_P (fun node -> str = (Lib.node_name node));
  backtrack_caches ();;

let reset_to str =
  Lib.reset_to_P (fun node -> str = (Lib.node_name node));
  backtrack_caches ()
;;

let find_nearest_module () =
  Lib.find_entry_P (function (_,Lib.OpenDir (_,{Lib.module_p=true})) -> true
                       | _ -> false)
;;

let find_nearest_section () =
  Lib.find_entry_P (function (_,Lib.OpenDir (_,{Lib.module_p=false})) -> true
                       | _ -> false)
;;

exception Unknown_dir;;

let find_dir s =
  try Lib.find_dir s
  with Failure _ -> raise Unknown_dir
;;

let is_module_p sp =
  match Lib.map sp with
    Lib.OpenDir(_,{Lib.module_p=true}) -> true
  | _ -> false
;;

let is_section_p sp =
  match Lib.map sp with
    Lib.OpenDir(_,{Lib.module_p=false}) -> true
  | _ -> false
;;

(* $Id: library.ml,v 1.32 1999/11/30 23:39:15 herbelin Exp $ *)
