(****************************************************************************)
(*                 The Calculus of Inductive Constructions                  *)
(*                                                                          *)
(*                                Projet Coq                                *)
(*                                                                          *)
(*                     INRIA        LRI-CNRS        ENS-CNRS                *)
(*              Rocquencourt         Orsay          Lyon                    *)
(*                                                                          *)
(*                                 Coq V6.3                                 *)
(*                               July 1st 1999                              *)
(*                                                                          *)
(****************************************************************************)
(*                                errors.ml                                 *)
(****************************************************************************)

open Std;;
open Pp;;
open Ast;;

let debug = ref false;;

let print_loc loc =
  if loc = dummy_loc then [< 'sTR"<unknown>" >]
  else [< 'iNT (fst loc); 'sTR"-"; 'iNT (snd loc) >]
;;

let guill s = "\""^s^"\""

(* assumption : explain_sys_exn does NOT end with a 'FNL anymore! *)
let rec explain_sys_exn_default e =
  match e with
    Stream.Failure -> [< 'sTR"Error: uncaught Parse.Failure." >]
  | Stream.Error txt -> hOV 0 [< 'sTR"Syntax error: "; 'sTR txt >]
  | Token.Error txt -> hOV 0 [< 'sTR"Lexer error: "; 'sTR txt >]
  | Clexer.BadToken tok ->
      hOV 0 [< 'sTR"Error: the token '"; 'sTR tok;
               'sTR"' does not respect the lexer rules." >]
  | Sys_error msg -> hOV 0 [< 'sTR"OS error: " ; 'sTR msg >]
  | UserError(s,pps) ->
      hOV 1 [< 'sTR"Error: ";
               (if !debug then  [< 'sTR s; 'sTR"."; 'sPC >] else [<>]);
               pps >]

  | Out_of_memory -> [< 'sTR"Out of memory" >]
  | Stack_overflow -> [< 'sTR"Stack overflow" >]

  | Ast.No_match s -> hOV 0 [< 'sTR"Ast matching error : "; 'sTR s >]

  | Anomaly (s,pps) ->
      hOV 1 [< 'sTR"System Anomaly: ";
               (if !debug then  [< 'sTR (guill s); 'sTR"."; 'sPC >] else [<>]);
               pps; 'fNL; 'sTR"Please report." >]

  | Match_failure(filename,pos1,pos2) ->
      hOV 1 [< 'sTR"Match failure in file " ;
               'sTR filename ; 'sTR " from char #" ;
               'iNT pos1 ; 'sTR " to #" ; 'iNT pos2 ;
               'sTR ": Please report." >]

  | Not_found -> [< 'sTR"Search error. Please Report." >]

  | Failure(s) -> hOV 0 [< 'sTR "System Error (Failure): " ; 'sTR (guill s) ;
                           'sTR ". Please report." >]

  | Invalid_argument(s) ->
      hOV 0 [< 'sTR"Invalid argument: " ; 'sTR (guill s) ;
	       'sTR ". Please report." >]

  | Sys.Break -> hOV 0 [< 'fNL; 'sTR"User Interrupt." >]

  | Stdpp.Exc_located (loc,exc) ->
      hOV 0 [< if loc = Ast.dummy_loc then [<>]
               else [< 'sTR"At location "; print_loc loc; 'sTR":"; 'fNL >];
               explain_sys_exn_default exc >]

  | reraise ->
      flush_all();
      (try Printexc.print raise reraise with _ -> ());
      flush_all();
      [< 'sTR "Please report." >]
;;

let fmt_disclaimer() = 
  [< 'sTR"If this is in user-written tactic code, then" ; 'sPC ;
     'sTR"it needs to be modified." ; 'sPC ;
     'sTR"If this is in system code, then"; 'sPC;
     'sTR"it needs to be reported." >];;

(* assumption : explain_user_exn does NOT end with a 'FNL anymore! *)
let rec explain_user_exn_default e =
  match e with
    Stream.Failure -> [< 'sTR"Error: uncaught Parse.Failure." >]

  | Stream.Error txt -> hOV 0 [< 'sTR"Syntax error: "; 'sTR txt >]
  | Token.Error txt -> hOV 0 [< 'sTR"Lexer error: "; 'sTR txt >]
  | Clexer.BadToken tok ->
      hOV 0 [< 'sTR"Error: the token '"; 'sTR tok;
               'sTR"' does not respect the lexer rules." >]

  | Sys_error msg -> hOV 0 [< 'sTR"OS error: " ; 'sTR msg >]

  | UserError(s,pps) ->
      hOV 1 [< 'sTR"Error: ";
               (if !debug then  [< 'sTR (guill s); 'sTR"."; 'sPC >] else [<>]);
               pps >]

  | Out_of_memory -> [< 'sTR"Out of memory" >]
  | Stack_overflow -> [< 'sTR"Stack overflow" >]

  | Anomaly (s,pps) ->
      hOV 1 [< 'sTR"System Anomaly: ";
               (if !debug then  [< 'sTR (guill s); 'sTR"."; 'sPC >] else [<>]);
               pps; 'fNL; fmt_disclaimer() >]

  | Match_failure(filename,pos1,pos2) ->
      hOV 1 [< 'sTR"Match failure in file " ;
               'sTR filename ; 'sPC; 'sTR "from char #" ;
               'iNT pos1 ; 'sTR " to #" ; 'iNT pos2 ; 'sTR ".";
               'fNL; fmt_disclaimer() >]

  | Ast.No_match s ->
      hOV 0 [< 'sTR"Ast matching error : "; 'sTR (guill s); 'sTR".";
               'fNL; fmt_disclaimer() >]

  | Not_found -> hOV 0 [< 'sTR"Search error."; 'fNL; fmt_disclaimer() >]

  | Failure(s) ->
      hOV 0 [< 'sTR "Somebody raised a Failure exception" ; 'sPC;
	       'sTR (guill s) ; 'sTR "." ; 'fNL; fmt_disclaimer() >]

  | Invalid_argument(s) ->
      hOV 0 [< 'sTR"Invalid argument: " ; 'sTR (guill s) ; 'sTR "." ;
               'fNL; fmt_disclaimer() >]

  | Sys.Break -> [<'fNL;  'sTR"User Interrupt." >]

  | Stdpp.Exc_located (loc,exc) ->
      hOV 0 [< if loc = Ast.dummy_loc then [<>]
               else [< 'sTR"At location "; print_loc loc; 'sTR":"; 'fNL >];
               explain_user_exn_default exc >]

  | reraise ->
      flush_all();
      (try Printexc.print raise reraise with _ -> ());
      flush_all();
      fmt_disclaimer()
;;

let reraise_user_exn = function
    Match_failure _ -> error "match_failure"
  | Not_found -> error "search"
  | Failure s -> error s
  | Invalid_argument s -> error s
  | e -> raise e
;;

let reraise_sys_exn = function
    Match_failure _ -> ()
  | Not_found -> ()
  | Failure s -> ()
  | Invalid_argument s -> ()
  | UserError _ -> ()
  | e -> raise e
;;

let raise_if_debug e =
  if !debug then raise e;;


let eXPLAIN_SYS_EXN = ref explain_sys_exn_default;;

let explain_sys_exn e = !eXPLAIN_SYS_EXN e;;

let eXPLAIN_USER_EXN = ref explain_user_exn_default;;

let explain_user_exn e = !eXPLAIN_USER_EXN e;;

(* $Id: errors.ml,v 1.15 1999/06/29 07:47:23 loiseleu Exp $ *)
