(****************************************************************************)
(*                 The Calculus of Inductive Constructions                  *)
(*                                                                          *)
(*                                Projet Coq                                *)
(*                                                                          *)
(*                     INRIA        LRI-CNRS        ENS-CNRS                *)
(*              Rocquencourt         Orsay          Lyon                    *)
(*                                                                          *)
(*                                 Coq V6.3                                 *)
(*                               July 1st 1999                              *)
(*                                                                          *)
(****************************************************************************)
(*                                 edit.ml                                  *)
(****************************************************************************)

open Pp;;
open Std;;


type ('a,'b,'c) t = {mutable focus : 'a option;
                     mutable last_focused_stk : 'a list;
                     buf : ('a, 'b Bstack.t * 'c) Mlm.t}
;;

let mt () = {focus = None;
             last_focused_stk = [];
             buf = Mlm.create()};;

let focus e nd =
  (match nd with
     None -> ()
   | Some f -> if not(Mlm.in_dom e.buf f) then
                  invalid_arg "Edit.focus");
  (match e.focus with
     None -> ()
   | Some foc -> if e.focus <> nd then
                  e.last_focused_stk <- (foc::(except foc e.last_focused_stk))
  );
  e.focus <- nd;;

let last_focused e =
  match e.last_focused_stk with
    [] -> None
  | (f::_) -> (Some f)
;;

let restore_last_focus e = focus e (last_focused e);;
(*
  match e.last_focused_stk with
    [] -> e.focus <- NONE
  | (f::r) -> (e.last_focused_stk <- r ; e.focus <- (SOME f))
;;
*)

let focusedp e =
    match e.focus with
    None -> false
  | _    -> true
;;

let read e =
    match e.focus with
    None -> None
  | Some d ->
    let (bs,c) = (Mlm.map e.buf d)
    in (match Bstack.top bs with
        None -> anomaly "Edit.read"
      | Some v -> Some(d,v,c))
;;

let mutate e f =
    match e.focus with
    None -> invalid_arg "Edit.mutate"
  | Some d ->
    let (bs,c) = (Mlm.map e.buf d)
    in Bstack.app_push bs (f c)
;;

let rev_mutate e f =
    match e.focus with
    None -> invalid_arg "Edit.rev_mutate"
  | Some d ->
    let (bs,c) = (Mlm.map e.buf d)
    in Bstack.app_repl bs (f c)
;;

let undo e n =
    match e.focus with
    None -> invalid_arg "Edit.undo"
  | Some d ->
    let (bs,_) = Mlm.map e.buf d
    in if Bstack.depth bs <= n then
        errorlabstrm "Edit.undo" [< 'sTR"Undo stack would be exhausted" >]
       else
           repeat n (fun () -> Bstack.pop bs)()
;;

let create e (d,b,c,udepth) =
    if Mlm.in_dom e.buf d then
        errorlabstrm "Edit.create" 
      	       	         [< 'sTR"Already editing something of that name" >]
    else let bs = Bstack.create udepth
         in (Bstack.push bs b;
             Mlm.add e.buf (d,(bs,c)))
;;

let delete e d =
    if not(Mlm.in_dom e.buf d) then
        errorlabstrm "Edit.delete" [< 'sTR"No such editor" >]
    else (Mlm.rmv e.buf d;
          e.last_focused_stk <- (except d e.last_focused_stk);
          match e.focus with
          Some d' -> if d = d' then (e.focus <- None ; (restore_last_focus e))
        | None -> ())
;;

let dom e = Mlm.dom e.buf;;

let empty e =
    e.focus <- None;
    e.last_focused_stk <- [];
    Mlm.empty e.buf
;;


(* $Id: edit.ml,v 1.8 1999/07/26 14:34:58 mohring Exp $ *)
