(****************************************************************************)
(*                 The Calculus of Inductive Constructions                  *)
(*                                                                          *)
(*                                Projet Coq                                *)
(*                                                                          *)
(*                     INRIA        LRI-CNRS        ENS-CNRS                *)
(*              Rocquencourt         Orsay          Lyon                    *)
(*                                                                          *)
(*                                 Coq V6.3                                 *)
(*                               July 1st 1999                              *)
(*                                                                          *)
(****************************************************************************)
(*                                declare.ml                                *)
(****************************************************************************)

open Std;;
open Vectops;;
open Names;;
open Impuniv;;
open Generic;;
open Term;;
open Constrtypes;;
open Himsg;;
open Pp;;
open More_util;;

open Library;;
open Environ;;
open Termenv;;
open Dischcore;;
open Variables;;
open Constants;;
open Indtypes;;
open Indrec;;
open Abstraction;;
open Reduction;;
open Initial;;

(********************************************************************)
(*                Declaring Variables and Constants                 *)
(********************************************************************)


let is_global id = 
  try let osp = Nametab.sp_of_id CCI id in
      prefix_of (List.rev (dirpath osp)) (List.rev (Lib.cwd()))
  with Not_found -> false
;;

let no_clash id =
  if is_global id then
    error ("Clash with previous constant " ^ (string_of_id id)) else
  if mem_sign (initial_sign ()) id then
    error ("Clash with previous variable " ^ (string_of_id id)) else
  true;;

let declare_constant id cobj =
  (* we forbid to hide a previous constant with a constant *)
  if no_clash id then add_named_object (id,OBJ) (inConstant cobj)
;;

let declare_variable (sticky,stre,impl) (name,vbody,u) =
    (match stre with
     NeverDischarge ->
     error "Cannot declare a variable which will appear at top-level - use Axiom"
   | DischargeAt disch_sp ->
     if not(is_section_p disch_sp) then
         error "Cannot declare a variable which will appear at top-level - use Axiom");
    if mem_sign (initial_sign()) name then 
        error ("Cannot redeclare global name "^(string_of_id name))
    else add_named_object (name,OBJ) (inVariable(name,vbody,stre,impl,sticky,u))
;;

let declare_minductive (id,lnames,cmap,u) =
   (* we forbid to hide a previous constant with a constant *)
   if List.for_all no_clash lnames then
     add_named_object (id,OBJ) (inMutualInductive (cmap,u))
;;

let declare_syntax_constant id cmap =
try Nametab.sp_of_id CCI id;
    error("Clash with previous name " ^ (string_of_id id))
with Not_found ->
    add_named_object (id,OBJ) (inSyntaxConstant cmap)
;;

let declare_abstraction id ao =
try Nametab.sp_of_id ao.abs_kind id;
    error("Clash with previous name " ^ (string_of_id id))
with Not_found ->
    add_named_object (id,OBJ) (inAbstraction ao)
;;

(* Var intro, Naming the hypothesis *)
let machine_variable penv (name,stre,sticky,t) =
  let (u,(v1,v2)) = infexecute_variable penv name t in
  let impl_fun x =
    if is_implicit_args() then IMPL_AUTO (poly_args x)
    else NO_IMPL in
  let v2' = option_app (fun x -> (x,impl_fun x.body)) v2 in
    declare_variable (sticky,stre,impl_fun v1.body) (name,(v1,v2'),u)
;;

let machine_constant penv cdef =
  let ((name,_,_),_) = cdef in
  let cobj = infexecute_constant penv cdef in
    declare_constant name cobj;;

let machine_constant_red penv cdef red_option =
  let ((name,_,_),_) = cdef in
  let cobj = infexecute_constant_red penv cdef red_option in
    declare_constant name cobj;;

let machine_constant_verbose  env ((name,_,_),_ as const) = 
  machine_constant env const;
  pPNL [<print_id name; 'sTR " is defined" >]
;;

let machine_parameter penv (name,cty) =
  let cobj = infexecute_parameter penv name cty in
    declare_constant name cobj
;;

(* Construction of a mutual recursive definition for compatibility *)

let declare_eliminations mindid = 
  let (sigma,(sign,fsign)) = initial_sigma_assumptions() in
  let sp = Nametab.sp_of_id CCI mindid in
  let mind = Machops.global_reference (gLOB sign) sp mindid in
  let redmind = minductype_spec sigma mind in
  let mindstr = string_of_id mindid in
  let declare na c =
    machine_constant_verbose (sign,fsign) ((na,false,NeverDischarge),c) in 
  let mispec = mind_specif_of_mind redmind in 
  let elim_scheme = strip_all_casts 
                      (mis_make_indrec sigma [] mispec).(0) in
  let npars = mis_nparams mispec in
  let make_elim s = instanciate_indrec_scheme s npars elim_scheme in
  let kd = mis_kd mispec and kn = mis_kn mispec
  in
    if (List.mem prop kd) or (List.mem prop kn) then
      declare (id_of_string(mindstr^"_ind")) (make_elim prop);
    if (List.mem spec kd) or (List.mem spec kn) then
      declare (id_of_string(mindstr^"_rec")) (make_elim spec);
    if (List.mem types kd) or (List.mem types kn)  then
      declare (id_of_string(mindstr^"_rect")) (make_elim (Type(new_univ())))
;;

let machine_minductive (hyps,fhyps) nparams mispecvec finite = 
  if Array.length mispecvec = 0 then anomaly "machine_minductive" 
  else 
    let mindidvec = 
      Array.map (fun (indid,_,_,_,_) -> indid) mispecvec
    and arlcvec = 
      Array.map
        (fun (_,_,indarity,indconstructors,_) -> (indarity,indconstructors))
        mispecvec
    and namesvec = 
      Array.map
        (fun (indid,indstamp,_,_,consnames) -> (indstamp,indid,consnames))
        mispecvec
    and lnames =  
      it_vect 
        (fun l (indid,_,_,_,consnames) -> (indid::(Array.to_list consnames)@l))
        []  mispecvec
    in (if not (distinct lnames)
        then error "Two inductive objects have the same name");
      let sp = Lib.make_path OBJ mindidvec.(0) in
      let (u',mimap) =
        with_universes 
          (infexecute_minductive (hyps,fhyps) nparams namesvec finite)
          (sp,empty_universes,arlcvec)
      in (declare_minductive (mindidvec.(0),lnames,mimap,u'); 
          if finite then
            let p = Array.length mispecvec in
            let rec decl_elim i =
              if i=p then () 
              else (declare_eliminations mindidvec.(i); decl_elim (i+1))
            in decl_elim 0)
;;


let machine_syntax_constant id cmap =
    declare_syntax_constant id cmap;;

let machine_abstraction id arity rhs =
    declare_abstraction id (execute_abstraction CCI arity rhs);;

(* $Id: declare.ml,v 1.19 1999/11/07 17:57:14 barras Exp $ *)
