(****************************************************************************)
(*                 The Calculus of Inductive Constructions                  *)
(*                                                                          *)
(*                                Projet Coq                                *)
(*                                                                          *)
(*                     INRIA        LRI-CNRS        ENS-CNRS                *)
(*              Rocquencourt         Orsay          Lyon                    *)
(*                                                                          *)
(*                                 Coq V6.3                                 *)
(*                               July 1st 1999                              *)
(*                                                                          *)
(****************************************************************************)
(*                               constants.ml                               *)
(****************************************************************************)

open Std;;
open Names;;
open Generic;;
open Library;;
open Constrtypes;;
open Term;;
open Reduction;;
open Pp;;
open More_util;;

open Dischcore;;
open Environ;;
open Termenv;;
open Initial;;
open Printer;;

let make_constant_body k opacity (hyps,j) =
  let evd = Evd.mt_evd() in
    { cONSTKIND   = k;
      cONSTHYPS   = hyps;
      cONSTBODY   = (Some (ref(COOKED j._VAL)));
      cONSTEVAL   = None;
      cONSTOPAQUE = opacity;
      cONSTTYPE   =
	(match hnf_constr evd j._KIND with
            DOP0 (Sort s) -> {body=j._TYPE; typ=s}
	  | _ -> anomaly "Not a type (make_constant)");
      cONSTIMPARGS =
        if (is_implicit_args())
        then IMPL_AUTO (poly_args j._TYPE)
        else NO_IMPL }
;;

let make_constant opacity ((hyps,j),(fhyps,inf)) =
  let cci = [(CCI,make_constant_body CCI opacity (hyps,j))] and
      fw = (match inf with
              Logic    -> []
            | Inf infj -> [(FW,make_constant_body FW false (fhyps,infj))])
  in cci@fw
;;

let make_cmap (hyps,fhyps) opacity (j,infj) =
  let ids = (auto_save_variables()) in
  let hyps' = thin_hyps ids (hyps,j) in
  let fhyps' = (match infj with
                  Logic -> nil_sign
                | Inf _ -> thin_to_match hyps' fhyps) in
  let fhyps'' = (match infj with
                   Logic    -> nil_sign
                 | Inf jinf -> thin_to_level fhyps' jinf._KIND) 
  in make_constant opacity ((hyps',j),(fhyps'',infj)) 
;;

let expmod_recipe modlist = function
    COOKED c -> COOKED(expmod_constr modlist c)
  | RECIPE dr ->
    let new_expands = map_succeed (function (Const sp,DO_REPLACE) -> sp
                                     | _ -> failwith "caught") modlist and
        new_modifs = map_succeed (function ((_,DO_ABSTRACT _) as x) -> x
                                    | _ -> failwith "caught") modlist
    in RECIPE
       {d_expand = new_expands@ dr.d_expand;
        d_modify = new_modifs@ dr.d_modify;
        d_abstract = dr.d_abstract;
        d_from = dr.d_from}
;;

let expmod_constant modlist cb =
   {cONSTKIND = cb.cONSTKIND;
    cONSTHYPS= map_sign_typ (expmod_type modlist) cb.cONSTHYPS;
    cONSTBODY= option_app (ref_app (expmod_recipe modlist)) cb.cONSTBODY;
    cONSTEVAL = None;
    cONSTOPAQUE = cb.cONSTOPAQUE;
    cONSTTYPE= expmod_type modlist cb.cONSTTYPE;
    cONSTIMPARGS =cb.cONSTIMPARGS}
;;

let expmod_cmap modlist cmap =
    List.map (fun (k,cb) ->
             (k,(expmod_constant modlist cb))) cmap
;;

let abstract_recipe_ref_option ((hyps,((copt,typ) as rpno)),modl) id =
  if isnull_sign hyps or id <> fst(hd_sign hyps) then ((hyps,rpno),modl)
  else
    match copt with
        (Some{contents=COOKED c}) ->
          let (c',typ') = Machops.abs_var id (snd(hd_sign hyps)) (c,typ)
          in ((tl_sign hyps,(Some{contents=COOKED c'},typ')), ABSTRACT::modl)

      | (Some{contents=RECIPE dr}) ->
          let typ' =  Machops.abs_var_type id (snd(hd_sign hyps)) typ in
          let dr' = {d_from = dr.d_from;
                     d_expand = dr.d_expand;
                     d_modify = dr.d_modify;
                     d_abstract = dr.d_abstract@[id]}
          in ((tl_sign hyps,(Some{contents=RECIPE dr'},typ')), ABSTRACT::modl)

      | None ->
          let typ' = Machops.abs_var_type id (snd(hd_sign hyps)) typ
          in ((tl_sign hyps,(None,typ')), ABSTRACT::modl)
;;

let abstract_constant ids_to_abs cb =
  let rp = (cb.cONSTBODY,cb.cONSTTYPE) in
  let ((hyps',rp'),revmodl) =
    List.fold_left abstract_recipe_ref_option
      ((cb.cONSTHYPS,rp),[]) ids_to_abs in
  let modl = List.rev revmodl
  in ({cONSTKIND = cb.cONSTKIND;
       cONSTHYPS = hyps';
       cONSTBODY = fst rp';
       cONSTTYPE = snd rp';
       cONSTOPAQUE = cb.cONSTOPAQUE;
       cONSTEVAL = None;
       cONSTIMPARGS = cb.cONSTIMPARGS },
      modl)
;;

let compute_implicits_const cb =
 {cONSTKIND = cb.cONSTKIND;
      cONSTHYPS = cb.cONSTHYPS;
      cONSTBODY = cb.cONSTBODY;
      cONSTTYPE = cb.cONSTTYPE; 
      cONSTOPAQUE = cb.cONSTOPAQUE;
      cONSTEVAL = cb.cONSTEVAL;
      cONSTIMPARGS = if (is_impl_auto cb.cONSTIMPARGS) 
                     then IMPL_AUTO (poly_args_type cb.cONSTTYPE)
                     else NO_IMPL}
;;

let abstract_cmap (osecsp,nsecsp) ids_to_abs cmap =
let (mods,cmap) =
     List.fold_left 
       (fun (mods,cmap) (k,cb) ->
          match k with
              CCI ->
                let (cb',modl) = abstract_constant ids_to_abs cb in 
		  ((Const(ccisp_of osecsp),
		    DO_ABSTRACT(Const(ccisp_of nsecsp),modl))
		   ::mods,(k,(compute_implicits_const cb'))::cmap)
            | FW ->
                let (cb',modl) = abstract_constant ids_to_abs cb in
                 ((Const(fwsp_of osecsp),
		   DO_ABSTRACT(Const(fwsp_of nsecsp),modl))
		  ::mods,(k, cb')::cmap)
	    | OBJ -> anomaly "abstract_cmap : found a constant of kind OBJ")
    ([],[]) cmap
in (mods,cmap)
;;

let recipe_cmap (sp,cmap) =
    List.map (fun (k,cb) ->
             let dr = {d_from = coerce_path k sp;
                       d_expand = [];
                       d_modify = [];
                       d_abstract = []}
             in (k,{cONSTKIND = cb.cONSTKIND;
                    cONSTHYPS = cb.cONSTHYPS;
                    cONSTBODY = option_app (fun _ -> ref(RECIPE dr))
      	       	       	       	       	   cb.cONSTBODY;
                    cONSTEVAL = None;
                    cONSTOPAQUE = true;
                    cONSTTYPE = cb.cONSTTYPE;
                    cONSTIMPARGS = cb.cONSTIMPARGS})) cmap
;;

let rec execute_recipe dr =
  let _ = cook_constant dr.d_from in
  let from_cb = snd(const_of_path dr.d_from) in
  let expands = dr.d_expand in
  let _ = List.iter cook_constant expands in
  let expmod_cb = expmod_constant
      	      ((List.map (fun sp -> (Const sp,DO_REPLACE)) expands)@dr.d_modify)
      	      from_cb in
  let (abs_cb,_) = abstract_constant dr.d_abstract expmod_cb
  in match abs_cb.cONSTBODY with
       Some{contents=COOKED c} -> c
     | _ -> anomaly "execute_recipe"

and cook_constant sp =
  let cb = (snd (const_of_path sp))
  in match cb.cONSTBODY with
       Some({contents=RECIPE dr} as r) ->
             (mSGNL [< 'sTR"[Cooking " ; 'sTR(string_of_path sp) ; 'sTR"]" >];
              r := COOKED (execute_recipe dr))
     | Some{contents=COOKED _} -> ()
     | None -> anomaly "cook_constant"
;;

let cooked_body cb = match cb.cONSTBODY with 
     Some({contents=RECIPE dr}) -> Some (execute_recipe dr)
  | Some{contents=COOKED c}     -> Some c
  | None                        -> None;;

(* Opaque/Transparent switching *)
let set_transparent_sp sp =
  if cookable_constant sp then
    let (_,cb) = const_of_path sp
    in (cook_constant sp;
        cb.cONSTOPAQUE <- false)
  else
    errorlabstrm "Constants.set_transparent_sp"
      [< 'sTR"Can not set transparent."; 'sPC ;
         'sTR"It is a constant from a required module or a parameter." >]
;;

let set_opaque_sp sp =
  if not (cookable_constant sp) then
    errorlabstrm "Constants.set_opaque_sp"
            [< 'sTR"Can not set opaque." ; 'sPC;
               'sTR"It is a constant from a required module or a parameter." >] ;
  let (_,cb) = const_of_path sp
  in cb.cONSTOPAQUE <- true
;;

let set_unfoldability (transparency,sp) =
  try if transparency then set_transparent_sp sp else set_opaque_sp sp
  with Not_found -> 
    errorlabstrm "Unfoldability (Opaque/Transparent)"
      [< 'sTR (string_of_path sp); 'sPC; 'sTR"does not exist" >]
;;

let load_unfoldability _ = ();;
let cache_unfoldability (_,tr_sp) = set_unfoldability tr_sp;;
let spec_unfoldability fp = fp;;

let (inTransparent,_) =
  Libobject.declare_object ("Unfoldability",
     {Libobject.load_function = load_unfoldability;
      Libobject.cache_function = cache_unfoldability;
      Libobject.specification_function = spec_unfoldability});;

let add_unfoldability tr id =
  let sp =
    try Nametab.sp_of_id CCI id
    with Not_found -> errorlabstrm "add_unfoldability"
	[< print_id id; 'sPC; 'sTR"is not defined." >] in
  add_anonymous_object (inTransparent (tr ,sp));;

let set_transparent = add_unfoldability true;;
let set_opaque      = add_unfoldability false;;


let set_transparent_extraction id = 
  try let sp = Nametab.sp_of_id FW id in set_transparent_sp sp
  with Not_found -> 
    errorlabstrm "Transparent Extraction"
      [< 'sTR "Extraction of"; 'sPC; 
	 'sTR (string_of_id id); 'sPC; 'sTR"does not exist" >];;

let process_recipe_cmap (sp,newsp) (ids_to_discard,work_alist) cmap =
  let raw_cmap = recipe_cmap (sp,cmap) in
  let expmod_raw_cmap = expmod_cmap work_alist raw_cmap
  in abstract_cmap (sp,newsp) ids_to_discard expmod_raw_cmap
;;

let infexecute_constant penv ((name,opacity,strength),c) =
  let sp = Lib.make_path OBJ name in
  let (u,(cj,infcj)) =
    Mach.infexecute_with_univ (Evd.mt_evd()) penv sp c in
  let cmap = make_cmap penv opacity (cj,infcj) in
    (cmap,strength,u)
;;

let infexecute_constant_red penv ((name,opacity,strength),c) red_option =
  let sp = Lib.make_path OBJ name in
  let evmap = Evd.mt_evd() in
  let (u,(cj,infcj)) =
    Mach.infexecute_with_univ evmap penv sp c in
  let (u,(cj,infcj)) = match red_option with 
      None -> (u,(cj,infcj))
    | Some red -> let redfun = reduction_of_redexp red evmap
      in let c=cj._VAL and t = cj._TYPE in 
	Mach.infexecute_with_univ evmap penv sp 
	  (mkCast (redfun c) t)
      in let cmap = make_cmap penv opacity (cj,infcj) in
    (cmap,strength,u)
;;

let make_parameter_body k (hyps,typ) =
    {cONSTKIND = k;
     cONSTHYPS = hyps;
     cONSTBODY = None;
     cONSTEVAL = None;
     cONSTOPAQUE = true;
     cONSTTYPE   = typ;
     cONSTIMPARGS =
       if (is_implicit_args()) 
       then IMPL_AUTO (poly_args_type typ)
       else NO_IMPL }
;;

let make_parameter ((hyps,j),(fhyps,inf)) =
let cci = [(CCI,make_parameter_body CCI (hyps,j))] and
    fw = (match inf with
          Logic -> []
        | Inf infj -> [(FW,make_parameter_body FW (fhyps,outjudge_type infj))])
in cci@fw
;;

let make_paramap (hyps,fhyps) (j,infj) =
let ids = (auto_save_variables()) in
let hyps' = thin_hyps_glob ids (hyps,global_vars j.body) in
let fhyps' = (match infj with
              Logic -> nil_sign
            | Inf _ -> thin_to_match hyps' fhyps) in
let fhyps'' = (match infj with
               Logic -> nil_sign
             | Inf jinf -> thin_to_level fhyps' jinf._TYPE)
in make_parameter ((hyps',j),(fhyps'',infj))
;;

let infexecute_parameter penv name t =
  let sp = Lib.make_path OBJ name in
  let (u,(j,infj)) =
    Mach.infexecute_type_with_univ (Evd.mt_evd()) penv sp t in
  let cmap = make_paramap penv (j,infj) in
    (cmap,NeverDischarge,u)
;;

(* $Id: constants.ml,v 1.19 1999/11/07 05:16:34 barras Exp $ *)
