/* File: int.c
 * Author: Arne John Glenstrup <panic@diku.dk>
 * Description: An interpreter for a small subset of C
 */

#include "cint.h"

#ifndef __CMIX
#  define __CMIX(X)
#endif

#pragma cmix spectime: readProg()
#pragma cmix pure spectime: spectimeerror() spectimeassign()
extern void spectimeassign(Name*, Name);
extern void spectimeerror(const char* errtxt);
extern void readProg(Prog** prog, char* filename);

/* Maximum number of variables ever on the call stack in the */
/* interpreted program                                       */
#define MAXVARS 32

struct {
Name debugglobalnames;
Name globalNames [MAXVARS]; /* interpreted program variable names  */
} ns;

struct {
Value debugglobalvalues;
Value globalValues[MAXVARS]; /* interpreted program variable values */
               /* (globalNames[0] and globalValues[0] are not used) */
} vs;
  
Value assign(const char* name, Value v, Name* names, Value* values) {
  char s[100];
  while (names != &ns.debugglobalnames/*globalNames*/) {
    if (!strcmp(*names, name)) return *values = v;
    names  = names - 1; values = values - 1;
  }
  __CMIX(pure)sprintf(s, "undefined variable `%s'\n", name);
  spectimeerror(s);
  __CMIX(pure)exit(-1);
  return int2val(0);
}

const char* evalLVal(Exp* exp) {
  switch (exp->tag) {
  case Var : return exp->varName;
  default  : spectimeerror 
	       ("non-lvalue expression in lvalue context encountered\n");
  }
  return "<error>";
}


Value evalExp(Exp* exp, Name* names, Value* values, Prog* p) {
  char s[100];int d;
  #pragma cmix residual: evalExp::d
  switch (exp->tag) {
  case Const : return exp->constValue;
  case Var   :
    while (names != &ns.debugglobalnames/*globalNames*/) {
      if (!strcmp(*names, exp->varName))
	return *values;
      names = names - 1; values = values - 1;
    }
    __CMIX(pure)sprintf(s, "undefined variable `%s'\n", exp->varName);
    spectimeerror(s);
    __CMIX(pure) exit(-1);
    return int2val(0);
  case UnOp  :
    switch (exp->operat) {
    case UnMinus : return uminus(evalExp(exp->exp1, names, values, p));
    case UnNeg   : return negate(evalExp(exp->exp1, names, values, p));
    default      : return int2val(0);
    }
  case BinOp :
    switch (exp->operat) {
    case Plus    : return plus    (evalExp(exp->exp1, names, values, p),
				   evalExp(exp->exp2, names, values, p));
    case Minus   : return minus   (evalExp(exp->exp1, names, values, p),
				   evalExp(exp->exp2, names, values, p));
    case Times   : return multiply(evalExp(exp->exp1, names, values, p),
				   evalExp(exp->exp2, names, values, p));
    case Divide  : return divide  (evalExp(exp->exp1, names, values, p),
				   evalExp(exp->exp2, names, values, p));
    case EQ      : return eq      (evalExp(exp->exp1, names, values, p),
				   evalExp(exp->exp2, names, values, p));
    case NE      : return ne      (evalExp(exp->exp1, names, values, p),
				   evalExp(exp->exp2, names, values, p));
    case GT      : return gt      (evalExp(exp->exp1, names, values, p),
				   evalExp(exp->exp2, names, values, p));
    case LT      : return lt      (evalExp(exp->exp1, names, values, p),
				   evalExp(exp->exp2, names, values, p));
    case GE      : return ge      (evalExp(exp->exp1, names, values, p),
				   evalExp(exp->exp2, names, values, p));
    case LE      : return le      (evalExp(exp->exp1, names, values, p),
				   evalExp(exp->exp2, names, values, p));
    case Assign  : return assign  (evalLVal(exp->exp1),
				   evalExp(exp->exp2, names, values, p),
				   names, values);
    default      : return int2val(0);
    }
  }
  return int2val(0);
}
  

Result evalStmts(Stmt* stmt, Name* names, Value* values, Prog* p) {
  /* Evaluate stmt. If this causes a function return,
   * result.returnAction is set to true, and any return value is
   * returned in result.value.
   */
  Result result;
  Decl* decl;
  while (stmt) {
    switch (stmt->tag) {
    case Expression :
      evalExp(stmt->exp, names, values, p);
      stmt = stmt->next;
      break;
    case IfElse     :
      
      /*****************************************************/
      /************* EXERCISE: CODE THIS UP ****************/
      spectimeerror("if-else-statements are not yet implemented.\n");
      __CMIX(pure) exit(-1);
      /*****************************************************/
      
      break;
    case While      :
      if (val2boolean(evalExp(stmt->exp, names, values, p))) {
	result = evalStmts(stmt->whileStmts, names, values, p);
	if (result.returnAction) return result;
      } else
	stmt = stmt->next;
	break;
    case Return     :
      if (stmt->exp) result.value = evalExp(stmt->exp, names, values, p);
	result.returnAction = true;
	return result;
    case Compound   :
      decl = stmt->decls;
      while (decl) { 
	names++; values++;
	spectimeassign(names, decl->name); *values = 0;
	decl = decl->next;
      }
      result = evalStmts(stmt->stmts, names, values, p);
      if (result.returnAction) return result; else stmt = stmt->next;
      break;
    }
  }
  result.returnAction = false;
  return result;
}

void ifTrueThenError(int condition, const char* errorStr, int exitcode) {
  if (condition) {
    __CMIX(pure)error(errorStr + (condition != condition));
    if (exitcode) exit(exitcode + (condition != condition));
  }
}

Value evalFunc(Func* f, Actual* args, Prog* p) {

  Name*   names = &ns.debugglobalnames /* globalNames*/;  /* local environment extends global */
  Value*  values = &vs.debugglobalvalues/* globalValues*/; /* local environment extends global */
  Formal* params = f->params;

  /* Build environment from actual arguments and formal parameters */
  
  while (params) {
    names++; values++;
    *names = params->name;
    ifTrueThenError(args == NULL, "too few arguments in function call\n", -1);
    *values = args->value;
    args   = args->next;
    params = params->next;
  }
  ifTrueThenError(args != NULL, "too many arguments in function call\n", 0);

  /* Run function */
  return evalStmts(f->stmts, names, values, p).value;

}

Value evalProg(char* filename, Actual* args) {
  Prog* prog, dummyprog; Func* f;
  Value temp;
  prog = &dummyprog;
  readProg(&prog, filename); filename = "dummyname";
  f = prog->funcs;
  while (f != NULL && strcmp(f->name, "main") != 0) f = f->next;
  if (strcmp(f->name, "main") == 0) {
    temp = evalFunc(f, args, prog);
    return temp;
  } else
  spectimeerror("there must be a function called `main'.\n");
  return -1;
}
      
