(* camlp4r q_MLast.cmo *)
(***********************************************************************)
(*                                                                     *)
(*                             Camlp4                                  *)
(*                                                                     *)
(*        Daniel de Rauglaudre, projet Cristal, INRIA Rocquencourt     *)
(*                                                                     *)
(*  Copyright 1998 Institut National de Recherche en Informatique et   *)
(*  Automatique.  Distributed only by permission.                      *)
(*                                                                     *)
(***********************************************************************)

(* $Id: pr_r.ml,v 2.4 1999/06/24 12:54:22 ddr Exp $ *)

open Pretty;

value gen_where = ref True;

value pr_fun x =
  let f = Pretty.pr_fun x in
  fun expr patt e k ->
    f (fun e (dg : string) k -> expr e k) (fun p (dg : string) k -> patt p k)
      e "" k
;

value null_f = fun [];

value not_impl name x =
  let desc =
    if Obj.is_block (Obj.repr x) then
      "tag = " ^ string_of_int (Obj.tag (Obj.repr x))
    else "int_val = " ^ string_of_int (Obj.magic x)
  in
  HVbox [: `S NO ("<pr_r: not impl: " ^ name ^ "; " ^ desc ^ ">") :]
;

external is_printable : char -> bool = "is_printable";

value char_escaped =
  fun
  [ '\\' -> "\\\\"
  | '\b' -> "\\b"
  | '\n' -> "\\n"
  | '\r' -> "\\r"
  | '\t' -> "\\t"
  | c ->
      if is_printable c then String.make 1 c
      else
        let n = Char.code c in
        let s = String.create 4 in
        do String.unsafe_set s 0 '\\';
           String.unsafe_set s 1 (Char.unsafe_chr (48 + n / 100));
           String.unsafe_set s 2 (Char.unsafe_chr (48 + n / 10 mod 10));
           String.unsafe_set s 3 (Char.unsafe_chr (48 + n mod 10));
        return s ]
;

value apply_it l f =
  apply_it_f l where rec apply_it_f =
    fun
    [ [] -> f
    | [a :: l] -> a (apply_it_f l) ]
;

value rec list elem el k =
  match el with
  [ [] -> k
  | [x] -> [: `elem x k :]
  | [x :: l] -> [: `elem x [: :]; list elem l k :] ]
;

value rec listws elem sep el k =
  match el with
  [ [] -> k
  | [x] -> [: `elem x k :]
  | [x :: l] -> [: `elem x [: `sep :]; listws elem sep l k :] ]
;

value rec listwbws elem b sep el k =
  match el with
  [ [] -> [: b; k :]
  | [x] -> [: `elem b x k :]
  | [x :: l] -> [: `elem b x [: :]; listwbws elem [: `sep :] sep l k :] ]
;

value level box elem next e k =
  let rec curr e k = elem curr next e k in box (curr e k)
;

value is_infix =
  let infixes = Hashtbl.create 73 in
  do List.iter (fun s -> Hashtbl.add infixes s True)
       ["=="; "!="; "+"; "+."; "-"; "-."; "*"; "*."; "/"; "/."; "**"; "**.";
        "="; "=."; "<>"; "<>."; "<"; "<."; ">"; ">."; "<="; "<=."; ">=";
        ">=."; "^"; "@"; "asr"; "land"; "lor"; "lsl"; "lsr"; "lxor"; "mod";
        "or"; "quo"; "&&"; "||"; "~-"; "~-."];
  return fun s -> try Hashtbl.find infixes s with [ Not_found -> False ]
;

value is_keyword =
  let keywords = Hashtbl.create 301 in
  do List.iter (fun s -> Hashtbl.add keywords s True)
       ["<>"; "<="; "struct"; "asr"; ":]"; "[:"; ":="; "type"; "::"; "return";
        "for"; "to"; "and"; "rec"; "||"; "of"; "with"; "while"; "module";
        "when"; "exception"; "lsr"; "lsl"; "done"; "/."; ".."; "->"; "in";
        "-."; "if"; "value"; "lor"; "external"; "sig"; "+."; "then"; "where";
        "*."; "**"; "match"; "parser"; "try"; "do"; "land"; "else"; "as";
        "open"; "}"; "|"; "end"; "{"; "lxor"; "`"; "_"; "^"; "]";
        "["; "let"; "!="; "@"; "?"; ">"; "="; "<"; ";"; ":"; "mutable"; "/";
        "[|"; "."; "-"; ","; "+"; "downto"; "*"; ")"; "|]"; "("; "'"; "&&";
        "functor"; ">="; "#"; "~-."; "~-"; "fun"; "mod"; "=="; "declare"];
  return fun s -> try Hashtbl.find keywords s with [ Not_found -> False ]
;

value has_special_chars v =
  match v.[0] with
  [ 'a'..'z' | 'A'..'Z' | '_' -> False
  | _ ->
      if String.length v >= 2 && v.[0] == '<' &&
         (v.[1] == '<' || v.[1] == ':') then
        False
      else True ]
;

value var_escaped v =
  if has_special_chars v || is_keyword v then "\\" ^ v
  else v
;

value flag n f = if f then [: `S LR n :] else [: :];

(* default global loc *)

value loc = (0, 0);

(* type core *)

value ctyp_f = ref null_f;
value ctyp t k = ctyp_f.val t k;

value rec labels b vl k = [: b; listws label (S RO ";") vl k :]
and label (f, m, t) k =
  let m = flag "mutable" m in
  HVbox [: `HVbox [: `S LR f; `S LR ":" :]; `HVbox [: m; `ctyp t k :] :]
;

value rec ctyp_list tel k = listws ctyp (S LR "and") tel k;

value rec variants b vl k = listwbws variant b (S LR "|") k vl
and variant b (c, tl) k =
  match tl with
  [ [] -> HVbox [: b; `HOVbox [: `S LR c; k :] :]
  | _ -> HVbox [: b; `HOVbox [: `S LR c; `S LR "of"; ctyp_list tl k :] :] ]
;

(* *)

value rec class_longident sl k =
  match sl with
  [ [i] -> HVbox [: `S LR i; k :]
  | [m :: sl] -> HVbox [: `S LR m; `S NO "."; `class_longident sl k :]
  | _ -> HVbox [: `not_impl "class_longident" sl; k :] ]
;

value rec clty_longident sl k =
  match sl with
  [ [i] -> HVbox [: `S LR i; k :]
  | [m :: sl] -> HVbox [: `S LR m; `S NO "."; `clty_longident sl k :]
  | _ -> HVbox [: `not_impl "clty_longident" sl; k :] ]
;

value rec meth_list (ml, v) k =
  match (ml, v) with
  [ ([f], False) -> [: `field f k :]
  | ([], _) -> [: `S LR ".."; k :]
  | ([f :: ml], v) -> [: `field f [: `S RO ";" :]; meth_list (ml, v) k :] ]
and field (lab, t) k =
  HVbox [: `S LR lab; `S LR ":"; `ctyp t k :]
;

ctyp_f.val :=
  apply_it
    [level (fun x -> HOVbox x)
       (fun curr next t k ->
          match t with
          [ <:ctyp< $t1$ == $t2$ >> ->
              [: curr t1 [: `S LR "==" :]; `next t2 k :]
          | t -> [: `next t k :] ]);
     level (fun x -> HOVbox x)
       (fun curr next t k ->
          match t with
          [ <:ctyp< $x$ as $y$ >> -> [: curr x [: `S LR "as" :]; `next y k :]
          | t -> [: `next t k :] ]);
     level (fun x -> HOVbox x)
       (fun curr next t k ->
          match t with
          [ <:ctyp< $x$ -> $y$ >> -> [: `next x [: `S LR "->" :]; curr y k :]
          | t -> [: `next t k :] ]);
     level (fun x -> HOVbox x)
       (fun curr next t k ->
          match t with
          [ <:ctyp< $t1$ $t2$ >> -> [: curr t1 [: :]; `next t2 k :]
          | MLast.TyXnd _ c t -> [: `S LR ("Xnd_" ^ c); `next t k :]
          | t -> [: `next t k :] ]);
     level (fun x -> HOVbox x)
       (fun curr next t k ->
          match t with
          [ <:ctyp< $t1$ . $t2$ >> ->
              [: curr t1 [: :]; `S NO "."; `next t2 k :]
          | t -> [: `next t k :] ]);
     level (fun x -> HOVbox x)
       (fun curr next t k ->
          match t with
          [ <:ctyp< ($list:tl$) >> ->
              [: `S LO "("; listws ctyp (S LR "*") tl [: `S RO ")"; k :] :]
          | <:ctyp< '$s$ >> -> [: `S LO "'"; `S LR (var_escaped s); k :]
          | <:ctyp< $lid:s$ >> -> [: `S LR s; k :]
          | <:ctyp< $uid:s$ >> -> [: `S LR s; k :]
          | <:ctyp< _ >> -> [: `S LR "_"; k :]
          | <:ctyp< { $list: ftl$ } >> ->
              [: `HVbox [: labels [: `S LR "{" :] ftl [: `S LR "}" :]; k :] :]
          | <:ctyp< [ $list:ctl$ ] >> ->
              [: `HVbox
                    [: `HVbox [: :];
                       variants [: `S LR "[" :] [: `S LR "]" :] ctl; k :] :]
          | <:ctyp< $_$ -> $_$ >> | <:ctyp< $_$ $_$ >> |
            <:ctyp< $_$ == $_$ >> | <:ctyp< $_$ . $_$ >> |
            <:ctyp< $_$ as $_$ >> | MLast.TyXnd _ _ _ ->
              [: `S LO "("; `ctyp t [: `HVbox [: `S RO ")"; k :] :] :]
          | MLast.TyCls _ id ->
              [: `S LO "#"; `class_longident id k :]
          | MLast.TyObj _ [] False -> [: `S LR "<>"; k :]
          | MLast.TyObj _ ml v ->
              [: `S LO "<"; meth_list (ml, v) [: `S RO ">"; k :] :] ])]
    (fun t k -> not_impl "ctyp" t);

(* patterns *)

value patt_f = ref null_f;
value patt p k = patt_f.val p k;

patt_f.val :=
  apply_it
    [level (fun x -> HOVbox [: `HVbox [: :]; x :])
       (fun curr next p k ->
          match p with
          [ <:patt< $x$ | $y$ >> -> [: curr x [: `S LR "|" :]; `next y k :]
          | p -> [: `next p k :] ]);
     level (fun x -> HOVbox [: `HVbox [: :]; x :])
       (fun curr next p k ->
          match p with
          [ <:patt< $x$ .. $y$ >> -> [: curr x [: `S NO ".." :]; `next y k :]
          | p -> [: `next p k :] ]);
     level (fun x -> HOVbox x)
       (fun curr next p k ->
          match p with
          [ <:patt< [$_$ :: $_$] >> -> [: `next p k :]
          | <:patt< $x$ $y$ >> -> [: curr x [: :]; `next y k :]
          | MLast.PaXnd _ c p -> [: `S LR ("Xnd_" ^ c); `next p k :]
          | p -> [: `next p k :] ]);
     level (fun x -> HOVbox x)
       (fun curr next p k ->
          match p with
          [ <:patt< $x$ . $y$ >> -> [: curr x [: `S NO "." :]; `next y k :]
          | p -> [: `next p k :] ]);
     level (fun x -> HOVbox x)
       (fun curr next p k ->
          match p with
          [ <:patt< [$_$ :: $_$] >> ->
              let (pl, c) =
                make_list p where rec make_list p =
                  match p with
                  [ <:patt< [$p$ :: $y$] >> ->
                      let (pl, c) = make_list y in ([p :: pl], c)
                  | <:patt< [] >> -> ([], None)
                  | x -> ([], Some p) ]
              in
              [: `HOVCbox
                    [: `S LO "[";
                       let rec glop pl k =
                         match pl with
                         [ [] -> failwith "simple_patt"
                         | [p] ->
                             match c with
                             [ None -> [: `patt p k :]
                             | Some x ->
                                 [: `patt p [: `S LR "::" :]; `patt x k :] ]
                         | [p :: pl] ->
                             [: `patt p [: `S RO ";" :]; glop pl k :] ]
                       in
                       glop pl [: `S RO "]"; k :] :] :]
          | <:patt< [| $list:pl$ |] >> ->
              [: `S LR "[|"; listws patt (S RO ";") pl [: `S LR "|]"; k :] :]
          | <:patt< { $list:fpl$ } >> ->
              [: `HVbox
                    [: `S LO "{";
                       listws
                         (fun (lab, p) k ->
                            HVbox
                              [: `patt lab [: `S LR "=" :]; `patt p k :])
                         (S RO ";") fpl [: `S RO "}"; k :] :] :]
          | <:patt< ($list:[p::pl]$) >> ->
              [: `HOVCbox
                    [: `S LO "(";
                       listws patt (S RO ",") [p :: pl]
                         [: `S RO ")"; k :] :] :]
          | <:patt< ($p$ : $ct$) >> ->
              [: `S LO "("; `patt p [: `S LR ":" :];
                 `ctyp ct [: `S RO ")"; k :] :]
          | <:patt< ($x$ as $y$) >> ->
              [: `S LO "("; `patt x [: `S LR "as" :];
                 `patt y [: `S RO ")"; k :] :]
          | <:patt< $int:s$ >> -> [: `S LR s; k :]
          | <:patt< $str:s$ >> ->
              [: `S LR ("\"" ^ String.escaped s ^ "\""); k :]
          | <:patt< $chr:c$ >> -> [: `S LR ("'" ^ char_escaped c ^ "'"); k :]
          | <:patt< $lid:s$ >> -> [: `S LR (var_escaped s); k :]
          | <:patt< $uid:s$ >> -> [: `S LR s; k :]
          | <:patt< _ >> -> [: `S LR "_"; k :]
          | <:patt< $_$ $_$ >> | <:patt< $_$ .. $_$ >>
          | <:patt< $_$ | $_$ >> | MLast.PaXnd _ _ _ ->
              [: `S LO "("; `patt p [: `HVbox [: `S RO ")"; k :] :] :]
          | p -> [: `next p k :] ])]
    (fun x k -> not_impl "patt" x);

value rec is_irrefut_patt =
  fun
  [ <:patt< $lid:_$ >> -> True
  | <:patt< () >> -> True
  | <:patt< _ >> -> True
  | <:patt< ($x$ as $_$) >> -> is_irrefut_patt x
  | <:patt< { $list:fpl$ } >> ->
      List.for_all (fun (_, p) -> is_irrefut_patt p) fpl
  | <:patt< ($p$ : $_$) >> -> is_irrefut_patt p
  | <:patt< ($list:pl$) >> -> List.for_all is_irrefut_patt pl
  | _ -> False ]
;

value rec get_defined_ident =
  fun
  [ <:patt< $_$ . $_$ >> -> []
  | <:patt< _ >> -> []
  | <:patt< $lid:x$ >> -> [x]
  | <:patt< ($p1$ as $p2$) >> -> get_defined_ident p1 @ get_defined_ident p2
  | <:patt< $int:_$ >> -> []
  | <:patt< $str:_$ >> -> []
  | <:patt< $chr:_$ >> -> []
  | <:patt< [| $list:pl$ |] >> -> List.flatten (List.map get_defined_ident pl)
  | <:patt< ($list:pl$) >> -> List.flatten (List.map get_defined_ident pl)
  | <:patt< $uid:_$ >> -> []
  | <:patt< $p1$ $p2$ >> -> get_defined_ident p1 @ get_defined_ident p2
  | <:patt< { $list:lpl$ } >> ->
      List.flatten (List.map (fun (lab, p) -> get_defined_ident p) lpl)
  | <:patt< $p1$ | $p2$ >> -> get_defined_ident p1 @ get_defined_ident p2
  | <:patt< $p1$ .. $p2$ >> -> get_defined_ident p1 @ get_defined_ident p2
  | <:patt< ($p$ : $_$) >> -> get_defined_ident p
  | MLast.PaAnt _ p -> get_defined_ident p
  | MLast.PaXnd _ _ p -> get_defined_ident p ]
;

value un_irrefut_patt p =
  match get_defined_ident p with
  [ [] -> (<:patt< _ >>, <:expr< () >>)
  | [i] -> (<:patt< $lid:i$ >>, <:expr< $lid:i$ >>)
  | il ->
      let (upl, uel) =
        List.fold_right
          (fun i (upl, uel) ->
             ([<:patt< $lid:i$ >> :: upl],
              [<:expr< $lid:i$ >> :: uel]))
          il ([], [])
      in
      (<:patt< ($list:upl$) >>, <:expr< ($list:uel$) >>) ]
;            

(* expressions *)

value rec expr_fun_args ge =
  match ge with
  [ <:expr< fun strm__ -> $_$ >> when Pretty.has_pr_fun "parser" -> ([], ge)
  | <:expr< fun [(strm__ : $_$) -> $_$] >> when Pretty.has_pr_fun "parser" ->
      ([], ge)
  | <:expr< fun [$p$ -> $e$] >> ->
      if is_irrefut_patt p then
        let (pl, e) = expr_fun_args e in ([p :: pl], e)
      else ([], ge)
  | _ -> ([], ge) ]
;

value expr_f = ref null_f;
value expr e k = expr_f.val e k;

value rec bind_list b pel k =
  match pel with
  [ [pe] -> let_binding b pe k
  | pel -> Vbox [: `HVbox [: :]; listwbws let_binding b (S LR "and") pel k :] ]
and let_binding b (p, e) k =
  let (p, e) =
    if is_irrefut_patt p then (p, e)
    else
      let (up, ue) = un_irrefut_patt p in
      (up, <:expr< match $e$ with [ $p$ -> $ue$ ] >>)
  in
  let (pl, e) = expr_fun_args e in
  match e with
  [ <:expr< let $rec:r$ $lid:f$ = fun [ $list:pel$ ] in $e$ >>
    when
      let rec call_f =
        fun
        [ <:expr< $lid:f'$ >> -> f = f'
        | <:expr< $e$ $_$ >> -> call_f e
        | _ -> False ]
      in
      gen_where.val && call_f e ->
      let (pl1, e1) = expr_fun_args <:expr< fun [ $list:pel$ ] >> in
      BEbox
        [: `HVbox
              [: `HVbox b; `HVbox (list patt [p :: pl] [: `S LR "=" :]) :];
           `HVbox
              [: `HOVbox
                    [: `expr e [: :]; `S LR "where"; flag "rec" r; `S LR f;
                       `HVbox (list patt pl1 [: `S LR "=" :]) :];
                 `expr e1 [: :] :];
           k :]
  | <:expr< ($e$ : $t$) >> ->
      BEbox
        [: `HVbox
              [: `HVbox b; `HVbox (list patt [p :: pl] [: `S LR ":" :]);
                 `ctyp t [: `S LR "=" :] :];
           `expr e [: :]; k :]
  | _ ->
      BEbox
        [: `HVbox
              [: `HVbox b; `HVbox (list patt [p :: pl] [: `S LR "=" :]) :];
           `expr e [: :]; k :] ]
and match_assoc_list pwel k =
  match pwel with
  [ [pwe] -> match_assoc [: `S LR "[" :] pwe [: `S LR "]"; k :]
  | pel ->
      Vbox
        [: `HVbox [: :];
           listwbws match_assoc [: `S LR "[" :] (S LR "|")
             pel [: `S LR "]"; k :] :] ]
and match_assoc b (p, w, e) k =
  let s =
    let (p, k) =
      match p with
      [ <:patt< ($p$ as $p2$) >> -> (p, [: `S LR "as"; `patt p2 [: :] :])
      | _ -> (p, [: :]) ]
    in
    match w with
    [ Some e1 ->
        [: `HVbox
              [: `HVbox [: :]; `patt p k;
                 `HVbox [: `S LR "when"; `expr e1 [: `S LR "->" :] :] :] :]
    | _ -> [: `patt p [: k; `S LR "->" :] :] ]
  in
  HVbox [: b; `HVbox [: `HVbox s; `expr e k :] :]
;

value label lab =
  S LR lab
;

value field_expr (lab, e) k =
  HVbox [: `label lab; `S LR "="; `expr e k :]
;

value fwd_module_binding = ref (fun []);

expr_f.val :=
  apply_it
    [level (fun x -> HOVbox x)
       (fun curr next e k ->
          match e with
          [ <:expr< let strm__ = $_$ in $_$ >>
            when Pretty.has_pr_fun "parser_m" ->
              let f = pr_fun "parser_m" in [: `f expr patt e k :]
          | <:expr< let $rec:r$ $p1$ = $e1$ in $e$ >> ->
              let r = flag "rec" r in
              [: `HVbox [: :];
                 `let_binding [: `S LR "let"; r :] (p1, e1) [: `S LR "in" :];
                 `expr e k :]
          | <:expr< let $rec:r$ $list:pel$ in $e$ >> ->
              let r = flag "rec" r in
              [: `Vbox
                    [: `HVbox [: :];
                       listwbws (fun b (p, e) k -> let_binding b (p, e) k)
                         [: `S LR "let"; r :] (S LR "and") pel
                         [: `S LR "in" :];
                       `expr e k :] :]
          | <:expr< let module $m$ = $mb$ in $e$ >> ->
              [: `HVbox
                    [: `HVbox [: :];
                       `fwd_module_binding.val
                          [: `S LR "let"; `S LR "module"; `S LR m :] mb [: :];
                       `S LR "in";
                       `expr e k :] :]
          | <:expr< fun strm__ -> $x$ >> when has_pr_fun "parser" ->
              let f = pr_fun "parser" in [: `f expr patt x k :]
          | <:expr< fun [ (strm__ : $_$) -> $x$ ] >>
            when has_pr_fun "parser" ->
              let f = pr_fun "parser" in [: `f expr patt x k :]
          | <:expr< fun [ $list:pel$ ] >> ->
              match pel with
              [ [] -> [: `S LR "fun"; `S LR "[]"; k :]
              | [(p, None, e)] ->
                  if is_irrefut_patt p then
                    let (pl, e) = expr_fun_args e in
                    [: `BEbox
                          [: `HOVbox
                                [: `S LR "fun";
                                   list patt [p :: pl] [: `S LR "->" :] :];
                             `expr e k :] :]
                  else
                    [: `HVbox [: `S LR "fun ["; `patt p [: `S LR "->" :] :];
                       `expr e [: `S LR "]"; k :] :]
              | _ ->
                  [: `Vbox
                        [: `HVbox [: :]; `S LR "fun";
                           listwbws match_assoc
                             [: `S LR "[" :] (S LR "|") pel
                             [: `S LR "]"; k :] :] :] ]
          | <:expr< match $e$ with $p1$ -> $e1$ >> when is_irrefut_patt p1 ->
              [: `BEbox
                    [: `S LR "match"; `expr e [: :];
                       `HVbox
                          [: `S LR "with"; `patt p1 [: `S LR "->" :] :] :];
                 `expr e1 k :]
          | <:expr< match $e$ with [ ] >> ->
              [: `HVbox [: :];
                 `BEbox
                    [: `S LR "match"; `expr e [: :]; `S LR "with"; `S LR "[]";
                       k :] :]
          | <:expr< match $e$ with [ $list:pel$ ] >> ->
              [: `HVbox [: :];
                 `BEbox [: `S LR "match"; `expr e [: :]; `S LR "with" :];
                 `match_assoc_list pel k :]
          | <:expr< try $e$ with [ ] >> ->
              [: `HVbox [: :];
                 `BEbox
                    [: `S LR "try"; `expr e [: :]; `S LR "with"; `S LR "[]";
                       k :] :]
          | <:expr< try $e$ with $p1$ -> $e1$ >> when is_irrefut_patt p1 ->
              [: `BEbox
                    [: `S LR "try"; `expr e [: :];
                       `HVbox
                          [: `S LR "with"; `patt p1 [: `S LR "->" :] :] :];
                 `expr e1 k :]
          | <:expr< try $e$ with [ $list:pel$ ] >> ->
              [: `HVbox [: :];
                 `BEbox [: `S LR "try"; `expr e [: :]; `S LR "with" :];
                 `match_assoc_list pel k :]
          | <:expr< if $_$ then ()
                    else raise (Pervasives.Assert_failure $_$) >> ->
              [: `next e k :]
          | <:expr< if $e1$ then $e2$ else $e3$ >> ->
              let (eel, e) =
                elseif e3 where rec elseif e =
                  match e with
                  [ <:expr< if $e1$ then $e2$ else $e3$ >> ->
                      let (eel, e) = elseif e3 in ([(e1, e2) :: eel], e)
                  | _ -> ([], e) ]
              in
              [: `HVbox
                    [: `HVbox [: :];
                       `HVbox
                          [: `HOVbox
                                [: `S LR "if"; `expr e1 [: `S LR "then" :] :];
                             `expr e2 [: :] :];
                       list
                         (fun (e1, e2) k ->
                            HVbox
                              [: `HOVbox
                                    [: `S LR "else"; `S LR "if";
                                       `expr e1 [: `S LR "then" :] :];
                                 `expr e2 [: :] :])
                         eel [: :];
                       `HVbox [: `S LR "else"; `expr e k :] :] :]
          | <:expr< do $list:el$ return $e$ >> ->
              [: `HOVCbox
                    [: `HVbox [: :];
                       `BEbox
                          [: `S LR "do";
                             `HVbox
                                [: `HVbox [: :];
                                   list (fun e k -> expr e [: `S RO ";"; k :])
                                     el [: :] :];
                             `S LR "return" :];
                       `expr e k :] :]
          | <:expr< for $i$ = $e1$ $to:d$ $e2$ do $list:el$ done >> ->
              let d = if d then "to" else "downto" in
              [: `BEbox
                    [: `HOVbox
                          [: `S LR "for"; `S LR i; `S LR "=";
                             `expr e1 [: `S LR d :];
                             `expr e2 [: `S LR "do" :] :];
                       `HVbox
                          [: `HVbox [: :];
                             list (fun e k -> expr e [: `S RO ";"; k :]) el
                               [: :] :];
                       `HVbox [: `S LR "done"; k :] :] :]
          | <:expr< while $e1$ do $list:el$ done >> ->
              [: `BEbox
                    [: `BEbox [: `S LR "while"; `expr e1 [: :]; `S LR "do" :];
                       `HVbox
                          [: `HVbox [: :];
                             list (fun e k -> expr e [: `S RO ";"; k :]) el
                               [: :] :];
                       `HVbox [: `S LR "done"; k :] :] :]
          | e -> [: `next e k :] ]);
     level (fun x -> HOVbox x)
       (fun curr next e k ->
          match e with
          [ <:expr< $x$ := $y$ >> -> [: `next x [: `S LR ":=" :]; `expr y k :]
          | e -> [: `next e k :] ]);
     level (fun x -> HOVbox [: `HVbox [: :]; x :])
       (fun curr next e k ->
          match e with
          [ <:expr< $lid:"||" | "or"$ $x$ $y$ >> ->
              [: `next x [: `S LR "||" :]; curr y k :]
          | e -> [: `next e k :] ]);
     level (fun x -> HOVbox [: `HVbox [: :]; x :])
       (fun curr next e k ->
          match e with
          [ <:expr< $lid:"&&" | "&"$ $x$ $y$ >> ->
              [: `next x [: `S LR "&&" :]; curr y k :]
          | e -> [: `next e k :] ]);
     level (fun x -> HOVbox x)
       (fun curr next e k ->
          match e with
          [ <:expr< $lid:op$ $x$ $y$ >> ->
              match op with
              [ "<" | ">" | "<=" | ">=" | "=" | "<>" | "==" | "!=" ->
                  [: curr x [: `S LR op :]; `next y k :]
              | _ -> [: `next e k :] ]
          | e -> [: `next e k :] ]);
     level (fun x -> HOVbox x)
       (fun curr next e k ->
          match e with
          [ <:expr< $lid:op$ $x$ $y$ >> ->
              match op with
              [ "^" | "@" -> [: `next x [: `S LR op :]; curr y k :]
              | _ -> [: `next e k :] ]
          | e -> [: `next e k :] ]);
     level (fun x -> HOVbox x)
       (fun curr next e k ->
          match e with
          [ <:expr< $lid:op$ $x$ $y$ >> ->
              match op with
              [ "+" | "+." | "-" | "-." ->
                  [: curr x [: `S LR op :]; `next y k :]
              | _ -> [: `next e k :] ]
          | e -> [: `next e k :] ]);
     level (fun x -> HOVbox x)
       (fun curr next e k ->
          match e with
          [ <:expr< $lid:op$ $x$ $y$ >> ->
              match op with
              [ "*" | "/" | "*." | "/." | "land" | "lor" | "lxor" | "mod" ->
                  [: curr x [: `S LR op :]; `next y k :]
              | _ -> [: `next e k :] ]
          | e -> [: `next e k :] ]);
     level (fun x -> HOVbox x)
       (fun curr next e k ->
          match e with
          [ <:expr< $lid:op$ $x$ $y$ >> ->
              match op with
              [ "**" | "asr" | "lsl" | "lsr" ->
                  [: `next x [: `S LR op :]; curr y k :]
              | _ -> [: `next e k :] ]
          | e -> [: `next e k :] ]);
     level (fun x -> HOVbox x)
       (fun curr next e k ->
          match e with
          [ <:expr< $lid:"~-"$ $x$ >> -> [: `S LR "-"; curr x k :]
          | <:expr< $lid:"~-."$ $x$ >> -> [: `S LR "-."; curr x k :]
          | e -> [: `next e k :] ]);
     level (fun x -> HOVbox x)
       (fun curr next e k ->
          match e with
          [ <:expr< $int:x$ >> -> [: `S LR x; k :]
          | _ -> [: `next e k :] ]);
     level (fun x -> HOVbox x)
       (fun curr next e k ->
          match e with
          [ <:expr< [$_$ :: $_$] >> -> [: `next e k :]
          | <:expr< Stream.iapp $_$ $_$ >> |
            <:expr< Stream.icons $_$ $_$ >> |
            <:expr< Stream.ising $_$ >> |
            <:expr< Stream.lapp (fun _ -> $_$) $_$ >> |
            <:expr< Stream.lcons (fun _ -> $_$) $_$ >> |
            <:expr< Stream.lsing (fun _ -> $_$) >> |
            <:expr< Stream.sempty >> |
            <:expr< Stream.slazy $_$ >>
            when has_pr_fun "stream" ->
              [: `next e k :]
          | <:expr< Grammar.extend $_$ >> when has_pr_fun "extend" ->
              [: `next e k :]
          | <:expr< Pervasives.ref (Lazy.Delayed (fun () -> $x$)) >> ->
              [: `S LR "lazy"; `next x k :]
          | <:expr< if $e$ then ()
                    else raise (Pervasives.Assert_failure $_$) >> ->
               [: `S LR "assert"; `next e k :]
          | <:expr< raise (Pervasives.Assert_failure $_$) >> ->
               [: `S LR "assert"; `S LR "False"; k :]
         | <:expr< $lid:n$ $x$ $y$ >> ->
              if is_infix n then [: `next e k :]
              else [: curr <:expr< $lid:n$ $x$ >> [: :]; `next y k :]
          | <:expr< $x$ $y$ >> -> [: curr x [: :]; `next y k :]
          | MLast.ExNew _ sl -> [: `S LR "new"; `class_longident sl k :]
          | MLast.ExXnd _ c e -> [: `S LR ("Xnd_" ^ c); `next e k :]
          | _ -> [: `next e k :] ]);
     level (fun x -> HOVbox x)
       (fun curr next e k ->
          match e with
          [ <:expr< $x$ . ( $y$ ) >> ->
              [: curr x [: :]; `S NO ".("; `expr y [: `S RO ")"; k :] :]
          | <:expr< $x$ . [ $y$ ] >> ->
              [: curr x [: :]; `S NO ".["; `expr y [: `S RO "]"; k :] :]
          | <:expr< Stream.sempty >> when has_pr_fun "stream" ->
              [: `next e k :]
          | <:expr< $e1$ . $e2$ >> ->
              [: curr e1 [: :]; `S NO "."; curr e2 k :]
          | MLast.ExSnd _ e lab ->
              [: curr e [: :]; `S NO "#"; `label lab; k :]
          | e -> [: `next e k :] ]);
     level (fun x -> HOVbox x)
       (fun curr next e k ->
          match e with
          [ (<:expr< Stream.iapp $_$ $_$ >> |
             <:expr< Stream.icons $_$ $_$ >> |
             <:expr< Stream.ising $_$ >> |
             <:expr< Stream.lapp (fun _ -> $_$) $_$ >> |
             <:expr< Stream.lcons (fun _ -> $_$) $_$ >> |
             <:expr< Stream.lsing (fun _ -> $_$) >> |
             <:expr< Stream.sempty >> |
             <:expr< Stream.slazy $_$ >> as
             e) when
              has_pr_fun "stream" ->
              let f = pr_fun "stream" in [: `f expr patt e k :]
          | (<:expr< Grammar.extend $_$ >> as e) when has_pr_fun "extend" ->
              let f = pr_fun "extend" in [: `f expr patt e k :]
          | <:expr< $int:x$ >> ->
              if x.[0] = '-' then [: `S LO "("; `S LR x; `S RO ")"; k :]
              else [: `S LR x; k :]
          | <:expr< $flo:s$ >> -> [: `S LR s; k :]
          | <:expr< $str:s$ >> ->
              [: `S LR ("\"" ^ String.escaped s ^ "\""); k :]
          | <:expr< $chr:c$ >> -> [: `S LR ("'" ^ char_escaped c ^ "'"); k :]
          | <:expr< $uid:s$ >> -> [: `S LR s; k :]
          | <:expr< $lid:s$ >> -> [: `S LR (var_escaped s); k :]
          | (<:expr< [$_$ :: $_$] >> as e) ->
              let (el, c) =
                make_list e where rec make_list e =
                  match e with
                  [ <:expr< [$e$ :: $y$] >> ->
                      let (el, c) = make_list y in ([e :: el], c)
                  | <:expr< [] >> -> ([], None)
                  | x -> ([], Some e) ]
              in
              match c with
              [ None ->
                  [: `S LO "[";
                     listws expr (S RO ";") el [: `S RO "]"; k :] :]
              | Some x ->
                  [: `S LO "["; listws expr (S RO ";") el [: `S LR "::" :];
                     `expr x [: `S RO "]"; k :] :] ]
          | <:expr< [| $list:el$ |] >> ->
              [: `S LR "[|"; listws expr (S RO ";") el [: `S LR "|]"; k :] :]
          | <:expr< { $list:fel$ } >> ->
              [: `S LO "{";
                 listws
                   (fun (lab, e) k ->
                      HVbox [: `expr lab [: `S LR "=" :]; `expr e k :])
                   (S RO ";") fel [: `S RO "}"; k :] :]
          | <:expr< { ($e$) with $list:fel$ } >> ->
              [: `HVbox
                    [: `S LO "{"; `S LO "(";
                       `expr e [: `S RO ")"; `S LR "with" :] :];
                 listws
                   (fun (lab, e) k ->
                      HVbox [: `expr lab [: `S LR "=" :]; `expr e k :])
                   (S RO ";") fel [: `S RO "}"; k :] :]
          | <:expr< ($e$ : $t$) >> ->
              [: `S LO "("; `expr e [: `S LR ":" :];
                 `ctyp t [: `S RO ")"; k :] :]
          | MLast.ExCoe _ e t ->
              [: `S LO "("; `expr e [: `S LR ":>" :];
                 `ctyp t [: `S RO ")"; k :] :]
          | MLast.ExOvr _ [] -> [: `S LR "{< >}"; k :]
          | MLast.ExOvr _ fel ->
              [: `S LR "{<";
                 listws field_expr (S RO ";") fel [: `S LR ">}"; k :] :]
          | <:expr< ($list:el$) >> ->
              [: `S LO "("; listws expr (S RO ",") el [: `S RO ")"; k :] :]
          | (<:expr< $_$ $_$ >> | <:expr< $uid:_$ $_$ $_$ >> |
             <:expr< fun [ $list:_$ ] >> |
             <:expr< match $_$ with [ $list:_$ ] >> |
             <:expr< if $_$ then $_$ else $_$ >> |
             <:expr< do $list:_$ return $_$ >> |
             <:expr< try $_$ with [ $list:_$ ] >> |
             <:expr< for $_$ = $_$ $to:_$ $_$ do $list:_$ done >> |
             <:expr< let $rec:_$ $list:_$ in $_$ >> |
             MLast.ExNew _ _ | MLast.ExXnd _ _ _ as
             e) ->
              [: `S LO "("; `expr e [: `HVbox [: `S RO ")"; k :] :] :]
          | e -> [: `next e k :] ])]
    (fun e k -> not_impl "expr" e);

value rec type_params sl k =
  list (fun s k -> HVbox [: `S LO "'"; `S LR s; k :]) sl k
;

value constrain (t1, t2) k =
  HVbox [: `S LR "constraint"; `ctyp t1 [: `S LR "=" :]; `ctyp t2 k :]
;

value type_list b tdl k =
  HVbox
    [: `HVbox [: :];
       listwbws
         (fun b (tn, tp, te, cl) k ->
            HVbox
              [: `HVbox [: b; `S LR tn; type_params tp [: `S LR "=" :] :];
                 `ctyp te [: :]; list constrain cl k :])
         b (S LR "and") tdl [: :];
       k :]
;

value external_def s t pl k =
  let ls = list (fun s k -> HVbox [: `S LR ("\"" ^ s ^ "\""); k :]) pl k in
  HVbox
    [: `HVbox [: `S LR "external"; `S LR (var_escaped s); `S LR ":" :];
       `ctyp t [: `S LR "="; ls :] :]
;

value value_description s t k =
  HVbox
    [: `HVbox [: `S LR "value"; `S LR (var_escaped s); `S LR ":" :];
       `ctyp t k :]
;

value rec mod_ident sl k =
  match sl with
  [ [] -> k
  | [s] -> [: `S LR s; k :]
  | [s :: sl] -> [: `S LR s; `S NO "."; mod_ident sl k :] ]
;

value rec module_type mt k =
  let next = module_type1 in
  match mt with
  [ <:module_type< functor ( $s$ : $mt1$ ) -> $mt2$ >> ->
      let head =
        HVbox
          [: `S LR "functor"; `S LO "("; `S LR s; `S LR ":";
             `module_type mt1 [: `S RO ")" :]; `S LR "->" :]
      in
      HVbox [: `head; `module_type mt2 k :]
  | _ -> next mt k ]
and module_type1 mt k =
  let curr = module_type1 in
  let next = module_type2 in
  match mt with
  [ <:module_type< $mt$ with $list:icl$ >> ->
      HVbox
        [: `curr mt [: :]; `with_constraints [: `S LR "with" :] icl k :]
  | _ -> next mt k ]
and module_type2 mt k =
  let curr = module_type2 in
  let next = module_type3 in
  match mt with
  [ <:module_type< sig $list:s$ end >> ->
      BEbox
        [: `S LR "sig"; `HVbox [: `HVbox [: :]; list sig_item s [: :] :];
           `HVbox [: `S LR "end"; k :] :]
  | _ -> next mt k ]
and module_type3 mt k =
  let curr = module_type3 in
  let next = module_type4 in
  match mt with
  [ <:module_type< $mt1$ $mt2$ >> -> HVbox [: `curr mt1 [: :]; `next mt2 k :]
  | _ -> next mt k ]
and module_type4 mt k =
  let curr = module_type4 in
  let next = module_type5 in
  match mt with
  [ <:module_type< $mt1$ . $mt2$ >> ->
      HVbox [: `curr mt1 [: `S NO "." :]; `next mt2 k :]
  | _ -> next mt k ]
and module_type5 mt k =
  match mt with
  [ <:module_type< $lid:s$ >> -> HVbox [: `S LR s; k :]
  | <:module_type< $uid:s$ >> -> HVbox [: `S LR s; k :]
  | _ -> HVbox [: `S LO "("; `module_type mt [: `S RO ")"; k :] :] ]
and sig_item si k =
  let k = [: `S RO ";"; k :] in
  match si with
  [ <:sig_item< type $list:stl$ >> -> type_list [: `S LR "type" :] stl k
  | <:sig_item< declare $list:s$ end >> ->
      BEbox
        [: `S LR "declare"; `HVbox [: `HVbox [: :]; list sig_item s [: :] :];
           `HVbox [: `S LR "end"; k :] :]
  | <:sig_item< exception $c$ of $list:tl$ >> ->
      variant [: `S LR "exception" :] (c, tl) k
  | <:sig_item< value $s$ : $t$ >> -> value_description s t k
  | <:sig_item< include $mt$ >> ->
      HVbox [: `S LR "include"; `module_type mt k :]
  | <:sig_item< external $s$ : $t$ = $list:pl$ >> -> external_def s t pl k
  | <:sig_item< module $s$ : $mt$ >> ->
      module_declaration [: `S LR "module"; `S LR s :] mt k
  | <:sig_item< module type $s$ = $mt$ >> -> modtype_declaration s mt k
  | <:sig_item< open $sl$ >> -> HVbox [: `S LR "open"; mod_ident sl k :]
  | MLast.SgCls _ cd ->
      HVbox
        [: `HVbox [: :];
           listwbws class_description [: `S LR "class" :] (S LR "and") cd
             k :]
  | MLast.SgClt _ cd ->
      HVbox
        [: `HVbox [: :];
           listwbws class_type_declaration [: `S LR "class"; `S LR "type" :]
             (S LR "and") cd k :] ]
and module_declaration b mt k =
  match mt with
  [ <:module_type< functor ( $i$ : $t$ ) -> $mt$ >> ->
      module_declaration
        [: b; `S LO "("; `S LR i; `S LR ":"; `module_type t [: `S RO ")" :] :]
        mt k
  | _ ->
      HVbox
        [: `HVbox [: :];
           `HVbox
              [: `HVbox [: b; `S LR ":" :];
                 `module_type mt [: :] :];
            k :] ]
and modtype_declaration s mt k =
  HVbox
    [: `HVbox [: :];
       `HVbox
          [: `HVbox [: `S LR "module"; `S LR "type"; `S LR s; `S LR "=" :];
             `module_type mt [: :] :];
       k :]
and with_constraints b icl k =
  HVbox [: `HVbox [: :]; listwbws with_constraint b (S LR "and") icl k :]
and with_constraint b wc k =
  match wc with
  [ MLast.WcTyp _ p al e ->
      let params =
        match al with
        [ [] -> [: :]
        | [s] -> [: `S LO "'"; `S LR s :]
        | sl -> [: `S LO "("; type_params sl [: `S RO ")" :] :] ]
      in
      HVbox
        [: `HVbox
              [: `HVbox b; `S LR "type"; params;
                 mod_ident p [: `S LR "=" :] :];
           `ctyp e k :]
  | MLast.WcMod _ sl mt ->
      HVbox
        [: b; `S LR "module"; mod_ident sl [: `S LR "=" :];
           `module_type mt k :] ]
and module_expr me k =
  match me with
  [ <:module_expr< struct $list:s$ end >> ->
      let s = HVbox [: `S LR "struct"; list str_item s [: :] :] in
      HVbox [: `HVbox [: :]; `s; `S LR "end"; k :]
  | <:module_expr< functor ($s$ : $mt$) -> $me$ >> ->
      let head =
        HVbox
          [: `S LR "functor"; `S LO "("; `S LR s; `S LR ":";
             `module_type mt [: `S RO ")" :]; `S LR "->" :]
      in
      HVbox [: `head; `module_expr me k :]
  | _ -> module_expr1 me k ]
and module_expr1 me k =
  let curr = module_expr1 in
  let next = module_expr2 in
  match me with
  [ <:module_expr< $me1$ $me2$ >> ->
      HVbox [: `curr me1 [: :]; `next me2 k :]
  | _ -> next me k ]
and module_expr2 me k =
  let curr = module_expr2 in
  let next = module_expr3 in
  match me with
  [ <:module_expr< $me1$ . $me2$ >> ->
      HVbox [: `curr me1 [: `S NO "." :]; `next me2 k :]
  | _ -> next me k ]
and module_expr3 me k =
  let curr = module_expr3 in
  match me with
  [ <:module_expr< $uid:s$ >> -> HVbox [: `S LR s; k :]
  | <:module_expr< ( $me$ : $mt$ ) >> ->
      HVbox
        [: `S LO "("; `module_expr me [: `S LR ":" :];
           `module_type mt [: `S RO ")"; k :] :]
  | <:module_expr< struct $list:_$ end >> ->
      HVbox [: `S LO "("; `module_expr me [: `S RO ")"; k :] :]
  | x -> not_impl "module_expr3" x ]
and str_item si k =
  let k = [: `S RO ";"; k :] in
  match si with
  [ <:str_item< open $i$ >> -> HVbox [: `S LR "open"; mod_ident i k :]
  | <:str_item< $exp:e$ >> -> HVbox [: `HVbox [: :]; `expr e k :]
  | <:str_item< declare $list:s$ end >> ->
      BEbox
        [: `S LR "declare"; `HVbox [: `HVbox [: :]; list str_item s [: :] :];
           `HVbox [: `S LR "end"; k :] :]
  | <:str_item< exception $c$ of $list:tl$ >> ->
      variant [: `S LR "exception" :] (c, tl) k
  | <:str_item< type $list:tdl$ >> -> type_list [: `S LR "type" :] tdl k
  | <:str_item< value $rec:rf$ $list:pel$ >> ->
      bind_list [: `S LR "value"; flag "rec" rf :] pel k
  | <:str_item< external $s$ : $t$ = $list:pl$ >> -> external_def s t pl k
  | <:str_item< module $s$ = $me$ >> ->
      module_binding [: `S LR "module"; `S LR s :] me k
  | <:str_item< module type $s$ = $mt$ >> ->
      HVbox
        [: `HVbox [: :];
           `HVbox
              [: `HVbox
                    [: `S LR "module"; `S LR "type"; `S LR s; `S LR "=" :];
                 `module_type mt [: :] :];
           k :]
  | MLast.StCls _ cd ->
      HVbox
        [: `HVbox [: :];
           listwbws class_declaration [: `S LR "class" :] (S LR "and") cd
             k :]
  | MLast.StClt _ cd ->
      HVbox
        [: `HVbox [: :];
           listwbws class_type_declaration [: `S LR "class"; `S LR "type" :]
             (S LR "and") cd k :] ]
and module_binding b me k =
  match me with
  [ <:module_expr< functor ($s$ : $mt$) -> $mb$ >> ->
      module_binding
        [: `HVbox
             [: b; `S LO "("; `S LR s; `S LR ":";
                `module_type mt [: `S RO ")" :] :] :]
        mb k
  | <:module_expr< ( $me$ : $mt$ ) >> ->
      HVbox
        [: `HVbox [: :];
           `HVbox
              [: `HVbox
                   [: `HVbox [: b; `S LR ":" :];
                      `module_type mt [: `S LR "=" :] :];
                 `module_expr me [: :] :];
           k :]
  | _ ->
      HVbox
        [: `HVbox [: :];
           `HVbox [: `HVbox [: b; `S LR "=" :]; `module_expr me [: :] :];
           k :] ]
and class_declaration b ci k =
  class_fun_binding 
    [: b; flag "virtual" ci.MLast.ciVir;
       `S LR ci.MLast.ciNam;
       class_type_parameters ci.MLast.ciPrm :]
    ci.MLast.ciExp k
and class_fun_binding b ce k =
  match ce with
  [ MLast.CeFun _ p cfb ->
      class_fun_binding [: b; `patt p [: :] :] cfb k
  | ce -> HVbox [: `HVbox [: b; `S LR "=" :]; `class_expr ce k :] ]
and class_type_parameters (loc, tpl) =
  match tpl with
  [ [] -> [: :]
  | tpl ->
      [: `S LO "[";
         listws type_parameter (S RO ",") tpl [: `S RO "]" :] :] ]
and type_parameter tp k =
  HVbox [: `S LO "'"; `S LR tp; k :]
and class_expr ce k =
  match ce with
  [ MLast.CeLet _ rf lb ce ->
      HVbox
        [: `HVbox [: :];
           `bind_list [: `S LR "let"; flag "rec" rf :] lb [: `S LR "in" :];
           `class_expr ce k :]
  | ce -> class_expr1 ce k ]
and class_expr1 ce k =
  match ce with
  [ MLast.CeApp _ ce sel ->
      HVbox [: `class_expr1 ce [: :]; list simple_expr sel k :]
  | ce -> class_expr2 ce k ]
and class_expr2 ce k =
  match ce with
  [ MLast.CeCon _ ci [] -> class_longident ci k
  | MLast.CeCon _ ci ctcl ->
      HVbox
        [: `class_longident ci [: :];
           `S LO "["; listws ctyp (S RO ",") ctcl [: `S RO "]"; k :] :]
  | MLast.CeStr _ csp cf ->
      class_structure [: `S LR "object"; `class_self_patt_opt csp :] cf
        [: `S LR "end"; k :]
  | MLast.CeTyc _ ce ct ->
      HVbox
        [: `S LO "("; `class_expr ce [: `S LR ":" :];
           `class_type ct [: `S RO ")"; k :] :]
  | _ -> HVbox [: `not_impl "class_expr" ce; k :] ]
and simple_expr e k =
  match e with
  [ <:expr< $lid:_$ >> -> expr e k
  | _ -> HVbox [: `S LO "("; `expr e [: `S RO ")"; k :] :] ]
and class_structure b cf k =
  BEbox
    [: `HVbox [: b :];
       `HVbox [: `HVbox [: :]; list class_str_item cf [: :] :];
       `HVbox [: k :] :]
and class_self_patt_opt csp =
  match csp with
  [ Some p -> HVbox [: `S LO "("; `patt p [: `S RO ")" :] :]
  | None -> HVbox [: :] ]
and class_str_item cf k =
  let k = [: `S RO ";"; k :] in
  match cf with
  [ MLast.CrInh _ ce pb ->
      HVbox
        [: `S LR "inherit"; `class_expr ce [: :];
           match pb with
           [ Some i -> [: `S LR "as"; `S LR i :]
           | _ -> [: :] ];
           k :]           
  | MLast.CrVal _ lab mf e -> HVbox [: `S LR "value"; `cvalue (lab, mf, e) k :]
  | MLast.CrVir _ lab pf t ->
      HVbox
        [: `S LR "method"; `S LR "virtual"; flag "private" pf; `label lab;
           `S LR ":"; `ctyp t k :]
  | MLast.CrMth _ lab pf fb ->
      fun_binding [: `S LR "method"; flag "private" pf; `label lab :] fb k
  | MLast.CrCtr _ t1 t2 ->
      HVbox
        [: `HVbox [: `S LR "type"; `ctyp t1 [: `S LR "=" :] :];
           `ctyp t2 k :]
  | MLast.CrIni _ se ->
      HVbox [: `S LR "initializer"; `expr se k :] ]
and label lab =
  S LR lab
and cvalue (lab, mf, e) k =
  HVbox [: flag "mutable" mf; `label lab; `S LR "="; `expr e k :]
and fun_binding b fb k =
  match fb with
  [ <:expr< fun $p$ -> $e$ >> ->
      fun_binding [: b; `simple_patt p [: :] :] e k
  | e -> HVbox [: `HVbox [: b; `S LR "=" :]; `expr e k :] ]
and simple_patt p k =
  match p with
  [ <:patt< $lid:_$ >> -> patt p k
  | _ -> HVbox [: `S LO "("; `patt p [: `S RO ")"; k :] :] ]
and class_type ct k =
  match ct with
  [ MLast.CtFun _ t ct ->
      HVbox
        [: `S LO "["; `ctyp t [: `S RO "]"; `S LR "->" :]; `class_type ct k :]
  | _ -> class_signature ct k ]
and class_signature cs k =
  match cs with
  [ MLast.CtCon _ id [] -> clty_longident id k
  | MLast.CtSig _ cst csf ->
      class_self_type [: `S LR "object" :] cst
        [: `HVbox [: `HVbox [: :]; list class_sig_item csf [: :] :];
           `HVbox [: `S LR "end"; k :] :]
  | _ -> HVbox [: `not_impl "class_signature" cs; k :] ]
and class_self_type b cst k =
  BEbox
    [: `HVbox
         [: b;
            match cst with
            [ None -> [: :]
            | Some t -> [: `S LO "("; `ctyp t [: `S RO ")" :] :] ] :];
       k :]
and class_sig_item csf k =
  let k = [: `S RO ";"; k :] in
  match csf with
  [ MLast.CgMth _ lab pf t ->
      HVbox
        [: `S LR "method"; flag "private" pf; `label lab; `S LR ":";
           `ctyp t k :]
  | _ -> HVbox [: `not_impl "class_sig_item" csf; k :] ]
and class_description b ci k =
  HVbox
    [: `HVbox
          [: b; flag "virtual" ci.MLast.ciVir;
             `S LR ci.MLast.ciNam; class_type_parameters ci.MLast.ciPrm;
             `S LR ":" :];
       `class_type ci.MLast.ciExp k :]
and class_type_declaration b ci k =
  HVbox
    [: `HVbox
          [: b; flag "virtual" ci.MLast.ciVir;
             `S LR ci.MLast.ciNam; class_type_parameters ci.MLast.ciPrm;
             `S LR "=" :];
       `class_signature ci.MLast.ciExp k :]
;

fwd_module_binding.val := module_binding;

value output_string_eval oc s =
  loop 0 where rec loop i =
    if i == String.length s then ()
    else if i == String.length s - 1 then output_char oc s.[i]
    else
      match (s.[i], s.[i+1]) with
      [ ('\\', 'n') -> do output_char oc '\n'; return loop (i + 2)
      | (c, _) -> do output_char oc c; return loop (i + 1) ]
;

value maxl = ref 78;
value sep = ref None;

value copy_source ic oc first bp ep =
  match sep.val with
  [ Some str ->
      if first then ()
      else if ep == in_channel_length ic then output_string oc "\n"
      else output_string_eval oc str
  | None ->
      let vp = pos_in ic in
      do seek_in ic bp;
         for i = bp to pred ep do output_char oc (input_char ic); done;
         seek_in ic vp;
      return () ]
;

value copy_to_end ic oc first bp =
  copy_source ic oc first bp (in_channel_length ic)
;

value apply_printer printer ast =
  let oc =
    match Pcaml.output_file.val with
    [ Some f -> open_out_bin f
    | None -> stdout ]
  in
  let cleanup () =
    match Pcaml.output_file.val with
    [ Some _ -> close_out oc
    | None -> () ]
  in
  if Pcaml.input_file.val <> "-" then
    let ic = open_in_bin Pcaml.input_file.val in
    do try
         let (first, last_pos) =
           List.fold_left
             (fun (first, last_pos) (si, (bp, ep)) ->
                do copy_source ic oc first last_pos bp;
                   flush oc;
                   print_pretty oc "" maxl.val (printer si [: :]);
                   flush oc;
                return (False, ep))
             (True, 0) ast
         in
         do copy_to_end ic oc first last_pos; flush oc; return ()
       with x ->
         do close_in ic; cleanup (); return raise x;
       close_in ic;
       cleanup ();
    return ()
  else
    do List.iter
         (fun (si, _) ->
            do print_pretty oc "" maxl.val (printer si [: :]);
               output_char oc '\n';
               flush oc;
            return ())
         ast;
       cleanup ();
    return ()
;

Pcaml.print_interf.val := apply_printer sig_item;
Pcaml.print_implem.val := apply_printer str_item;

Pcaml.add_option "-l" (Arg.Int (fun x -> maxl.val := x))
  "<length>   Maximum line length for pretty printing."
;

Pcaml.add_option "-sep" (Arg.String (fun x -> sep.val := Some x))
  "<string> Use this string between phrases instead of reading source."
;
