------------------------------------------------------------------------------
--                                                                          --
--                 ASIS-for-GNAT IMPLEMENTATION COMPONENTS                  --
--                                                                          --
--                           A 4 G . V C H E C K                            --
--                                                                          --
--                                 B o d y                                  --
--                                                                          --
--            Copyright (c) 1995-1999, Free Software Foundation, Inc.       --
--                                                                          --
-- ASIS-for-GNAT is free software; you can redistribute it and/or modify it --
-- under terms of the  GNU General Public License  as published by the Free --
-- Software Foundation;  either version 2,  or  (at your option)  any later --
-- version. ASIS-for-GNAT is distributed  in the hope  that it will be use- --
-- ful, but WITHOUT ANY WARRANTY; without even the implied warranty of MER- --
-- CHANTABILITY or  FITNESS FOR A PARTICULAR PURPOSE.  See the GNU General  --
-- Public License for more details. You should have received a copy of the  --
-- GNU General Public License  distributed with ASIS-for-GNAT; see file     --
-- COPYING. If not, write to the Free Software Foundation,  59 Temple Place --
-- - Suite 330,  Boston, MA 02111-1307, USA.                                --
--                                                                          --
-- As a special exception,  if other files  instantiate  generics from this --
-- unit, or you link  this unit with other files  to produce an executable, --
-- this  unit  does not  by itself cause  the resulting  executable  to  be --
-- covered  by the  GNU  General  Public  License.  This exception does not --
-- however invalidate  any other reasons why  the executable file  might be --
-- covered by the  GNU Public License.                                      --
--                                                                          --
-- ASIS-for-GNAT was originally developed  by the ASIS-for-GNAT team at the --
-- Software  Engineering  Laboratory  of  the Swiss  Federal  Institute  of --
-- Technology (LGL-EPFL) in Lausanne,  Switzerland, in cooperation with the --
-- Scientific  Research  Computer  Center of  Moscow State University (SRCC --
-- MSU), Russia,  with funding partially provided  by grants from the Swiss --
-- National  Science  Foundation  and  the  Swiss  Academy  of  Engineering --
-- Sciences.  ASIS-for-GNAT is now maintained by  Ada Core Technologies Inc --
-- (http://www.gnat.com).                                                   --
--                                                                          --
------------------------------------------------------------------------------

with Asis.Errors;             use Asis.Errors;
with Asis.Exceptions;         use Asis.Exceptions;
with Asis.Elements;           use Asis.Elements;
with Asis.Text;               use Asis.Text;

with Asis.Set_Get;            use Asis.Set_Get;
with Asis.Text.Set_Get;       use Asis.Text.Set_Get;
with A4G.A_Output;            use A4G.A_Output;

with Types;                   use Types;

package body A4G.Vcheck is

   LT : String renames A4G.A_Types.ASIS_Line_Terminator;

   Argument_Postponed : Boolean;
   --  this boolean flag is used by the (new) Raise_ASIS_Failed procedure
   --  to handle the situation when failure happens in some internal
   --  implementation routine, which does not know the ASIS Element
   --  being processed. If this flag is set ON, then the first
   --  routine which knows this ASIS Element should add it to diagnosis
   --  and reset this flag OFF

   ---------
   -- Add --
   ---------
   procedure Add (Phrase : String) is
   begin
      if Diagnosis_Len = Max_Diagnosis_Length then
         return;
      end if;
      for I in Phrase'Range loop
         Diagnosis_Len := Diagnosis_Len + 1;
         Diagnosis_Buffer (Diagnosis_Len) := Phrase (I);
         if Diagnosis_Len = Max_Diagnosis_Length then
            exit;
         end if;
      end loop;
   end Add;

------------------------------------------------------------

   procedure Check_Validity
     (Compilation_Unit : Asis.Compilation_Unit;
      Query            : String) is
   begin

      if Not_Nil (Compilation_Unit) and then
         not Valid (Compilation_Unit)
      then

         Set_Error_Status (Status    => Value_Error,
                           Diagnosis => "Invalid Unit value in " & Query);

         raise ASIS_Inappropriate_Compilation_Unit;
      end if;

   end Check_Validity;

------------------------------------------------------------
   procedure Check_Validity (Element : Asis.Element;
                             Query   : String) is
   begin

      if Kind (Element) /= Not_An_Element and then
         not Valid (Element)
      then
         Set_Error_Status (Status    => Value_Error,
                           Diagnosis => "Invalid Element value in " & Query);

         raise ASIS_Inappropriate_Element;

      end if;

   end Check_Validity;
---------------------------------------------------------------
   procedure Check_Validity
     (Line  : Asis.Text.Line;
      Query : String)
   is
   begin
      if not Asis.Text.Is_Nil (Line) and then not Valid (Line) then
         Set_Error_Status (Status    => Value_Error,
                           Diagnosis => "Invalid Line value in " & Query);
         raise ASIS_Inappropriate_Line;
      end if;
   end Check_Validity;
---------------------------------------------------------------
   procedure Check_Validity (Context : Asis.Context;
                             Query   : String) is
   begin
      if not Valid (Context) then
         Set_Error_Status (Status    => Value_Error,
                           Diagnosis => "Unopened or unassociated Context in "
                                       & Query);

         raise ASIS_Inappropriate_Context;

      end if;

   end Check_Validity;
----------------------------------------------------------------------
   procedure Raise_ASIS_Failed (Diagnosis : String) is

   begin

      Set_Error_Status (Status    => Internal_Error,
                        Diagnosis => "Internal implementation error: "
                                    & Diagnosis);

      raise ASIS_Failed;

   end Raise_ASIS_Failed;

   -----------------------------
   -- Raise_ASIS_Failed (new) --
   -----------------------------

   procedure Raise_ASIS_Failed
     (Argument : Asis.Element;
      Diagnosis : String;
      Stat      : Asis.Errors.Error_Kinds := Internal_Error;
      Bool_Par  : Boolean := False)
   is
   begin
      Diagnosis_Len := 0;

      --  forming the standard diagnosis prefix:

      Add ("Internal implementation error: ");
      Add (Diagnosis);

      if Bool_Par then
         Add (LT & "(Boolean parameter is TRUE)");
      end if;

      if Is_Nil (Argument) then
         Argument_Postponed := True;
      else
         Add (LT & "when processing ");
         Debug_String (Argument);
         Add (Debug_Buffer (1 .. Debug_Buffer_Len));
      end if;

      Status_Indicator := Stat;

      Argument_Postponed := False; --  ??? temporary solution

      raise ASIS_Failed;

   end Raise_ASIS_Failed;

---------------------------------------------------------------

   procedure Raise_ASIS_Inappropriate_Compilation_Unit
                         (Diagnosis : String) is
   begin

      Set_Error_Status (Status    => Value_Error,
                        Diagnosis => "Inappropriate Unit Kind in "
                                    & Diagnosis);

      raise ASIS_Inappropriate_Compilation_Unit;

   end Raise_ASIS_Inappropriate_Compilation_Unit;
----------------------------------------------------------------------
   procedure Raise_ASIS_Inappropriate_Element
     (Diagnosis : String;
      Status    : Error_Kinds      := Value_Error) is
   begin

      Set_Error_Status (Status    => Status,
                        Diagnosis => "Inappropriate Element Kind in "
                                    & Diagnosis);

      raise ASIS_Inappropriate_Element;

   end Raise_ASIS_Inappropriate_Element;
----------------------------------------------------------------------
   procedure Raise_ASIS_Inappropriate_Line_Number
     (Diagnosis : String;
      Status    : Error_Kinds      := Value_Error) is
   begin
      Set_Error_Status (Status    => Status,
                        Diagnosis => "Inappropriate Lines/Span Kind in "
                                    & Diagnosis);
      raise ASIS_Inappropriate_Line_Number;
   end Raise_ASIS_Inappropriate_Line_Number;
----------------------------------------------------------------------
   procedure Not_Implemented_Yet (Diagnosis : String) is
   begin

      Set_Error_Status (Status    => Not_Implemented_Error,
                        Diagnosis => "Not Implemented Query:" & LT
                                    & Diagnosis);

      raise ASIS_Failed;

   end Not_Implemented_Yet;
--------------------------------------------------------------------
   procedure Set_Error_Status
     (Status    : Error_Kinds      := Not_An_Error;
      Diagnosis : String := Nil_Asis_String)
   is
   begin
      if Status     = Not_An_Error and
         Diagnosis /= Nil_Asis_String
      then
         Status_Indicator := Internal_Error;
         Diagnosis_Len    := Incorrect_Setting_Len + ASIS_Line_Terminator_Len;
         Diagnosis_Buffer (1 .. Diagnosis_Len)
                          := Incorrect_Setting & ASIS_Line_Terminator;
         raise ASIS_Failed;
      end if;

      Status_Indicator := Status;

      Diagnosis_Len    := Diagnosis'Length + ASIS_Line_Terminator_Len;

      Diagnosis_Buffer (1 .. Diagnosis_Len)
                                   := Diagnosis & ASIS_Line_Terminator;

   end Set_Error_Status;
----------------------------------------------------------------------

--   procedure Add_Call_Information (Outer_Call : String) is

--      New_Diagnosis_Len : Natural;

--   begin

--      if Diagnosis_Len = Max_Diagnosis_Length then -- Diagnosis_Buffer
--         return;  -- is full, no new context information could be added
--      end if;

--      New_Diagnosis_Len := ASIS_Line_Terminator_Len +
--                           Diagnosis_Len +
--                           10 +           -- "called in " ' Length
--                           Outer_Call'Length;
--                           --  ASIS_Line_Terminator_Len;

--      if New_Diagnosis_Len <
--         Max_Diagnosis_Length - ASIS_Line_Terminator_Len
--      then
--         Diagnosis_Buffer (Diagnosis_Len + 1 .. New_Diagnosis_Len) :=
--            ASIS_Line_Terminator &
--            "called in " &
--            Outer_Call;
--            --  ASIS_Line_Terminator;

--         Diagnosis_Len := New_Diagnosis_Len;

--      else
--         for I in Diagnosis_Len + 1 ..
--                  Max_Diagnosis_Length - ASIS_Line_Terminator_Len loop

--            Diagnosis_Buffer (I) := '*';
--            --  indication of the "overflow" of the Diagnosis_Buffer ???
--         end loop;

--         Diagnosis_Buffer
--           (Max_Diagnosis_Length - ASIS_Line_Terminator_Len + 1 ..
--            Max_Diagnosis_Length) :=
--            ASIS_Line_Terminator;

--         Diagnosis_Len := Max_Diagnosis_Length;

--      end if;

--   end Add_Call_Information;

   --------------------------------
   -- Add_Call_Information (new) --
   --------------------------------

   procedure Add_Call_Information
     (Outer_Call : String;
      Argument   : Asis.Element := Nil_Element;
      Bool_Par   : Boolean := False)
   is
   begin
      Add (LT & "called in " & Outer_Call);
      if Bool_Par then
         Add (LT & "(Boolean parameter is TRUE)");
      end if;
      if Argument_Postponed and then not Is_Nil (Argument) then
         Add (LT & "with the argument : ");
         Debug_String (Argument);
         Add (Debug_Buffer (1 .. Debug_Buffer_Len));
         Argument_Postponed := False;
      end if;
   end Add_Call_Information;

   ----------------------
   -- Reset_Postponing --
   ----------------------

   procedure Reset_Postponing is
   begin
      Argument_Postponed := False;
   end Reset_Postponing;
----------------------------------------------------------------------
end A4G.Vcheck;