/* util.c: misc utility functions for Atari fdisk
 *
 * Copyright (C) 1995-97 Roman Hodek <Roman.Hodek@informatik.uni-erlangen.de>
 *               1996-97 Michael Schlueter <schlue00@marvin.informatik.uni-dortmund.de>
 *
 * This program is free software.  You can redistribute it and/or
 * modify it under the terms of the GNU General Public License as
 * published by the Free Software Foundation: either version 2 or
 * (at your option) any later version.
 *
 */

/* $Id: util.c,v 1.5 1997/07/08 18:59:06 rnhodek Exp $
 *
 * $Log: util.c,v $
 * Revision 1.5  1997/07/08 18:59:06  rnhodek
 * Ouch, too many changes without commits in between...
 * Implemented moving of partitions
 * Implemented checking of "extended start condition" (great changes in
 *   new_partition())
 * Some more tests in verify()
 *
 * Revision 1.4  1997/06/22 10:31:00  rnhodek
 * Add __attribute__((unused)) to cvid
 *
 * Revision 1.3  1997/06/21 20:47:51  rnhodek
 * Added RCS keywords
 *
 * Revision 1.2  1997/06/12 13:43:39  rnhodek
 * Fix many small bugs here and there. The ones you find when first running a
 * program...
 * 
 * Revision 1.1  1997/06/11 14:36:36  rnhodek
 * Initial revision
 * 
 * Revision 1.1.1.1  1997/06/11 14:36:36  rnhodek
 * Started using CVS for atafdisk
 *
 */

#ifndef lint
static char vcid[] __attribute__ ((unused)) =
"$Id: util.c,v 1.5 1997/07/08 18:59:06 rnhodek Exp $";
#endif /* lint */

#include <stdio.h>
#include <stdlib.h>
#include <string.h>
#include <unistd.h>
#include <ctype.h>

#include "fdisk.h"
#include "util.h"

void fatal( enum failure why )
{
    char error[LINE_LENGTH];
    char *message = error;

    if (listing) {
	close(fd);
	longjmp( listingbuf, 1 );
    }
    
    switch( why ) {
      case usage:
	message = "Usage: fdisk [-lfv] [-s /dev/hdxn] [-I sectorfile] "
		  "[-O sectorfile]\n"
		  "             [-o option] [/dev/hdx]\n";
	break;
      case unable_to_open:
	sprintf(error, "Unable to open %s\n", disk_device);
	break;
      case unable_to_read:
	sprintf(error, "Unable to read %s\n", disk_device);
	break;
      case unable_to_seek:
	sprintf(error, "Unable to seek on %s\n", disk_device);
	break;
      case unable_to_write:
	sprintf(error, "Unable to write %s\n", disk_device);
	break;
      case out_of_memory:
	message = "Unable to allocate any more memory\n";
	break;
      case couldnt_save_sectors:
	message = "Failed saving the old sectors - aborting\n";
	break;
      default:
	message = "Fatal error\n";
    }
    
    fputc( '\n', stderr );
    fputs( message, stderr );
    exit( 1 );
}

char *round_mb( unsigned long secs )
{
    static char buf[20];

    sprintf( buf, "%lu%s", secs/2048, (secs%2048) ? "+" : "" );
    return( buf );
}

/* ------------------------------------------------------------------------ */
/*			   Atari specific functions			    */

/* translate PID to long name */
char *atari_partition_type(char *type)
{
    int high = n_partition_IDs, tmp;

    for(tmp = 0; tmp <= high; tmp++) {
	if (PID_EQ(partition_IDs[tmp].id,type)) {
	    return partition_IDs[tmp].name;
	}
    };
    return NULL;
}

int is_valid_PID( const char *id )
{
    return( isalnum(id[0]) && isalnum(id[1]) && isalnum(id[2]) );
}

/* does a partition ID look like a known PID? */
int is_reasonable_PID( const char *id )
{
    int i, max = n_partition_IDs;

    for( i = 0; i <= max; i++) {
	if (PID_EQ(partition_IDs[i].id,id))
	    return 1;
    };
    return 0;
}

int is_valid_part_entry( struct apartition *pi )
{
    unsigned long end = pi->start + pi->size;
    return( is_valid_PID(pi->id) &&
	    0 < pi->start && pi->start <= hd_size &&
	    0 < end && end <= hd_size &&
	    pi->size > 0 );
}

/* list known partition types */
void list_atari_types( void )
{
    int i, size = n_partition_IDs;

    printf( "\nKnown/usual partition IDs\n" );
    for( i = 0; i < size; ++i ) {
	printf( "  %s  %-15.15s\n",
		partition_IDs[i].id, partition_IDs[i].name );
    };
}

/* return name for boot flag */
char *atari_boot_type(unsigned char flag)
{
    int j, max = n_bootflags;
    
    for( j = 0; j < max; ++j )
	if (bootflags[j].flag == (flag & ~PART_FLAG_VALID))
	    return( bootflags[j].name );
    return( "???" );
}

/* return short name for boot flag */
char *atari_boot_type_short(unsigned char flag)
{
    int j, max = n_bootflags;
    static char buf[6];
    
    for( j = 0; j < max; ++j )
	if (bootflags[j].flag == (flag & ~PART_FLAG_VALID))
	    return( bootflags[j].short_name );
    sprintf( buf, "$%02x", flag );
    return( buf );
}

#define	MAGIC_CHECKSUM	0x1234

int check_rootsec_checksum( char *buf )
{
    signed short *p, sum;
    signed short *pend = (signed short *)&buf[SECTOR_SIZE];
    
    for( sum = 0, p = (signed short *)buf; p < pend; ++p )
	sum += *p;
    return( sum == MAGIC_CHECKSUM );
}


void recalc_rootsec_checksum( char *buf )
{
    signed short *p, sum;
    signed short *pend = (signed short *)&buf[SECTOR_SIZE-2];
    
    for( sum = 0, p = (signed short *)buf; p < pend; ++p )
	sum += *p;
    *pend = MAGIC_CHECKSUM - sum;
}

/* Local Variables: */
/* tab-width: 8     */
/* End:             */
