#include <stdio.h>
#include <stdlib.h>
#include <syslog.h>
#include <ctype.h>
#include <limits.h>
#include <signal.h>
#include <unistd.h>
#include <string.h>
#include <sys/wait.h>
#include <sys/stat.h>
#include <sys/file.h>
#include <pwd.h>
#include <grp.h>
#include <errno.h>
#include <string.h>
#include "pop3.h"
#include "misc.h"
#include "../paths.h"

// Error code returned to sendmail
#define VERR_USER_UNKNOWN	67
#define VERR_CANTCREAT		73
#define VERR_NOPERM			77

static void vdeliver_checkdir(const char *dirpath)
{
	struct stat st;
	if (stat(dirpath,&st)==-1){
		syslog (LOG_ERR,"Directory %s does not exist",dirpath);
		exit (-1);
	}else if (!S_ISDIR(st.st_mode)){
		syslog (LOG_ERR,"%s is not a directory",dirpath);
		exit (-1);
	}
}

struct SEEN_LOOKUP{
	const char *user;
	struct SEEN_LOOKUP *next;
};


static struct SEEN_LOOKUP *new_SEEN_LOOKUP (
	const char *user,
	struct SEEN_LOOKUP *next)
{
	struct SEEN_LOOKUP *ret = (struct SEEN_LOOKUP*)calloc(1
		,sizeof(struct SEEN_LOOKUP));
	ret->user = strdup(user);
	ret->next = next;
	return ret;
}

struct VDEV_CTX{
	FILE *faliases[3];
	char fallback[500];
	struct SEEN_LOOKUP *seen;
	long quota;
	long mailsize;
	bool match_gecos;
	char filter[1000];
};

/*
	Is that alias was already processed
	Return != 0 if this alias was processed once
*/
static int vdeliver_wasseen (struct VDEV_CTX *ctx, const char *user)
{
	int ret = 0;
	struct SEEN_LOOKUP *e = ctx->seen;
	while (e != NULL){
		if (strcmp(e->user,user)==0){
			ret = 1;
			break;
		}
		e = e->next;
	}
	return ret;
}

static void vdeliver_copy (FILE *fin, FILE *fout)
{
	char buf[1000];
	rewind (fin);
	while (fgets(buf,sizeof(buf)-1,fin)!=NULL){
		fputs (buf,fout);
	}
}

// Copy a message to a filter, itself pointed to the open mailbox
static void vdeliver_filter (FILE *fin, const char *filter, FILE *fout)
{
	int tb[2];
	if (pipe(tb)==-1){
		syslog (LOG_ERR,"Can't setup pipe for filtering, filtering disabled");
		vdeliver_copy(fin,fout);
	}else{
		pid_t pid = fork();
		if (pid == 0){
			close (tb[1]);
			dup2 (tb[0],0);
			dup2 (fileno(fout),1);
			system (filter);
		}else if (pid != (pid_t)-1){
			close (tb[0]);
			FILE *newfout = fdopen (tb[1],"w");
			vdeliver_copy (fin,newfout);
			fclose (newfout);
			close (tb[1]);
			int status;
			wait (&status);
		}else{
			syslog (LOG_ERR,"Can't fork, filtering disabled");
			vdeliver_copy(fin,fout);
			close (tb[0]);
			close (tb[1]);
		}
	}
}


// Copy a message, and escape lines starting with From in the body
static void vdeliver_copy_from (FILE *fin, FILE *fout)
{
	char buf[1000];
	rewind (fin);
	bool first = true;
	while (fgets(buf,sizeof(buf)-1,fin)!=NULL){
		if (!first && strncmp(buf,"From ",5)==0){
			fputc ('>',fout);
		}
		first = false;
		fputs (buf,fout);
	}
}

static FILE *vdeliver_openex (const char *fname, struct passwd *pwd)
{
	FILE *ret = NULL;
	int i;
	for (i=0; i<20; i++){
		ret = fopen (fname,"a");
		if (ret == NULL){
			break;
		}else{
			int fd = fileno(ret);
			if (flock(fd, LOCK_EX|LOCK_NB) != -1) {
				struct group *grp = getgrnam ("mail");
				int mailgid = 0;
				if (grp != NULL) mailgid = grp->gr_gid;
				fchown (fd,pwd->pw_uid,mailgid);
				fchmod (fd,0600);
				break;
			}else{
				fclose(ret);
				ret = NULL;
				sleep(1);
			}
		}
	}
	if (i==20 && ret == NULL) syslog (LOG_ERR,"Can't lock %s (%m)",fname);
	return ret;
}

static int vdeliver_doaliases (
	struct VDEV_CTX *ctx,
	const char *user,
	const char *domain,
	FILE *mailin);
/*
	Send one copy of the message to a remote user

	Return -1 if any error.
*/
static int vdeliver_send (const char *adr, FILE *fin)
{
	int ret = -1;
	char cmd[1000];
	FILE *fout;
	sprintf (cmd,"%s %s",USR_SBIN_SENDMAIL,adr);
	fout = popen (cmd,"w");
	if (fout == NULL){
		syslog (LOG_ERR,"Can't talk back to sendmail for user %s (%m)",adr);
	}else{
		vdeliver_copy (fin,fout);
		ret = pclose(fout);
		if (ret != 0){
			ret = WEXITSTATUS(ret);
			syslog (LOG_ERR,"cmd return %d: Was relaying to %s",ret,adr);
		}
	}
	return ret;
}

/*
	Read multiple lines of an aliases file and return a single one
	Skip comments and empty lines.
*/
static int fgets_cont (
	char *line,
	int ,
	FILE *fin,
	char *accum)
{
	char buf[1000];
	int noline = 0;
	int empty;
	strcpy (line,accum);
	accum[0] = '\0';
	while (fgets_strip (buf,sizeof(buf)-1,fin
			,'\0','#',&noline,&empty)!=NULL){
		if (line[0] == '\0'){
			strcpy (line,buf);
		}else if (isspace(buf[0])){
			strcat (line,buf);
		}else{
			strcpy (accum,buf);
			break;
		}
	}
	return line[0] != '\0' ? 0 : -1;
}

static int vdeliver_splitline(
	struct VDEV_CTX *ctx,
	char *ptpt,
	const char *domain,
	FILE *mailin);


/*
	Expands aliases.
	Return true if one aliases was found
*/
static int vdeliver_checkaliases(
	struct VDEV_CTX *ctx,
	const char *user,
	const char *domain,
	FILE *mailin,
	bool &found)
{
	int ret = 0;
	found = false;
	for (int i=0; i<3 && !found; i++){
		FILE *fin = ctx->faliases[i];
		if (fin != NULL){
			char buf[2000],accum[2000];
			rewind (fin);
			accum[0] = '\0';
			while (fgets_cont (buf,sizeof(buf)-1,fin,accum)!=-1){
				char *pt = str_skip (buf);
				char *ptpt = strchr (pt,':');
				if (ptpt != NULL){
					*ptpt++ = '\0';
					strip_end (pt);
					if (strcasecmp(user,pt)==0
						&& !vdeliver_wasseen(ctx,pt)){
						found = true;
						ctx->seen = new_SEEN_LOOKUP(pt,ctx->seen);
						ret |= vdeliver_splitline (ctx
							,ptpt,domain,mailin);
						break;
					}
				}
			}
		}
	}
	return ret;
}

const long QUOTA_NOLIMIT=0x7fffffffl;

/*
	Write the message to the inbox
*/
static int vdeliver_writemsg (
	struct VDEV_CTX *ctx,
	struct passwd *pwd,
	const char *domain,
	FILE *filein)
{
	int ret = 0;
	char dirpath[PATH_MAX],filepath[PATH_MAX];
	snprintf (dirpath,sizeof(dirpath)-1,"%s/%s",VAR_SPOOL_VMAIL,domain);
	vdeliver_checkdir (dirpath);
	snprintf (filepath,sizeof(filepath)-1,"%s/%s",dirpath,pwd->pw_name);
	FILE *fout = vdeliver_openex (filepath,pwd);
	if (fout != NULL){
		bool deliver = true;
		// Wait until we have the lock to check the quota
		long uquota = QUOTA_NOLIMIT;
		char uquotap[PATH_MAX];
		snprintf (uquotap,sizeof(uquotap)-1,"%s/%s.quota"
			,dirpath,pwd->pw_name);
		FILE *fin = fopen (uquotap,"r");
		if (fin != NULL){
			fscanf (fin,"%ld",&uquota);
			uquota <<= 10;	// In K, put it in bytes
			fclose (fin);
		}
		if (ctx->quota != QUOTA_NOLIMIT || uquota != QUOTA_NOLIMIT){
			struct stat st;
			if (fstat(fileno(fout),&st)!=-1
				&& (st.st_size+ctx->mailsize >= ctx->quota
					|| st.st_size+ctx->mailsize > uquota)){
				deliver = false;
				ret = VERR_CANTCREAT;
				fprintf (stderr,"Out of disk quota for this user inbox\n");
			}
		}
		if (deliver){
			if (ctx->filter[0] == '\0'){
				vdeliver_copy (filein,fout);
			}else{
				vdeliver_filter (filein,ctx->filter,fout);
			}
			ret = fclose (fout);
		}else{
			fclose (fout);
		}
	}else{
		syslog (LOG_ERR,"Can't open file %s (%m)",filepath);
	}
	return ret;
}


/*
	Do the final delivery if the user exist in the mailbox
	Return -1 if any errors.
*/
static int vdeliver_do (
	struct VDEV_CTX *ctx,
	const char *user,
	const char *domain,
	FILE *fin)
{
	int ret = -1;
	char pwdfile[PATH_MAX],shadowfile[PATH_MAX];
	struct passwd *pwd;
	sprintf (pwdfile,"%s/passwd.%s",ETC_VMAIL,domain);
	sprintf (shadowfile,"%s/shadow.%s",ETC_VMAIL,domain);
	pwd = vmail_getpwnam (pwdfile,shadowfile,user,true,ctx->match_gecos);
	if (pwd == NULL){
		/* #Specification: vdeliver / fallback management
			If a virtual account does not exist (and there were no
			aliases defined, the fallback is used. There are 3 cases

			#
			-The fallback is empty. The mail is simply rejected
			 like sendmail is normally doing.
			-The fallback start with the character @. This is taken
			 as the new destination domain. The user account is used
			 so unknown@this_domain becomes unknowns@fallback_domain.
			-The fallback contain a @. This is taken as a full email
			 address and all email to unknown account is sent to
			 this address
			-The fallback is a local account name. Then all processing
			 is done once again with the fallback replacing for the 
			 target account. This means that that fallback may itself
			 be an alias if needed.
		*/
		const char *fallback = ctx->fallback;
		if (fallback[0] == '\0'){
			syslog (LOG_INFO,"Unknown user: %s",user);
			ret = VERR_USER_UNKNOWN;
		}else if (fallback[0] == '@'){
			char newdest[1000];
			snprintf (newdest,sizeof(newdest)-1,"%s%s",user,fallback);
			ret = vdeliver_send (newdest,fin);			
		}else if (strchr(fallback,'@') != NULL){
			ret = vdeliver_send (fallback,fin);			
		}else{
			static char fallbacking = 0;
			if (!fallbacking){
				fallbacking = 1;
				ret = vdeliver_doaliases (ctx,fallback,domain,fin);
				fallbacking = 0;
			}else{
				syslog (LOG_ERR,"Invalid fallback destination for domain %s",domain);
			}
		}
	}else{
		if (strcasecmp(user,pwd->pw_name)==0){
			ret = vdeliver_writemsg (ctx,pwd,domain,fin);
		}else{
			// Ok, we got there using the gecos. Now we know the
			// real user id, so we check the aliases again.
			// If no aliases matches, then we can deliver to the inbox.
			bool found;
			ret = vdeliver_checkaliases (ctx,pwd->pw_name,domain
				,fin,found);
			if (!found){
				ret |= vdeliver_writemsg (ctx,pwd,domain,fin);
			}
		}
	}
	return ret;
}

static void fctsig(int )
{
}
/*
	Deliver one copy of the message to one recipient
	The recipient may be one of those
		-A filter program
		-A file holding a list of recipient
		-One user either local to this domain or remote

	Local recipient are check recursivly to see if they are not aliases
	themselves.
*/
static int vdeliver_doone (
	struct VDEV_CTX *ctx,
	const char *dest,
	const char *domain,
	FILE *mailin)
{
	int ret = 0;
	dest = str_skip(dest);
	if (dest[0] == '|'){
		int pid;
		signal (SIGCHLD,fctsig);
		pid = fork();
		if (pid == 0){
			FILE * out;
			dest = str_skip (dest+1);
			setgid (65535);
			setuid (65535);
			out = popen (dest,"w");
			if (out != NULL){
				vdeliver_copy (mailin,out);
				pclose (out);
				_exit (0);
			}
			_exit (-1);
		}else if (pid != -1){
			int status;
			while (wait (&status) != pid);
			ret |= status;
		}
	}else if (strncmp(dest,":include:",9)==0){
		FILE *list;
		struct stat st;
		dest = str_skip(dest+9);
		list = fopen (dest,"r");
		if (list == NULL){
			syslog (LOG_ERR,"Can't open list file %s (%m) for domain %s"
				,dest,domain);
		}else{
			/* #Specification: vdeliver / aliases / list file
				Only world readable file will be used by vdeliver.
				Since privilege users may set aliases themselves and
				virtual domain co-administrator are somewhat trusted,
				but not much, they can only set list file using paths
				to world readable file.
			*/
			if (fstat (fileno(list),&st)!=-1
				&& st.st_mode & 4){
				char buf[1000];
				while (fgets(buf,sizeof(buf)-1,list)!=NULL){
					char alias[1000];
					if (sscanf(buf,"%s",alias)==1){
						ret |= vdeliver_doone (ctx,alias,domain,mailin);
					}
				}
			}
			fclose (list);
		}
	}else if (strchr (dest,'@')!=NULL){
		ret = vdeliver_send (dest,mailin);
	}else{
		ret = vdeliver_doaliases (ctx,dest,domain,mailin);
	}
	return ret;
}

static char *str_copyupto (char *dest, const char *src, char stop)
{
	while (*src > ' ' && *src != stop) *dest++ = *src++;
	*dest = '\0';
	return ((char*) src);
}

static int vdeliver_splitline(
	struct VDEV_CTX *ctx,
	char *ptpt,
	const char *domain,
	FILE *mailin)
{
	int ret = 0;
	while (1){
		ptpt = str_skip (ptpt);
		if (ptpt[0] == '\0'){
			break;
		}else if (ptpt[0] == ','){
			ptpt++;
		}else if (ptpt[0] == '"'){
			char word[200],*ptw;
			ptpt++;
			ptw = word;
			while (*ptpt != '\0' && *ptpt != '"') *ptw++ = *ptpt++;
			*ptw = '\0';
			if (*ptpt == '"') ptpt++;
			ret |= vdeliver_doone (ctx,word,domain,mailin);
		}else{
			char word[200];
			ptpt = str_copyupto (word,ptpt,',');
			ret |= vdeliver_doone (ctx,word,domain,mailin);
		}
	}
	return ret;
}




/*
	Process aliases for this user.
	Return > 0 if at least one aliases was processed.

	A missing alias file is not an error
*/
static int vdeliver_doaliases (
	struct VDEV_CTX *ctx,
	const char *user,
	const char *domain,
	FILE *mailin)
{
	int ret = 0;
	static int recur = 0;
	if (recur == 16){
		syslog (LOG_ERR,"Broken recursive alias %s for vdomain %s",user,domain);
		ret = -1;
	}else{
		recur++;
		bool found;
		ret |= vdeliver_checkaliases(ctx,user,domain,mailin,found);
		if (!found){
			ret |= vdeliver_do (ctx,user,domain,mailin);
		}
		recur--;
	}
	return ret;
}

/*
	Open all possible aliases files for a domain and initialise the
	VDEV_CTX structure.
*/
static void vdeliver_openaliases(
	const char *domain,
	struct VDEV_CTX *ctx)
{
	// Aliases file are optionnal, so we open the file and do not
	// care if it succeed
	char fname[PATH_MAX];
	ctx->fallback[0] = '\0';
	ctx->faliases[0] = ctx->faliases[1] = ctx->faliases[2] = NULL;
	ctx->quota = QUOTA_NOLIMIT;
	ctx->match_gecos = 0;
	ctx->filter[0] = '\0';
	sprintf (fname,"%s/aliases.%s",ETC_VMAIL,domain);
	ctx->faliases[0] = fopen (fname,"r");
	FILE *fin = fopen (ETC_CONF_LINUXCONF,"r");
	if (fin != NULL){
		// Looks for alternate alias file, up to 2 per domains
		// so a domain may have 3 aliases files
		char key[200],keyf[200],keyquota[200],keygecos[200],buf[1000];
		char keyfilter[200];
		int noalias = 1;
		snprintf (key,sizeof(key)-1,"vdomain_alias.%s",domain);
		snprintf (keyf,sizeof(keyf)-1,"vdomain_fallback.%s",domain);
		snprintf (keyquota,sizeof(keyquota)-1,"vdomain_quota.%s",domain);
		snprintf (keygecos,sizeof(keygecos)-1,"vdomain_gecos.%s",domain);
		snprintf (keyfilter,sizeof(keyfilter)-1,"vdomain_filter.%s",domain);
		while (fgets(buf,sizeof(buf)-1,fin)!=NULL){
			char v1[1000],v2[1000];
			if (sscanf (buf,"%s %s",v1,v2)==2){
				if (strcmp(keyf,v1)==0){
					strcpy (ctx->fallback,v2);
				}else if (strcmp(key,v1)==0){
					if (noalias < 3){
						ctx->faliases[noalias++] = fopen (v2,"r");
					}
				}else if (strcmp(keyquota,v1)==0){
					ctx->quota = atoi(v2)*1024l;
				}else if (strcmp(keygecos,v1)==0){
					ctx->match_gecos = v2[0] != '0';
				}else if (strcmp(keyfilter,v1)==0){
					char *pt = buf;
					while (*pt > ' ') pt++;
					pt = str_skip(pt);
					strip_end (pt);
					strcpy (ctx->filter,pt);
				}
			}
		}
		fclose (fin);
	}
}
/*
	Find the official email domain associate (optionally) to a domain
*/
static void vdeliver_alias2domain (
	const char *domain,
	char realdomain[PATH_MAX])
{
	strcpy (realdomain,domain);
	FILE *fin = fopen (ETC_CONF_LINUXCONF,"r");
	if (fin != NULL){
		// Look for a line of the following format
		// vdomain_other.A_DOMAIN alias
		char buf[2000];
		while (fgets(buf,sizeof(buf)-1,fin)!=NULL){
			if (strncmp(buf,"vdomain_other.",14)==0){
				char v1[1000],v2[1000];
				if (sscanf (buf,"%s %s",v1,v2)==2
					&& strcmp(domain,v2)==0){
					strcpy (realdomain,v1+14);
					break;
				}
			}
		}
		fclose (fin);
	}
}


int main (int argc, char *argv[])
{
	int ret = -1;
	openlog ("vdeliver",LOG_PID,LOG_MAIL);
	if (argc != 3){
		syslog (LOG_ERR,"vdeliver: Invalid arguments: expected user domain");
	}else{
		char tmpfile[PATH_MAX];
		FILE *fout;
		sprintf (tmpfile,"%s.%d",VAR_RUN_VDELIVER_TMP,getpid());
		fout = fopen (tmpfile,"w");
		if (fout == NULL){
			syslog (LOG_ERR,"Can't open temporary file %s (%m)",tmpfile);
		}else{
			struct VDEV_CTX ctx;
			const char *user = argv[1];
			char domain[PATH_MAX],aliasdomain[PATH_MAX];
			ctx.seen = NULL;
			strlwr (aliasdomain,argv[2],sizeof(aliasdomain));
			vdeliver_alias2domain (aliasdomain,domain);
			vdeliver_copy_from (stdin,fout);
			ctx.mailsize = ftell(fout);
			fclose (fout);
			fout = fopen (tmpfile,"r");
			vdeliver_openaliases(domain,&ctx);
			ret = vdeliver_doaliases (&ctx,user,domain,fout);
		}
		unlink (tmpfile);
	}
	return ret;
}
		

