/* -*- Mode: C++; tab-width: 4; indent-tabs-mode: nil; c-basic-offset: 2 -*-
 *   Gnome Apt frontend
 *
 *   Copyright (C) 1998 Havoc Pennington <hp@pobox.com>
 *
 * This program is free software; you can redistribute it and/or 
 * modify it under the terms of the GNU General Public License as 
 * published by the Free Software Foundation; either version 2 of the
 * License, or (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA 02111-1307
 * USA
 */

#include "childprocess.h"
#include <zvt/zvtterm.h>

#include <iostream> 
#include <stdio.h>

#include <unistd.h>

void 
ChildDialog::child_died_cb(GtkWidget* w, gpointer data)
{
  ChildDialog* cd = static_cast<ChildDialog*>(data);
  
  cd->child_died();
}

void 
ChildDialog::child_died()
{
  child_done_ = true;

  
}

gint 
ChildDialog::close_cb(GtkWidget* w, gpointer data)
{
  ChildDialog* cd = static_cast<ChildDialog*>(data);
  
  return cd->close();
}

gint 
ChildDialog::close()
{
  if (child_done_) return FALSE; 

  GtkWidget* dialog = 
    gnome_message_box_new(_("The child process is still running. Do you want to kill it?\nIf you do, you'll have to restart Gnome Apt."),
                          GNOME_MESSAGE_BOX_QUESTION,
                          GNOME_STOCK_BUTTON_YES,
                          GNOME_STOCK_BUTTON_NO, 
                          NULL);
  gnome_dialog_set_default(GNOME_DIALOG(dialog), 1);
  gnome_apt_setup_dialog(dialog);
  gint response = gnome_dialog_run(GNOME_DIALOG(dialog));
  if (response == 0) {
    close_anyway_ = true;
    return FALSE;
  }

  return TRUE; // block the close
}

ChildDialog::ChildDialog(const char* title,
                         const char* label)
  : we_are_parent_(false), child_done_(false), close_anyway_(false)
{
  dialog_ = gnome_dialog_new(title, 
                             GNOME_STOCK_BUTTON_CLOSE,
                             NULL);
  gnome_dialog_set_close(GNOME_DIALOG(dialog_), TRUE);
  gnome_dialog_close_hides(GNOME_DIALOG(dialog_), TRUE);
  gtk_window_set_modal(GTK_WINDOW(dialog_), TRUE);
  gnome_apt_setup_dialog(dialog_);

  gtk_signal_connect(GTK_OBJECT(dialog_),
                     "close",
                     GTK_SIGNAL_FUNC(close_cb),
                     this);

  label_ = gtk_label_new(label);

  gtk_box_pack_start(GTK_BOX(GNOME_DIALOG(dialog_)->vbox),
                     label_,
                     FALSE, FALSE, 0);

  term_ = zvt_term_new();

  zvt_term_set_size (ZVT_TERM(term_), 80, 24);

  gtk_signal_connect(GTK_OBJECT(term_),
                     "child_died",
                     GTK_SIGNAL_FUNC(child_died_cb),
                     this);

  zvt_term_set_scrollback(ZVT_TERM(term_), 10000);

  zvt_term_set_scroll_on_keystroke (ZVT_TERM(term_), TRUE);
  zvt_term_set_scroll_on_output (ZVT_TERM(term_), TRUE);

  GtkWidget* scrollbar = 
    gtk_vscrollbar_new (GTK_ADJUSTMENT (ZVT_TERM(term_)->adjustment));
  GTK_WIDGET_UNSET_FLAGS (scrollbar, GTK_CAN_FOCUS);

  GtkWidget* hbox = gtk_hbox_new(FALSE,0);

  gtk_box_pack_start(GTK_BOX(hbox), term_, 
                     TRUE, TRUE, 0);
  gtk_box_pack_end(GTK_BOX(hbox), scrollbar,
                   FALSE, FALSE, 0);

  gtk_box_pack_start(GTK_BOX(GNOME_DIALOG(dialog_)->vbox),
                     hbox,
                     TRUE, TRUE, 0);  

  gtk_widget_realize(term_);
  gtk_widget_realize(dialog_);

  gtk_widget_show_all(dialog_);

  while (gtk_events_pending())
    gtk_main_iteration();

  gtk_widget_show_now(dialog_);

  while (gtk_events_pending())
    gtk_main_iteration();
}

ChildDialog::~ChildDialog()
{
  if (we_are_parent_)
    {
      if (dialog_) gtk_widget_destroy(dialog_);
    }
}

int
ChildDialog::fork()
{
  // The ZvtTerm *must* get a size_allocate before the 
  //  child starts sending output, or it will 
  //  insert line breaks in the wrong place.
  gtk_widget_show_now(dialog_);

  // make sure we are all synced up here. weirdo paranoia.
  cout << flush;
  cerr << flush;
  fflush(stdout);
  fflush(stderr);

  int retval = zvt_term_forkpty (ZVT_TERM(term_), FALSE); // FALSE == no utmp
  switch (retval) {
  case -1:
    perror ("Error: unable to fork");
    break;
  case 0: 
    {
      we_are_parent_ = false;
      // touching the GUI == flaming death in the child
      dialog_ = 0;
      term_ = 0;
      label_ = 0;
      // Close all file descriptors but first 3 - total paranoia kludge
      int open_max = sysconf (_SC_OPEN_MAX);
      for (int i = 3; i < open_max; i++){
        ::close (i);
      }
    }
    break;
  default:
    we_are_parent_ = true;
    break;
  }

  return retval;
}


bool 
ChildDialog::run()
{
  g_return_val_if_fail(we_are_parent_, false);

  while (child_done_ == false)
    {
      int reply = gnome_dialog_run(GNOME_DIALOG(dialog_));

      if (close_anyway_) break;
    }

  gtk_widget_destroy(dialog_);

  dialog_ = 0; // got destroyed

  // for now, we just return true
  return true;
}

