/*
   fbgetty : a getty for framebuffer 
   Copyright (C) 1999 Yann DRONEAUD (lch@multimania.com). 

   fbgetty is free software; you can redistribute it and/or modify
   it under the terms of the GNU General Public License as published by
   the Free Software Foundation; either version 2, or (at your option)
   any later version.

   fbgetty is distributed in the hope that it will be useful, but 
   WITHOUT ANY WARRANTY; without even the implied warranty of
   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
   GNU General Public License for more details.

   You should have received a copy of the GNU General Public License
   along with this program; if not, write to the Free Software Foundation,
   Inc., 59 Temple Place - Suite 330, Boston, MA 02111-1307, USA.  

 */

#include "global.h"
#include "options.h"
#include "errors.h"
#include <stdio.h>
#include <stdlib.h>
#include <unistd.h>
#include <errno.h>
#include <string.h>

#ifdef HAVE_GETOPT_LONG
#include <getopt.h>
#else
#include "getopt.h"
#endif

void usage (int status);
void version (void);
void compile (void);
void parse_command_line (int argc, char *argv[]);

#define merge_fgoptions_str(var,deflt) fgoptions-> ## var = merge_str (fgoptions_options-> ## var, \
								       fgoptions_environment-> ## var, \
								       fgoptions_arguments-> ## var, deflt)


#define merge_fgoptions_int(var,deflt) fgoptions-> ## var = merge_int (fgoptions_options-> ## var, \
								       fgoptions_environment-> ## var, \
								       fgoptions_arguments-> ## var,deflt)

char *merge_str(char *,char *,char *,char *);
int merge_int(int,int,int,int);

fbgetty_options_t *fgoptions;

void 
parse_command_line (int argc, char *argv[])
{
  fbgetty_options_t *fgoptions_options;
  fbgetty_options_t *fgoptions_arguments;
  fbgetty_options_t *fgoptions_environment;

  int count;
  int arg;
  int arg_readed;

#define OPTION_TTY     't'
#ifdef USE_FRAME_BUFFER
#define OPTION_FB      'f'
#endif
#define OPTION_ISSUE   'i'
#define OPTION_REFRESH 'r'
#define OPTION_PROGRAM 'l'
#define OPTION_PROMPT  'p'
#define OPTION_TIMEOUT 'o'

#define OPTION_CLEAR   'C'
#define OPTION_NOCLEAR 'X'

#define OPTION_VERSION 'V'
#define OPTION_COMPILE 'O'
#define OPTION_HELP    'H'

#ifdef USE_FRAME_BUFFER
#define SHORT_OPTIONS "t:f:p:l:i:o:"
#else
#define SHORT_OPTIONS "t:p:l:i:o:"
#endif /* USE_FRAME_BUFFER */

  struct option long_options[] =
  {
    {"tty",           required_argument, NULL, OPTION_TTY},
#ifdef USE_FRAME_BUFFER
    {"framebuffer",   required_argument, NULL, OPTION_FB},
    {"fb",            required_argument, NULL, OPTION_FB},
#endif 
    {"clear-screen",  optional_argument, NULL, OPTION_CLEAR}, 
    {"no-clear-screen",  optional_argument, NULL, OPTION_CLEAR}, 
    {"issue",         required_argument, NULL, OPTION_ISSUE}, 
    {"refresh",       required_argument, NULL, OPTION_REFRESH}, 
    {"login-program", required_argument, NULL, OPTION_PROGRAM},
    {"login-prompt",  required_argument, NULL, OPTION_PROMPT},
    {"login-timeout", required_argument, NULL, OPTION_TIMEOUT},
    {"version",       no_argument,       NULL, OPTION_VERSION},
    {"compile",       no_argument,       NULL, OPTION_COMPILE},
    {"help",          no_argument,       NULL, OPTION_HELP},
    {0, 0, 0, 0}
  };

  int option_index = 0;
  int option = -1 ;

  int show_version = 0;
  int show_help = 0;
  int show_error = 0;
  int show_compile = 0;


  fgoptions_arguments = (fbgetty_options_t *) malloc(sizeof(struct fbgetty_options_t));
  if (fgoptions_arguments == NULL)
    fatal_error("can't allocate memory to fbgetty config");

  memset(fgoptions_arguments, 0x00,sizeof (fbgetty_options_t));

  fgoptions_options = (fbgetty_options_t *) malloc(sizeof(struct fbgetty_options_t));
  if (fgoptions_options == NULL)
    fatal_error("can't allocate memory to fbgetty config");

  memset(fgoptions_options, 0x00,sizeof (fbgetty_options_t));

  fgoptions_environment = (fbgetty_options_t *) malloc(sizeof(struct fbgetty_options_t));
  if (fgoptions_environment == NULL)
    fatal_error("can't allocate memory to fbgetty config");

  memset(fgoptions_environment, 0x00,sizeof (fbgetty_options_t));
 
#ifdef FB_GETTY_DEBUG
  error("parsing configuration");
#endif

#if 0
  /* fill with default */
  fgoptions->login_program=LOGIN_PROGRAM;
  fgoptions->login_prompt=LOGIN_PROMPT;
  fgoptions->issue_file=ISSUE_FILE;
  fgoptions->issue_refresh=ISSUE_REFRESH;
  fgoptions->login_timeout=LOGIN_TIMEOUT;
  fgoptions->tty_device=TTY_DEVICE;
  fgoptions->screen_clear=TRUE;
#ifdef USE_FRAME_BUFFER
  fgoptions->fb_device=FB_DEVICE;
#endif
#endif /* 0 */

#ifdef FB_GETTY_DEBUG
  error(" parsing options");
#endif

  /* Parse command line */
  while ((option = getopt_long (argc, argv, SHORT_OPTIONS, long_options, &option_index)) != -1)
    {
      switch (option)
        {
        case OPTION_VERSION:
          show_version = 1;
          break;

        case OPTION_HELP:
          show_help = 1;
          break;

	case OPTION_COMPILE:
          show_compile = 1;
          break;

	case OPTION_TIMEOUT:
          fgoptions_options->login_timeout = strtol (optarg, NULL, 0);  /* convert timeout */
          break;

	case OPTION_REFRESH:
          fgoptions_options->issue_refresh = strtol (optarg, NULL, 0);  /* convert refresh */
          break;

	case OPTION_PROMPT:
          fgoptions_options->login_prompt = optarg;  /* login prompt */
          break;

	case OPTION_PROGRAM:
          fgoptions_options->login_program = optarg;  /* login program */
          break;

	case OPTION_ISSUE:
          fgoptions_options->issue_file = optarg;  /* issue file */
          break;

#ifdef USE_FRAME_BUFFER
	case OPTION_FB:
          fgoptions_options->fb_device = optarg;  /* frame buffer device */
          break;
#endif

	case OPTION_TTY:
          fgoptions_options->tty_device = optarg;  /* tty device */
          break;

	case OPTION_CLEAR:
	  fgoptions_options->screen_clear = TRUE;
	  break;

	case OPTION_NOCLEAR:
	  fgoptions_options->screen_clear = FALSE;
	  break;

        case EOF:
          break;

        case '?':
          show_error = 1;
          break;

        case ':':
          error ("Missing argument");
          show_error = 1;
          break;

        default:
#ifdef FB_GETTY_DEBUG
          error ("%s:%i Error in getopt_long : %c", __FILE__, __LINE__, option);
#endif
          break;
        }
}

  if (show_help)
    usage (0);
  if (show_version)
    version ();
  if (show_compile)
   compile ();
  if (show_error)
    usage (1);

  /*
   * Get the arguments 
   */
#ifdef FB_GETTY_DEBUG
  error(" parsing arguments");
#endif

  count = 0;
  arg_readed = 0;
  for (arg = optind; arg < argc; arg ++)
    {
       arg_readed = 0;
       if (strchr(argv[arg],'=') != NULL)
	 { 
           putenv(argv[arg]);
	   arg_readed = 1;
	 }
       else
	 {
	   if ((count == 0) && (arg_readed == 0)) 
	     {
	       fgoptions_arguments->tty_device = argv[arg];
               count++;
               arg_readed = 1;
	     }
#ifdef USE_FRAME_BUFFER           
	   if ((count == 1) && (arg_readed == 0))
             {
	       fgoptions_arguments->fb_device = argv[arg];
	       count++;
	       arg_readed = 1;
	     }
#endif /* USE_FRAME_BUFFER */
           
	 }

       if (arg_readed == 0)
         error("unknown parameter %s",argv[arg]);
   }


  /* 
   * Get parameter from environment 
   */
#ifdef FB_GETTY_DEBUG
  error(" parsing environment");
#endif

  fgoptions_environment->tty_device = getenv("tty");

#ifdef USE_FRAME_BUFFER
  fgoptions_environment->fb_device = getenv("framebuffer");
#endif 

  if (getenv("clear_screen") != NULL)
    {
      if (strcasecmp(getenv("clear_screen"),"no") == 0)  
	fgoptions_environment->screen_clear = FALSE;
      else
	if (strcasecmp(getenv("clear_screen"),"yes") == 0)
	  fgoptions_environment->screen_clear = TRUE;
	else
	  error("unknown argument for clean_screen: %s",getenv("clear_screen"));
    }

  fgoptions_environment->issue_file = getenv("issue");

  if (getenv("refresh") != NULL)
    {
      fgoptions_environment->issue_refresh = strtol(getenv("refresh"),NULL,0);
    }
  else
    {
      fgoptions_environment->issue_refresh = 0;
    }
  fgoptions_environment->login_program = getenv("login_program");
  fgoptions_environment->login_prompt = getenv("login_prompt");
  
  if (getenv("login_timeout") != NULL)
    {
      fgoptions_environment->login_timeout = strtol(getenv("login_timeout"),NULL,0);
    }
  else
    {
      fgoptions_environment->login_timeout = 0;
    }
 
  /* 
   * Merge the different type of options 
   */
#ifdef FB_GETTY_DEBUG
  error(" merging differents options");
#endif

  merge_fgoptions_str(tty_device,NULL);

#ifdef USE_FRAME_BUFFER
  merge_fgoptions_str(fb_device,NULL);
#endif /* USE_FRAME_BUFFER */
  merge_fgoptions_str(login_program,LOGIN_PROGRAM);
  merge_fgoptions_str(login_prompt,LOGIN_PROMPT);
  merge_fgoptions_str(issue_file,ISSUE_FILE);

  merge_fgoptions_int(login_timeout,LOGIN_TIMEOUT);
  merge_fgoptions_int(issue_refresh,ISSUE_REFRESH);

  free(fgoptions_environment);
  free(fgoptions_arguments);
  free(fgoptions_options);

#ifdef FB_GETTY_DEBUG
  error("Analizing options");
#endif

  if ((fgoptions->issue_file != NULL) && (strcasecmp(fgoptions->issue_file,"none") == 0))
    fgoptions->issue_file=NULL;
  
  if ((fgoptions->login_prompt != NULL) && (strcasecmp(fgoptions->login_prompt,"none") == 0))
    fgoptions->login_prompt=NULL;
  
  if (fgoptions_options->login_timeout < 0 )
    error("Invalid login timeout");
   
  if (fgoptions_options->issue_refresh < 0 )
    error("Invalid issue refresh interval");

#ifdef FB_GETTY_DEBUG
  error("Options :");
  error("tty device    : %s",fgoptions->tty_device);
#ifdef USE_FRAME_BUFFER
  error("fb device     : %s",fgoptions->fb_device);
#endif
  error("login program : %s",fgoptions->login_program);
  error("login prompt  : %s",fgoptions->login_prompt);
  error("login timeout : %d",fgoptions->login_timeout);
  error("issue file    : %s",fgoptions->issue_file);
  error("issue refresh : %d",fgoptions->issue_refresh);
#endif
  
  if (fgoptions->tty_device  == NULL)
    {
      fprintf(stderr,"You must specify a tty device\n");
      usage(1);
    }


#ifdef FB_GETTY_DEBUG
  count = 0;
  while (environ[count] != NULL)
   error("%s",environ[count++]);

  error("options parsing finish");
#endif

}


void 
usage (int status)
{
  if (status != 0)
    {
      fprintf (stderr, "Try `%s --help' for more information.\n", program_invocation_short_name);
    }
  else
    {
      fprintf (stderr, "\
Usage: %s [options] [tty] [framebuffer]\n\
\n\
with options in\n\
  -t, --tty=TTY                use tty device `TTY'\n"
#ifdef USE_FRAME_BUFFER
"  -f, --fb=FB
      --framebuffer=FB         use frame buffer device `FB'\n"
#endif
"      --screen-clear           clear screen when launched
      --no-screen-clear        do not clear the screen 
  -i, --issue=ISSUE            display ISSUE\n\
  -l, --login-program=LOGIN    use LOGIN program\n\
  -p, --login-prompt=PROMPT    use login PROMPT\n\
  -o, --login-timeout=SECONDE  set login timeout to SECONDE\n\
      --compile                display compilation information\n\
      --help                   display this help and exit\n\
      --version                display version information and exit\n\
\n\
Report bugs, comment, or what you want to "AUTHOR_MAIL"\n\
\n\
", program_invocation_short_name);

    }

  exit (EXIT_SUCCESS);
}

void 
version (void)
{
  fprintf (stderr,"\
%s version %s,\n\
Copyright (C) 1999 " AUTHOR_NAME " <" AUTHOR_MAIL ">\n\
This is free software, and you are welcome to redistribute it under\n\
the GNU General Public License.\n\
%s comes with ABSOLUTELY NO WARRANTY.\n\n\
see info page or man page for more information\n\
go to %s for new release\n\
\n\
", PACKAGE , VERSION, PACKAGE ,FBGETTY_URL);

  exit(EXIT_SUCCESS);
}
 
void 
compile (void)
{
  fprintf (stderr,"%s version %s compilation informations\n", 
                  PACKAGE, VERSION);

  fprintf(stderr,"\n");
#ifdef BUILD_USER
  fprintf(stderr,"compile by          : " BUILD_USER);
#ifdef BUILD_HOSTNAME
  fprintf(stderr,"@" BUILD_HOSTNAME);
#endif
  fprintf(stderr,"\n");
#endif

#ifdef __DATE__
  fprintf(stderr,"compile date        : " __DATE__ " " __TIME__ "\n");
#endif

#ifdef TARGET_SYSTEM
  fprintf(stderr,"target              : " TARGET_SYSTEM "\n"); 
#endif 

#ifdef __linux__
#include <linux/version.h>
#ifdef UTS_RELEASE
  fprintf(stderr,"kernel version      : " UTS_RELEASE "\n");
#endif
#endif

#ifdef __GNUC__
  fprintf(stderr,"C compiler          : GNU CC " __VERSION__ "\n" );
#endif

#ifdef __GLIBC__
  fprintf(stderr,"C library           : GNU C library %d.%d\n", __GLIBC__, __GLIBC_MINOR__);
#endif

  fprintf(stderr,"framebuffer         : ");

#ifdef USE_FRAME_BUFFER
  fprintf(stderr,"used\n");
#else
  fprintf(stderr,"not used\n");
#endif

  fprintf(stderr,"debug support       : ");
#ifdef FB_GETTY_DEBUG
  fprintf(stderr,"yes\n");
#else
  fprintf(stderr,"no\n");
#endif

  fprintf(stderr,"\n");

  exit(EXIT_SUCCESS);
}



char *merge_str(char *first, char *second, char *third, char *deflt)
{
#if 1
  if (first != NULL)
    return first;
  else
    if (second != NULL)
      return second;
    else
      if (third != NULL)
	return third;
      else
	return deflt;
#else
  if (first == NULL)
    {
      if (second == NULL)
	{
	if (third == NULL)
	  return deflt;
	else
	  return third;
	}
      else
	return second;
    }
  else
    return first;
#endif

}

int merge_int(int first, int second, int third, int deflt)
{
#if 1
  if (first != -1)
    return first;
  else
    if (second != -1)
      return second;
    else
      if (third != -1)
	return third;
      else
	return deflt;
#else
  if (first == -1)
    {
      if (second == -1)
	{
	if (third == -1)
	  return deflt;
	else
	  return third;
	}
      else
	return second;
    }
  else
    return first;
#endif

}

