#include <ctype.h>
#include <fcntl.h>
#include <glob.h>
#include <limits.h>
#include <linux/fd.h>
#include <net/if.h>
#include <netinet/in.h>
#include <setjmp.h>
#include <signal.h>
#include <stdio.h> 
#include <string.h>
#include <sys/ioctl.h>
#include <sys/param.h>
#include <sys/socket.h>
#include <sys/stat.h>
#include <sys/types.h>
#include <sys/wait.h>
#include <syslog.h>
#include <termios.h>
#include <unistd.h>

#ifdef USE_LANGUAGE_CHOOSER
#include <wchar.h>
#endif

#include "dbootstrap.h"
#include "lang.h"
#include "util.h"

/* for documentation see util.h */

int execlog (const char *incmd, int priority)
{ 
  FILE *output; 
  char line[MAXLINE]; 
  char cmd[strlen(incmd) + 6]; 
  strcpy(cmd, incmd); 

  if (bootargs.isdebug) {
    openlog(SYSLOG_IDENT, LOG_PID, LOG_USER);
    syslog(LOG_DEBUG, "running cmd '%s'", cmd);
  }
  else {
    openlog(SYSLOG_IDENT, LOG_PID, LOG_USER);
  }

  /* FIXME: this can cause the shell command if there's redirection
            already in the passed string */
  strcat(cmd, " 2>&1"); 
  output = popen(cmd, "r"); 
  while (fgets(line, MAXLINE, output) != NULL ) { 
    syslog(priority, line); 
  }
  closelog();
  return(pclose(output)); 
}


int fullscreen_execlog  (const char *cmd)
{
  int ret;

  DEBUGMSG("running cmd fullscreen '%s'", cmd);

  boxSuspend();
  printf(CLEAR);
  ret = system(cmd);
  boxResume();

  if (ret != 0) {
    ERRMSG("command %s returned %d", cmd, ret);
  }

  return(ret);
}


int check_target_path(void)
{
  if (!Root) {
    /* need to create the temp dir */
    return mkdir(TARGET_NOPREFIX, 0755);
  }
  return 0;
}

const char* target_path(const char* s)
{
  static char buf[PATH_MAX + 1];
  const char* prefix;
#ifndef _TESTING_
  if (Root)
    prefix = TARGET_PREFIX;
  else
    prefix = TARGET_NOPREFIX;
#else
  prefix = TARGET_TESTDIR;
#endif
  strcpy(buf, prefix);
  strcat(buf, s);
  return buf;
}


int check_dir(const char* dirname)
{
  struct stat check;
  if(stat(dirname, &check) == -1)
    return -1;
  else
    return S_ISDIR(check.st_mode);
}


/* for documentation see util.h */
int write_userconfig (const char *key, const char *value)
{
  FILE *conffile;
  const char *path;
  
  if ( check_target_path() != 0 ) {
    ERRMSG("unable to write settings, target root not mounted yet?");
  }

  path = target_path(USER_SETTINGS_DIR);
  if (check_dir(path) == -1)
    mkdir(path, 0700);
  
  if ((conffile = fopen(target_path(USER_SETTINGS_FILE), "a")) == NULL)
      return 1;

  fprintf(conffile, "%s='%s'\n", key, value);
  fclose(conffile);
  
  return 0;
}

/* for documentation see util.h */
int write_userconfig_va (const char *key, const char *fmt, ...)
{
  char *p;
  va_list ap;
  if ((p = malloc(MAXLINE)) == NULL)
    return -1;
  
  va_start(ap, fmt);
  (void) vsnprintf(p, MAXLINE, fmt, ap);
  va_end(ap);

  return write_userconfig(key, p);
}


/* for documentation see util.h */
char *normalize_dir (const char *dir, char *buf, size_t bufsize)
{
    char *result;

    if (dir[0] == '/')
    {
        struct stat ds;

        if (stat (dir, &ds) == 0 && S_ISDIR (ds.st_mode))
        {
            strncpy (buf, dir, bufsize - 1);
            buf[bufsize - 1] = '\0';

            result = buf;
        }
        else
            result = NULL;
    }
    else
        for (;;)
        {
            result = NULL;

            {
                int dd = open (dir, O_RDONLY);
                struct stat ds;

                if (dd == -1)
                    break;

                if (fstat (dd, &ds) == -1 || !S_ISDIR (ds.st_mode))
                {
                    close (dd);

                    break;
                }

                {
                    char *tempo = malloc (PATH_MAX);

                    getcwd (tempo, PATH_MAX);   /* Store the current directory for later reuse */

                    fchdir (dd);

                    getcwd (buf, bufsize);

                    chdir (tempo);

                    free (tempo);
                }
            }

            result = buf;

            break;
        }

    return result;
}

int
is_network_configured(void)
{
    struct stat statbuf;

    if ( (NAME_ISREG(target_path("/etc/network/interfaces"),&statbuf)) &&
	((NAME_ISEXE("/sbin/ifconfig",&statbuf)) ||
	 (NAME_ISEXE(target_path("/sbin/ifconfig"),&statbuf))))
	return 1;
    return 0;
}

int
is_interface_up(char *inter)
{
    struct ifreq ifr;
    int sfd = -1, ret = -1;

    if ((sfd = socket(AF_INET, SOCK_DGRAM, 0)) == -1)
	goto int_up_done;

    strncpy(ifr.ifr_name, inter, sizeof(ifr.ifr_name));

    if (ioctl(sfd, SIOCGIFFLAGS, &ifr) < 0)
	goto int_up_done;

    ret = (ifr.ifr_flags & IFF_UP) ? 1 : 0;

int_up_done:
    if (sfd != -1)
	close(sfd);
    return ret;
}

int
is_network_up(int all) {
    int ret = 0;
    char *buf;

    getif_start();
    while ((buf = getif(1)) != NULL)
      if (all || is_interface_up(buf))
	    ret++;
        
    getif_end();

    return ret;
}

static FILE *ifs = NULL;
static char ibuf[512];

void
getif_start(void) {
    if (ifs != NULL) {
	fclose(ifs);
	ifs = NULL;
    }
    if ((ifs = fopen("/proc/net/dev", "r")) != NULL) {
	fgets(ibuf, sizeof(ibuf), ifs); /* eat header */
	fgets(ibuf, sizeof(ibuf), ifs); /* ditto */
    }
    return;
}

void
getif_end(void) {
    if (ifs != NULL) {
	fclose(ifs);
	ifs = NULL;
    }
    return;
}

char *
getif(int all) {
    char rbuf[512];
    if (ifs == NULL)
	return NULL;

    if (fgets(rbuf, sizeof(rbuf), ifs) != NULL ) {
	get_name(ibuf, rbuf);
	if (!strcmp(ibuf,"lo")) /* ignore the loopback */
	    return getif(all); /* seriously doubt there is an infinite number of lo devices */
	if (all || is_interface_up(ibuf) == 1)
	    return ibuf;
    }
    return NULL;
}

void
get_name(char *name, char *p)
{
    while (isspace(*p))
	p++;
    while (*p) {
	if (isspace(*p))
	    break;
	if (*p == ':') {        /* could be an alias */
	    char *dot = p, *dotname = name;
	    *name++ = *p++;
	    while (isdigit(*p))
		*name++ = *p++;
	    if (*p != ':') {    /* it wasn't, backup */
		p = dot;
		name = dotname;
	    }
	    if (*p == '\0')
		return;
	    p++;
	    break;
	}
	*name++ = *p++;
    }
    *name++ = '\0';
    return;
}

int
initchroot_cap(void)
{
    struct stat statbuf;
    return 0; /* default for now */
    if (NAME_ISREG("/proc/sys/kernel/init-chroot",&statbuf))
	return 1;
    else
	return 0;
}

/* This may get more complex...leave it simple for now */
int
is_cdrom_image(void)
{
    struct stat statbuf;
    if (NAME_ISREG("/cdrom_image",&statbuf) || bootargs.cdrom) {
	if (NAME_ISDIR("/dists",&statbuf))
	    return 2;
	else
	    return 1;
    } else
	return 0;
}

int
check_pending_config(void)
{
  const char *path;
  struct stat statbuf;

  if (check_dir(TARGET_NOPREFIX)!=-1) {
    /*
     * /tmp/notarget exists: network was previously configured with no
     * target system mounted.
     * Just move the notarget tree to /target
     */
    execlog("cp -pR " TARGET_NOPREFIX "/. " TARGET_PREFIX, LOG_INFO);
    /* also, readjust the symlink if needed for cdrom filesystem */
    if (is_cdrom_image() == 2) {
	unlink("/tmp/resolv.conf");
	symlink(target_path("/etc/resolv.conf"), "/tmp/resolv.conf");
    }
  }

  /* 
   * first create our config file, and the USER_SETTINGS_DIR too, incidentally
   */
  if ( write_userconfig("BUILDTIME", BUILDTIME) != 0 ) { /* check that we can write config  */
    ERRMSG("failed to create " USER_SETTINGS_FILE);
    vaproblemBox(_("Failed while saving dbootstrap settings in %s"), USER_SETTINGS_FILE);
    return 1;                   /* might as well give up now */
  }

  /* 
   * copy settings if any from the kbdconfig step
   */
  if (NAME_ISREG(KEYBD_SETTINGS_FILE, &statbuf)) {
    path = target_path(USER_SETTINGS_FILE);
    snprintf(prtbuf, sizeof(prtbuf) - 1, "cat " KEYBD_SETTINGS_FILE " >> %s", path);
    execlog(prtbuf, LOG_INFO);
  }

  /* 
   * note any significant boot flags
   */
  if ( bootargs.isverbose )
    write_userconfig("VERBOSE", "yes");
  if ( bootargs.isquiet )
    write_userconfig("VERBOSE", "quiet");
  if ( bootargs.isdebug )
    write_userconfig("DEBUG", "true");
  if ( bootargs.cdrom )
    write_userconfig("CDROM", "true");

#ifdef USE_LANGUAGE_CHOOSER
  if ( lang != NULL )
    write_userconfig("LANGUAGE", lang);
#endif

  return 0;
}

/*
 * Used, so far, by http-fetch.c:nf_http_fetchfile() and
 * choose_medium.c:choose_NFS_server().
 *
 * See "util.h" for documentation.
 */
sigjmp_buf env;
struct termios term_settings;
void (*old_sigint) (int);

sig_atomic_t threw;

static void
sigint (int signum)
{
  threw = 1;
  siglongjmp (env, 1);
}

/*
 * Potentially the `KDSIGACCEPT' console ioctl could be used for
 * something like this also.  One or several keys could be temporarily
 * rebound so that they generate the `KeyboardSignal' keysym, with an
 * "unwind-protect" type mechanism in the signal handler that resets
 * them to their normal behaviour before it throws, or something like
 * that. - karlheg
 *
 */

int
do_interruptable (const char *message,
		  void (*thunk) (void *result),
		  void *result)
{
  int saved_errno;
  const char *press_cc = _("  Press Ctrl-C to interrupt.");
  char msg_buf[ strlen (message) + strlen (press_cc) + 1 ];
  snprintf (msg_buf, sizeof (msg_buf), "%s%s", message, press_cc);
  if (! sigsetjmp (env, 1)) {
    struct termios new_term_settings;
    tcgetattr (0, &term_settings);
    memcpy (&new_term_settings, &term_settings, sizeof (struct termios));
    new_term_settings.c_lflag |= NOFLSH;
    new_term_settings.c_lflag |= ISIG;
    /* Allow only VINTR (C-c) */
    new_term_settings.c_cc[VINTR]  = '\003';
    new_term_settings.c_cc[VQUIT]  = _POSIX_VDISABLE;
    new_term_settings.c_cc[VSUSP]  = _POSIX_VDISABLE;
    tcsetattr (0, TCSANOW, &new_term_settings);
    threw = 0;
    old_sigint = signal (SIGINT, sigint);
    pleaseWaitBox (msg_buf);
    thunk (result);
    saved_errno = errno;
  }
  tcsetattr (0, TCSANOW, &term_settings);
  signal (SIGINT, old_sigint);
  boxPopWindow();
  errno = saved_errno;
  return (threw);
}

#ifdef _UTIL_TEST_
int main (int argc, char **argv) { 
  int i=0; 
  int ret; 
  openlog ("execlog test", LOG_CONS | LOG_PID, LOG_USER);
  i++; 
  ret=execlog("echo 'I AM the pumpkin king'", LOG_INFO); 
  printf("Command %d returned %d\n", i, ret); 
  i++; 
  ret=execlog("cat /no_such_goddamn_file", LOG_INFO); 
  printf("Command %d returned %d\n", i, ret); 
  i++; 
  ret=execlog("echo 'P1i1p1e1s 1w1o1r1k' | sed -e 's/1//g'", LOG_INFO); 
  printf("Command %d returned %d\n", i, ret); 
  i++; 
  ret=execlog("echo 'LOG_INFO message'", LOG_INFO); 
  printf("Command %d returned %d\n", i, ret); 
  i++; 
  ret=execlog("echo 'LOG_NOTICE message'", LOG_NOTICE); 
  printf("Command %d returned %d\n", i, ret); 
  i++; 
  ret=execlog("echo 'LOG_ALERT message'", LOG_ALERT); 
  i++; 
  ret=execlog("echo 'Command with stderr redirected' 2> /tmp/test.out", LOG_INFO); 
  i++;
  ret=execlog("cat /etc/motd command_with_stdout_redirected > /tmp/test.out", LOG_INFO); 
}; 

#endif 
  
 


/*
 * ---------------------------------------------------------
 * Utility routines swiped from busybox, because I'm sick of
 * my changes to busybox internals breaking dbootstrap.
 * That is just stupid.
 *  -Erik <andersee@debian.org>
 */


/*
 * Return TRUE if a fileName is a directory.
 * Nonexistant files return FALSE.
 */
int isDirectory(const char *name)
{
    struct stat statBuf;

    if (stat(name, &statBuf) < 0)
	return FALSE;
    if (S_ISDIR(statBuf.st_mode))
	return TRUE;
    return(FALSE);
}


/*
 * Copy one file to another, while possibly preserving its modes, times,
 * and modes.  Returns TRUE if successful, or FALSE on a failure with an
 * error message output.  (Failure is not indicted if the attributes cannot
 * be set.)
 *
 * Code written by Erik Andersen <andersee@debian.org> and
 * swiped from busybox.
 */
int
copyFile( const char *srcName, const char *destName, 
	 int setModes, int followLinks)
{
    int rfd;
    int wfd;
    int rcc;
    int result;
    char buf[BUF_SIZE];
    struct stat srcStatBuf;
    struct stat dstStatBuf;
    struct utimbuf times;

    if (followLinks == FALSE)
	result = stat(srcName, &srcStatBuf);
    else 
	result = lstat(srcName, &srcStatBuf);
    if (result < 0) {
	perror(srcName);
	return FALSE;
    }

    if (followLinks == FALSE)
	result = stat(destName, &dstStatBuf);
    else 
	result = lstat(destName, &dstStatBuf);
    if (result < 0) {
	dstStatBuf.st_ino = -1;
	dstStatBuf.st_dev = -1;
    }

    if ((srcStatBuf.st_dev == dstStatBuf.st_dev) &&
	(srcStatBuf.st_ino == dstStatBuf.st_ino)) {
	fprintf(stderr, "Copying file \"%s\" to itself\n", srcName);
	return FALSE;
    }

    if (S_ISDIR(srcStatBuf.st_mode)) {
	//fprintf(stderr, "copying directory %s to %s\n", srcName, destName);
	/* Make sure the directory is writable */
	result = mkdir(destName, 0777777 ^ umask(0));
	if (result < 0 && errno != EEXIST) {
	    perror(destName);
	    return (FALSE);
	}
    } else if (S_ISLNK(srcStatBuf.st_mode)) {
	char *link_val;
	int link_size;

	//fprintf(stderr, "copying link %s to %s\n", srcName, destName);
	link_val = (char *) alloca(PATH_MAX + 2);
	link_size = readlink(srcName, link_val, PATH_MAX + 1);
	if (link_size < 0) {
	    perror(srcName);
	    return (FALSE);
	}
	link_val[link_size] = '\0';
	link_size = symlink(link_val, destName);
	if (link_size != 0) {
	    perror(destName);
	    return (FALSE);
	}
    } else if (S_ISFIFO(srcStatBuf.st_mode)) {
	//fprintf(stderr, "copying fifo %s to %s\n", srcName, destName);
	if (mkfifo(destName, 0644)) {
	    perror(destName);
	    return (FALSE);
	}
    } else if (S_ISBLK(srcStatBuf.st_mode) || S_ISCHR(srcStatBuf.st_mode) 
	    || S_ISSOCK (srcStatBuf.st_mode)) {
	//fprintf(stderr, "copying soc, blk, or chr %s to %s\n", srcName, destName);
	if (mknod(destName, srcStatBuf.st_mode, srcStatBuf.st_rdev)) {
	    perror(destName);
	    return (FALSE);
	}
    } else if (S_ISREG(srcStatBuf.st_mode)) {
	//fprintf(stderr, "copying regular file %s to %s\n", srcName, destName);
	rfd = open(srcName, O_RDONLY);
	if (rfd < 0) {
	    perror(srcName);
	    return FALSE;
	}

	wfd = creat(destName, srcStatBuf.st_mode);
	if (wfd < 0) {
	    perror(destName);
	    close(rfd);
	    return FALSE;
	}

	while ((rcc = read(rfd, buf, sizeof(buf))) > 0) {
	    if (fullWrite(wfd, buf, rcc) < 0)
		goto error_exit;
	}
	if (rcc < 0) {
	    goto error_exit;
	}

	close(rfd);
	if (close(wfd) < 0) {
	    return FALSE;
	}
    }

    if (setModes == TRUE) {
	//fprintf(stderr, "Setting permissions for %s\n", destName);
	chmod(destName, srcStatBuf.st_mode);
#if (__GLIBC__ >= 2) && (__GLIBC_MINOR__ >= 1)
	if (followLinks == FALSE)
	    lchown(destName, srcStatBuf.st_uid, srcStatBuf.st_gid);
	else
#endif
	    chown(destName, srcStatBuf.st_uid, srcStatBuf.st_gid);

	times.actime = srcStatBuf.st_atime;
	times.modtime = srcStatBuf.st_mtime;

	utime(destName, &times);
    }

    return TRUE;


  error_exit:
    perror(destName);
    close(rfd);
    close(wfd);

    return FALSE;
}



/*
 * Walk down all the directories under the specified 
 * location, and do something (something specified
 * by the fileAction and dirAction function pointers).
 *
 * Unfortunatly, while nftw(3) could replace this and reduce 
 * code size a bit, nftw() wasn't supported before GNU libc 2.1, 
 * and so isn't sufficiently portable to take over since glibc2.1
 * is so stinking huge.
 *
 * Code written by Erik Andersen <andersee@debian.org> and
 * swiped from busybox.
 */
int
recursiveAction(const char *fileName, int recurse, int followLinks, int depthFirst,
		int (*fileAction) (const char *fileName, struct stat* statbuf),
		int (*dirAction) (const char *fileName, struct stat* statbuf))
{
    int status;
    struct stat statbuf, statbuf1;
    struct dirent *next;

    if (followLinks == TRUE)
	status = stat(fileName, &statbuf);
    else
	status = lstat(fileName, &statbuf);

    if (status < 0) {
	perror(fileName);
	return (FALSE);
    }

    if ( (followLinks == FALSE) && (S_ISLNK(statbuf.st_mode)) ) {
	if (fileAction == NULL)
	    return (TRUE);
	else
	    return (fileAction(fileName, &statbuf));
    }

    if (recurse == FALSE) {
	if (S_ISDIR(statbuf.st_mode)) {
	    if (dirAction != NULL)
		return (dirAction(fileName, &statbuf));
	    else
		return (TRUE);
	} 
    }
    
    status = lstat(fileName, &statbuf1);
    if (status < 0) {
	perror(fileName);
	return (FALSE);
    }

    if (S_ISDIR(statbuf.st_mode) && S_ISDIR(statbuf1.st_mode)) {
	DIR *dir;
	dir = opendir(fileName);
	if (!dir) {
	    perror(fileName);
	    return (FALSE);
	}
	if (dirAction != NULL && depthFirst == FALSE) {
	    status = dirAction(fileName, &statbuf);
	    if (status == FALSE) {
		perror(fileName);
		return (FALSE);
	    }
	}
	while ((next = readdir(dir)) != NULL) {
	    char nextFile[NAME_MAX];
	    if ((strcmp(next->d_name, "..") == 0)
		|| (strcmp(next->d_name, ".") == 0)) {
		continue;
	    }
	    sprintf(nextFile, "%s/%s", fileName, next->d_name);
	    status =
		recursiveAction(nextFile, TRUE, followLinks, depthFirst, 
			fileAction, dirAction);
	    if (status < 0) {
		closedir(dir);
		return (FALSE);
	    }
	}
	status = closedir(dir);
	if (status < 0) {
	    perror(fileName);
	    return (FALSE);
	}
	if (dirAction != NULL && depthFirst == TRUE) {
	    status = dirAction(fileName, &statbuf);
	    if (status == FALSE) {
		perror(fileName);
		return (FALSE);
	    }
	}
    } else {
	if (fileAction == NULL)
	    return (TRUE);
	else
	    return (fileAction(fileName, &statbuf));
    }
    return (TRUE);
}




/*
 * Write all of the supplied buffer out to a file.
 * This does multiple writes as necessary.
 * Returns the amount written, or -1 on an error.
 */
int fullWrite(int fd, const char *buf, int len)
{
    int cc;
    int total;

    total = 0;

    while (len > 0) {
	cc = write(fd, buf, len);

	if (cc < 0)
	    return -1;

	buf += cc;
	total += cc;
	len -= cc;
    }

    return total;
}

/* Is the target an NFS mountpoint? */
int
is_nfs_partition( const char* target )
{
    struct fdisk_partition* p;
    mounted_reread();
    p = fdisk_find_partition_by_mntpoint(target);
    if ((p) && (p->type == PTYPE_NFS))
	return 1;
    return 0;
}

/* Is the root filesystem on a floppy drive (not cd or initrd) */
int is_root_a_floppy (void) {
    if (   !strcmp(InstallationRootDevice,"/dev/fd0") 
	|| !strcmp(InstallationRootDevice,"/dev/fd1") 
#ifdef SCSI_FLOPPY
	|| !strcmp(InstallationRootDevice,"/dev/sfd0") 
#endif
       ) {
	problemBox(_("You have bootstrapped the Installation Root Disk without using the RAM disk. If this is a low-memory installation, that's OK, but the installation is going to run a whole lot slower this way. If you didn't intend to boot without the RAM disk, it probably happened because you didn't have the root floppy inserted when the system first asked for it at boot time. You might want to reboot with the RAM disk if that's the case."),_("No RAM Disk?"));
    }
    return 0;
}

int no_match(const char *path, const char *name, __mode_t mode){
  int i,status;
  glob_t globbuf;
  struct stat statbuf;
  snprintf(prtbuf,PRTBUFSIZE,"%s/*/%s",path,name);
  glob( prtbuf, GLOB_NOSORT, NULL, &globbuf);
  status=1;
  if ( globbuf.gl_pathc > 0 ) {
    for (i=0;i<globbuf.gl_pathc;i++) {
      if ( (! stat(globbuf.gl_pathv[i],&statbuf) ) &&
           ( (statbuf.st_mode & S_IFMT) == mode ) )  {
        status=0;
        break;
      }
    }
  }
  globfree(&globbuf); 
  return status;
}

/* flush out the buffers */
inline int 
fdflush(const char *filename) 
{
	int	fd = open(filename, 0);
	if ( fd < 0 ) {
		return 1;
	}
	ioctl(fd, FDFLUSH, 0);
	close(fd);
	return 0;
}

/* safely add item to a list */
int addtolist(char **list, const char *element){
  char *tmp= (char*)malloc(strlen(*list)+3+strlen(element));
  if (NULL == tmp) {
    fprintf(stderr,_("Memory full!"));
    return (-1);
  }
  tmp[0]='\0';
  strcat(tmp,*list);
  strcat(tmp,", ");
  strcat(tmp,element);
  free(*list);
  *list=tmp;
  return (0);
}

#ifdef USE_LANGUAGE_CHOOSER
/* Helper functions */
size_t
strwidth (const char *what)
{
    size_t res;
    int k;
    const char *p;
    wchar_t c;

    for (res = 0, p = what ; (k = mbtowc (&c, p, MB_LEN_MAX)) > 0 ; p += k)
        res += wcwidth (c);

    return res;
}

void
wpadit (char *dest, const char *source, size_t width)
{
    wchar_t c;
    int k;
    const char *p;
    char *q;
    size_t len;

    for (len = 0, p = source, q = dest ; (k = mbtowc (&c, p, MB_LEN_MAX)) > 0 && len < width ; p += k)
    {
        len += wcwidth (c);

        q += wctomb (q, c);
    }

    if (k == 0)
        for (; len < width ; ++len)
        {
            *q++ = ' '; /* I hope this is ok */
        }

    *q = '\0';
}
#endif
