// vscreen.h                       (This is -*-c++-*-)
// Copyright 1999 Daniel Burrows
//
//  This program is free software; you can redistribute it and/or modify
//  it under the terms of the GNU General Public License as published by
//  the Free Software Foundation; either version 2 of the License, or
//  (at your option) any later version.
//
//  This program is distributed in the hope that it will be useful,
//  but WITHOUT ANY WARRANTY; without even the implied warranty of
//  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
//  GNU General Public License for more details.
//
//  You should have received a copy of the GNU General Public License
//  along with this program; see the file COPYING.  If not, write to
//  the Free Software Foundation, Inc., 59 Temple Place - Suite 330,
//  Boston, MA 02111-1307, USA.
//
//  This file defines the backbone of the IO system in aptitude, the vscreen
// class.  vscreens are rather similar to the terminals provided by
// screen(1), but instead of being full terminals they're just curses entities.
// They're meant to be used as base classes (and in fact are abstract)
//
//  A vscreen at any moment can have exactly one (1) WINDOW object associated
// with it; all output of the task using the vscreen should be placed within
// this window.  Sometimes, though, the window object is NULL; in this case,
// the process running in the vscreen may still attempt to print output, but
// that output will be silently discarded (and the routine will return
// success).  [ output is performed through the vscreen's routines rather
// than through the window's ]
//
//  Input is handled through an event-loop mechanism.  If the vscreen does most
// of its processing in another thread, it may want to use internal
// synchronization to handle this.

#ifndef VSCREEN_H
#define VSCREEN_H

#include "curses++.h"

#include <assert.h>

class vscreen;

class vscreen_widget
// This provides support for pushing widgets onto a vscreen which take over the
// keyboard input (short-circuiting the normal dispatch_char)
// ONLY input support is provided -- policy as to location on the screen,
// drawing order, etc, is handled by subclasses.
{
  vscreen *owner;
protected:
  vscreen *get_owner() {return owner;}
public:
  vscreen_widget(vscreen *_owner):owner(_owner) {}
  virtual bool dispatch_char(chtype ch) {return false;}
  // Returns true if it consumed the character (otherwise, it'll be passed on)
  virtual ~vscreen_widget() {}
};

class vscreen
{
  cwindow win;

  bool use_nodelay;
  bool use_leaveok;
protected:
  bool winavail() {return win;}

  void nodelay(bool bf) {use_nodelay=bf; if(win) win.nodelay(bf);}

  bool get_nodelay() {return use_nodelay;}
public:
  vscreen();

  void syncup() {if(win) win.syncup();}
  int syncok(bool bf) {if(win) return win.syncok(bf); else return 0;}
  void cursyncup() {if(win) win.cursyncup();}
  void syncdown() {if(win) win.syncdown();}

  int scroll(int n=1) {return win?win.scroll(n):0;}

  int leaveok(bool bf) {use_leaveok=bf; return win?win.leaveok(bf):0;}

  int addch(chtype ch) {return win?win.addch(ch):0;}
  int mvaddch(int y, int x, chtype ch) {return win?win.mvaddch(y,x,ch):0;}
  int addstr(const char *str) {return win?win.addstr(str):0;}
  int addnstr(const char *str, int n) {return win?win.addnstr(str, n):0;}
  int mvaddstr(int y, int x, const char *str) {return win?win.mvaddstr(y, x, str):0;}
  int mvaddnstr(int y, int x, const char *str, int n) {return win?win.mvaddnstr(y, x, str, n):0;}

  int attroff(int attrs) {return win?win.attroff(attrs):0;}
  int attron(int attrs) {return win?win.attron(attrs):0;}
  int attrset(int attrs) {return win?win.attrset(attrs):0;}

  void bkgdset(const chtype ch) {if(win) win.bkgdset(ch);}
  int bkgd(const chtype ch) {return win?win.bkgd(ch):0;}
  chtype getbkgd() {return win?win.getbkgd():0;}

  int border(chtype ls, chtype rs, chtype ts, chtype bs, chtype tl, chtype tr, chtype bl, chtype br)
    {return win?win.border(ls,rs,ts,bs,tl,tr,bl,br):0;}
  int box(chtype verch, chtype horch) {return win?win.box(verch,horch):0;}
  int hline(chtype ch, int n) {return win?win.hline(ch,n):0;}
  int vline(chtype ch, int n) {return win?win.vline(ch,n):0;}
  int mvhline(int y, int x, chtype ch, int n) {return win?win.mvhline(y, x, ch, n):0;}
  int mvvline(int y, int x, chtype ch, int n) {return win?win.mvvline(y,x,ch,n):0;}

  int delch() {return win?win.delch():0;}
  int mvdelch(int y, int x) {return win?win.mvdelch(y, x):0;}

  int deleteln() {return win?win.deleteln():0;}
  int insdelln(int n) {return win?win.insdelln(n):0;}
  int insertln() {return win?win.insertln():0;}

  int echochar(chtype ch) {return win?win.echochar(ch):0;}

  int move(int y, int x) {return win?win.move(y,x):0;}
  void getyx(int &y, int &x) {if(win) win.getyx(y,x); else y=x=0;}
  void getbegyx(int &y, int &x) {if(win) win.getbegyx(y,x); else y=x=0;}
  void getmaxyx(int &y, int &x) {if(win) win.getmaxyx(y,x); else y=x=0;}

  void show_string_as_progbar(int x, int y, string s, int attr1, int attr2, int size1, int totalsize) {if(win) win.show_string_as_progbar(x, y, s, attr1, attr2, size1, totalsize);}

  void display_header(string s, unsigned int attr) {if(win) win.display_header(s, attr);}
  void display_status(string s, unsigned int attr) {if(win) win.display_status(s, attr);}

  int refresh() {return win?win.refresh():0;}
  int noutrefresh() {return win?win.noutrefresh():0;}
  // FIXME: does this make sense?  Should the wrapper be more intelligent?

  int touch() {return win?win.touch():0;}
  int untouch() {return win?win.untouch():0;}
  int touchln(int y, int n, int changed) {return win?win.touchln(y,n,changed):0;}
  int touchline(int start, int count) {return win?win.touchline(start,count):0;}
  int untouchline(int start, int count) {return win?win.untouchline(start,count):0;}

  int erase() {return win?win.erase():0;}
  int clear() {return win?win.clear():0;}
  int clrtobot() {return win?win.clrtobot():0;}
  int clrtoeol() {return win?win.clrtoeol():0;}

  // FIXME: we should preserve these settings ourselves and restore them on
  // set_win().  ?
  int keypad(bool bf) {return win?win.keypad(bf):0;}
  int meta(bool bf) {return win?win.meta(bf):0;}

  int clearok(bool bf) {return win?win.clearok(bf):0;}
  int idlok(bool bf) {return win?win.idlok(bf):0;}
  void idcok(bool bf) {if(win) win.idcok(bf);}
  void immedok(bool bf) {if(win) win.immedok(bf);}
  int setscrreg(int top, int bot) {return win?win.setscrreg(top,bot):0;}
  int scrollok(bool bf) {return win?win.scrollok(bf):0;}

  bool enclose(int y, int x) {return win?win.enclose(y,x):0;}

  cwindow set_win(cwindow _win)
    {
      cwindow tmp=win; win=_win;
      if(win)
	{
	  win.nodelay(use_nodelay);
	  win.leaveok(use_leaveok);
	  repaint();
	}
      return tmp;
    }

  virtual vscreen_widget *get_focus() {return NULL;}
  // Returns where the focus is or NULL if nothing has it (thus allowing the
  // vscreen to catch keystrokes)
  virtual void dispatch_char(chtype ch) {}
  // Subclasses should override this to handle character input.

  virtual void repaint()=0;
  // This routine is called whenever a new cwindow is assigned to this

  virtual ~vscreen() {if(win && win!=rootwin) assert(win.delwin()!=ERR);}

  friend void vscreen_mainloop(int synch=0);
};

void init_vscreen();
// Performs initialization tasks (including calling init_curses())

void vscreen_mainloop(int synch=0);
// Enters a loop, calling getch() over and over and over again..
// A valid vscreen must be currently displayed.

vscreen *vscreen_show(vscreen *newscreen);
// Displays the given vscreen instead of the current one.  Note that the
// available vscreens aren't managed in any way aside from simply hiding one
// and displaying another -- if some sort of circular-list mechanism is
// required, the user should handle this.
//
// Returns the previously displayed vscreen.

void vscreen_preparedelete(vscreen *victim);
// Prepares a screen for deletion.  When the current event-handler exits, this
// screen will be deleted.  Note that deleting the visible screen is not so
// bright; you should display another screen first.

void vscreen_exitmain();
// Exits the main loop.

void vscreen_handleresize();
// Does anything needed to handle a window resize event.  Normally only called
// from an internal signal handler, but at least one vscreen also needs this.

void vscreen_suspend();
// Hides all virtual screens and suspends Curses operation until vscreen_resume
// is called.

void vscreen_resume();

void vscreen_refresh();
// Redraws the screen from scratch.

#endif
