// download_screen.cc
//
//  Copyright 1999 Daniel Burrows
//
//  This program is free software; you can redistribute it and/or modify
//  it under the terms of the GNU General Public License as published by
//  the Free Software Foundation; either version 2 of the License, or
//  (at your option) any later version.
//
//  This program is distributed in the hope that it will be useful,
//  but WITHOUT ANY WARRANTY; without even the implied warranty of
//  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
//  GNU General Public License for more details.
//
//  You should have received a copy of the GNU General Public License
//  along with this program; see the file COPYING.  If not, write to
//  the Free Software Foundation, Inc., 59 Temple Place - Suite 330,
//  Boston, MA 02111-1307, USA.
//
//  Handles a download by acting as a progress meter.

#include "download_screen.h"

#include <assert.h>
#include <signal.h>

#include <apt-pkg/configuration.h>
#include <apt-pkg/error.h>
#include <apt-pkg/strutl.h>
#include <apt-pkg/acquire-worker.h>
#include <apt-pkg/dpkgpm.h>
#include <apt-pkg/packagemanager.h>

#include "apt.h"
#include "download_item.h"

#define CRITICAL_ENTER	\
  sigset_t ___signals,___oldset;\
  sigemptyset(&___signals);	\
  sigaddset(&___signals, SIGWINCH);\
  sigprocmask(SIG_BLOCK, &___signals, &___oldset);\
  assert(!sigismember(&___oldset, SIGWINCH));

#define CRITICAL_EXIT	\
  sigprocmask(SIG_SETMASK, &___oldset, &___signals);\
  assert(sigismember(&___signals, SIGWINCH));

void download_screen::IMSHit(pkgAcquire::ItemDesc &itmdesc)
{
  CRITICAL_ENTER

  downloadmap::iterator found=active_items.find(itmdesc.Owner);

  if(found==active_items.end())
    {
      download_item *newitm=new download_item(itmdesc);
      newitm->download_done(true);
      contents->add_child(newitm);
      sync_bounds();
    }
  else
    found->second->download_done(true);
  repaint();

  CRITICAL_EXIT
}

void download_screen::Fetch(pkgAcquire::ItemDesc &itmdesc)
{
  CRITICAL_ENTER
  downloadmap::iterator found=active_items.find(itmdesc.Owner);

  if(found==active_items.end())
    {
      download_item *newitm=new download_item(itmdesc);
      active_items[itmdesc.Owner]=newitm;
      contents->add_child(newitm);
      sync_bounds();
    }

  repaint();
  CRITICAL_EXIT
}

void download_screen::Done(pkgAcquire::ItemDesc &itmdesc)
{
  CRITICAL_ENTER
  download_item *itm=get_itm(itmdesc);

  itm->download_done(false);
  itm->set_worker(NULL);
  repaint();
  CRITICAL_EXIT
}

void download_screen::Fail(pkgAcquire::ItemDesc &itmdesc)
{
  CRITICAL_ENTER
  downloadmap::iterator found=active_items.find(itmdesc.Owner);
  if(found!=active_items.end())
    found->second->set_worker(NULL);

  // Nothing really to do??
  repaint();
  CRITICAL_EXIT
}

bool download_screen::Pulse(pkgAcquire *Owner)
{
  CRITICAL_ENTER
  pkgAcquireStatus::Pulse(Owner);

  for(pkgAcquire::Worker *i=Owner->WorkersBegin(); i; i=Owner->WorkerStep(i))
    {
      if(i->CurrentItem)
	get_itm(*i->CurrentItem)->set_worker(i);
    }

  bool old_delaystate=get_nodelay();
  chtype ch;
  nodelay(true);

  while((ch=getch())!=(chtype) ERR)
    {
      if(global_bindings.key_matches(ch, "ExitScreen"))
	{
	  CRITICAL_EXIT
	    return false;
	}
      else
	vs_tree::dispatch_char(ch);
    }

  nodelay(old_delaystate);

  repaint();
  CRITICAL_EXIT
  return true;
}

void download_screen::Start()
{
  CRITICAL_ENTER
  pkgAcquireStatus::Start();
  CRITICAL_EXIT
}

void download_screen::Stop()
{
  CRITICAL_ENTER
  pkgAcquireStatus::Stop();

  finished=true;

  set_status("Downloaded "+SizeToStr(FetchedBytes)+"B in "+TimeToStr(ElapsedTime)+" ("+SizeToStr(CurrentCPS)+"B/s).  Press any key to continue...");
  repaint();
  refresh();
  CRITICAL_EXIT

  getch();
}

void download_screen::paint_status()
{
  if(finished)
    vs_tree::paint_status();
  else
    {
      int width,height;
      getmaxyx(height,width);
      string output="";

      output+="Total Progress: ";

      int barsize=0;

      if((TotalBytes+TotalItems)>0)
	{
	  char progress_string[50]; // More'n enough chars.
	  unsigned long ETA=(unsigned long)((TotalBytes-CurrentBytes)/CurrentCPS);
	  // Straight from acqprogress.cc, that is..

	  barsize=int((double(width*(CurrentBytes+CurrentItems)))/(TotalBytes+TotalItems));
	  if(barsize>width)
	    barsize=width;

	  attrset(COLOR_PAIR(get_status_color())|A_BOLD);
	  snprintf(progress_string, 50, " [ %i%% ] (%sB/s, %s remaining)", int(double((100.0*(CurrentBytes+CurrentItems)))/(TotalBytes+TotalItems)), SizeToStr(CurrentCPS).c_str(), TimeToStr(ETA).c_str());
	  output+=progress_string;
	}

      show_string_as_progbar(0,
			     height-1,
			     output,
			     COLOR_PAIR(get_color("download_progress", COLOR_BLUE, COLOR_YELLOW)),
			     COLOR_PAIR(get_status_color())|A_BOLD,
			     barsize,
			     width);
    }
}

download_screen::~download_screen()
{
}

void do_pkglist_update(OpProgress *load_progress)
  // I made heavy reference here to apt-get and console-apt's code..
{
  pkgSourceList src_list;
  // Read the source list

  if(!(*apt_cache_file)->save_selection_list(*load_progress))
    return;

  if(src_list.ReadMainList()==false)
    // FIXME: show an error message
    return;

  // Lock the list directory
  FileFd lock;
  if(_config->FindB("Debug::NoLocking", false)==false)
    {
      lock.Fd(GetLock(_config->FindDir("Dir::States::Lists")+"lock"));
      if(_error->PendingError()==true)
	// FIXME: show an error message
	return;
    }

  // Create and initialize the download screen.
  download_screen *ui=new download_screen;
  ui->set_header("Updating package lists");
  vscreen *prev=vscreen_show(ui);
  pkgAcquire fetcher(ui);
  for(pkgSourceList::const_iterator i=src_list.begin(); i!=src_list.end(); i++)
    {
      new pkgAcqIndex(&fetcher, i);
      if(_error->PendingError()==true)
	{
	  // FIXME: show an error message
	  // FIXME: will this deallocate all the workers and stuff?
	  vscreen_show(prev);
	  vscreen_preparedelete(ui);
	  return;
	}
    }

  sigset_t signals,oldsigs;
  sigemptyset(&signals);
  sigaddset(&signals, SIGWINCH);
  sigprocmask(SIG_UNBLOCK, &signals, &oldsigs);

  switch(fetcher.Run())
    // Uhh, is this really the best way to do this?
    {
    case pkgAcquire::Failed:
    case pkgAcquire::Cancelled:
      sigprocmask(SIG_SETMASK, &oldsigs, &signals);
      vscreen_show(prev);
      vscreen_preparedelete(ui);
      return;
    case pkgAcquire::Continue:
      sigprocmask(SIG_SETMASK, &oldsigs, &signals);
      break;
    }

  // Clean old stuff out (?)
  if(fetcher.Clean(_config->FindDir("Dir::State::lists"))==false ||
     fetcher.Clean(_config->FindDir("Dir::State::lists")+"partial/")==false)
    {
      vscreen_show(prev);
      vscreen_preparedelete(ui);
      // FIXME: print an error message
      return;
    }

  vscreen_show(prev);
  vscreen_preparedelete(ui);
}

void do_install_run(OpProgress *load_progress)
{
  if(!(*apt_cache_file)->save_selection_list(*load_progress))
    return;

  // Lock the archive directory..
  FileFd lock;
  if(_config->FindB("Debug::NoLocking", false)==false)
    {
      lock.Fd(GetLock(_config->FindDir("Dir::States::Lists")+"lock"));
      if(_error->PendingError()==true)
	// FIXME: show an error message
	return;
    }

  // Get ready to download
  download_screen *ui=new download_screen;
  ui->set_header("Downloading and installing packages");
  vscreen *prev=vscreen_show(ui);
  pkgAcquire fetcher(ui);

  // Get source lists.
  pkgSourceList list;
  if(!list.ReadMainList())
    {
      vscreen_show(prev);
      vscreen_preparedelete(ui);

      // FIXME: show an error message
      return;
    }

  // Make a package manager, get ready to download
  pkgDPkgPM pm(*apt_cache_file);
  if(!pm.GetArchives(&fetcher, &list, apt_package_records) || _error->PendingError())
    {
      vscreen_show(prev);
      vscreen_preparedelete(ui);

      // FIXME: show an error message
      return;
    }

  // FIXME: check for free space

  // FIXME: console-apt checks for errors here.  I don't see anything that
  // could cause errors :)

  sigset_t signals,oldsigs;
  sigemptyset(&signals);
  sigaddset(&signals, SIGWINCH);
  sigprocmask(SIG_UNBLOCK, &signals, &oldsigs);

  if(fetcher.FetchNeeded()>0)
    {
      if(fetcher.Run()!=pkgAcquire::Continue)
	{
	  sigprocmask(SIG_SETMASK, &oldsigs, &signals);
	  vscreen_show(prev);
	  vscreen_preparedelete(ui);

	  return;
	}

      // This loop appears to check whether anything failed.
      bool failed=false,transient=false;
      for(pkgAcquire::Item **i=fetcher.ItemsBegin(); i!=fetcher.ItemsEnd(); i++)
	{
	  if((*i)->Status==pkgAcquire::Item::StatDone && (*i)->Complete)
	    continue;

	  if((*i)->Status==pkgAcquire::Item::StatIdle)
	    {
	      transient=true;
	      failed=true;
	      continue;
	    }

	  // FIXME: print a diagnostic message
	  failed=true;
	}
    }

  sigprocmask(SIG_SETMASK, &oldsigs, &signals);

  if(!pm.FixMissing())
    {
      ui->set_status("Unrecoverable missing packages.  Press any key to continue...");
      ui->refresh();
      getch();
      vscreen_show(prev);
      vscreen_preparedelete(ui);
      return;
    }

  vscreen_suspend();

  //  Ewww!  This is a race condition waiting to happen!
  //  Ok, that's a little strong ;-)  But there's a potential problem if
  // someone tries to lock the file in the miniscule amount of time between
  // our releasing the lock and dpkg's taking it.  Not sure we can do much
  // about this, though..
  apt_cache_file->ReleaseLock();

  // I don't like this:
  if(pm.DoInstall()!=pkgPackageManager::Completed)
    {
      _error->DumpErrors();
      cerr<<"Ack!  Something bad happened while installing packages.  Trying to recover:"<<endl;
      // and this is really a hack:
      system("dpkg --configure -a");
      _error->Discard();
    }

  cerr<<"Press return to continue.\n";
  getchar();

  vscreen_show(prev);
  vscreen_preparedelete(ui);

  vscreen_resume();
}
