// apt.cc
//
//  Copyright 1999 Daniel Burrows
//
//  This program is free software; you can redistribute it and/or modify
//  it under the terms of the GNU General Public License as published by
//  the Free Software Foundation; either version 2 of the License, or
//  (at your option) any later version.
//
//  This program is distributed in the hope that it will be useful,
//  but WITHOUT ANY WARRANTY; without even the implied warranty of
//  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
//  GNU General Public License for more details.
//
//  You should have received a copy of the GNU General Public License
//  along with this program; see the file COPYING.  If not, write to
//  the Free Software Foundation, Inc., 59 Temple Place - Suite 330,
//  Boston, MA 02111-1307, USA.
//
//  Handles basic apt bookeeping that apt-pkg doesn't :)

#include "apt.h"

#include <unistd.h>

#include <apt-pkg/sourcelist.h>
#include <apt-pkg/pkgcachegen.h>
#include <apt-pkg/error.h>
#include <apt-pkg/configuration.h>
#include <apt-pkg/init.h>
#include <apt-pkg/depcache.h>

#include <assert.h>

#include "config/keybindings.h"
#include "config/colors.h"
#include "defaults.h"

aptitudeCacheFile *apt_cache_file=NULL;
pkgRecords *apt_package_records=NULL;

string *pendingerr;

void load_colors(const char *group)
{
  Configuration::Item const *cfg_grp=_config->Tree(group);

  if(!cfg_grp)
    return;

  for(Configuration::Item const *i=cfg_grp->Child; i; i=i->Next)
    {
      if(!i->Value.empty())
	_error->Error("Invalid entry in color definition group: \"%s\"", i->Tag.c_str());
      else if(i->Tag.empty())
	_error->Error("Invalid tagless entry in color definition group", i->Value.c_str());
      else if(!i->Child || !i->Child->Next || i->Child->Next->Next)
	_error->Error("Need exactly two colors in color definition \"%s\"", i->Tag.c_str());
      else
	{
	  short fg, bg;
	  if(!parse_color(i->Child->Value, fg))
	    _error->Error("Can't parse foreground of color \"%s\"", i->Tag.c_str());
	  else if(!parse_color(i->Child->Next->Value, bg))
	    _error->Error("Can't parse background of color \"%s\"", i->Tag.c_str());
	  else
	    set_color(i->Tag, fg, bg);
	}
    }
}

void load_bindings(const char *group, keybindings *toload)
{
  if(!group)
    {
      _error->Error("Internal error: NULL group passed to load_bindings()!");
      return;
    }

  Configuration::Item const *cfg_grp=_config->Tree(group);

  assert(group!=NULL);

  if(!cfg_grp)
    return;

  for(Configuration::Item const *i=cfg_grp->Child; i; i=i->Next)
    {
      if(i->Value.empty())
	_error->Error("Invalid entry in keybinding group: \"%s\"", i->Tag.c_str());
      else if(i->Tag.empty())
	_error->Error("Invalid entry in keybinding group: \"%s\"", i->Value.c_str());
      else
	{
	  unsigned int split=0,newsplit;
	  keybinding newbinding;
	  do
	    {
	      newsplit=i->Value.find(',',split);
	      string currval(i->Value, split, newsplit-split);

	      chtype ch=parse_key(currval);

	      if(ch!=(chtype) ERR)
		newbinding.push_back(ch);
	      else
		_error->Error("Ignoring invalid keybinding \"%s\" -> \"%s\"", i->Tag.c_str(), currval.c_str());
	      split=newsplit+1;
	    } while(newsplit!=string::npos);
	  global_bindings.set(i->Tag, newbinding);
	}
    }
}

void apt_init(OpProgress *progress_bar)
{
  init_defaults();

  pkgInitialize(*_config);
  // Magic Happens.. :)

  load_bindings("Aptitude::UI::Keybindings", &global_bindings);
  load_colors("Aptitude::UI::Colors");

  if(!apt_cache_file)
    apt_reload_cache(progress_bar);
}

void apt_reload_cache(OpProgress *progress_bar)
{
  if(apt_cache_file)
    delete apt_cache_file;

  apt_cache_file=new aptitudeCacheFile;

  if(!apt_cache_file->Open(*progress_bar, (getuid()==0)) ||
     _error->PendingError())
  // Only lock the cache if we're running as root.
    {
      delete apt_cache_file;
      apt_cache_file=NULL;
      return;
    }

  if(apt_package_records)
    delete apt_package_records;

  apt_package_records=new pkgRecords(*apt_cache_file);
}

bool pending_err()
{
  if(pendingerr)
    return true;
  else if(!_error->empty())
    {
      pendingerr=new string;
      _error->PopMessage(*pendingerr);
      return true;
    }
  else
    return false;
}

string first_err()
{
  if(pending_err())
    return *pendingerr;
  else
    return "";
}

void pop_err()
{
  if(pendingerr)
    {
      delete pendingerr;
      pendingerr=NULL;
    }
  else
    _error->PopMessage(*pendingerr);
}
