/* -*- mode: C++; tab-width: 4 -*- */
/* ================================================================================== */
/* Copyright (c) 1998-1999 3Com Corporation or its subsidiaries. All rights reserved. */
/* ================================================================================== */

#include "EmulatorCommon.h"
#include "TrapPatches.h"

#include "ATraps.h"				// ATrap::DoingCall
#include "Byteswapping.h"		// Canonical
#include "CGremlinsStubs.h" 	// StubAppEnqueueKey
#include "CPU_REG.h"			// LowMem::GetEvtMgrIdle, LowMem::TrapExists, gKeyQueue, DateToDays, gNeedPostLoad
#include "DebugMgr.h"			// Debug::SendMessage
#include "ErrorHandling.h"		// Errors::SysFatalAlert
#include "Hordes.h"				// Hordes::IsOn, Hordes::PostFakeEvent, Hordes::CanSwitchToApp
#include "HostControlPrv.h" 	// HandleHostControlCall
#include "Logging.h"			// LogEvtAddEventToQueue, etc.
#include "MetaMemory.h" 		// MetaMemory mark functions
#include "Miscellaneous.h"		// SetHotSyncUserName, DateToDays, SystemCallContext
#include "Platform.h"			// Platform::GetDate
#include "PreferenceMgr.h"		// Preference (kPrefKeyUserName)
#include "Profiling.h"			// StDisableAllProfiling
#include "RAM_ROM.h"			// CEnableFullAccess
#include "EmRPC.h"				// RPC::SignalWaiters
#include "SessionFile.h"		// SessionFile
#include "SLP.h"				// SLP
#include "Startup.h"			// Startup::GetAutoLoads
#include "Strings.r.h"			// kStr_ values
#include "SystemPacket.h"		// SystemPacket::SendMessage
#include "UAE_Utils.h"			// uae_memmove, uae_memset




// ======================================================================
// Patches for system functions.
// ======================================================================

class SysHeadpatch
{
	public:
		static CallROMType		RecordTrapNumber		(void); // EvtGetEvent & EvtGetPen

		static CallROMType		DbgMessage				(void);
		static CallROMType		DmInit					(void);
		static CallROMType		ErrDisplayFileLineMsg	(void);
		static CallROMType		EvtAddEventToQueue		(void);
		static CallROMType		EvtAddUniqueEventToQueue(void);
		static CallROMType		EvtEnqueueKey			(void);
		static CallROMType		EvtEnqueuePenPoint		(void);
		static CallROMType		FrmDrawForm				(void);
		static CallROMType		HostControl 			(void);
		static CallROMType		HwrBatteryLevel 		(void);
		static CallROMType		HwrDockStatus			(void);
		static CallROMType		HwrDoze					(void);
		static CallROMType		HwrGetROMToken			(void);
		static CallROMType		HwrSleep				(void);
		static CallROMType		KeyCurrentState 		(void);
		static CallROMType		PenOpen 				(void);
		static CallROMType		SysAppExit				(void);
		static CallROMType		SysAppLaunch			(void);
		static CallROMType		SysBinarySearch 		(void);
		static CallROMType		SysEvGroupWait			(void);
		static CallROMType		SysFatalAlert			(void);
		static CallROMType		SysLaunchConsole		(void);
		static CallROMType		SysReset				(void);
		static CallROMType		SysSemaphoreWait		(void);
		static CallROMType		SysUIAppSwitch			(void);
		static CallROMType		SysUIBusy				(void);
		static CallROMType		TimInit 				(void);
};


class SysTailpatch
{
	public:
		static void 	DmGetNextDatabaseByTypeCreator	(void);
		static void 	EvtGetEvent 			(void);
		static void 	EvtGetPen				(void);
		static void 	EvtGetSysEvent			(void);
		static void 	FtrInit 				(void);
		static void 	HwrMemReadable			(void);
		static void 	HwrSleep				(void);
		static void 	SysAppStartup			(void);
		static void 	SysBinarySearch 		(void);
		static void 	TimInit 				(void);
		static void 	UIInitialize			(void);
};


// ======================================================================
//	Proto patch table for the system functions.  This array will be used
//	to create a sparse array at runtime.
// ======================================================================

ProtoPatchTableEntry	gProtoSysPatchTable[] =
{
	{sysTrapDbgMessage, 			SysHeadpatch::DbgMessage,				NULL},
//	sysTrapDmGetNextDatabaseByTypeCreator,	NULL,							SysTailpatch::DmGetNextDatabaseByTypeCreator,
	{sysTrapDmInit, 				SysHeadpatch::DmInit,					NULL},
	{sysTrapErrDisplayFileLineMsg,	SysHeadpatch::ErrDisplayFileLineMsg,	NULL},
	{sysTrapEvtAddEventToQueue, 	SysHeadpatch::EvtAddEventToQueue,		NULL},
	{sysTrapEvtAddUniqueEventToQueue,SysHeadpatch::EvtAddUniqueEventToQueue,	NULL},
	{sysTrapEvtEnqueueKey,			SysHeadpatch::EvtEnqueueKey,			NULL},
	{sysTrapEvtEnqueuePenPoint, 	SysHeadpatch::EvtEnqueuePenPoint,		NULL},
	{sysTrapEvtGetEvent,			SysHeadpatch::RecordTrapNumber, 		SysTailpatch::EvtGetEvent},
	{sysTrapEvtGetPen,				SysHeadpatch::RecordTrapNumber, 		SysTailpatch::EvtGetPen},
	{sysTrapEvtGetSysEvent, 		NULL,									SysTailpatch::EvtGetSysEvent},
	{sysTrapFrmDrawForm,			SysHeadpatch::FrmDrawForm,				NULL},
	{sysTrapFtrInit,				NULL,									SysTailpatch::FtrInit},
	{sysTrapHostControl,			SysHeadpatch::HostControl,				NULL},
	{sysTrapHwrBatteryLevel,		SysHeadpatch::HwrBatteryLevel,			NULL},
	{sysTrapHwrDockStatus,			SysHeadpatch::HwrDockStatus,			NULL},
	{sysTrapHwrGetROMToken, 		SysHeadpatch::HwrGetROMToken,			NULL},
	{sysTrapHwrDoze,				SysHeadpatch::HwrDoze,					NULL},
	{sysTrapHwrMemReadable, 		NULL,									SysTailpatch::HwrMemReadable},
	{sysTrapHwrSleep,				SysHeadpatch::HwrSleep, 				SysTailpatch::HwrSleep},
	{sysTrapKeyCurrentState,		SysHeadpatch::KeyCurrentState,			NULL},
	{sysTrapPenOpen,				SysHeadpatch::PenOpen,					NULL},
	{sysTrapSysAppExit, 			SysHeadpatch::SysAppExit,				NULL},
	{sysTrapSysAppLaunch,			SysHeadpatch::SysAppLaunch, 			NULL},
	{sysTrapSysAppStartup,			NULL,									SysTailpatch::SysAppStartup},
	{sysTrapSysBinarySearch,		SysHeadpatch::SysBinarySearch,			SysTailpatch::SysBinarySearch},
	{sysTrapSysEvGroupWait, 		SysHeadpatch::SysEvGroupWait,			NULL},
	{sysTrapSysFatalAlert,			SysHeadpatch::SysFatalAlert,			NULL},
	{sysTrapSysLaunchConsole,		SysHeadpatch::SysLaunchConsole, 		NULL},
	{sysTrapSysReset,				SysHeadpatch::SysReset,			 		NULL},
	{sysTrapSysSemaphoreWait,		SysHeadpatch::SysSemaphoreWait, 		NULL},
	{sysTrapSysUIAppSwitch, 		SysHeadpatch::SysUIAppSwitch,			NULL},
	{sysTrapSysUIBusy,				SysHeadpatch::SysUIBusy,				NULL},
	{sysTrapTimInit,				SysHeadpatch::TimInit,					SysTailpatch::TimInit},
	{sysTrapUIInitialize,			NULL,									SysTailpatch::UIInitialize},

	{0, 							NULL,									NULL}
};


// ======================================================================
//	Patches for HtalLib functions
// ======================================================================

class HtalLibHeadpatch
{
	public:
		static CallROMType		HtalLibSendReply			(void);
};


#pragma mark -

// ===========================================================================
//		 ModulePatchTable
// ===========================================================================
//	A simple class for managing patches on a module.  A "module" is defined
//	a the main set of system functions (dispatch numbers 0xA000 - 0xA7FFF) or
//	a library (dispatch numbers 0xA800 - 0xAFFF, with a unique refnum to
//	select which library).

class ModulePatchTable
{
	public:
		void Clear ();

		void AddProtoPatchTable (ProtoPatchTableEntry protoPatchTable[]);

		// B.S. operators for VC++ so that we can put objects of
		// this class into STL collections.

		bool operator < (const ModulePatchTable& other) const
		{
			return this < &other;
		}

		bool operator == (const ModulePatchTable& other) const
		{
			return this == &other;
		}

		// Return the patch function for the given module function.  The given
		// module function *must* be given as a zero-based index.  If there is
		// no patch function for the modeule function, return NULL.

		HeadpatchProc GetHeadpatch (uae_u16 index) const
		{
			if (index < fHeadpatches.size ())
			{
				return fHeadpatches[index];
			}

			return NULL;
		}

		TailpatchProc GetTailpatch (uae_u16 index) const
		{
			if (index < fTailpatches.size ())
			{
				return fTailpatches[index];
			}

			return NULL;
		}

	private:
		vector<HeadpatchProc>	fHeadpatches;
		vector<TailpatchProc>	fTailpatches;
};


void ModulePatchTable::Clear (void)
{
	fHeadpatches.clear ();
	fTailpatches.clear ();
}


void ModulePatchTable::AddProtoPatchTable (ProtoPatchTableEntry protoPatchTable[])
{
	// Create a fast dispatch table for the managed module.  A "fast
	// dispatch table" is a table with a headpatch and tailpatch entry
	// for each possible function in the module.  If the function is
	// not head or tailpatched, the corresponding entry is NULL.  When
	// a patch function is needed, the trap dispatch number is used as
	// an index into the table in order to get the right patch function.
	//
	// For simplicity, "fast patch tables" are created from "proto patch
	// tables".  A proto patch table is a compact table containing the
	// information needed to create a fast patch table.  Each entry in
	// the proto patch table is a trap-number/headpatch/tailpatch tupple.
	// Each tuple is examined in turn.	If there is a head or tail patch
	// function for the indicated module function, that patch function
	// is entered in the fast dispatch table, using the trap number as
	// the index.

	for (long ii = 0; protoPatchTable[ii].fTrapWord; ++ii)
	{
		// If there is a headpatch function...

		if (protoPatchTable[ii].fHeadpatch)
		{
			// Get the trap number.

			uae_u16 index = SysTrapIndex (protoPatchTable[ii].fTrapWord);

			// If the trap number is 0xA800-based, make it zero based.

			if (IsLibraryTrap (index))
				index -= SysTrapIndex (sysLibTrapBase);

			// Resize the fast patch table, if necessary.

			if (index >= fHeadpatches.size ())
			{
				fHeadpatches.resize (index + 1);
			}

			// Add the headpatch function.

			fHeadpatches[index] = protoPatchTable[ii].fHeadpatch;
		}

		// If there is a tailpatch function...

		if (protoPatchTable[ii].fTailpatch)
		{
			// Get the trap number.

			uae_u16 index = SysTrapIndex (protoPatchTable[ii].fTrapWord);

			// If the trap number is 0xA800-based, make it zero based.

			if (IsLibraryTrap (index))
				index -= SysTrapIndex (sysLibTrapBase);

			// Resize the fast patch table, if necessary.

			if (index >= fTailpatches.size ())
			{
				fTailpatches.resize (index + 1);
			}

			// Add the tailpatch function.

			fTailpatches[index] = protoPatchTable[ii].fTailpatch;
		}
	}
}


// ===========================================================================
//		 TailpatchType
// ===========================================================================
// Structure used to hold tail-patch information.

struct TailpatchType
{
	SystemCallContext	fContext;
	uae_s32 			fCount;
	TailpatchProc		fTailpatch;

	// I hate VC++...really I do...
	bool operator< (const TailpatchType&) const {return false;}
	bool operator> (const TailpatchType&) const {return false;}
	bool operator== (const TailpatchType&) const {return false;}
	bool operator!= (const TailpatchType&) const {return false;}
};


// ===========================================================================
//		 Patches
// ===========================================================================

// ======================================================================
//	Globals and constants
// ======================================================================

const UInt							kMagicRefNum		= 0x666;	// See comments in HtalLibSendReply.
const ModulePatchTable* 			kNoPatchTable		= (ModulePatchTable*) -1;

static bool 						gUIInitialized;
static bool 						gHeapInitialized;
static bool							gEvtGetEventCalled;
static bool							gHaveJapanese;
static long 						gSysBinarySearchCount;

extern long 						gMemMgrCount;
extern long 						gMemSemaphoreCount;
extern unsigned long				gMemSemaphoreReserveTime;
extern ULong						gResizeOrigSize;
extern UInt 						gHeapID;

static UInt 						gNextAppCardNo;
static LocalID						gNextAppDbID;
static QuitStage					gQuitStage;

static EmuAppInfoList				gCurAppInfo;

static ModulePatchTable 			gSysPatchTable;
static ModulePatchTable 			gNetLibPatchTable;

typedef vector<const ModulePatchTable*> PatchTableType;
static PatchTableType				gLibPatches;

typedef vector<TailpatchType>		TailpatchTableType;
static TailpatchTableType			gInstalledTailpatches;

static uae_u16						gLastEvtTrap;
static DWord						gOSVersion;

static const DWord					kOSUndeterminedVersion = ~0;


#if defined (__MACOS__)
extern Bool 		gProhibitMemoryAllocation;
#endif


// ======================================================================
//	Private functions
// ======================================================================

static string		PrvToString 		(uaecptr s);
static void 		PrvAutoload 		(void);
static void 		PrvSetCurrentDate	(void);


// ========================================================================
// Macros for extracting parameters from the emulated stack
// Note that use of these has been superceded by the PARAM_FOO
// macros;	we should eventually move over to those macros.
// ========================================================================

#define PARAMETER_SIZE(x)	\
	(sizeof (((StackFrame*) 0)->x))

#define PARAMETER_OFFSET(x) \
	(m68k_areg (regs, 7) + offsetof (StackFrame, x))

#define GET_PARAMETER(x)		\
	((PARAMETER_SIZE(x) == sizeof (char)) ? get_byte (PARAMETER_OFFSET(x)) :	\
	 (PARAMETER_SIZE(x) == sizeof (short)) ? get_word (PARAMETER_OFFSET(x)) :	\
											get_long (PARAMETER_OFFSET(x)))

#define SET_PARAMETER(x, v) 	\
	((PARAMETER_SIZE(x) == sizeof (char)) ? put_byte (PARAMETER_OFFSET(x), v) : \
	 (PARAMETER_SIZE(x) == sizeof (short)) ? put_word (PARAMETER_OFFSET(x), v) :	\
											put_long (PARAMETER_OFFSET(x), v))


// ========================================================================
// The following functions define a bunch of StackFrame structs.
// These structs need to mirror the format of parameters pushed
// onto the stack by the emulated code, and so need to be packed
// to 2-byte boundaries.
//
// The pragmas are reversed at the end of the file.
// ========================================================================

#include "PalmPack.h"


/***********************************************************************
 *
 * FUNCTION:	Patches::Initialize
 *
 * DESCRIPTION: Standard initialization function.  Responsible for
 *				initializing this sub-system when a new session is
 *				created.  May also be called from the Load function
 *				to share common functionality.
 *
 * PARAMETERS:	none
 *
 * RETURNED:	nothing
 *
 ***********************************************************************/
void Patches::Initialize (void)
{
	gSysPatchTable.Clear ();
	gSysPatchTable.AddProtoPatchTable (gProtoSysPatchTable);
	gSysPatchTable.AddProtoPatchTable (gProtoMemMgrPatchTable);

	gNetLibPatchTable.Clear ();
	gNetLibPatchTable.AddProtoPatchTable (gProtoNetLibPatchTable);
}


/***********************************************************************
 *
 * FUNCTION:	Patches::Reset
 *
 * DESCRIPTION: Standard reset function.  Sets the sub-system to a
 *				default state.	This occurs not only on a Reset (as
 *				from the menu item), but also when the sub-system
 *				is first initialized (Reset is called after Initialize)
 *				as well as when the system is re-loaded from an
 *				insufficient session file.
 *
 * PARAMETERS:	none
 *
 * RETURNED:	nothing
 *
 ***********************************************************************/

void Patches::Reset (void)
{
	gLastEvtTrap = 0;
	gOSVersion = kOSUndeterminedVersion;

	gInstalledTailpatches.clear ();

	gUIInitialized				= false;
	gHeapInitialized			= false;
	gEvtGetEventCalled			= false;
	gHaveJapanese				= false;
	gSysBinarySearchCount		= 0;
	gMemMgrCount				= 0;
	gMemSemaphoreCount			= 0;
	gMemSemaphoreReserveTime	= 0;
	gResizeOrigSize 			= 0;
	gHeapID 					= 0;

	Patches::SetSwitchApp (0, 0);
	Patches::QuitOnAppExit (false);

	gLibPatches.clear ();

	// Clear out everything we know about the current applications.

	gCurAppInfo.clear ();
}


/***********************************************************************
 *
 * FUNCTION:	Patches::Save
 *
 * DESCRIPTION: Standard save function.  Saves any sub-system state to
 *				the given session file.
 *
 * PARAMETERS:	none
 *
 * RETURNED:	nothing
 *
 ***********************************************************************/

void Patches::Save (SessionFile& f)
{
	const long	kCurrentVersion = 2;

	Chunk		chunk;
	ChunkStream s (chunk);

	s << kCurrentVersion;

	s << gUIInitialized;
	s << gHeapInitialized;
	s << gSysBinarySearchCount;

	s << gMemMgrCount;
	s << gMemSemaphoreCount;
	s << gMemSemaphoreReserveTime;
	s << gResizeOrigSize;
	s << gHeapID;

	s << gNextAppCardNo;
	s << gNextAppDbID;
	s << (long) gQuitStage;

	s << (long) gCurAppInfo.size ();
	EmuAppInfoList::iterator	iter1;
	for (iter1 = gCurAppInfo.begin (); iter1 != gCurAppInfo.end (); ++iter1)
	{
		s << iter1->fCmd;
		s << (long) iter1->fDB;
		s << iter1->fCardNo;
		s << iter1->fDBID;
		s << iter1->fMemOwnerID;
		s << iter1->fStackP;
		s << iter1->fStackEndP;
		s << iter1->fStackSize;
		s << iter1->fName;
		s << iter1->fVersion;
	}

//	s << gSysPatchTable;
//	s << gNetLibPatchTable;

//	s << gLibPatches;

	s << (long) gInstalledTailpatches.size ();
	TailpatchTableType::iterator	iter2;
	for (iter2 = gInstalledTailpatches.begin (); iter2 != gInstalledTailpatches.end (); ++iter2)
	{
		s << iter2->fContext.fDestPC;
		s << iter2->fContext.fExtra;
		s << iter2->fContext.fNextPC;
		s << iter2->fContext.fPC;
		s << iter2->fContext.fTrapIndex;
		s << iter2->fContext.fTrapWord;
		s << iter2->fCount;
//		s << iter2->fTailpatch; // Patched up in ::Load
	}

	s << gLastEvtTrap;
	s << gOSVersion;

	// Added in version 2.

	s << gHaveJapanese;

	f.WritePatchInfo (chunk);
}


/***********************************************************************
 *
 * FUNCTION:	Patches::Load
 *
 * DESCRIPTION: Standard load function.  Loads any sub-system state
 *				from the given session file.
 *
 * PARAMETERS:	none
 *
 * RETURNED:	nothing
 *
 ***********************************************************************/

void Patches::Load (SessionFile& f)
{
	Chunk	chunk;
	if (f.ReadPatchInfo (chunk))
	{
		long		temp;
		long		version;
		ChunkStream s (chunk);

		s >> version;

		if (version >= 1)
		{
			s >> gUIInitialized;
			s >> gHeapInitialized;
			gEvtGetEventCalled = false;
			s >> gSysBinarySearchCount;

			s >> gMemMgrCount;
			s >> gMemSemaphoreCount;
			s >> gMemSemaphoreReserveTime;
			s >> gResizeOrigSize;
			s >> gHeapID;

			s >> gNextAppCardNo;
			s >> gNextAppDbID;
			s >> temp;	gQuitStage = (QuitStage) temp;

			long	numApps;
			s >> numApps;

			long	ii;
			for (ii = 0; ii < numApps; ++ii)
			{
				EmuAppInfo	info;

				s >> info.fCmd;
				s >> temp;	info.fDB = (VoidPtr) temp;
				s >> info.fCardNo;
				s >> info.fDBID;
				s >> info.fMemOwnerID;
				s >> info.fStackP;
				s >> info.fStackEndP;
				s >> info.fStackSize;
				s >> info.fName;
				s >> info.fVersion;

				gCurAppInfo.push_back (info);
			}

			long	numTailpatches;
			s >> numTailpatches;

			for (ii = 0; ii < numTailpatches; ++ii)
			{
				TailpatchType	patch;

				s >> patch.fContext.fDestPC;
				s >> patch.fContext.fExtra;
				s >> patch.fContext.fNextPC;
				s >> patch.fContext.fPC;
				s >> patch.fContext.fTrapIndex;
				s >> patch.fContext.fTrapWord;
				s >> patch.fCount;

				// Patch up the tailpatch proc.

				HeadpatchProc	dummy;
				GetPatches (patch.fContext, dummy, patch.fTailpatch);

				gInstalledTailpatches.push_back (patch);
			}

			s >> gLastEvtTrap;
			s >> gOSVersion;
		}

		if (version >= 2)
		{
			s >> gHaveJapanese;
		}
		else
		{
			gHaveJapanese = false;
		}
	}
	else
	{
		f.SetCanReload (false);
	}
}


/***********************************************************************
 *
 * FUNCTION:	Patches::Dispose
 *
 * DESCRIPTION: Standard dispose function.	Completely release any
 *				resources acquired or allocated in Initialize and/or
 *				Load.
 *
 * PARAMETERS:	none
 *
 * RETURNED:	nothing
 *
 ***********************************************************************/

void Patches::Dispose (void)
{
	gUIInitialized				= false;
	gHeapInitialized			= false;
	gEvtGetEventCalled			= false;
	gHaveJapanese				= false;

	gSysPatchTable.Clear ();
	gNetLibPatchTable.Clear ();
}


/***********************************************************************
 *
 * FUNCTION:	Patches::PostLoad
 *
 * DESCRIPTION: Do some stuff that is normally taken care of during the
 *				process of resetting the device (autoloading
 *				applications, setting the device date, installing the
 *				HotSync user-name, and setting the 'gdbS' feature).
 *
 * PARAMETERS:	none
 *
 * RETURNED:	nothing
 *
 ***********************************************************************/

void Patches::PostLoad (void)
{
	if (Patches::UIInitialized ())
	{
		// If we're listening on a socket, install the 'gdbS' feature.	The
		// existance of this feature causes programs written with the prc tools
		// to enter the debugger when they're launched.

		if (Debug::ConnectedToTCPDebugger ())
		{
			FtrSet ('gdbS', 0, 0x12BEEF34);
		}
		else
		{
			FtrUnregister ('gdbS', 0);
		}

		// Install the HotSync user-name.

		Preference<string>	userName (kPrefKeyUserName);
		::SetHotSyncUserName (userName->c_str ());

		// Auto-load any files in the Autoload[Foo] directories.

		::PrvAutoload ();

		// Install the current date.

		::PrvSetCurrentDate ();

		// Wake up any current application so that they can respond
		// to events we pump in at EvtGetEvent time.

		::EvtWakeup ();
	}
}


/***********************************************************************
 *
 * FUNCTION:	Patches::GetLibPatchTable
 *
 * DESCRIPTION: .
 *
 * PARAMETERS:	none
 *
 * RETURNED:	nothing
 *
 ***********************************************************************/

const ModulePatchTable* Patches::GetLibPatchTable (uae_u16 refNum)
{
	string	libName = ::GetLibraryName (refNum);

	if (libName == "Net.lib")
	{
		return &gNetLibPatchTable;
	}

	return kNoPatchTable;
}


/***********************************************************************
 *
 * FUNCTION:	Patches::HandleSystemCall
 *
 * DESCRIPTION: If this is a trap we could possibly have head- or
 *				tail-patched, handle those cases.
 *
 * PARAMETERS:	none
 *
 * RETURNED:	nothing
 *
 ***********************************************************************/

CallROMType Patches::HandleSystemCall (const SystemCallContext& context)
{
	if (gNeedPostLoad)
	{
		gNeedPostLoad = false;
		Patches::PostLoad ();
	}

	HeadpatchProc	hp;
	TailpatchProc	tp;
	Patches::GetPatches (context, hp, tp);

	CallROMType handled = Patches::HandlePatches (context, hp, tp);

	return handled;
}


/***********************************************************************
 *
 * FUNCTION:	Patches::GetPatches
 *
 * DESCRIPTION: 
 *
 * PARAMETERS:	none
 *
 * RETURNED:	nothing
 *
 ***********************************************************************/

void Patches::GetPatches (	const SystemCallContext& context,
							HeadpatchProc& hp,
							TailpatchProc& tp)
{
	const ModulePatchTable* patchTable = NULL;

	// If this is in the system function range, check our table of
	// system function patches.

	if (::IsSystemTrap (context.fTrapWord))
	{
		patchTable = &gSysPatchTable;
	}

	// Otherwise, see if this is a call to a library function

	else
	{
		if (context.fExtra == kMagicRefNum) // See comments in HtalLibSendReply.
		{
			hp = HtalLibHeadpatch::HtalLibSendReply;
			tp = NULL;
			return;
		}

		if (context.fExtra >= gLibPatches.size ())
		{
			gLibPatches.resize (context.fExtra + 1);
		}

		patchTable = gLibPatches[context.fExtra];

		if (patchTable == NULL)
		{
			patchTable = gLibPatches[context.fExtra] = GetLibPatchTable (context.fExtra);
		}
	}

	// Now that we've got the right patch table for this module, see if
	// that patch table has head- or tailpatches for this function.

	if (patchTable != kNoPatchTable)
	{
		hp = patchTable->GetHeadpatch (context.fTrapIndex);
		tp = patchTable->GetTailpatch (context.fTrapIndex);
	}
	else
	{
		assert (patchTable == kNoPatchTable);

		hp = NULL;
		tp = NULL;
	}
}


/***********************************************************************
 *
 * FUNCTION:	Patches::HandlePatches
 *
 * DESCRIPTION: 
 *
 * PARAMETERS:	none
 *
 * RETURNED:	nothing
 *
 ***********************************************************************/

CallROMType Patches::HandlePatches (const SystemCallContext& context,
									HeadpatchProc hp,
									TailpatchProc tp)
{
	CallROMType handled = kExecuteROM;

	// First, see if we have a SysHeadpatch for this function. If so, call
	// it. If it returns true, then that means that the head patch
	// completely handled the function.

	// !!! May have to mess with PC here in case patches do something
	// to enter the debugger.

	if (hp)
	{
		handled = CallHeadpatch (hp);
	}

	// Next, see if there's a SysTailpatch function for this trap. If
	// so, install the TRAP that will cause us to regain control
	// after the trap function has executed.

	if (tp)
	{
		if (handled == kExecuteROM)
		{
			SetupForTailpatch (tp, context);
		}
		else
		{
			CallTailpatch (tp);
		}
	}

	return handled;
}


/***********************************************************************
 *
 * FUNCTION:	Patches::HandleCPUBreak
 *
 * DESCRIPTION: Handle a tail patch, if any is registered for this
 *				memory location.
 *
 * PARAMETERS:	none
 *
 * RETURNED:	nothing
 *
 ***********************************************************************/

void Patches::HandleCPUBreak (void)
{
	// Get the address of the tailpatch to call.  May return NULL if
	// there is no tailpatch for this memory location.

	TailpatchProc	tp = RecoverFromTailpatch (m68k_getpc ());

	// Call the tailpatch handler for the trap that just returned.

	CallTailpatch (tp);
}


/***********************************************************************
 *
 * FUNCTION:	Patches::InstallCPUBreaks
 *
 * DESCRIPTION: Set the MetaMemory bit that tells the CPU loop to stop
 *				when we get to the desired locations.
 *
 * PARAMETERS:	none
 *
 * RETURNED:	nothing
 *
 ***********************************************************************/

void Patches::InstallCPUBreaks (void)
{
	TailpatchTableType::iterator	iter = gInstalledTailpatches.begin ();

	while (iter != gInstalledTailpatches.end ())
	{
		MetaMemory::MarkCPUBreak (iter->fContext.fNextPC);
		++iter;
	}
}


/***********************************************************************
 *
 * FUNCTION:	Patches::RemoveCPUBreaks
 *
 * DESCRIPTION: Clear the MetaMemory bit that tells the CPU loop to stop
 *				when we get to the desired locations.
 *
 * PARAMETERS:	none
 *
 * RETURNED:	nothing
 *
 ***********************************************************************/

void Patches::RemoveCPUBreaks (void)
{
	TailpatchTableType::iterator	iter = gInstalledTailpatches.begin ();

	while (iter != gInstalledTailpatches.end ())
	{
		MetaMemory::UnmarkCPUBreak (iter->fContext.fNextPC);
		++iter;
	}
}


/***********************************************************************
 *
 * FUNCTION:	Patches::SetupForTailpatch
 *
 * DESCRIPTION: Set up the pending TRAP $F call to be tailpatched.
 *
 * PARAMETERS:	none
 *
 * RETURNED:	nothing
 *
 ***********************************************************************/

void Patches::SetupForTailpatch (TailpatchProc tp, const SystemCallContext& context)
{
	// See if this function is already tailpatched.  If so, merely increment
	// the use-count field.

	TailpatchTableType::iterator	iter = gInstalledTailpatches.begin ();

	while (iter != gInstalledTailpatches.end ())
	{
		if (iter->fContext.fNextPC == context.fNextPC)
		{
			++(iter->fCount);
			return;
		}

		++iter;
	}

	// This function is not already tailpatched, so add a new entry
	// for the the PC/opcode we want to save.

	TailpatchType	newTailpatch;

	newTailpatch.fContext	= context;
	newTailpatch.fCount 	= 1;
	newTailpatch.fTailpatch = tp;

	gInstalledTailpatches.push_back (newTailpatch);

#if defined (__MACOS__)
	// For the Mac, make sure there's always at least 10 spaces available.
	// We use these spaces in times when we can't resize the array, as
	// when we're handling debugger packets.

	if (!gProhibitMemoryAllocation)
	{
		if (gInstalledTailpatches.size () + 10 > gInstalledTailpatches.capacity ())
		{
			gInstalledTailpatches.reserve (gInstalledTailpatches.capacity () + 10);
		}
	}
#endif

	Emulator::InstallCPUBreaks ();
}


/***********************************************************************
 *
 * FUNCTION:	Patches::RecoverFromTailpatch
 *
 * DESCRIPTION: .
 *
 * PARAMETERS:	none
 *
 * RETURNED:	nothing
 *
 ***********************************************************************/

TailpatchProc Patches::RecoverFromTailpatch (uaecptr startPC)
{
	// Get the current PC so that we can find the record for this tailpatch.

	uaecptr patchPC = startPC;

	// Find the PC.

	TailpatchTableType::iterator	iter = gInstalledTailpatches.begin ();

	while (iter != gInstalledTailpatches.end ())
	{
		if (iter->fContext.fNextPC == patchPC)
		{
			TailpatchProc	result = iter->fTailpatch;

			// Decrement the use-count.  If it reaches zero, remove the
			// patch from our list.

			if (--(iter->fCount) == 0)
			{
				gInstalledTailpatches.erase (iter);

				Emulator::InstallCPUBreaks ();
			}

			return result;
		}

		++iter;
	}

	return NULL;
}


/***********************************************************************
 *
 * FUNCTION:	Patches::CallHeadpatch
 *
 * DESCRIPTION: If the given system function is head patched, then call
 *				the headpatch.	Return "handled" (which means whether
 *				or not to call the ROM function after this one).
 *
 * PARAMETERS:	none
 *
 * RETURNED:	nothing
 *
 ***********************************************************************/

CallROMType Patches::CallHeadpatch (HeadpatchProc hp)
{
	CallROMType handled = kExecuteROM;

	if (hp)
	{
		if (hp != &SysHeadpatch::HostControl)
		{
			// Stop all profiling activities. Stop cycle counting and stop the
			// recording of function entries and exits.  We want our trap patches
			// to be as transparent as possible.

			StDisableAllProfiling	stopper;

			handled = hp ();
		}
		else
		{
			handled = hp ();
		}
	}

	return handled;
}


/***********************************************************************
 *
 * FUNCTION:	Patches::CallTailpatch
 *
 * DESCRIPTION: If the given function is tail patched, then call the
 *				tailpatch.
 *
 * PARAMETERS:	none
 *
 * RETURNED:	nothing
 *
 ***********************************************************************/

void Patches::CallTailpatch (TailpatchProc tp)
{
	if (tp)
	{
		// Stop all profiling activities. Stop cycle counting and stop the
		// recording of function entries and exits.  We want our trap patches
		// to be as transparent as possible.

		StDisableAllProfiling	stopper;

		tp ();
	}
}


/***********************************************************************
 *
 * FUNCTION:	Patches::OSVersion
 *
 * DESCRIPTION: 
 *
 * PARAMETERS:	none
 *
 * RETURNED:	nothing
 *
 ***********************************************************************/

DWord Patches::OSVersion (void)
{
	assert (gOSVersion != kOSUndeterminedVersion);

	return gOSVersion;
}


/***********************************************************************
 *
 * FUNCTION:	Patches::OSMajorMinorVersion
 *
 * DESCRIPTION: 
 *
 * PARAMETERS:	none
 *
 * RETURNED:	nothing
 *
 ***********************************************************************/

DWord Patches::OSMajorMinorVersion (void)
{
	return OSMajorVersion () * 10 + OSMinorVersion ();
}


/***********************************************************************
 *
 * FUNCTION:	Patches::OSMajorVersion
 *
 * DESCRIPTION: 
 *
 * PARAMETERS:	none
 *
 * RETURNED:	nothing
 *
 ***********************************************************************/

DWord Patches::OSMajorVersion (void)
{
	assert (gOSVersion != kOSUndeterminedVersion);

	return sysGetROMVerMajor (gOSVersion);
}


/***********************************************************************
 *
 * FUNCTION:	Patches::OSMinorVersion
 *
 * DESCRIPTION: 
 *
 * PARAMETERS:	none
 *
 * RETURNED:	nothing
 *
 ***********************************************************************/

DWord Patches::OSMinorVersion (void)
{
	assert (gOSVersion != kOSUndeterminedVersion);

	return sysGetROMVerMinor (gOSVersion);
}


/***********************************************************************
 *
 * FUNCTION:	SetSwitchApp
 *
 * DESCRIPTION: Sets an application or launchable document to switch to
 *				the next time the system can manage it.
 *
 * PARAMETERS:	cardNo - the card number of the app to switch to.
 *
 *				dbID - the database id of the app to switch to.
 *
 * RETURNED:	nothing
 *
 ***********************************************************************/

void Patches::SetSwitchApp (UInt cardNo, LocalID dbID)
{
	gNextAppCardNo = cardNo;
	gNextAppDbID = dbID;
}


/***********************************************************************
 *
 * FUNCTION:	Patches::SwitchToApp
 *
 * DESCRIPTION: Switches to the given application or launchable document.
 *
 * PARAMETERS:	cardNo - the card number of the app to switch to.
 *
 *				dbID - the database id of the app to switch to.
 *
 * RETURNED:	Err number of any errors that occur
 *
 ***********************************************************************/

Err Patches::SwitchToApp (UInt cardNo, LocalID dbID)
{
	UInt	dbAttrs;
	ULong	type, creator;

	Err err = ::DmDatabaseInfo (
				cardNo,
				dbID,
				NULL,				/*name*/
				&dbAttrs,
				NULL,				/*version*/
				NULL,				/*create date*/
				NULL,				/*modDate*/
				NULL,				/*backup date*/
				NULL,				/*modNum*/
				NULL,				/*appInfoID*/
				NULL,				/*sortInfoID*/
				&type,
				&creator);

	if (err)
		return err;

	//---------------------------------------------------------------------
	// If this is an executable, call SysUIAppSwitch
	//---------------------------------------------------------------------
	if (::IsExecutable (type, creator, dbAttrs))
	{
		err = ::SysUIAppSwitch (cardNo, dbID,
						sysAppLaunchCmdNormalLaunch, NULL);

		if (err)
			return err;
	}

	//---------------------------------------------------------------------
	// else, this must be a launchable data database. Find it's owner app
	//	and launch it with a pointer to the data database name.
	//---------------------------------------------------------------------
	else
	{
		DmSearchStateType	searchState;
		UInt				appCardNo;
		LocalID 			appDbID;

		err = ::DmGetNextDatabaseByTypeCreator (true, &searchState,
						sysFileTApplication, creator,
						true, &appCardNo, &appDbID);
		if (err)
			return err;

		// Create the param block

		uaecptr cmdPBP = (uaecptr) ::MemPtrNew (sizeof (SysAppLaunchCmdOpenDBType));
		if (cmdPBP == UAE_NULL)
			return memErrNotEnoughSpace;

		// Fill it in

		::MemPtrSetOwner ((VoidPtr) cmdPBP, 0);
		put_word (cmdPBP + offsetof (SysAppLaunchCmdOpenDBType, cardNo), cardNo);
		put_long (cmdPBP + offsetof (SysAppLaunchCmdOpenDBType, dbID), dbID);

		// Switch now

		err = ::SysUIAppSwitch (appCardNo, appDbID, sysAppLaunchCmdOpenDB, (Ptr) cmdPBP);
		if (err)
			return err;
	}

	if (GetQuitStage () == kWaitingForSysUIAppSwitch)
	{
		SetQuitStage (kWaitingForSysAppStartup);
	}

	return errNone;
}


/***********************************************************************
 *
 * FUNCTION:	Patches::QuitOnAppExit
 *
 * DESCRIPTION: Tells out patching mechanisms that the emulator should
 *				quit once the current application quits.
 *
 * PARAMETERS:	none
 *
 * RETURNED:	nothing
 *
 ***********************************************************************/

void Patches::QuitOnAppExit (Bool val)
{
	if (val)
		SetQuitStage (kWaitingForSysUIAppSwitch);
	else
		SetQuitStage (kNoQuit);
}


/***********************************************************************
 *
 * FUNCTION:	Patches::GetQuitStage
 *
 * DESCRIPTION: .
 *
 * PARAMETERS:	none
 *
 * RETURNED:	nothing
 *
 ***********************************************************************/

QuitStage Patches::GetQuitStage (void)
{
	return gQuitStage;
}


/***********************************************************************
 *
 * FUNCTION:	Patches::SetQuitStage
 *
 * DESCRIPTION: .
 *
 * PARAMETERS:	none
 *
 * RETURNED:	nothing
 *
 ***********************************************************************/

void Patches::SetQuitStage (QuitStage q)
{
	gQuitStage = q;
}


/***********************************************************************
 *
 * FUNCTION:	Patches::PuppetString
 *
 * DESCRIPTION: Puppet stringing function for inserting events into
 *				the system.  We want to insert events when:
 *
 *				- Gremlins is running
 *				- The user types characters
 *				- The user clicks with the mouse
 *				- We need to trigger a switch another application
 *
 *				This function is called from headpatches to
 *				SysEvGroupWait and SysSemaphoreWait.  When the Palm OS
 *				needs an event, it calls EvtGetEvent.  EvtGetEvent
 *				looks in all the usual places for events to return. If
 *				it doesn't find any, it puts the system to sleep by
 *				calling SysEvGroupWait (or SysSemaphoreWait on 1.0
 *				systems).  SysEvGroupWait will wake up and return when
 *				an event is posted via something like EvtEnqueuePenPoint,
 *				EvtEnqueueKey, or KeyHandleInterrupt.
 *
 *				To puppet-string Palm OS, we therefore patch those
 *				functions and post events, preventing them from actually
 *				going to sleep.
 *
 * PARAMETERS:	callROM - return here whether or not the original ROM
 *					function still needs to be called.	Normally, the
 *					answer is "yes".
 *
 *				clearTimeout - set to true if the "timeout" parameter
 *					of the function we've patched needs to be prevented
 *					from being "infinite".
 *
 * RETURNED:	nothing
 *
 ***********************************************************************/

static void PrvForceNilEvent (void)
{
	// No event was posted.  What we'd like right now is to force
	// EvtGetEvent to return a nil event.  We can do that by returning
	// a non-zero result code from SysEvGroupWait.  EvtGetEvent doesn't
	// look too closely at the result, but let's try to be as close to
	// reality as possible. SysEvGroupWait currently returns "4"
	// (CJ_WRTMOUT) to indicate a timeout condition.  It should
	// probably get turned into sysErrTimeout somewhere along the way,
	// but that translation doesn't seem to occur.

	m68k_dreg(regs, 0) = 4;
}

void Patches::PuppetString (CallROMType& callROM, Bool& clearTimeout)
{
	callROM = kExecuteROM;
	clearTimeout = false;

	// Set the return value (Err) to zero in case we return
	// "true" (saying that we handled the trap).

	m68k_dreg (regs, 0) = 0;

	// If the low-memory global "idle" is true, then we're being
	// called from EvtGetEvent or EvtGetPen, in which case we
	// need to check if we need to post some events.

	if (LowMem::GetEvtMgrIdle ())
	{
		// If there's an RPC request waiting for a nilEvent,
		// let it know that it happened.

		if (gLastEvtTrap == sysTrapEvtGetEvent)
		{
			RPC::SignalWaiters (hostSignalIdle);
		}

		// If we're in the middle of calling a Palm OS function ourself,
		// and we are someone at the point where the system is about to
		// doze, then just return now. Don't let it doze!  Interrupts are
		// off, and HwrDoze will never return!

		if (ATrap::DoingCall())
		{
			::PrvForceNilEvent();
			callROM = kSkipROM;
			return;
		}

		if (Hordes::IsOn ())
		{
			if (gLastEvtTrap == sysTrapEvtGetEvent)
			{
				if (!Hordes::PostFakeEvent ())
				{
					if (LogEnqueuedEvents ())
					{
						LogAppendMsg ("Hordes::PostFakeEvent did not post an event.");
					}

					::PrvForceNilEvent();
					callROM = kSkipROM;
					return;
				}
			}
			else if (gLastEvtTrap == sysTrapEvtGetPen)
			{
				Hordes::PostFakePenEvent ();
			}
			else
			{
				if (LogEnqueuedEvents ())
				{
					LogAppendMsg ("Last event was 0x%04X, so not posting event.", gLastEvtTrap);
				}
			}
#if 0
			// Ensure that there is an event posted.  If there isn't
			// a timeout of zero (= forever) would kill us.

			assert (get_word (objID + 0x1E) == 1);
#endif

			// Never let the timeout be infinite.  If the above event-posting
			// attempts failed (which could happen, for instance, if we attempted
			// to post a pen event with the same coordinates as the previous
			// pen event), we'd end up waiting forever.

			clearTimeout = true;
		}

		// Gremlins aren't on; let's see if the user has typed some
		// keys that we need to pass on to the Palm device.

		else if (gKeyQueue.GetUsed () > 0)
		{
			StubAppEnqueueKey (gKeyQueue.Get (), 0, 0);
		}

		// No key events, let's see if there are pen events.

		else if (Hardware::HavePenEvent ())
		{
			PointType	pen = { -1, -1 };
			if (Hardware::PenIsDown ())
			{
				pen = Hardware::PenLocation ();
			}

			Hardware::SetHavePenEvent (false);

			StubAppEnqueuePt (&pen);
		}

		// E. None of the above.  Let's see if there's an app
		//	  we're itching to switch to.

		else if (gNextAppDbID != 0)
		{
			Err err = SwitchToApp (gNextAppCardNo, gNextAppDbID);

			gNextAppCardNo = 0;
			gNextAppDbID = 0;

			clearTimeout = true;
		}
	}
	else
	{
		if (Hordes::IsOn () && LogEnqueuedEvents ())
		{
			LogAppendMsg ("Event Manager not idle, so not posting an event.");
		}
	}
}


/***********************************************************************
 *
 * FUNCTION:	Patches::HasWellBehavedMemSemaphoreUsage
 *
 * DESCRIPTION: 
 *
 * PARAMETERS:	none
 *
 * RETURNED:	nothing
 *
 ***********************************************************************/

Bool Patches::HasWellBehavedMemSemaphoreUsage (void)
{
	// Palm OS 3.0 and later should not be holding the memory manager
	// semaphore for longer than 1 minute.	I imagine that older ROMs
	// don't hold the semaphore for longer than this, but Roger still
	// suggested testing for 3.0.

	return gOSVersion != kOSUndeterminedVersion && OSMajorMinorVersion () >= 30;
}


/***********************************************************************
 *
 * FUNCTION:	Patches::EnterMemMgr
 *
 * DESCRIPTION: 
 *
 * PARAMETERS:	none
 *
 * RETURNED:	nothing
 *
 ***********************************************************************/

void Patches::EnterMemMgr (const char* fnName)
{
	UNUSED_PARAM(fnName)

	++gMemMgrCount;
	assert (gMemMgrCount < 10);
}


/***********************************************************************
 *
 * FUNCTION:	Patches::ExitMemMgr
 *
 * DESCRIPTION: 
 *
 * PARAMETERS:	none
 *
 * RETURNED:	nothing
 *
 ***********************************************************************/

void Patches::ExitMemMgr (const char* fnName)
{
	UNUSED_PARAM(fnName)

	--gMemMgrCount;
	assert (gMemMgrCount >= 0);
}


/***********************************************************************
 *
 * FUNCTION:	Patches::IsInSysBinarySearch
 *
 * DESCRIPTION: 
 *
 * PARAMETERS:	none
 *
 * RETURNED:	nothing
 *
 ***********************************************************************/

Bool Patches::IsInSysBinarySearch (void)
{
	return gSysBinarySearchCount > 0;
}


/***********************************************************************
 *
 * FUNCTION:	Patches::EnterSysBinarySearch
 *
 * DESCRIPTION: 
 *
 * PARAMETERS:	none
 *
 * RETURNED:	nothing
 *
 ***********************************************************************/

void Patches::EnterSysBinarySearch (void)
{
	assert (gSysBinarySearchCount < 10);
	++gSysBinarySearchCount;
}


/***********************************************************************
 *
 * FUNCTION:	Patches::ExitSysBinarySearch
 *
 * DESCRIPTION: 
 *
 * PARAMETERS:	none
 *
 * RETURNED:	nothing
 *
 ***********************************************************************/

void Patches::ExitSysBinarySearch (void)
{
	--gSysBinarySearchCount;
	assert (gSysBinarySearchCount >= 0);
}


/***********************************************************************
 *
 * FUNCTION:	Patches::UIInitialized
 *
 * DESCRIPTION: 
 *
 * PARAMETERS:	none
 *
 * RETURNED:	nothing
 *
 ***********************************************************************/

Bool Patches::UIInitialized (void)
{
	return gUIInitialized;
}


/***********************************************************************
 *
 * FUNCTION:	Patches::SetUIInitialized
 *
 * DESCRIPTION: 
 *
 * PARAMETERS:	none
 *
 * RETURNED:	nothing
 *
 ***********************************************************************/

void Patches::SetUIInitialized (Bool b)
{
	gUIInitialized = b != 0;
}


/***********************************************************************
 *
 * FUNCTION:	Patches::HeapInitialized
 *
 * DESCRIPTION: 
 *
 * PARAMETERS:	none
 *
 * RETURNED:	nothing
 *
 ***********************************************************************/

Bool Patches::HeapInitialized (void)
{
	return gHeapInitialized;
}


/***********************************************************************
 *
 * FUNCTION:	Patches::SetHeapInitialized
 *
 * DESCRIPTION: 
 *
 * PARAMETERS:	none
 *
 * RETURNED:	nothing
 *
 ***********************************************************************/

void Patches::SetHeapInitialized (Bool b)
{
	gHeapInitialized = b != 0;
}


/***********************************************************************
 *
 * FUNCTION:	Patches::TurningJapanese
 *
 * DESCRIPTION: 
 *
 * PARAMETERS:	none
 *
 * RETURNED:	nothing
 *
 ***********************************************************************/

Bool Patches::TurningJapanese (void)
{
	return gHaveJapanese;
}


/***********************************************************************
 *
 * FUNCTION:	Patches::EvtGetEventCalled
 *
 * DESCRIPTION: 
 *
 * PARAMETERS:	none
 *
 * RETURNED:	nothing
 *
 ***********************************************************************/

Bool Patches::EvtGetEventCalled (void)
{
	return gEvtGetEventCalled;
}


/***********************************************************************
 *
 * FUNCTION:	Patches::CollectCurrentAppInfo
 *
 * DESCRIPTION: 
 *
 * PARAMETERS:	none
 *
 * RETURNED:	nothing
 *
 ***********************************************************************/

EmuAppInfo Patches::CollectCurrentAppInfo (uaecptr appInfoP)
{
	EmuAppInfo	newAppInfo;
	memset (&newAppInfo, 0, sizeof (newAppInfo));

	// Scarf some information out of the app info block.

	newAppInfo.fDB			= (DmOpenRef) get_long (appInfoP + offsetof (SysAppInfoType, dbP));
	newAppInfo.fStackP		= get_long (appInfoP + offsetof (SysAppInfoType, stackP));
	newAppInfo.fMemOwnerID	= get_word (appInfoP + offsetof (SysAppInfoType, memOwnerID));

	// Determine the current stack range.  Under Palm OS 3.0 and later, this information
	// is in the DatabaseInfo block.  Under earlier OSes, we only get the low-end of the stack
	// (that is, the address that was returned by MemPtrNew).  To get the high-end of
	// the stack, assume that stackP pointed to a chunk of memory allocated by MemPtrNew
	// and call MemPtrSize.

	if (DatabaseInfoHasStackInfo ())
	{
		if (newAppInfo.fStackP)
		{
			ULong	stackSize = ::MemPtrSize ((VoidPtr) newAppInfo.fStackP);
			if (stackSize)
			{
				newAppInfo.fStackEndP = newAppInfo.fStackP + stackSize;
			}
			else
			{
				newAppInfo.fStackEndP = UAE_NULL;
			}
		}
	}
	else
	{
		newAppInfo.fStackEndP = get_long (appInfoP + offsetof (SysAppInfoType, stackEndP));
	}

	newAppInfo.fStackSize = newAppInfo.fStackEndP - newAppInfo.fStackP;

	// Remember the current application name and version information.  We use
	// this information when telling users that something has gone haywire.  Collect
	// this information now instead of later (on demand) as we can't be sure that
	// we can make the necessary DataMgr calls after an error occurs.
	//
	// If the database has a 'tAIN' resource, get the name from there.
	// Otherwise, use the database name.
	//
	// (Write the name into a temporary local variable.  The local variable is
	// on the stack, which will get "mapped" into the emulated address space so
	// that the emulated DmDatabaseInfo can get to it.)

	UInt	cardNo;
	LocalID dbID;

	Err err = ::DmOpenDatabaseInfo (newAppInfo.fDB, &dbID, NULL, NULL, &cardNo, NULL);
	if (err)
		return newAppInfo;

	newAppInfo.fCardNo = cardNo;
	newAppInfo.fDBID = dbID;

	char	appName[dmDBNameLength] = {0};
	char	appVersion[256] = {0};	// <gulp> I hope this is big enough...

//	DmOpenRef	dbP = DmOpenDatabase (cardNo, dbID, dmModeReadOnly);

//	if (dbP)
	{
		VoidHand	strH;

		// Get the app name from the 'tAIN' resource.

		strH = ::DmGet1Resource (ainRsc, ainID);
		if (strH)
		{
			uaecptr strP = (uaecptr) ::MemHandleLock (strH);
			uae_strcpy (appName, strP);
			::MemHandleUnlock (strH);
			::DmReleaseResource (strH);
		}

		// Get the version from the 'tver' resource, using ID's 1 and 1000

		strH = ::DmGet1Resource (verRsc, appVersionID);
		if (strH == NULL)
			strH = ::DmGet1Resource (verRsc, appVersionAlternateID);
		if (strH)
		{
			uaecptr strP = (uaecptr) ::MemHandleLock (strH);
			uae_strcpy (appVersion, strP);
			::MemHandleUnlock (strH);
			::DmReleaseResource (strH);
		}

//		::DmCloseDatabase (dbP);
	}

	if (appName[0] == 0)	// No 'tAIN' resource, so use database name
	{
		::DmDatabaseInfo (cardNo, dbID,
						appName,
						NULL, NULL, NULL, NULL, NULL,
						NULL, NULL, NULL, NULL, NULL);
	}

	// Copy the strings from the stack to their permanent homes.

	strcpy (newAppInfo.fName, appName);
	strcpy (newAppInfo.fVersion, appVersion);

	return newAppInfo;
}

/***********************************************************************
 *
 * FUNCTION:	Patches::GetCurrentAppInfo
 *
 * DESCRIPTION: Return information on the last application launched
 *				with SysAppLaunch (and that hasn't exited yet).
 *
 * PARAMETERS:	none
 *
 * RETURNED:	nothing
 *
 ***********************************************************************/


EmuAppInfo	Patches::GetCurrentAppInfo		(void)
{
	EmuAppInfo	result;
	memset (&result, 0, sizeof (result));

	if (gCurAppInfo.size () > 0)
		result = *(gCurAppInfo.rbegin ());

	return result;
}


/***********************************************************************
 *
 * FUNCTION:	Patches::GetRootAppInfo
 *
 * DESCRIPTION: Return information on the last application launched
 *				with SysAppLaunch and with the launch code of
 *				sysAppLaunchCmdNormalLaunch (and that hasn't exited yet).
 *
 * PARAMETERS:	none
 *
 * RETURNED:	nothing
 *
 ***********************************************************************/

EmuAppInfo	Patches::GetRootAppInfo 		(void)
{
	EmuAppInfo	result;
	memset (&result, 0, sizeof (result));

	EmuAppInfoList::reverse_iterator iter = gCurAppInfo.rbegin ();

	while (iter != gCurAppInfo.rend ())
	{
		if ((*iter).fCmd == sysAppLaunchCmdNormalLaunch)
		{
			result = *iter;
			break;
		}

		++iter;
	}

	return result;
}

#pragma mark -

// ===========================================================================
//		 SysHeadpatch
// ===========================================================================

/***********************************************************************
 *
 * FUNCTION:	SysHeadpatch::RecordTrapNumber
 *
 * DESCRIPTION: Record the trap we're executing for our patch to
 *				SysEvGroupWait later.
 *
 * PARAMETERS:	none
 *
 * RETURNED:	nothing
 *
 ***********************************************************************/

CallROMType SysHeadpatch::RecordTrapNumber (void)
{
	struct StackFrame
	{
	};

	uae_u8* realMem = get_real_address (m68k_getpc ());

	assert (do_get_mem_word (realMem - 2) == (m68kTrapInstr + sysDispatchTrapNum));

	gLastEvtTrap = do_get_mem_word (realMem);

	return kExecuteROM;
}


/***********************************************************************
 *
 * FUNCTION:	SysHeadpatch::DbgMessage
 *
 * DESCRIPTION: 
 *
 * PARAMETERS:	none
 *
 * RETURNED:	nothing
 *
 ***********************************************************************/

CallROMType SysHeadpatch::DbgMessage (void)
{
	// void DbgMessage(CharPtr aStr);

	struct StackFrame
	{
		CharPtr aStr;
	};

	uaecptr msg = GET_PARAMETER (aStr);

	if (msg)
	{
		string	msgCopy;
		size_t	msgLen = uae_strlen (msg);

		if (msgLen > 0)
		{
			msgCopy.resize (msgLen);
			uae_strcpy (&msgCopy[0], msg);
		}

		SLP	slp (Debug::GetDebuggerSocket ());
		SystemPacket::SendMessage (slp, msgCopy.c_str ());
	}

	return kSkipROM;
}


/***********************************************************************
 *
 * FUNCTION:	SysHeadpatch::DmInit
 *
 * DESCRIPTION: After MemInit is called, we need to sync up with the
 *				initial state of the heap(s).  However, MemInit is not
 *				called via the trap table, so we can't easily tailpatch
 *				it.  DmInit is the first such function called after
 *				MemInit, so we headpatch *it* instead of tailpatching
 *				MemInit.
 *
 *				(Actually, MemHeapCompact is called as one of the last
 *				 things MemInit does which makes it an interesting
 *				 candidate for patching in order to sync up with the
 *				 heap state.  However, if we were to do a full sync on
 *				 that call, a full sync would occur on *every* call to
 *				 MemHeapCompact, which we don't really want to do.)
 *
 * PARAMETERS:	none
 *
 * RETURNED:	nothing
 *
 ***********************************************************************/

CallROMType SysHeadpatch::DmInit (void)
{
	MetaMemory::SyncAllHeaps ();

	Patches::SetHeapInitialized (true);

	return kExecuteROM;
}


/***********************************************************************
 *
 * FUNCTION:	SysHeadpatch::ErrDisplayFileLineMsg
 *
 * DESCRIPTION: 
 *
 * PARAMETERS:	none
 *
 * RETURNED:	nothing
 *
 ***********************************************************************/

CallROMType SysHeadpatch::ErrDisplayFileLineMsg (void)
{
	// void ErrDisplayFileLineMsg(CharPtr filename, UInt lineno, CharPtr msg)

	CEnableFullAccess	munge;	// Remove blocks on memory access.

	// Force this guy to true.	If it's false, ErrDisplayFileLineMsg will
	// just try to enter the debugger.

	Word	sysMiscFlags = LowMem_GetGlobal (sysMiscFlags);
	LowMem_SetGlobal (sysMiscFlags, sysMiscFlags | sysMiscFlagUIInitialized);

	// Clear this low-memory flag so that we force the dialog to display.
	// If this flag is true, ErrDisplayFileLineMsg will just try to enter
	// the debugger.

	LowMem_SetGlobal (dbgWasEntered, false);

	return kExecuteROM;
}


/***********************************************************************
 *
 * FUNCTION:	SysHeadpatch::EvtAddEventToQueue
 *
 * DESCRIPTION: 
 *
 * PARAMETERS:	none
 *
 * RETURNED:	nothing
 *
 ***********************************************************************/

CallROMType SysHeadpatch::EvtAddEventToQueue (void)
{
	// void EvtAddEventToQueue (const EventPtr event)

	struct StackFrame
	{
		const EventPtr	event;
	};

	uaecptr event	= GET_PARAMETER (event);

	LogEvtAddEventToQueue (event);

	return kExecuteROM;
}


/***********************************************************************
 *
 * FUNCTION:	SysHeadpatch::EvtAddUniqueEventToQueue
 *
 * DESCRIPTION: 
 *
 * PARAMETERS:	none
 *
 * RETURNED:	nothing
 *
 ***********************************************************************/

CallROMType SysHeadpatch::EvtAddUniqueEventToQueue (void)
{
	// void EvtAddUniqueEventToQueue(const EventPtr eventP, const DWord id, const Boolean inPlace)

	struct StackFrame
	{
		const EventPtr	event;
		const DWord 	id;
		const Boolean	inPlace;
	};

	uaecptr event	= GET_PARAMETER (event);
	DWord	id		= GET_PARAMETER (id);
	Boolean inPlace = GET_PARAMETER (inPlace);

	LogEvtAddUniqueEventToQueue (event, id, inPlace);

	return kExecuteROM;
}


/***********************************************************************
 *
 * FUNCTION:	SysHeadpatch::EvtEnqueueKey
 *
 * DESCRIPTION: 
 *
 * PARAMETERS:	none
 *
 * RETURNED:	nothing
 *
 ***********************************************************************/

CallROMType SysHeadpatch::EvtEnqueueKey (void)
{
	// Err EvtEnqueueKey(UInt ascii, UInt keycode, UInt modifiers)

	struct StackFrame
	{
		UInt	ascii;
		UInt	keycode;
		UInt	modifiers;
	};

	UInt	ascii		= GET_PARAMETER (ascii);
	UInt	keycode 	= GET_PARAMETER (keycode);
	UInt	modifiers	= GET_PARAMETER (modifiers);

	LogEvtEnqueueKey (ascii, keycode, modifiers);

	return kExecuteROM;
}


/***********************************************************************
 *
 * FUNCTION:	SysHeadpatch::EvtEnqueuePenPoint
 *
 * DESCRIPTION: 
 *
 * PARAMETERS:	none
 *
 * RETURNED:	nothing
 *
 ***********************************************************************/

CallROMType SysHeadpatch::EvtEnqueuePenPoint (void)
{
	// Err EvtEnqueuePenPoint(PointType* ptP)

	struct StackFrame
	{
		PointType*	ptP;
	};

	uaecptr ptP 	= GET_PARAMETER (ptP);

	LogEvtEnqueuePenPoint (ptP);

	return kExecuteROM;
}

/***********************************************************************
 *
 * FUNCTION:	SysHeadpatch::FrmDrawForm
 *
 * DESCRIPTION: 
 *
 * PARAMETERS:	none
 *
 * RETURNED:	nothing
 *
 ***********************************************************************/

CallROMType SysHeadpatch::FrmDrawForm (void)
{
	// void FrmDrawForm (const FormPtr frm)

	struct StackFrame
	{
		FormPtr	frm;
	};

	FormPtr frm 	= (FormPtr) GET_PARAMETER (frm);

	vector<Word>	okObjects;
	::CollectOKObjects (frm, okObjects, true);

	return kExecuteROM;
}


/***********************************************************************
 *
 * FUNCTION:	SysHeadpatch::HostControl
 *
 * DESCRIPTION: This one's kind of odd.  Originally, there was
 *				SysGremlins, which was declared as follows:
 *
 *					DWord SysGremlins(GremlinFunctionType selector,
 *										GremlinParamsType *params)
 *
 *				Also originally, the only defined selector was
 *				GremlinIsOn.
 *
 *				Now, SysGremlins is extended to be SysHostControl,
 *				which allows the Palm environment to access host
 *				functions if it's actually running under the simulator
 *				or emulator.
 *
 *				Because of this extension, functions implemented via
 *				this trap are not limited to pushing a selector and
 *				parameter block on the stack. Now, they will all push
 *				on a selector, but what comes after is dependent on the
 *				selector.
 *
 * PARAMETERS:	none
 *
 * RETURNED:	nothing
 *
 ***********************************************************************/

CallROMType SysHeadpatch::HostControl (void)
{
	return HandleHostControlCall ();
}


/***********************************************************************
 *
 * FUNCTION:	SysHeadpatch::HwrBatteryLevel
 *
 * DESCRIPTION: Return that the battery is always full.  HwrBatteryLevel
 *				is the bottleneck function called to determine the
 *				battery level.	By patching it this way, we don't have
 *				to emulate the hardware registers that report the
 *				battery level.
 *
 * PARAMETERS:	none
 *
 * RETURNED:	nothing
 *
 ***********************************************************************/

CallROMType SysHeadpatch::HwrBatteryLevel (void)
{
	// UInt HwrBatteryLevel(void)

	struct StackFrame
	{
	};

	m68k_dreg (regs, 0) = 255;	// Hardcode a maximum level

	return kSkipROM;
}


/***********************************************************************
 *
 * FUNCTION:	SysHeadpatch::HwrDockStatus
 *
 * DESCRIPTION: Always return hwrDockStatusUsingExternalPower.  We
 *				could fake this out by twiddling the right bits in the
 *				Dragonball and DragonballEZ emulation units, but those
 *				bits are different for almost every device.
 *
 * PARAMETERS:	none
 *
 * RETURNED:	nothing
 *
 ***********************************************************************/

CallROMType SysHeadpatch::HwrDockStatus (void)
{
	// hwrDockStatusState HwrDockStatus(void)
	//	(added in Palm OS 3.1)
	//	(changed later to return UInt16)

	struct StackFrame
	{
	};

	// Old enumerated values from Hardware.h:
	//
	//		DockStatusNotDocked = 0,
	//		DockStatusInModem,
	//		DockStatusInCharger,
	//		DockStatusUnknown = 0xFF

	// New defines from HwrDock.h
#define	hwrDockStatusUndocked			0x0000	// nothing is attached
#define	hwrDockStatusModemAttached		0x0001	// some type of modem is attached
#define	hwrDockStatusDockAttached		0x0002	// some type of dock is attached
#define	hwrDockStatusUsingExternalPower	0x0004	// using some type of external power source
#define	hwrDockStatusCharging			0x0008	// internal power cells are recharging

	m68k_dreg (regs, 0) = hwrDockStatusUsingExternalPower;

	return kSkipROM;
}


/***********************************************************************
 *
 * FUNCTION:	SysHeadpatch::HwrDoze
 *
 * DESCRIPTION: If we're the one responsible for this function being
 *				called, then return immediately.  If HwrDoze were
 *				called, it would never return, because we have
 *				interrupts turned off.
 *
 * PARAMETERS:	none
 *
 * RETURNED:	nothing
 *
 ***********************************************************************/

CallROMType SysHeadpatch::HwrDoze (void)
{
//	if (ATrap::DoingCall())
//		return kSkipROM;

	return kExecuteROM;
}


/***********************************************************************
 *
 * FUNCTION:	SysHeadpatch::HwrGetROMToken
 *
 * DESCRIPTION: Patch this guy so that we never return the 'irda' token.
 *				We should take this out when some sort of IR support is
 *				added.
 *
 *				NOTE: This patch is useless for diverting the ROM. It
 *				calls HwrGetROMToken directly, which means that it will
 *				bypass this patch.
 *
 * PARAMETERS:	none
 *
 * RETURNED:	nothing
 *
 ***********************************************************************/

CallROMType SysHeadpatch::HwrGetROMToken (void)
{
	//	Err HwrGetROMToken (Word cardNo, DWord token, BytePtr *dataP, WordPtr sizeP)

	struct StackFrame
	{
		Word cardNo;
		DWord token;
		BytePtr *dataP;
		WordPtr sizeP;
	};

	Word	cardNo = GET_PARAMETER (cardNo);
	DWord	token = GET_PARAMETER (token);
	uaecptr dataP = GET_PARAMETER (dataP);
	uaecptr sizeP = GET_PARAMETER (sizeP);

	if (cardNo == 0 && token == hwrROMTokenIrda)
	{
		if (dataP)
			put_long (dataP, 0);

		if (sizeP)
			put_long (sizeP, 0);

		m68k_dreg (regs, 0) = ~0;		// token not found.

		return kSkipROM;
	}

	return kExecuteROM;
}


/***********************************************************************
 *
 * FUNCTION:	SysHeadpatch::HwrSleep
 *
 * DESCRIPTION: Record whether or not we are sleeping and update the
 *				boolean that determines if low-memory access is OK.
 *
 * PARAMETERS:	none
 *
 * RETURNED:	nothing
 *
 ***********************************************************************/

CallROMType SysHeadpatch::HwrSleep (void)
{
	// void HwrSleep(Boolean untilReset, Boolean emergency)

	struct StackFrame
	{
		Boolean untilReset;
		Boolean dummy;
		Boolean emergency;
	};

	// HwrSleep changes the exception vectors for for the interrupts,
	// so temporarily unlock those.  We'll re-establish them in the
	// HwrSleep tailpatch.

	MetaMemory::MarkTotalAccess (offsetof (M68KExcTableType, busErr),
								 offsetof (M68KExcTableType, busErr) + sizeof (uaecptr));

	MetaMemory::MarkTotalAccess (offsetof (M68KExcTableType, addressErr),
								 offsetof (M68KExcTableType, addressErr) + sizeof (uaecptr));

	MetaMemory::MarkTotalAccess (offsetof (M68KExcTableType, illegalInstr),
								 offsetof (M68KExcTableType, illegalInstr) + sizeof (uaecptr));

	MetaMemory::MarkTotalAccess (offsetof (M68KExcTableType, autoVec1),
								 offsetof (M68KExcTableType, trapN[0]));

	return kExecuteROM;
}


/***********************************************************************
 *
 * FUNCTION:	SysHeadpatch::KeyCurrentState
 *
 * DESCRIPTION: 
 *
 * PARAMETERS:	none
 *
 * RETURNED:	nothing
 *
 ***********************************************************************/

// From Roger:
//
//	I was thinking a bit more about Gremlins and games yesterday.  Games which
//	use the buttons as input have been a special case for Gremlins because
//	Gremlins only faked events and pen points.	Normally games have to fake
//	their own button mask, and they get better results if have some buttons
//	held down more often than others.
//
//	Now I'm thinking different.  With Poser, the KeyCurrentState call should
//	be stolen when Gremlins is running.  All of the keys possible should have
//	their button bits on half the time.  This will allow users to test games.
//	By not tuning how often buttons should be held down, the testing process
//	will take longer to excerise all app functionality, but it's better than
//	now.  App developers can override the default Gremlin values with their
//	own for better results.
//
//	To successfully test this, A game like SubHunt should play on it's own for
//	a least a while.  HardBall is an example of a game which would really
//	benefit from recognizing Gremlins is running and tune itself to make the
//	testing more effective.  It should grant infinite balls until after the
//	last level is finished.
//
//	I actually think this is important enough to be for Acorn because it
//	enables users to test a large class of apps which otherwise can't.	I
//	think it's easy to implement.  Basically just have KeyCurrentState return
//	the int from the random number generator.  Each bit should be on about
//	half the time.
//
// Follow-up commentary: it turns out that this patch is not having the
// effect we hoped.  SubHunt was so overwhelmed with events from Gremlins
// that it rarely had the chance to call KeyCurrentState.  We're thinking
// of Gremlins backing off on posting events if the SysGetEvent sleep time
// is small (i.e., not "forever"), but we'll have to think about the impact
// on other apps first.

CallROMType SysHeadpatch::KeyCurrentState (void)
{
	// DWord KeyCurrentState(void)

	struct StackFrame
	{
	};

	if (Hordes::IsOn ())
	{
		// Let's try setting each bit 1/4 of the time.

		uae_u32 bits = rand () & rand ();

		m68k_dreg (regs, 0) = bits;

		return kSkipROM;
	}

	return kExecuteROM;
}


/***********************************************************************
 *
 * FUNCTION:	SysHeadpatch::PenOpen
 *
 * DESCRIPTION: This is where the pen calibration information is read.
 *				Preflight this call to add the calibration information
 *				to the preferences database if it doesn't exist.  That
 *				way, we don't have to continually calibrate the screen
 *				when we boot up.
 *
 * PARAMETERS:	none
 *
 * RETURNED:	nothing
 *
 ***********************************************************************/

CallROMType SysHeadpatch::PenOpen (void)
{
	// Err PenOpen(void)

	struct StackFrame
	{
	};

#if !TIME_STARTUP

	::InstallCalibrationInfo ();

#endif

	return kExecuteROM;
}


/***********************************************************************
 *
 * FUNCTION:	SysHeadpatch::SysAppExit
 *
 * DESCRIPTION: If the application calling SysAppExit was launched as a
 *				full application, then "forget" any information we have
 *				about it.  When the next application is launched, we'll
 *				collect information on it in SysAppStartup.
 *
 * PARAMETERS:	none
 *
 * RETURNED:	nothing
 *
 ***********************************************************************/

CallROMType SysHeadpatch::SysAppExit (void)
{
	// Err SysAppExit(SysAppInfoPtr appInfoP, Ptr prevGlobalsP, Ptr globalsP)

	struct StackFrame
	{
		SysAppInfoPtr appInfoP;
		Ptr prevGlobalsP;
		Ptr globalsP;
	};

	uaecptr appInfoP		= GET_PARAMETER (appInfoP);
//	uaecptr prevGlobalsP	= GET_PARAMETER (prevGlobalsP);
//	uaecptr globalsP		= GET_PARAMETER (globalsP);

	if (!appInfoP)
		return kExecuteROM;

	Int cmd = get_word (appInfoP + offsetof (SysAppInfoType, cmd));

	if (cmd == sysAppLaunchCmdNormalLaunch)
	{
		if (Patches::GetQuitStage () == kWaitingForSysAppExit)
		{
			Patches::SetQuitStage (kTimeToQuit);
		}
	}

	gCurAppInfo.pop_back ();	// !!! should probably make sure appInfoP matches
								// the top guy on the stack.

	return kExecuteROM;
}


/***********************************************************************
 *
 * FUNCTION:	SysHeadpatch::SysAppLaunch
 *
 * DESCRIPTION: Log information app launches and action codes.
 *
 * PARAMETERS:	none
 *
 * RETURNED:	nothing
 *
 ***********************************************************************/

CallROMType SysHeadpatch::SysAppLaunch (void)
{
	struct StackFrame
	{
		UInt	cardNo;
		LocalID dbID;
		UInt	launchFlags;
		Word	cmd;
		Ptr 	cmdPBP;
		DWord*	resultP;
	};

	if (false)
	{
		UInt	cardNo		= GET_PARAMETER (cardNo);
		LocalID dbID		= GET_PARAMETER (dbID);
		UInt	launchFlags = GET_PARAMETER (launchFlags);
		Word	cmd 		= GET_PARAMETER (cmd);
//		uaecptr cmdPBP		= GET_PARAMETER (cmdPBP);
//		uaecptr resultP 	= GET_PARAMETER (resultP);

		const char* launchStr = ::LaunchCmdToString (cmd);

		LogAppendMsg ("SysAppLaunch called:");
		LogAppendMsg ("	cardNo:			%ld",	  (long) cardNo);
		LogAppendMsg ("	dbID:			0x%08X",	(long) dbID);
		LogAppendMsg ("	launchFlags:	0x%08X",   (long) launchFlags);
		LogAppendMsg ("	cmd:			%ld (%s)",	 (long) cmd, launchStr);

		switch (cmd)
		{
			case sysAppLaunchCmdNormalLaunch:
				// No parameter block
				break;

			case sysAppLaunchCmdFind:
			{
				// FindParamsType
				LogAppendMsg ("	FindParamsType:");
				LogAppendMsg ("		dbAccesMode:		%ld",		 (long) cmd);
				LogAppendMsg ("		recordNum:			%ld",	  (long) cmd);
				LogAppendMsg ("		more:				%ld",	  (long) cmd);
				LogAppendMsg ("		strAsTyped:			%ld",		 (long) cmd);
				LogAppendMsg ("		strToFind:			%ld",	  (long) cmd);
				LogAppendMsg ("		numMatches:			%ld",		 (long) cmd);
				LogAppendMsg ("		lineNumber:			%ld",		 (long) cmd);
				LogAppendMsg ("		continuation:		%ld",		(long) cmd);
				LogAppendMsg ("		searchedCaller:		%ld",	  (long) cmd);
				LogAppendMsg ("		callerAppDbID:		%ld",	   (long) cmd);
				LogAppendMsg ("		callerAppCardNo:	%ld",	  (long) cmd);
				LogAppendMsg ("		appDbID:			%ld",		(long) cmd);
				LogAppendMsg ("		newSearch:			%ld",	  (long) cmd);
				LogAppendMsg ("		searchState:		%ld",		 (long) cmd);
				LogAppendMsg ("		match:				%ld",		 (long) cmd);

				break;
			}

			case sysAppLaunchCmdGoTo:
				// GoToParamsType
				break;

			case sysAppLaunchCmdSyncNotify:
				// No parameter block
				break;

			case sysAppLaunchCmdTimeChange:
				// No parameter block
				break;

			case sysAppLaunchCmdSystemReset:
				// SysAppLaunchCmdSystemResetType
				break;

			case sysAppLaunchCmdAlarmTriggered:
				// SysAlarmTriggeredParamType
				break;

			case sysAppLaunchCmdDisplayAlarm:
				// SysDisplayAlarmParamType
				break;

			case sysAppLaunchCmdCountryChange:
				// Not sent?
				break;

			case sysAppLaunchCmdSyncRequestLocal:
//			case sysAppLaunchCmdSyncRequest:
				// No parameter block (I think...)
				break;

			case sysAppLaunchCmdSaveData:
				// SysAppLaunchCmdSaveDataType
				break;

			case sysAppLaunchCmdInitDatabase:
				// SysAppLaunchCmdInitDatabaseType
				break;

			case sysAppLaunchCmdSyncCallApplicationV10:
				// SysAppLaunchCmdSyncCallApplicationTypeV10
				break;

			case sysAppLaunchCmdPanelCalledFromApp:
				// Panel specific?
				//	SvcCalledFromAppPBType
				//	NULL
				break;

			case sysAppLaunchCmdReturnFromPanel:
				// No parameter block
				break;

			case sysAppLaunchCmdLookup:
				// App-specific (see AppLaunchCmd.h)
				break;

			case sysAppLaunchCmdSystemLock:
				// No parameter block
				break;

			case sysAppLaunchCmdSyncRequestRemote:
				// No parameter block (I think...)
				break;

			case sysAppLaunchCmdHandleSyncCallApp:
				// SysAppLaunchCmdHandleSyncCallAppType
				break;

			case sysAppLaunchCmdAddRecord:
				// App-specific (see AppLaunchCmd.h)
				break;

			case sysSvcLaunchCmdSetServiceID:
				// ServiceIDType
				break;

			case sysSvcLaunchCmdGetServiceID:
				// ServiceIDType
				break;

			case sysSvcLaunchCmdGetServiceList:
				// serviceListType
				break;

			case sysSvcLaunchCmdGetServiceInfo:
				// serviceInfoType
				break;

			case sysAppLaunchCmdFailedAppNotify:
				// SysAppLaunchCmdFailedAppNotifyType
				break;

			case sysAppLaunchCmdEventHook:
				// EventType
				break;

			case sysAppLaunchCmdExgReceiveData:
				// ExgSocketType
				break;

			case sysAppLaunchCmdExgAskUser:
				// ExgAskParamType
				break;

			case sysDialLaunchCmdDial:
				// DialLaunchCmdDialType
				break;

			case sysDialLaunchCmdHangUp:
				// DialLaunchCmdDialType
				break;

			case sysSvcLaunchCmdGetQuickEditLabel:
				// SvcQuickEditLabelInfoType
				break;

			case sysAppLaunchCmdURLParams:
				// Part of the URL
				break;

			case sysAppLaunchCmdNotify:
				// SysNotifyParamType
				break;

			case sysAppLaunchCmdOpenDB:
				// SysAppLaunchCmdOpenDBType
				break;

			case sysAppLaunchCmdAntennaUp:
				// No parameter block
				break;

			case sysAppLaunchCmdGoToURL:
				// URL
				break;
		}
	}

	return kExecuteROM;
}


/***********************************************************************
 *
 * FUNCTION:	SysHeadpatch::SysBinarySearch
 *
 * DESCRIPTION: There's a bug in pre-3.0 versions of SysBinarySearch
 *				that cause it to call the user callback function with a
 *				pointer just past the array to search.	Make a note of
 *				when we enter SysBinarySearch so that we can make
 *				allowances for that in MetaMemory's memory checking.
 *
 * PARAMETERS:	none
 *
 * RETURNED:	nothing
 *
 ***********************************************************************/

CallROMType SysHeadpatch::SysBinarySearch (void)
{
	Patches::EnterSysBinarySearch ();

	return kExecuteROM;
}


/***********************************************************************
 *
 * FUNCTION:	SysHeadpatch::SysEvGroupWait
 *
 * DESCRIPTION: We patch SysEvGroupWait as the mechanism for feeding
 *				the Palm OS new events.  See comments in
 *				Patches::PuppetString.
 *
 * PARAMETERS:	none
 *
 * RETURNED:	nothing
 *
 ***********************************************************************/

CallROMType SysHeadpatch::SysEvGroupWait (void)
{
	// Err SysEvGroupWait(DWord evID, DWord mask, DWord value, SDWord matchType,
	//						 SDWord timeout)

	struct StackFrame
	{
		DWord evID;
		DWord mask;
		DWord value;
		SDWord matchType;
		SDWord timeout;
	};

	// Only do this under 2.0 and later.  Under Palm OS 1.0, EvtGetSysEvent
	// called SysSemaphoreWait instead.  See our headpatch of that function
	// for a chunk of pretty similar code.

	if (Patches::OSMajorVersion () == 1)
	{
		return kExecuteROM;
	}

	CallROMType result;
	Bool		clearTimeout;

	Patches::PuppetString (result, clearTimeout);

	// If timeout is infinite, the kernel wants 0.
	// If timeout is none, the kernel wants -1.

	if (clearTimeout && GET_PARAMETER (timeout) == 0)
	{
		SET_PARAMETER (timeout, (uae_u32) -1);
	}

	return result;
}


/***********************************************************************
 *
 * FUNCTION:	SysHeadpatch::SysFatalAlert
 *
 * DESCRIPTION: Intercept this and show the user a dialog.
 *
 * PARAMETERS:	none
 *
 * RETURNED:	nothing
 *
 ***********************************************************************/

CallROMType SysHeadpatch::SysFatalAlert (void)
{
	// UInt SysFatalAlert (CharPtr msg)

	struct StackFrame
	{
		CharPtr msg;
	};

	Preference<bool>	pref (kPrefKeyInterceptSysFatalAlert);
	if (!*pref)
	{
		// Palm OS will display a dialog with just a Reset button
		// in it.  So *always* turn off the Gremlin, as the user
		// won't be able to select "Continue".

		Hordes::Stop ();
		return kExecuteROM;
	}

	uaecptr msg = GET_PARAMETER (msg);

	string	msgString;

	if (msg)
	{
		msgString = PrvToString (msg);
	}
	else
	{
		msgString = Platform::GetString (kStr_UnknownFatalError);
	}

	int button = Errors::SysFatalAlert (msgString.c_str ());

	switch (button)
	{
		case Errors::kDebug:		m68k_dreg (regs, 0) = fatalEnterDebugger;	break;
		case Errors::kReset:		m68k_dreg (regs, 0) = fatalReset;			break;
		case Errors::kContinue:		m68k_dreg (regs, 0) = fatalDoNothing;		break;
		case Errors::kNextGremlin:	m68k_dreg (regs, 0) = fatalDoNothing;		break;
	}

	if (button == Errors::kNextGremlin)
	{
		Hordes::ErrorEncountered();
	}
	else if (button != Errors::kContinue)
	{
		Hordes::Stop ();
	}

	return kSkipROM;
}


/***********************************************************************
 *
 * FUNCTION:	SysHeadpatch::SysLaunchConsole
 *
 * DESCRIPTION: Stub this function out so that it doesn't do anything.
 *				We completely handle the console in DebugMgr, so there's
 *				no need to get the ROM all heated up.  Also, there are
 *				problems with actually letting the ROM try to launch its
 *				console task, at least on the Mac.	It will try to open
 *				a serial port socket and do stuff with it.	That attempt
 *				will fail, as much of the serial port processing on the
 *				Mac is handled at idle time, and idle time processing is
 *				inhibited when handling debugger packets
 *				(SysLaunchConsole is usually called by a debugger via
 *				the RPC packet).  Since serial port processing doesn't
 *				occur, SysLaunchConsole hangs.
 *
 * PARAMETERS:	none
 *
 * RETURNED:	nothing
 *
 ***********************************************************************/

CallROMType SysHeadpatch::SysLaunchConsole (void)
{
	// Err SysLaunchConsole(void)

	struct StackFrame
	{
	};

	m68k_dreg (regs, 0) = 0;		// no error

	return kSkipROM;
}


/***********************************************************************
 *
 * FUNCTION:	SysHeadpatch::SysReset
 *
 * DESCRIPTION: Reset the device our way.  This way, we can keep track
 *				of the fact that we're booting and need to make those
 *				"special allowances" that are necessary at boot time.
 *
 * PARAMETERS:	none
 *
 * RETURNED:	nothing
 *
 ***********************************************************************/

CallROMType SysHeadpatch::SysReset (void)
{
	// Err SysReset(void)

	struct StackFrame
	{
	};

	// Causes us to reset the next time we're in the CPU loop.

	regs.spcflags |= SPCFLAG_RESET;

	return kSkipROM;
}


/***********************************************************************
 *
 * FUNCTION:	SysHeadpatch::SysSemaphoreWait
 *
 * DESCRIPTION: We patch SysSemaphoreWait as the mechanism for feeding
 *				the Palm OS new events. See comments in
 *				Patches::PuppetString.
 *
 * PARAMETERS:	none
 *
 * RETURNED:	nothing
 *
 ***********************************************************************/

CallROMType SysHeadpatch::SysSemaphoreWait (void)
{
	// Err SysSemaphoreWait(DWord smID, DWord priority, SDWord timeout)

	struct StackFrame
	{
		DWord smID;
		DWord priority;
		SDWord timeout;
	};

	// Only do this under 1.0.	Under Palm OS 2.0 and later, EvtGetSysEvent
	// calls SysEvGroupWait instead.  See our headpatch of that function
	// for a chunk of pretty similar code.

	if (Patches::OSMajorVersion () != 1)
	{
		return kExecuteROM;
	}

	CallROMType result;
	Bool		clearTimeout;

	Patches::PuppetString (result, clearTimeout);

	// If timeout is infinite, the kernel wants 0.
	// If timeout is none, the kernel wants -1.

	if (clearTimeout && GET_PARAMETER (timeout) == 0)
	{
		SET_PARAMETER (timeout, (uae_u32) -1);
	}

	return result;
}


/***********************************************************************
 *
 * FUNCTION:	SysHeadpatch::SysUIAppSwitch
 *
 * DESCRIPTION: SysUIAppSwitch is called from the following locations
 *				for the given reasons.	When running Gremlins, we want
 *				to prevent SysUIAppSwitch from doing its job, which is
 *				to record information about the application to switch
 *				to and to then post an appStopEvent to the current
 *				application.
 *
 *				There are a couple of places where SysUIAppSwitch is
 *				called to quit and re-run the current application.
 *				Therefore, we want to stub out SysUIAppSwitch only when
 *				the application being switched to is not the currently
 *				running application.
 *
 * PARAMETERS:	none
 *
 * RETURNED:	nothing
 *
 ***********************************************************************/

//	Places where this is called:
//
//	- LauncherMain.c (AppsViewSwitchApp) to launch new app.
//	- PrefApp (PilotMain) to launch a panel.
//	- SyncApp (prvLaunchExternalModule) to launch a "service owner"
//	- AppLaunchCmd.h (AppLaunchWithCommand) ???
//	- ModemPanel.h (ModemPanelLaunch) ???
//	- callApp.h (LaunchApp) ???
//	- ExgMgr.c (ExgNotifyReceive) to send a sysAppLaunchCmdGoTo message.
//	- GraffitiGlue.c (PrvLaunchGraffitiDemo) to launch Graffiti demo.
//	- Keyboard.c (PrvLaunchGraffitiDemo) to launch Graffiti demo.
//	- Launcher.c (LauncherFormHandleEvent) handles taps on launcher icons.
//	- SystemMgr.c (SysHandleEvent) to send sysAppLaunchCmdSystemLock
//	  in response to seeing a lockChr keyboard message.
//	- SystemMgr.c (SysHandleEvent) to switch apps in response to hard#Chr
//	  keyboard messages.
//	- Find.c (Find) to send sysAppLaunchCmdGoTo message.
//
//	- ButtonsPanel.c (ButtonsFormHandleEvent) switch to another panel.
//	- FormatsPanel.c (FormatsFormHandleEvent) switch to another panel.
//	- GeneralPanel.c (GeneralFormHandleEvent) switch to another panel.
//	- ModemPanel.c (ModemFormHandleEvent) switch to another panel.
//	- NetworkPanel.c (NetworkFormHandleEvent) switch to another panel.
//	- OwnerPanel.c (OwnerViewHandleEvent) switch to another panel.
//	- ShortCutsPanel.c (ShortCutFormHandleEvent) switch to another panel.

CallROMType SysHeadpatch::SysUIAppSwitch (void)
{
	// Err SysUIAppSwitch(UInt cardNo, LocalID dbID, Word cmd, Ptr cmdPBP)

	struct StackFrame
	{
		UInt cardNo;
		LocalID dbID;
		Word cmd;
		Ptr cmdPBP;
	};

	UInt	cardNo	= (UInt) GET_PARAMETER (cardNo);
	LocalID dbID	= (LocalID) GET_PARAMETER (dbID);
//	Word	cmd 	= (Word) GET_PARAMETER (cmd);

	// We are headpatching SysUIAppSwitch; if we skip the ROM version, we
	// need to replicate at least this part of its functionality:
	//
	// If the last launch attempt failed, release the command parameter block, if
	//	any. When a launch succeeds, the UIAppShell will clear this global
	//	and free the chunk itself  when the app quits.
	{
		CEnableFullAccess	munge;	// Remove blocks on memory access.
		uaecptr nextUIAppCmdPBP = LowMem_GetGlobal (nextUIAppCmdPBP);
		if (nextUIAppCmdPBP) {
			MemPtrFree((VoidPtr) nextUIAppCmdPBP);
			LowMem_SetGlobal (nextUIAppCmdPBP, 0);
			}
	}

	// Get the current application card and db.  If we are attempting to switch
	// to the same app, allow it.

	UInt	currentCardNo;
	LocalID currentDbID;
	Err 	err = SysCurAppDatabase (&currentCardNo, &currentDbID);

	// Got an error? Give up and let default handling occur.

	if (err != 0)
		return kExecuteROM;

	// Switching to current app; let default handling occur.

	if ((cardNo == currentCardNo) && (dbID == currentDbID))
		return kExecuteROM;

	// OK, we're switching to a different application.	If Gremlins is running
	// and running in "single app" mode, then stub out SysUIAppSwitch to do nothing.

	if (Hordes::IsOn () && !Hordes::CanSwitchToApp (cardNo, dbID))
	{
		m68k_dreg (regs, 0) = 0;
		return kSkipROM;
	}

	// Do the normal app switch.

	return kExecuteROM;
}


/***********************************************************************
 *
 * FUNCTION:	SysHeadpatch::SysUIBusy
 *
 * DESCRIPTION: 
 *
 * PARAMETERS:	none
 *
 * RETURNED:	nothing
 *
 ***********************************************************************/

extern uae_u32	gStart;

CallROMType SysHeadpatch::SysUIBusy (void)
{
	// Word SysUIBusy(Boolean set, Boolean value)

	struct StackFrame
	{
		Boolean set;
		Boolean value;
	};


#if TIME_STARTUP

	static calledOnce;

	if (!calledOnce)
	{
		calledOnce = true;

		uae_u32 now = Platform::GetMilliseconds ();
		uae_u32 elapsed = now - gStart;

		char	buffer[200];

		sprintf (buffer, "startup = %ld milliseconds  "
						"instructions = %ld "
						elapsed, Emulator::GetInstructionCount ());

		Platform::ReportString (buffer);
	}

#endif

	return kExecuteROM;
}


/***********************************************************************
 *
 * FUNCTION:	SysHeadpatch::TimInit
 *
 * DESCRIPTION: TimInit is where the date is read from non-volatile
 *				memory into private Time Manager memory.  After that,
 *				the Time Manager uses its private copy to update the
 *				non-volatile copy.	Thus, stashing correct values of
 *				the date into non-volatile memory should be done just
 *				before TimInit.
 *
 * PARAMETERS:	none
 *
 * RETURNED:	nothing
 *
 ***********************************************************************/

CallROMType SysHeadpatch::TimInit (void)
{
	// Err TimInit(void)

	struct StackFrame
	{
	};

	::PrvSetCurrentDate ();

	return kExecuteROM;
}


#pragma mark -

// ===========================================================================
//		 SysTailpatch
// ===========================================================================

void SysTailpatch::DmGetNextDatabaseByTypeCreator (void)
{
	static int	recursing;

	if (recursing)
		return;

	struct StackFrame
	{
		Boolean newSearch;
		DmSearchStatePtr stateInfoP;
		ULong type;
		ULong creator;
		Boolean onlyLatestVers;
		UIntPtr cardNoP;
		LocalID* dbIDP;
	};

//	Boolean newSearch		= GET_PARAMETER (newSearch);
	uaecptr stateInfoP		= GET_PARAMETER (stateInfoP);
	ULong	type			= GET_PARAMETER (type);
	ULong	creator 		= GET_PARAMETER (creator);
	Boolean onlyLatestVers	= GET_PARAMETER (onlyLatestVers);
	uaecptr cardNoP 		= GET_PARAMETER (cardNoP);
	uaecptr dbIDP			= GET_PARAMETER (dbIDP);
	Err 	result			= m68k_dreg (regs, 0);

	if (result == errNone && type == sysFileTExtension && creator == 0)
	{
		// Make sure we are called from within PrvLoadExtensions.

		// If so, get information on the returned cardNoP and dbIDP.

		UInt	cardNo	= get_word (cardNoP);
		LocalID dbID	= get_long (dbIDP);

		ULong	actualCreator;
		Err 	err = ::DmDatabaseInfo (cardNo, dbID, NULL, NULL, NULL, NULL,
							NULL, NULL, NULL, NULL, NULL, NULL, &actualCreator);

		// If the returned database has the creator of 'xxxx'
		// call DmGetNextDatabaseByTypeCreator again to skip
		// past this entry.

		if (err == errNone && actualCreator == 'xxxx')
		{
			recursing = true;

			err = ::DmGetNextDatabaseByTypeCreator (false,
						(DmSearchStatePtr) stateInfoP,
						type, creator, onlyLatestVers,
						(UIntPtr) cardNoP, (LocalID*) dbIDP);
			m68k_dreg (regs, 0) = (uae_u32) err;

			recursing = false;
		}
	}
}


/***********************************************************************
 *
 * FUNCTION:	SysTailpatch::EvtGetEvent
 *
 * DESCRIPTION: 
 *
 * PARAMETERS:	none
 *
 * RETURNED:	nothing
 *
 ***********************************************************************/

void SysTailpatch::EvtGetEvent (void)
{
	// void EvtGetEvent(const EventPtr event, SDWord timeout);

	struct StackFrame
	{
		const EventPtr event;
		SDWord timeout;
	};

	uaecptr event = GET_PARAMETER (event);
	SDWord	timeout = GET_PARAMETER (timeout);

	LogEvtGetEvent (event, timeout);

	gEvtGetEventCalled = true;
}


/***********************************************************************
 *
 * FUNCTION:	SysTailpatch::EvtGetPen
 *
 * DESCRIPTION: 
 *
 * PARAMETERS:	none
 *
 * RETURNED:	nothing
 *
 ***********************************************************************/

void SysTailpatch::EvtGetPen (void)
{
	// void EvtGetPen(SWord *pScreenX, SWord *pScreenY, Boolean *pPenDown)

	struct StackFrame
	{
		SWord *pScreenX;
		SWord *pScreenY;
		Boolean *pPenDown;
	};

	uaecptr pScreenX = GET_PARAMETER (pScreenX);
	uaecptr pScreenY = GET_PARAMETER (pScreenY);
	uaecptr pPenDown = GET_PARAMETER (pPenDown);

	LogEvtGetPen (pScreenX, pScreenY, pPenDown);
}


/***********************************************************************
 *
 * FUNCTION:	SysTailpatch::EvtGetSysEvent
 *
 * DESCRIPTION: 
 *
 * PARAMETERS:	none
 *
 * RETURNED:	nothing
 *
 ***********************************************************************/

void SysTailpatch::EvtGetSysEvent (void)
{
	// void EvtGetSysEvent(EventPtr eventP, Long timeout)


	struct StackFrame
	{
		EventPtr event;
		Long timeout;
	};

	uaecptr event = GET_PARAMETER (event);
	Long	timeout = GET_PARAMETER (timeout);

	LogEvtGetSysEvent (event, timeout);
}


/***********************************************************************
 *
 * FUNCTION:	SysTailpatch::FtrInit
 *
 * DESCRIPTION: 
 *
 * PARAMETERS:	none
 *
 * RETURNED:	nothing
 *
 ***********************************************************************/

void SysTailpatch::FtrInit (void)
{
	// void FtrInit(void)

	struct StackFrame
	{
	};

	// Get information about the current OS so that we know
	// what features are implemented (for those cases when we
	// can't make other tests, like above where we test for
	// the existance of a trap before calling it).

	// Read the version into a local variable; the ROM Stub facility
	// automatically maps local variables into Palm space so that ROM
	// functions can get to them.  If we were to pass &gOSVersion,
	// the DummyBank functions would complain about an invalid address.

	DWord	value;
	Err 	err = FtrGet (sysFtrCreator, sysFtrNumROMVersion, &value);

	if (err == errNone)
	{
		gOSVersion = value;
	}
	else
	{
		gOSVersion = kOSUndeterminedVersion;
	}

	err = FtrGet (sysFtrCreator, sysFtrNumEncoding, &value);
	gHaveJapanese = (err == errNone);
}


/***********************************************************************
 *
 * FUNCTION:	SysTailpatch::HwrMemReadable
 *
 * DESCRIPTION: Patch this function so that it returns non-zero if the
 *				address is in the range of memory that we've mapped in
 *				to emulated space.
 *
 * PARAMETERS:	none
 *
 * RETURNED:	nothing
 *
 ***********************************************************************/

void SysTailpatch::HwrMemReadable (void)
{
	// DWord HwrMemReadable(VoidPtr address)

	struct StackFrame
	{
		VoidPtr address;
	};

	uaecptr address = GET_PARAMETER (address);
	DWord	result = m68k_dreg (regs, 0);

	if (result == 0)
	{
		void*	addrStart;
		uae_u32 addrLen;

		Memory::GetMappingInfo ((void*) address, &addrStart, &addrLen);

		m68k_dreg (regs, 0) = addrLen;
	}
}


/***********************************************************************
 *
 * FUNCTION:	SysTailpatch::HwrSleep
 *
 * DESCRIPTION: 
 *
 * PARAMETERS:	none
 *
 * RETURNED:	nothing
 *
 ***********************************************************************/

void SysTailpatch::HwrSleep (void)
{
	// void HwrSleep(Boolean untilReset, Boolean emergency)

	struct StackFrame
	{
		Boolean untilReset;
		Boolean dummy;
		Boolean emergency;
	};

	MetaMemory::MarkLowMemory (offsetof (M68KExcTableType, busErr),
							   offsetof (M68KExcTableType, busErr) + sizeof (uaecptr));

	MetaMemory::MarkLowMemory (offsetof (M68KExcTableType, addressErr),
							   offsetof (M68KExcTableType, addressErr) + sizeof (uaecptr));

	MetaMemory::MarkLowMemory (offsetof (M68KExcTableType, illegalInstr),
							   offsetof (M68KExcTableType, illegalInstr) + sizeof (uaecptr));

	MetaMemory::MarkLowMemory (offsetof (M68KExcTableType, autoVec1),
							   offsetof (M68KExcTableType, trapN[0]));
}


/***********************************************************************
 *
 * FUNCTION:	SysTailpatch::SysAppStartup
 *
 * DESCRIPTION: 
 *
 * PARAMETERS:	none
 *
 * RETURNED:	nothing
 *
 ***********************************************************************/

void SysTailpatch::SysAppStartup (void)
{
	// Err SysAppStartup(SysAppInfoPtr* appInfoPP, Ptr* prevGlobalsP, Ptr* globalsPtrP)

	struct StackFrame
	{
		SysAppInfoPtr* appInfoPP;
		Ptr* prevGlobalsP;
		Ptr* globalsPtrP;
	};

	if (m68k_dreg (regs, 0) != errNone)
		return;

	uaecptr appInfoPP = GET_PARAMETER (appInfoPP);
	if (!appInfoPP)
		return;

	uaecptr appInfoP = get_long (appInfoPP);
	if (!appInfoP)
		return;

	Int cmd = get_word (appInfoP + offsetof (SysAppInfoType, cmd));

	EmuAppInfo	newAppInfo = Patches::CollectCurrentAppInfo (appInfoP);
	newAppInfo.fCmd = cmd;
	gCurAppInfo.push_back (newAppInfo);

	if (cmd == sysAppLaunchCmdNormalLaunch)
	{
		if (Patches::GetQuitStage () == kWaitingForSysAppStartup)
		{
			Patches::SetQuitStage (kWaitingForSysAppExit);
		}

		// Clear the flags that tell us which warnings we've already issued.
		// We reset these flags any time a new application is launched.
		//
		// !!! Put these flags into EmuAppInfo?  That way, we can keep
		// separate sets for sub-launched applications.

		Errors::ClearWarningFlags ();
	}
}


/***********************************************************************
 *
 * FUNCTION:	SysTailpatch::SysBinarySearch
 *
 * DESCRIPTION: There's a bug in pre-3.0 versions of SysBinarySearch
 *				that cause it to call the user callback function with a
 *				pointer just past the array to search.	Make a note of
 *				when we enter SysBinarySearch so that we can make
 *				allowances for that in MetaMemory's memory checking.
 *
 * PARAMETERS:	none
 *
 * RETURNED:	nothing
 *
 ***********************************************************************/

void SysTailpatch::SysBinarySearch (void)
{
	Patches::ExitSysBinarySearch ();
}


/***********************************************************************
 *
 * FUNCTION:	SysTailpatch::TimInit
 *
 * DESCRIPTION: TimInit is where a special boolean is set to trigger a
 *				bunch of RTC bug workarounds in the ROM (that is, there
 *				are RTC bugs in the Dragonball that the ROM needs to
 *				workaround).  We're not emulating those bugs, so turn
 *				off that boolean.  Otherwise, we'd have to add support
 *				for the 1-second, 1-minute, and 1-day RTC interrupts in
 *				order to get those workarounds to work.
 *
 * PARAMETERS:	none
 *
 * RETURNED:	nothing
 *
 ***********************************************************************/

void SysTailpatch::TimInit (void)
{
	// Err TimInit(void)

	struct StackFrame
	{
	};

	// Turn off the RTC bug workaround flag.

	uaecptr timGlobalsP = LowMem_GetGlobal (timGlobalsP);

	if (get_byte (timGlobalsP + 8) == 0x01)
	{
		put_byte (timGlobalsP + 8, 0x00);
	}
}


/***********************************************************************
 *
 * FUNCTION:	SysTailpatch::UIInitialize
 *
 * DESCRIPTION: 
 *
 * PARAMETERS:	none
 *
 * RETURNED:	nothing
 *
 ***********************************************************************/

void SysTailpatch::UIInitialize (void)
{
	// Void UIInitialize (void)

	struct StackFrame
	{
	};

	Patches::SetUIInitialized (true);

	// Prevent the device from going to sleep.

	SysSetAutoOffTime (0);

	// Can't call PrefSetPreference on 1.0 devices....

	if (LowMem::TrapExists (sysTrapPrefSetPreference))
	{
		PrefSetPreference (prefAutoOffDuration, 0);
	}

	// Install a 'pose' feature so that people can tell if they're running
	// under the emulator or not.  We *had* added a HostControl function
	// for this, but people can call that only under Poser or under 3.0
	// and later ROMs.	Which means that there's no way to tell under 2.0
	// and earlier ROMs when running on an actual device.
	//
	// Note that we do this here instead of in a tailpatch to FtrInit because
	// of a goofy inter-dependency.  FtrInit is called early in the boot process
	// before a valid stack has been allocated and switched to. During this time,
	// the ROM is using a range of memory that happens to reside in a free
	// memory chunk.  Routines in MetaMemory know about this and allow the use
	// of this unallocate range of memory.	However, in order to work, they need
	// to know the current stack pointer value.  If we were to call FtrSet from
	// the FtrInit tailpatch, the stack pointer will point to the stack set up
	// by the ATrap object, not the faux stack in use at the time of FtrInit.
	// Thus, the MetaMemory routines get confused and end up getting in a state
	// where accesses to the temporary OS stack are flagged as invalid.  Hence,
	// we defer adding these Features until much later in the boot process (here).

	FtrSet ('pose', 0, 0);

	// If we're listening on a socket, install the 'gdbS' feature.	The
	// existance of this feature causes programs written with the prc tools
	// to enter the debugger when they're launched.

	if (Debug::ConnectedToTCPDebugger ())
	{
		FtrSet ('gdbS', 0, 0x12BEEF34);
	}

	// Install the HotSync user-name.

	Preference<string>	userName (kPrefKeyUserName);
	::SetHotSyncUserName (userName->c_str ());

	// Auto-load any files in the Autoload[Foo] directories.

	::PrvAutoload ();
}


#pragma mark -

/***********************************************************************
 *
 * FUNCTION:	HtalLibHeadpatch::HtalLibSendReply
 *
 * DESCRIPTION: Ohhh...I'm going to Programmer Hell for this one...
 *				We call DlkDispatchRequest to install the user name in
 *				our UIInitialize patch.  DlkDispatchRequest will
 *				eventually call HtalLibSendReply to return a result
 *				code.  Well, we haven't fired up the Htal library, and
 *				wouldn't want it to send a response even if we had.
 *				Therefore, I'll subvert the whole process by setting
 *				the HTAL library refNum passed in to the Desktop Link
 *				Manager to an invalid value.  I'll look for this
 *				value in the SysTrap handling code and no-op the call
 *				by calling this stub.
 *
 * PARAMETERS:	none
 *
 * RETURNED:	nothing
 *
 ***********************************************************************/

CallROMType HtalLibHeadpatch::HtalLibSendReply (void)
{
	m68k_dreg (regs, 0) = errNone;

	return kSkipROM;
}


#pragma mark -

/***********************************************************************
 *
 * FUNCTION:	PrvToString
 *
 * DESCRIPTION: 
 *
 * PARAMETERS:	none
 *
 * RETURNED:	nothing
 *
 ***********************************************************************/

string PrvToString (uaecptr s)
{
	string	result;

	size_t	sLen = uae_strlen (s);
	if (sLen > 0)
	{
		result.resize (sLen);
		uae_strcpy (&result[0], s);
	}

	return result;
}


/***********************************************************************
 *
 * FUNCTION:	PrvAutoload
 *
 * DESCRIPTION: Install the files in the various Autoload directories.
 *				If there are any designated to be run, pick on to run.
 *				If the emulator should exit when the picked application
 *				quits, schedule it to do so.
 *
 * PARAMETERS:	none
 *
 * RETURNED:	nothing
 *
 ***********************************************************************/

void PrvAutoload (void)
{
	// Load all the files in one blow.

	FileRefList	fileList;
	Startup::GetAutoLoads (fileList);

	::LoadPalmFileList (fileList);

	// Get the application to switch to (if any);

	string	appToRun = Startup::GetAutoRunApp ();

	if (!appToRun.empty())
	{
		char	name[dmDBNameLength];
		strcpy (name, appToRun.c_str());

		UInt		cardNo = 0;
		LocalID	dbID = ::DmFindDatabase (cardNo, name);

		if (dbID != 0)
		{
			Patches::SetSwitchApp (cardNo, dbID);

			// If we're supposed to quit after running this application,
			// start that process in motion.

			if (Startup::QuitOnExit ())
			{
				Patches::QuitOnAppExit (true);
			}
		}
	}
}


/***********************************************************************
 *
 * FUNCTION:	PrvSetCurrentDate
 *
 * DESCRIPTION: .
 *
 * PARAMETERS:	None.
 *
 * RETURNED:	Nothing.
 *
 ***********************************************************************/

void PrvSetCurrentDate (void)
{
	// Get the current date.

	long	year, month, day;
	::GetHostDate (&year, &month, &day);

	// Get the current non-volatile variables.

	SysNVParamsType params[100];	// Allocate much more memory than we need
									// in case the ROM's notion of SysNVParamsType
									// is different from ours.
	::MemNVParams (false, &params[0]);

	// Update the "hours adjustment" value to contain the current date.

	params[0].rtcHours = ::DateToDays (year, month, day) * 24;

	// Re-write the non-volatile variables.

	::MemNVParams (true, &params[0]);
}


#include "PalmPackPop.h"
