/* -*- mode: C++; tab-width: 4 -*- */
/* ================================================================================== */
/* Copyright (c) 1998-1999 3Com Corporation or its subsidiaries. All rights reserved. */
/* ================================================================================== */

#include "EmulatorCommon.h"
#include "PalmHeap.h"

#include "CPU_REG.h"			// Emulator::HandleDlgButton
#include "ErrorHandling.h"		// Errors::ReportCorruptedHeap
#include "RAM_ROM.h"			// CEnableFullAccess


// ===========================================================================
//		 PalmHeap
// ===========================================================================

UInt					gNumHeaps;
static vector<uaecptr>	gHeapPtrs;

const UInt				kUnknownNumHeaps = (UInt) -1;


/***********************************************************************
 *
 * FUNCTION:	PalmHeap::Initialize
 *
 * DESCRIPTION:	Standard initialization function.  Responsible for
 *				initializing this sub-system when a new session is
 *				created.  May also be called from the Load function
 *				to share common functionality.
 *
 * PARAMETERS:	nothing
 *
 * RETURNED:	nothing
 *
 ***********************************************************************/

void PalmHeap::Initialize (void)
{
}


/***********************************************************************
 *
 * FUNCTION:	PalmHeap::Reset
 *
 * DESCRIPTION:	Re-initializes the PalmHeap sub-system.  This function
 *				is called when the ROM is "rebooted" (which includes
 *				the initial boot-up after a ROM is loaded).
 *
 * PARAMETERS:	nothing
 *
 * RETURNED:	nothing
 *
 ***********************************************************************/

void PalmHeap::Reset (void)
{
	gNumHeaps = kUnknownNumHeaps;
	gHeapPtrs.clear ();
}


/***********************************************************************
 *
 * FUNCTION:	PalmHeap::Save
 *
 * DESCRIPTION:	Standard save function.  Saves any sub-system state to
 *				the given session file.
 *
 * PARAMETERS:	nothing
 *
 * RETURNED:	nothing
 *
 ***********************************************************************/

void PalmHeap::Save (SessionFile&)
{
}


/***********************************************************************
 *
 * FUNCTION:	PalmHeap::Load
 *
 * DESCRIPTION:	Standard load function.  Loads any sub-system state
 *				from the given session file.
 *
 * PARAMETERS:	nothing
 *
 * RETURNED:	nothing
 *
 ***********************************************************************/

void PalmHeap::Load (SessionFile&)
{
}


/***********************************************************************
 *
 * FUNCTION:	PalmHeap::Dispose
 *
 * DESCRIPTION:	Standard dispose function.  Completely release any
 *				resources acquired or allocated in Initialize and/or
 *				Load.
 *
 * PARAMETERS:	nothing
 *
 * RETURNED:	nothing
 *
 ***********************************************************************/

void PalmHeap::Dispose (void)
{
	gHeapPtrs.clear ();
}


/***********************************************************************
 *
 * FUNCTION:	PalmHeap::GetHeapInfo
 *
 * DESCRIPTION:	Returns information about the given heap.
 *
 * PARAMETERS:	heapPtr - a pointer to the base of the heap to get
 *					information on.  The pointer points to the Palm OS
 *					heap header (that is, the Mem1HeapHeaderType,
 *					Mem2HeapHeaderType, or MemHeapHeaderType).
 *
 *				info - a pointer to a HeapInfo structure.  This
 *					structure will be filled with the returned
 *					information.
 *
 * RETURNED:	Information is returned in the HeapInfo structure.
 *
 ***********************************************************************/

void PalmHeap::GetHeapInfo (uaecptr heapPtr, HeapInfo* heapInfo)
{
	assert (heapInfo);

	CEnableFullAccess	munge;	// Remove blocks on memory access.


	// Save the pointer to the beginning of the heap.  This pointer
	// points to a struct, the definition of which changes depending
	// on the heap version.

	heapInfo->heapHdrStart	= heapPtr;


	// Get the heap version so we know the layout of the header.

	heapInfo->version = GetHeapVersion (heapInfo->heapHdrStart);


	/*
		Heap layout

			flags:					// Always here, always 2 bytes
			size					// 2 bytes in V1, 4 otherwise
			firstFreeChunkOffset	// Only in V3
			mstrPtrTbl
				numEntries			// always 2 bytes long
				nextTblOffset		// 2 bytes in V1, 4 otherwise
				mstrP [numEntries]
			...chunks...
	*/

	MPTInfo	mptInfo;

	switch (heapInfo->version)
	{
		case kHeapVersion1:
			heapInfo->flags			= get_word (heapInfo->heapHdrStart);
			heapInfo->size			= get_word (heapInfo->heapHdrStart + offsetof (Mem1HeapHeaderType, size));
			heapInfo->firstFree		= UAE_NULL;
			heapInfo->chunkHdrSize	= sizeof (Mem1ChunkHeaderType);

			GetMPTInfo (heapInfo->heapHdrStart + offsetof (Mem1HeapHeaderType, mstrPtrTbl), *heapInfo, &mptInfo);
			heapInfo->heapHdrSize	= sizeof (Mem1HeapHeaderType) - mptInfo.mptHdrSize;
			heapInfo->firstChunk	= heapInfo->heapHdrStart + heapInfo->heapHdrSize + mptInfo.mptHdrSize + mptInfo.numEntries * sizeof (Ptr);

			// In version 1 heaps, a zero in the size field means 64K.

			if (heapInfo->size == 0)
			{
				heapInfo->size = 64 * 1024L;
			}
			break;

		case kHeapVersion2:
			heapInfo->flags			= get_word (heapInfo->heapHdrStart);
			heapInfo->size			= get_long (heapInfo->heapHdrStart + offsetof (Mem2HeapHeaderType, size));
			heapInfo->firstFree		= UAE_NULL;
			heapInfo->chunkHdrSize	= sizeof (MemChunkHeaderType);

			GetMPTInfo (heapInfo->heapHdrStart + offsetof (Mem2HeapHeaderType, mstrPtrTbl), *heapInfo, &mptInfo);
			heapInfo->heapHdrSize	= sizeof (Mem2HeapHeaderType) - mptInfo.mptHdrSize;
			heapInfo->firstChunk	= heapInfo->heapHdrStart + heapInfo->heapHdrSize + mptInfo.mptHdrSize + mptInfo.numEntries * sizeof (Ptr);
			break;

		case kHeapVersion3:
		case kHeapVersion4:
			heapInfo->flags			= get_word (heapInfo->heapHdrStart);
			heapInfo->size			= get_long (heapInfo->heapHdrStart + offsetof (MemHeapHeaderType, size));
			heapInfo->firstFree		= get_long (heapInfo->heapHdrStart + offsetof (MemHeapHeaderType, firstFreeChunkOffset));
			heapInfo->chunkHdrSize	= sizeof (MemChunkHeaderType);

			GetMPTInfo (heapInfo->heapHdrStart + offsetof (MemHeapHeaderType, mstrPtrTbl), *heapInfo, &mptInfo);
			heapInfo->heapHdrSize	= sizeof (MemHeapHeaderType) - mptInfo.mptHdrSize;
			heapInfo->firstChunk	= heapInfo->heapHdrStart + heapInfo->heapHdrSize + mptInfo.mptHdrSize + mptInfo.numEntries * sizeof (Ptr);
			break;

		default:
			assert (false);
			memset (heapInfo, 0, sizeof (*heapInfo));
			break;
	}
}


/***********************************************************************
 *
 * FUNCTION:	PalmHeap::GetMPTInfo
 *
 * DESCRIPTION:	Returns information about the given master pointer table.
 *
 * PARAMETERS:	mptHdrStart - a pointer to the master pointer table to
 *					get information on (that is, the Mem1MstrPtrTableType
 *					or MemMstrPtrTableType).  This pointer can be
 *					obtained by adding HeapInfo::heapHdrSize to to
 *					HeapInfo::heapHdrStart (where HeapInfo is returned
 *					by GetHeapInfo).  Since there can be more than one
 *					master pointer table in a heap, subsequent pointers
 *					can be found by adding MPTInfo::nextTblOffset to
 *					HeapInfo::heapHdrStart (if MPTInfo::nextTblOffset is
 *					NULL, there is no next table).
 *
 *				info - a pointer to an MPTInfo structure.  This
 *					structure will be filled with the returned
 *					information.
 *
 * RETURNED:	Information is returned in the MPTInfo structure.
 *
 ***********************************************************************/

void PalmHeap::GetMPTInfo (uaecptr mptHdrStart, const HeapInfo& heapInfo, MPTInfo* mptInfo)
{
	assert (mptInfo);

	CEnableFullAccess	munge;	// Remove blocks on memory access.

	switch (heapInfo.version)
	{
		case kHeapVersion1:
			mptInfo->version		= kMPTVersion1;

			mptInfo->mptHdrStart	= mptHdrStart;
			mptInfo->mptHdrSize		= sizeof (Mem1MstrPtrTableType);

			mptInfo->numEntries		= get_word (mptHdrStart + offsetof (Mem1MstrPtrTableType, numEntries));
			mptInfo->nextTblOffset	= get_word (mptHdrStart + offsetof (Mem1MstrPtrTableType, nextTblOffset));
			break;

		case kHeapVersion2:
		case kHeapVersion3:
		case kHeapVersion4:
			mptInfo->version		= kMPTVersion2;

			mptInfo->mptHdrStart	= mptHdrStart;
			mptInfo->mptHdrSize		= sizeof (MemMstrPtrTableType);

			mptInfo->numEntries		= get_word (mptHdrStart + offsetof (MemMstrPtrTableType, numEntries));
			mptInfo->nextTblOffset	= get_long (mptHdrStart + offsetof (MemMstrPtrTableType, nextTblOffset));
			break;

		default:
			assert (false);
			memset (mptInfo, 0, sizeof (*mptInfo));
			break;
	}
}


/***********************************************************************
 *
 * FUNCTION:	PalmHeap::GetChunkInfo
 *
 * DESCRIPTION:	Returns information about the given chunk.
 *
 * PARAMETERS:	hdrP - a pointer to the header information about the
 *					chunk to return information on.
 *
 *				heapVersion - the heap version, as returned by
 *					GetHeapInfo (HeapInfo::version).
 *
 *				info - a pointer to a ChunkInfo structure.  This
 *					structure will be filled with the returned
 *					information.
 *
 * RETURNED:	Information is returned in the ChunkInfo structure.
 *
 ***********************************************************************/

void PalmHeap::GetChunkInfo (	uaecptr			hdrP,
								const HeapInfo& heapInfo,
								ChunkInfo*		chunkInfo)
{
	assert (chunkInfo);

	CEnableFullAccess	munge;	// Remove blocks on memory access.

	// Clear out all the fields in case we have to make a hasty retreat.

	memset (chunkInfo, 0, sizeof (*chunkInfo));

	switch (heapInfo.version)
	{
		case kHeapVersion1:
		{
			chunkInfo->version			= kChunkVersion1;
			chunkInfo->chunkHdrStart	= hdrP;
			chunkInfo->chunkHdrSize		= sizeof (Mem1ChunkHeaderType);

			Word	size				= get_word (hdrP + offsetof (Mem1ChunkHeaderType, size));

			if (size == 0)
			{
				// Leave now.  We're at the end of the heap.  If we're at the end of RAM,
				// too, attempts to access subsequent fields will cause a bus error.

				break;
			}

			Byte	lockOwner		= get_byte (hdrP + offsetof (Mem1ChunkHeaderType, lockOwner));
			Byte	flags			= get_byte (hdrP + offsetof (Mem1ChunkHeaderType, flags));
			SWord	hOffset			= get_word (hdrP + offsetof (Mem1ChunkHeaderType, hOffset));

			chunkInfo->free			= (flags & memChunkFlagFree) != 0;
			chunkInfo->moved		= (flags & memChunkFlagUnused1) != 0;
			chunkInfo->unused2		= (flags & memChunkFlagUnused2) != 0;
			chunkInfo->unused3		= (flags & memChunkFlagUnused3) != 0;
			chunkInfo->sizeAdj		= (flags & mem1ChunkFlagSizeAdj);
			chunkInfo->size			= size;
			chunkInfo->lockCount	= (lockOwner & mem1ChunkLockMask) >> 4;
			chunkInfo->owner		= (lockOwner & mem1ChunkOwnerMask);
			chunkInfo->hOffset		= hOffset;

			break;
		}

		case kHeapVersion2:
		case kHeapVersion3:
		case kHeapVersion4:
		{
			chunkInfo->version			= kChunkVersion1;
			chunkInfo->chunkHdrStart	= hdrP;
			chunkInfo->chunkHdrSize		= sizeof (MemChunkHeaderType);

			DWord	part1				= get_long (hdrP);

			if ((part1 & 0x00FFFFFF) == 0)
			{
				// Leave now.  We're at the end of the heap.  If we're at the end of RAM,
				// too, attempts to access subsequent fields will cause a bus error.

				break;
			}

			DWord	part2			= get_long (hdrP + 4);

			chunkInfo->free			= (part1 & 0x80000000) != 0;
			chunkInfo->moved		= (part1 & 0x40000000) != 0;
			chunkInfo->unused2		= (part1 & 0x20000000) != 0;
			chunkInfo->unused3		= (part1 & 0x10000000) != 0;
			chunkInfo->sizeAdj		= (part1 & 0x0F000000) >> 24;
			chunkInfo->size			= (part1 & 0x00FFFFFF);
			chunkInfo->lockCount	= (part2 & 0xF0000000) >> 28;
			chunkInfo->owner		= (part2 & 0x0F000000) >> 24;
			chunkInfo->hOffset		= (part2 & 0x00FFFFFF);

			// hOffset is a signed value; make sure we keep it that way
			// when we extract it by hand.

			if ((chunkInfo->hOffset & 0x00800000) != 0)
			{
				chunkInfo->hOffset |= 0xFF000000;
			}

			break;
		}

		default:
			assert (false);
	}
}


/***********************************************************************
 *
 * FUNCTION:	PalmHeap::GetHeapHeader
 *
 * DESCRIPTION:	Return the pointer to the heap header structure for
 *				the given heap.
 *
 * PARAMETERS:	heapID - the ID of the heap whose base pointer is to
 *					be returned.
 *
 * RETURNED:	The base pointer to the given heap.
 *
 ***********************************************************************/

uaecptr PalmHeap::GetHeapHeader (UInt heapID)
{
	if (gHeapPtrs.size () < heapID + 1)
		gHeapPtrs.resize (heapID + 1);

	if (gHeapPtrs[heapID] == UAE_NULL)
	{
		gHeapPtrs[heapID] = (uaecptr) MemHeapPtr (heapID);
	}

	return gHeapPtrs[heapID];
}


/***********************************************************************
 *
 * FUNCTION:	PalmHeap::FindHeapHeader
 *
 * DESCRIPTION:	Return the beginning of the heap containing the given
 *				test address.
 *
 * PARAMETERS:	testAddress - the address that should be contained
 *					by the returned heap.
 *
 * RETURNED:	The base pointer to the given heap.
 *
 ***********************************************************************/

uaecptr PalmHeap::FindHeapHeader (uaecptr testAddress)
{
	FindHeapData	data = { testAddress, UAE_NULL };

	WalkHeaps (FindHeapCallback, &data);

	return data.heapHeader;
}


/***********************************************************************
 *
 * FUNCTION:	PalmHeap::GetChunkHeader
 *
 * DESCRIPTION:	Given a pointer to the body of a memory chunk (that is,
 *				the user-accessible part of it), return a pointer to
 *				the chunk header.
 *
 *				This function does not assume anything about the
 *				pointer passed to it; it just subtracts a fixed amount
 *				from it.  This function *does* require a valid heap
 *				header, however, so that that fixed amount can be
 *				determined from the heap version number.
 *
 * PARAMETERS:	p - the pointer to the chunk body.
 *
 * RETURNED:	A pointer to the chunk header.
 *
 ***********************************************************************/

uaecptr PalmHeap::GetChunkHeader (VoidPtr p, const HeapInfo& heapInfo)
{
	if (!p)
		return UAE_NULL;

	uaecptr	chunkHdrPtr = ((uaecptr) p) - heapInfo.chunkHdrSize;

	return chunkHdrPtr;
}


/***********************************************************************
 *
 * FUNCTION:	PalmHeap::GetChunkHeader
 *
 * DESCRIPTION:	Given a handle to the body of a memory chunk (that is,
 *				the user-accessible part of it), return a pointer to
 *				the chunk header.
 *
 *				This function does not assume anything about the
 *				handle passed to it; it just dereferences it and passes
 *				the resulting VoidPtr to the other version of
 *				GetChunkHeader.
 *
 * PARAMETERS:	h - the handle to the chunk.
 *
 * RETURNED:	A pointer to the chunk header.
 *
 ***********************************************************************/

uaecptr PalmHeap::GetChunkHeader (VoidHand h, const HeapInfo& heapInfo)
{
	return GetChunkHeader (DerefHandle (h), heapInfo);
}


/***********************************************************************
 *
 * FUNCTION:	PalmHeap::FindChunkHeader
 *
 * DESCRIPTION:	Return information about the chunk containing the given
 *				test address.
 *
 * PARAMETERS:	testAddress - the address that should be contained by
 *					the returned chunk.
 *
 * RETURNED:	A ChunkInfo structure describing the chunk.  If the chunk
 *				could not be found, the ChunkInfo members are all zero.
 *
 ***********************************************************************/

uaecptr PalmHeap::FindChunkHeader (uaecptr testAddress, const HeapInfo& heapInfo)
{
	FindChunkData	data = { testAddress, UAE_NULL };

	WalkChunks (heapInfo, FindChunkCallback, &data);

	return data.chunkHeader;
}


/***********************************************************************
 *
 * FUNCTION:	PalmHeap::ValidateChunk
 *
 * DESCRIPTION:	Validate a chunk, ensuring that the data in the header
 *				appears to make sense.  Aspects that are checked:
 *
 *					- chunk is in the right heap.
 *					- chunk's size is not outrageous.
 *					- unused2 and unused3 flags are clear.
 *					- the hOffset field of an allocated, movable
 *					  chunk points to a pointer back to the chunk.
 *					- an allocated, non-movable chunk has the
 *					  appropriate lock-count.
 *
 * PARAMETERS:	heapInfo - a HeapInfo block on the heap that should
 *					contain the chunk.
 *
 *				chunkInfo - a ChunkInfo block containing the information
 *					to be validated.
 *
 *				validateHOffset - ensure that heapStart + hOffset is
 *					some sensible value before trying to use it.  Since
 *					this operation can be expensive (it involves making
 *					sure that the resulting pointer is in some master
 *					pointer table somewhere), it is optional.
 *
 * RETURNED:	An error code if an improper condition is detected.
 *
 ***********************************************************************/

PalmHeap::EPalmHeapResult PalmHeap::ValidateChunk (	const HeapInfo&		heapInfo,
													const ChunkInfo&	chunkInfo,
													Bool				validateHOffset)
{
	// Make sure the chunk is in the heap (this should always be true).
	// !!! It's not...determine why and document.

	if (chunkInfo.chunkHdrStart < heapInfo.heapHdrStart ||
		chunkInfo.chunkHdrStart > heapInfo.heapHdrStart + heapInfo.size)
	{
		return kCorruptedHeap_ChunkNotInHeap;
	}

	// Check the size.

	if (chunkInfo.chunkHdrStart + chunkInfo.size >
		heapInfo.heapHdrStart + heapInfo.size)
	{
		char	buffer[20];

		// !!! There's a problem here.  These variables are set before the
		// variable that uses them.  Thus, replacements based on these variables
		// are made before they're ready.  The result is that %chunk_size and
		// %chunk_max still appear in the final message.  I'm not sure how, but
		// this needs to be addressed and fixed...

		sprintf (buffer, "%0x08X", (int) chunkInfo.size);
		Errors::SetParameter ("%chunk_size", buffer);

		sprintf (buffer, "%0x08X", (int) heapInfo.size);
		Errors::SetParameter ("%chunk_max", buffer);

		return kCorruptedHeap_ChunkTooLarge;
	}

	if (chunkInfo.unused2 || chunkInfo.unused3)
	{
		return kCorruptedHeap_InvalidFlags;
	}

	if (!chunkInfo.free)
	{
		// If it's a movable chunk, validate that the handle offset
		// references the handle that points back to the chunk.

		if (chunkInfo.hOffset)
		{
			// Get the master pointer to this block.  If hOffset is bogus, then
			// this pointer may be bogus, too.

			uaecptr	h = chunkInfo.chunkHdrStart - chunkInfo.hOffset * 2;

			if (validateHOffset)
			{
				// Make sure it is in a master pointer block.

				// Get information about the first master pointer table.

				MPTInfo		mptInfo;
				GetMPTInfo (heapInfo.heapHdrStart + heapInfo.heapHdrSize, heapInfo, &mptInfo);

				// Loop over the MPT's.  Usually there's just one, but there could
				// be more (linked together in a single-linked list).

				while (1)
				{
					// Skip over the MPT header and get to the array of master pointers.

					uaecptr	mptArray = mptInfo.mptHdrStart + mptInfo.mptHdrSize;

					// See if "h" is in this array.  If so, break from the loop.

					if (h >= mptArray && h < mptArray + mptInfo.numEntries * sizeof (Ptr))
					{
						break;
					}

					// If there isn't another MPT, return with failure.

					if (mptInfo.nextTblOffset == 0)
					{
						return kCorruptedHeap_HOffsetNotInMPT;
					}

					// Get information on the next MPT.

					GetMPTInfo (heapInfo.heapHdrStart + mptInfo.nextTblOffset, heapInfo, &mptInfo);
				}
			}

			// "h" is in a master pointer table (or we've skipped the test). Make
			// sure "h" points back to the block it's supposed to.

			CEnableFullAccess	munge;	// Remove blocks on memory access.

			if (get_long (h) != chunkInfo.chunkHdrStart + chunkInfo.chunkHdrSize)
			{
				return kCorruptedHeap_HOffsetNotBackPointing;
			}
		}

		// If it's not a movable chunk, it must have a max lock count

		else
		{
			if (chunkInfo.lockCount != memPtrLockCount)
			{
				return kCorruptedHeap_InvalidLockCount;
			}
		}
	}

	// !!! Walk the heap and make sure that chunkHdrStart is found?

	return kChunkOK;
}


/***********************************************************************
 *
 * FUNCTION:	PalmHeap::ValidatePointer
 *
 * DESCRIPTION:	Ensure that the given pointer points to the body of
 *				a memory manager chunk.
 *
 * PARAMETERS:	p - the pointer to test.
 *
 * RETURNED:	An error code if an improper condition is detected.
 *
 ***********************************************************************/

PalmHeap::EPalmHeapResult PalmHeap::ValidatePointer (VoidPtr p, const HeapInfo& heapInfo)
{
	// See if the pointer is NULL or not.

	if (!p)
		return kNULLPointerOrHandle;

	// Get the chunk header for this chunk.

	uaecptr		chunkHdr = GetChunkHeader (p, heapInfo);
	ChunkInfo	chunkInfo;
	GetChunkInfo (chunkHdr, heapInfo, &chunkInfo);

	// Validate the chunk.

	PalmHeap::EPalmHeapResult	result = ValidateChunk (heapInfo, chunkInfo, true);

	return result;
}


/***********************************************************************
 *
 * FUNCTION:	PalmHeap::ValidateHandle
 *
 * DESCRIPTION:	Ensure that the given handle is indeed a handle and that
 *				it references a valid memory manager chunk.
 *
 * PARAMETERS:	h - the handle to test.
 *
 * RETURNED:	An error code if an improper condition is detected.
 *
 ***********************************************************************/

PalmHeap::EPalmHeapResult PalmHeap::ValidateHandle (VoidHand h, const HeapInfo& heapInfo)
{
	// See if the handle is NULL or not.

	if (!h)
		return kNULLPointerOrHandle;

	// Dereference the handle to get the pointer.
	// !!! Need to validate this operation.

	VoidPtr	p = DerefHandle (h);

	// Call the pointer-based version of this function to do the next
	// set of work.

	return ValidatePointer (p, heapInfo);
}


/***********************************************************************
 *
 * FUNCTION:	PalmHeap::ValidateHeaps
 *
 * DESCRIPTION:	.
 *
 * PARAMETERS:	.
 *
 * RETURNED:	.
 *
 ***********************************************************************/

void PalmHeap::ValidateHeaps (void)
{
	static Bool	validateInProgress;

	if (!validateInProgress)
	{
		validateInProgress = true;

		WalkHeaps (ValidateHeapCallback, NULL);

		validateInProgress = false;
	}
}


/***********************************************************************
 *
 * FUNCTION:	PalmHeap::WalkChunks
 *
 * DESCRIPTION:	"Walk" the heap.  That is, iterate over the chunks in
 *				the heap, calling a user-defined callback function for
 *				each one.
 *
 *				If an error occurs while iterating (usually because of
 *				some heap corruption), an error dialog is presented.
 *
 * PARAMETERS:	heapID - the Palm OS ID of the heap to iterate over.
 *
 *				fn - a pointer to the function to call for each chunk.
 *					This function is passed a HeapInfo block, a ChunkInfo
 *					block, and the userData value.  The function should
 *					return either kKeepIterating or kStopIterating.
 *
 * RETURNED:	The result of the iteration.  If a heap corruption is
 *				detected while iterating, an error value is returned.
 *				Otherwise, a value indicating whether or not the full
 *				heap was iterated is returned.
 *
 ***********************************************************************/

PalmHeap::EPalmHeapResult PalmHeap::WalkHeaps (HeapWalker fn, void* userData)
{
	EPalmHeapResult	result = kWalkHeapFinished;

	UInt	numHeaps = GetNumHeaps (0);

//	numHeaps = 1;

	for (UInt heapID = 0; heapID < numHeaps; ++heapID)
	{
		uaecptr	heapHdr = GetHeapHeader (heapID);

		if (heapHdr != UAE_NULL)
		{
			HeapInfo	heapInfo;
			GetHeapInfo (heapHdr, &heapInfo);

			// !!! Should validate the heap structures.

			if (fn (heapInfo, userData) == kStopIterating)
			{
				result = kWalkHeapAborted;
				break;
			}
		}
	}

	return result;
}


/***********************************************************************
 *
 * FUNCTION:	PalmHeap::WalkChunks
 *
 * DESCRIPTION:	"Walk" the heap.  That is, iterate over the chunks in
 *				the heap, calling a user-defined callback function for
 *				each one.
 *
 *				If an error occurs while iterating (usually because of
 *				some heap corruption), an error dialog is presented.
 *
 * PARAMETERS:	heapID - the Palm OS ID of the heap to iterate over.
 *
 *				fn - a pointer to the function to call for each chunk.
 *					This function is passed a HeapInfo block, a ChunkInfo
 *					block, and the userData value.  The function should
 *					return either kKeepIterating or kStopIterating.
 *
 * RETURNED:	The result of the iteration.  If a heap corruption is
 *				detected while iterating, an error value is returned.
 *				Otherwise, a value indicating whether or not the full
 *				heap was iterated is returned.
 *
 ***********************************************************************/

PalmHeap::EPalmHeapResult PalmHeap::WalkChunks (const HeapInfo& heapInfo, ChunkWalker fn, void* userData)
{
	EPalmHeapResult	result;

	uaecptr			prevChunkHdr = UAE_NULL;
	uaecptr			chunkHdr = heapInfo.firstChunk;
	ChunkInfo		chunkInfo;

	while (1)
	{
		// Get information about the current chunk.

		GetChunkInfo (chunkHdr, heapInfo, &chunkInfo);

		// If the size is zero, we've reached the sentinel at the end.

		if (!chunkInfo.size)
		{
			result = kWalkHeapFinished;
			break;
		}

		// See if this chunk looks valid.

		result = ValidateChunk (heapInfo, chunkInfo, true);
		if (result >= kWalkHeapError)
		{
			break;
		}

		// Call the callback proc for this chunk.  If it returns kStopIterating,
		// it's signalling that it wants us to stop iterating.

		if (fn (heapInfo, chunkInfo, userData) == kStopIterating)
		{
			result = kWalkHeapAborted;
			break;
		}

		// Go on to next chunk

		prevChunkHdr = chunkHdr;	// Keep this, for debugging.
		chunkHdr += chunkInfo.size;
	}

	// If an error occured, handle it now.

	if (result >= kWalkHeapError)
	{
		int	button = Errors::ReportCorruptedHeap (result, chunkHdr);

		Emulator::HandleDlgButton (button, m68k_getpc ());
	}

	return result;
}


/***********************************************************************
 *
 * FUNCTION:	PalmHeap::GetNumHeaps
 *
 * DESCRIPTION:	Return the number of heaps created by the emulated
 *				Palm OS.
 *
 * PARAMETERS:	cardNo - the ID of the "card".
 *
 * RETURNED:	The number of heaps.
 *
 ***********************************************************************/

UInt PalmHeap::GetNumHeaps (UInt cardNo)
{
	if (gNumHeaps == kUnknownNumHeaps)
		gNumHeaps = MemNumHeaps (cardNo);

	return gNumHeaps;
}


/***********************************************************************
 *
 * FUNCTION:	PalmHeap::GetHeapVersion
 *
 * DESCRIPTION:	Determine and return the version number of the given
 *				heap.
 *
 * PARAMETERS:	heapPtr - a pointer to the header for the heap whose
 *					version is to be determined.
 *
 * RETURNED:	The version number of the heap.  Currently, these values
 *				are 1 (original heap format), 2 (allows for >64K heaps),
 *				and 3 (uses a free list for faster allocations).
 *
 ***********************************************************************/

int PalmHeap::GetHeapVersion (uaecptr heapPtr)
{
	CEnableFullAccess	munge;	// Remove blocks on memory access.

	int heapVersion = kHeapVersionUnknown;

	// Get the heap version.  If bit 14 is set, it's a version 3 heap.
	// If bit 15 is set, it's a version 2 heap.  Otherwise, it's a
	// version 1 heap.

// This isn't in my version of MemMgrPrv.h, yet.
#define memHeapFlagVers4 0x2000

	uae_u16	flags = get_word (heapPtr);
	
	if ((flags & memHeapFlagVers4) != 0)
	{
		heapVersion = kHeapVersion4;	// free MP's on a list
	}
	else if ((flags & memHeapFlagVers3) != 0)
	{
		heapVersion = kHeapVersion3;	// has free list
	}
	else if ((flags & memHeapFlagVers2) != 0)
	{
		heapVersion = kHeapVersion2;	// > 64K
	}
	else
	{
		heapVersion = kHeapVersion1;	// original
	}

	return heapVersion;
}


/***********************************************************************
 *
 * FUNCTION:	PalmHeap::DerefHandle
 *
 * DESCRIPTION:	Dereferences a handle and returns the pointer
 *
 * PARAMETERS:	h - the handle to dereference
 *
 * RETURNED:	The pointer associated with the handle.
 *
 ***********************************************************************/

VoidPtr PalmHeap::DerefHandle (VoidHand h)
{
	if (!h)
		return NULL;

	CEnableFullAccess	munge;	// Remove blocks on memory access.

	h = memHandleUnProtect(h);
	return 	(VoidPtr) get_long ((uaecptr) h);
}


/***********************************************************************
 *
 * FUNCTION:	PalmHeap::FindHeapCallback
 *
 * DESCRIPTION:	Callback function passed to WalkHeap by FindHeapHeader.
 *
 * PARAMETERS:	heapInfo, userData - standard HeapWalker
 *					parameters.
 *
 * RETURNED:	A EWalkerProcResult indicating whether or not the
 *				iteration should continue.
 *
 ***********************************************************************/

PalmHeap::EWalkerProcResult PalmHeap::FindHeapCallback (	const HeapInfo& heapInfo,
															void* userData)
{
	FindHeapData*	data = (FindHeapData*) userData;

	// See if the test pointer is within the given heap.
	// If so, stop iterating and return information on that heap.

	if (data->testAddress >= heapInfo.heapHdrStart &&
		data->testAddress < heapInfo.heapHdrStart + heapInfo.size)
	{
		data->heapHeader = heapInfo.heapHdrStart;
		return kStopIterating;
	}

	return kKeepIterating;
}


/***********************************************************************
 *
 * FUNCTION:	PalmHeap::FindChunkCallback
 *
 * DESCRIPTION:	Callback function passed to WalkHeap by FindChunkHeader.
 *
 * PARAMETERS:	heapInfo, chunkInfo, userData - standard ChunkWalker
 *					parameters.
 *
 * RETURNED:	A EWalkerProcResult indicating whether or not the
 *				iteration should continue.
 *
 ***********************************************************************/

PalmHeap::EWalkerProcResult PalmHeap::FindChunkCallback (	const HeapInfo& heapInfo,
															const ChunkInfo& chunkInfo,
															void* userData)
{
	UNUSED_PARAM(heapInfo)

	FindChunkData*	data = (FindChunkData*) userData;

	// See if the test pointer is within the given chunk.
	// If so, stop iterating and return information on that chunk.

	uaecptr	bodyStart = chunkInfo.chunkHdrStart;	// + chunkInfo.chunkHdrSize;
	uaecptr	bodyEnd = chunkInfo.chunkHdrStart + chunkInfo.size;	// - chunkInfo.sizeAdj;

	if ((data->testAddress >= bodyStart) && (data->testAddress < bodyEnd))
	{
		data->chunkHeader = chunkInfo.chunkHdrStart;
		return kStopIterating;
	}

	return kKeepIterating;
}


/***********************************************************************
 *
 * FUNCTION:	PalmHeap::ValidateHeapCallback
 *
 * DESCRIPTION:	Callback function passed to WalkHeap by ValidateHeaps.
 *
 * PARAMETERS:	heapInfo, userData - standard HeapWalker
 *					parameters.
 *
 * RETURNED:	A EWalkerProcResult indicating whether or not the
 *				iteration should continue.
 *
 ***********************************************************************/

PalmHeap::EWalkerProcResult PalmHeap::ValidateHeapCallback (	const HeapInfo& heapInfo,
																void* userData)
{
	UNUSED_PARAM(userData)

	if (WalkChunks (heapInfo, ValidateChunkCallback, NULL) != kWalkHeapFinished)
		return kStopIterating;

	return kKeepIterating;
}


/***********************************************************************
 *
 * FUNCTION:	PalmHeap::ValidateChunkCallback
 *
 * DESCRIPTION:	Callback function passed to WalkHeap by ValidateHeaps.
 *
 * PARAMETERS:	heapInfo, chunkInfo, userData - standard ChunkWalker
 *					parameters.
 *
 * RETURNED:	A EWalkerProcResult indicating whether or not the
 *				iteration should continue.
 *
 ***********************************************************************/

PalmHeap::EWalkerProcResult PalmHeap::ValidateChunkCallback (	const HeapInfo& heapInfo,
																const ChunkInfo& chunkInfo,
																void* userData)
{
	UNUSED_PARAM(userData)

	EPalmHeapResult	result = ValidateChunk (heapInfo, chunkInfo, true);

	if (result >= kWalkHeapError)
	{
		return kStopIterating;
	}

	return kKeepIterating;
}
